#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <malloc.h>
#include <stdlib.h>
#include <stdarg.h>

#include "structure.h"

struct namelist {
    char *name;
    struct namelist *next;
};

struct information {
    char *name;
    struct namelist *childnames;
    struct information *next;
};

union property {
    int i;
    float f;
};

#define PROPERTY_INT 0
#define PROPERTY_FLOAT 1

struct propertylist {
    int type;
    char *name;
    union property value;
    struct propertylist *next;
};

struct detectors {
    struct information *info;
    struct detectors *parent;
    struct detectors *children;
    struct propertylist *properties;
    struct detectors *next;
};

struct alllist {
    struct detectors *dp;
    struct alllist *next;
};

static struct information *infohead = NULL;
static struct alllist *allhead = NULL;

static int loaddetectors(char *);
static void loaddetector(FILE *, char *);
static struct detectors *instance(char *, struct detectors *);

static void destroyall(void)
{
    struct alllist *ap = allhead, *atmp;
    struct information *ip = infohead, *itmp;
    struct namelist *np, *ntmp;

    while(ap != NULL)
    {
	free(ap->dp);
	atmp = ap->next;
	free(ap);
	ap = atmp;
    }

    allhead = NULL;

    while (ip != NULL)
    {
	free(ip->name);
	for(np = ip->childnames; np != NULL; )
	{
	    free(np->name);
	    ntmp = np->next;
	    free(np);
	    np = ntmp;
	}
	itmp = ip->next;
	free(ip);
	ip = itmp;
    }

    infohead = NULL;
}

struct detectors *loadexperiment(char *filename)
{
    if (loaddetectors(filename) < 0)
    {
	fprintf(stderr, "Unable to load experiment file.\n");
	return(NULL);
    }

    return(instance("experiment", NULL));
}

#define TOKENBUF_LEN 40
#define TOKEN_STRING 256
#define TOKEN_INTEGER 257

static int token, line;
static char tokenbuf[TOKENBUF_LEN], *currentfile;

static int gettoken(FILE *fp)
{
    int c;
    char *p = tokenbuf;

    while(isspace(c = getc(fp)))
	if (c == '\n')
	    line++;
    if (isalpha(c) || c == '_')
    {
	do {
	    if (p - tokenbuf < TOKENBUF_LEN-1)
		*p++ = tolower(c);
	    c = getc(fp);
	} while(isalnum(c) || c == '_');
	ungetc(c, fp);
	*p++ = '\0';
	return(TOKEN_STRING);
    }
    if (isdigit(c))
    {
	do {
	    if (p - tokenbuf < TOKENBUF_LEN-1)
		*p++ = c;
	    c = getc(fp);
	} while(isdigit(c));
	ungetc(c, fp);
	*p++ = '\0';
	return(TOKEN_INTEGER);
    }
    return(c);
}

static void unexpectedtoken()
{
    if (token == EOF)
	fprintf(stderr, "Unexpected end of file");
    else if (token < 256)
    {
	if (!isprint(token))
	    fprintf(stderr, "Unexpected character %d", token);
	else
	    fprintf(stderr, "Unexpected token \"%c\"", token);
    }
    else
	fprintf(stderr, "Unexpected token \"%s\"", tokenbuf);
    fprintf(stderr, " at line %d of file %s\n", line, currentfile);
}

static int loaddetectors(char *filename)
{
    FILE *fp;

    currentfile = filename;
    line = 1;

    destroyall();

    if ((fp = fopen(filename, "r")) == NULL)
    {
	perror(filename);
	return(-1);
    }

    token = gettoken(fp);
    while(token != EOF)
    {
	switch(token)
	{
	case TOKEN_STRING:
	    if (!strcmp(tokenbuf, "detector"))
		loaddetector(fp, NULL);
	    else if (!strcmp(tokenbuf, "experiment"))
		loaddetector(fp, "experiment");
	    else
	    {
		unexpectedtoken();
		token = EOF;
	    }
	    break;
	default:
	    unexpectedtoken();
	    token = EOF;
	    break;
	}
    }

    fclose(fp);
    return(0);
}

static void loaddetector(FILE *fp, char *name)
{
    struct information *ip;
    struct namelist *nametail;
    int num;

    if (name == NULL)
    {
	token = gettoken(fp);

	if (token != TOKEN_STRING)
	{
	    unexpectedtoken();
	    return;
	}
	name = tokenbuf;
    }

    if ((ip = (struct information *) malloc(sizeof(struct information)))
	== NULL)
    {
	fprintf(stderr, "Out of memory at line %d of file %s.\n", line,
		currentfile);
	return;
    }

    ip->name = strdup(name);
    ip->childnames = NULL;
    ip->next = infohead;
    infohead = ip;

    token = gettoken(fp);

    while(token == TOKEN_STRING)
    {
	if (!strcmp(tokenbuf, "comprises"))
	{
	    token = gettoken(fp);
	    if (token == TOKEN_INTEGER)
	    {
		num = atoi(tokenbuf);
		token = gettoken(fp);
		if (token != '*')
		{
		    unexpectedtoken();
		    return;
		}
		token = gettoken(fp);
	    }
	    else
		num = 1;
	    if (token != TOKEN_STRING)
	    {
		unexpectedtoken();
		return;
	    }
	    while(num--)
	    {
		if (ip->childnames == NULL)
		{
		    ip->childnames = nametail = (struct namelist *)
			malloc(sizeof(struct namelist));
		}
		else
		{
		    nametail->next = (struct namelist *)
			malloc(sizeof(struct namelist));
		    nametail = nametail->next;
		}
		nametail->name = strdup(tokenbuf);
		nametail->next = NULL;
	    }
	    token = gettoken(fp);
	}
	else if (!strcmp(tokenbuf, "detector"))
	    return;
	else if (!strcmp(tokenbuf, "experiment"))
	    return;
	else
	{
	    unexpectedtoken();
	    return;
	}
    }

    if (token != EOF)
	unexpectedtoken();
}

static struct detectors *instance(char *name, struct detectors *parent)
{
    struct detectors *dp, *child;
    struct information *ip;
    struct namelist *cname;
    struct alllist *ap;

    for(ip = infohead; ip != NULL; ip = ip->next)
	if (!strcasecmp(ip->name, name))
	    break;
    
    if (ip == NULL)
    {
	fprintf(stderr, "Detector \"%s\" is not defined.\n", name);
	destroyall();
	return(NULL);
    }

    if ((ap = (struct alllist *) malloc(sizeof(struct alllist))) == NULL)
    {
	destroyall();
	return(NULL);
    }

    if ((dp = (struct detectors *) malloc(sizeof(struct detectors))) == NULL)
    {
	fprintf(stderr, "Out of memory trying to build detector structure.\n");
	free(ap);
	destroyall();
	return(NULL);
    }

    ap->dp = dp;
    ap->next = allhead;
    allhead = ap;

    dp->info = ip;
    dp->parent = parent;
    dp->children = NULL;
    dp->properties = NULL;
    dp->next = NULL;

    for(cname = ip->childnames; cname != NULL; cname = cname->next)
    {
	if (dp->children == NULL)
	{
	    if ((dp->children = child = instance(cname->name, dp)) == NULL)
		return(NULL);
	}
	else
	{
	    if ((child->next = instance(cname->name, dp)) == NULL)
		return(NULL);
	    child = child->next;
	}
    }
    return(dp);
}

static struct detectors *vgetchild(struct detectors *dp, char *reference,
				   int ptrs, va_list ap)
{
    int len, n;
    struct detectors *child;

    if (dp == NULL)
	return(NULL);

    while(*reference != '\0')
    {
	if (*reference == '%')
	{
	    if (ptrs)
		n = *(va_arg(ap, int *));
	    else
		n = va_arg(ap, int);
	    for(child = dp->children; child != NULL && --n;
		child = child->next)
		;
	    if (child == NULL)
		return(NULL);
	    reference++;
	}
	else
	{
	    for(child = dp->children; child != NULL; child = child->next)
	    {
		if (!strncasecmp(reference, child->info->name,
				 len = strlen(child->info->name))
		    && (reference[len] == '.' || reference[len] == '\0'))
		    break;
	    }
	    if (child == NULL)
		return(NULL);
	    reference += len;
	}
	dp = child;
	if (*reference == '.')
	    reference++;
    }

    return(dp);
}

struct detectors *getchild(struct detectors *dp, char *reference, ...)
{
    struct detectors *child;

    va_list ap;
    va_start(ap, reference);

    child = vgetchild(dp, reference, 0, ap);

    va_end(ap);

    return(child);
}

struct detectors *getparent(struct detectors *dp)
{
    return(dp == NULL ? NULL : dp->parent);
}

char *getname(struct detectors *dp)
{
    return(dp == NULL ? NULL : dp->info->name);
}

struct detectors *getnext(struct detectors *dp)
{
    return(dp == NULL ? NULL : dp->next);
}

static struct propertylist *getp(struct detectors *dp,
				 char *name, int create)
{
    struct propertylist *pp, **plast;

    plast = &(dp->properties);
    pp = *plast;

    for( ; pp != NULL; pp = pp->next)
    {
	if (!strcasecmp(name, pp->name))
	    break;
	plast = &(pp->next);
    }
    if (pp == NULL && create)
    {
	if ((pp = *plast = (struct propertylist *)
	     malloc(sizeof(struct propertylist))) == NULL)
	{
	    fprintf(stderr, "Out of memory trying to add property %s.\n",
		    name);
	    return(NULL);
	}
	pp->name = strdup(name);
	pp->type = PROPERTY_INT;
	pp->value.i = 0;
	pp->next = NULL;
    }

    return(pp);
}

void iset(char *name, int value, struct detectors *dp, char *reference, ...)
{
    struct detectors *child;
    struct propertylist *pp;

    va_list ap;
    va_start(ap, reference);

    child = vgetchild(dp, reference, 0, ap);

    va_end(ap);

    if (child != NULL)
    {
	if ((pp = getp(child, name, 1)) != NULL)
	{
	    pp->type = PROPERTY_INT;
	    pp->value.i = value;
	}
    }
}

void isetthis(char *name, int value, struct detectors *dp)
{
    struct propertylist *pp;

    if (dp != NULL)
    {
	if ((pp = getp(dp, name, 1)) != NULL)
	{
	    pp->type = PROPERTY_INT;
	    pp->value.i = value;
	}
    }
}

void fset(char *name, double value, struct detectors *dp, char *reference, ...)
{
    struct detectors *child;
    struct propertylist *pp;

    va_list ap;
    va_start(ap, reference);

    child = vgetchild(dp, reference, 0, ap);

    va_end(ap);

    if (child != NULL)
    {
	if ((pp = getp(child, name, 1)) != NULL)
	{
	    pp->type = PROPERTY_FLOAT;
	    pp->value.f = value;
	}
    }
}

void fsetthis(char *name, double value, struct detectors *dp)
{
    struct propertylist *pp;

    if (dp != NULL)
    {
	if ((pp = getp(dp, name, 1)) != NULL)
	{
	    pp->type = PROPERTY_FLOAT;
	    pp->value.f = value;
	}
    }
}

int hasproperty(char *name, struct detectors *dp, char *reference, ...)
{
    struct detectors *child;
    struct propertylist *pp;

    va_list ap;
    va_start(ap, reference);

    child = vgetchild(dp, reference, 0, ap);

    va_end(ap);

    if (child != NULL)
	if ((pp = getp(child, name, 0)) != NULL)
	    return(1);
    return(0);
}

int iget(char *name, struct detectors *dp, char *reference, ...)
{
    struct detectors *child;
    struct propertylist *pp;

    va_list ap;
    va_start(ap, reference);

    child = vgetchild(dp, reference, 0, ap);

    va_end(ap);

    if (child != NULL)
	if ((pp = getp(child, name, 0)) != NULL)
	    return(pp->value.i);
    return(-1);
}

int igetthis(char *name, struct detectors *dp)
{
    struct propertylist *pp;
    
    if (dp != NULL)
	if ((pp = getp(dp, name, 0)) != NULL)
	    return(pp->value.i);
    return(-1);
}

float fget(char *name, struct detectors *dp, char *reference, ...)
{
    struct detectors *child;
    struct propertylist *pp;

    va_list ap;
    va_start(ap, reference);

    child = vgetchild(dp, reference, 0, ap);

    va_end(ap);

    if (child != NULL)
	if ((pp = getp(child, name, 0)) != NULL)
	    return(pp->value.f);
    return(0);
}

float fgetthis(char *name, struct detectors *dp)
{
    struct propertylist *pp;

    if (dp != NULL)
	if ((pp = getp(dp, name, 0)) != NULL)
	    return(pp->value.f);
    return(0);
}

void dumpdetectors(struct detectors *dp, int depth)
{
    int i;
    struct propertylist *pp;

    if (dp == NULL)
    {
	printf("Null pointer\n");
	return;
    }

    for(i=depth; i--; )
	printf("  ");
    printf("Name    : %s\n", dp->info->name);
    for(i=depth; i--; )
	printf("  ");
    printf("Address : %08x\n", (unsigned int) dp);
    for(i=depth; i--; )
	printf("  ");
    printf("Parent  : (%08x) %s\n", (unsigned int) dp->parent,
	   (dp->parent == NULL) ? "none" : dp->parent->info->name);
    for(i=depth; i--; )
	printf("  ");
    printf("Properties:\n");
    for(pp = dp->properties; pp != NULL; pp = pp->next)
    {
	for(i=depth+1; i--; )
	    printf("  ");
	printf("%s = ", pp->name);
	switch(pp->type)
	{
	case PROPERTY_INT:
	    printf("(int) %d\n", pp->value.i);
	    break;
	case PROPERTY_FLOAT:
	    printf("(float) %f\n", pp->value.f);
	    break;
	default:
	    printf("(unknown) 0x%08x\n", pp->value.i);
	    break;
	}
    }
    for(i=depth; i--; )
	printf("  ");
    printf("Children:\n");
    for(dp = dp->children; dp != NULL; dp = dp->next)
	dumpdetectors(dp, depth+1);
}

#ifdef DEBUG

int main()
{
    struct information *ip;
    struct namelist *np;
    struct detectors *dp;

    printf("Loading structure\n");
    dp = loadexperiment("testexpt.det");

    printf("Getting child pssd.strip\n");
    dumpdetectors(getchild(dp, "pssd.strip"), 0);
    printf("Setting property foo = 57.\n");
    iset("foo", 57, dp, "pssd.strip");

    printf("Getting child 2.3\n");
    dumpdetectors(getchild(dp, "%.%", 2, 3), 0);
    printf("Setting property bar = 123.21.\n");
    fset("bar", 123.21, dp, "%.%", 2, 3);

    printf("property foo of pssd.strip = %d\n",
	   iget("foo", dp, "pssd.strip"));
    printf("property bar of 2.3 = %f\n",
	   fget("bar", dp, "%.%", 2,3));

    printf("Dumping information list\n");
    for(ip = infohead; ip != NULL; ip = ip->next)
    {
	printf("Name    : %s\n", ip->name);
	printf("Children:");
	for(np = ip->childnames; np != NULL; np=np->next)
	    printf(" %s", np->name);
	printf("\n\n");
    }

    printf("Dumping experiment\n");
    dumpdetectors(dp, 0);

    return(0);
}
#endif

/* FORTRAN interface routines */

struct detectors *loadexperiment_(char *filename)
{
    return(loadexperiment(filename));
}

void dumpdetectors_(struct detectors **dp)
{
    dumpdetectors(*dp, 0);
}

void iset_(char *name, int value, struct detectors **dp, char *reference, ...)
{
    struct detectors *child;
    struct propertylist *pp;

    va_list ap;
    va_start(ap, reference);

    child = vgetchild(*dp, reference, 1, ap);

    va_end(ap);

    if (child != NULL)
    {
	if ((pp = getp(child, name, 1)) != NULL)
	{
	    pp->type = PROPERTY_INT;
	    pp->value.i = value;
	}
    }
}

void fset_(char *name, double value, struct detectors **dp, char *reference,
	   ...)
{
    struct detectors *child;
    struct propertylist *pp;

    va_list ap;
    va_start(ap, reference);

    child = vgetchild(*dp, reference, 1, ap);

    va_end(ap);

    if (child != NULL)
    {
	if ((pp = getp(child, name, 1)) != NULL)
	{
	    pp->type = PROPERTY_FLOAT;
	    pp->value.f = value;
	}
    }
}

int hasproperty_(char *name, struct detectors **dp, char *reference, ...)
{
    struct detectors *child;
    struct propertylist *pp;

    va_list ap;
    va_start(ap, reference);

    child = vgetchild(*dp, reference, 1, ap);

    va_end(ap);

    if (child != NULL)
	if ((pp = getp(child, name, 0)) != NULL)
	    return(1);
    return(0);
}

int iget_(char *name, struct detectors **dp, char *reference, ...)
{
    struct detectors *child;
    struct propertylist *pp;

    va_list ap;
    va_start(ap, reference);

    child = vgetchild(*dp, reference, 1, ap);

    va_end(ap);

    if (child != NULL)
	if ((pp = getp(child, name, 0)) != NULL)
	    return(pp->value.i);
    return(-1);
}

float fget_(char *name, struct detectors **dp, char *reference, ...)
{
    struct detectors *child;
    struct propertylist *pp;

    va_list ap;
    va_start(ap, reference);

    child = vgetchild(*dp, reference, 1, ap);

    va_end(ap);

    if (child != NULL)
	if ((pp = getp(child, name, 0)) != NULL)
	    return(pp->value.f);
    return(0);
}

