/*
 *  projecting routine for sunsort.
 *  using given 2d spectrum AND it with a 2d window spectrum
 *  to produce two 1d spectra containing the projection of the
 *  AND'ed spectrum on the X and Y axes.... assume square 2d spectrum.
 *  arguments -spectrum <2d data file> -window <2d window file>
 *
 *  last modified   18/1/95
 */

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <stdlib.h>
#include <memory.h>

#include <sys/param.h>
#include <sys/types.h>

#include <errno.h>

#include "sort_def.h"
#include "sort_mem.h"

/********************** global variables and symbols **************************/

int     DBX_val;

extern   void read_args();
extern   void project();
extern   int  write_slice();
extern   int  *resize_window();
#ifdef SVR4
extern   pid_t  lookup_proc();
#define NeedFunctionPrototype 1
#define PTR_t void *
#else
extern   int    lookup_proc();
#define PTR_t char *
#endif

/***************************** static data ***********************************/

static struct sm_data_buf    spec_data;
static char                  *spectrum = NULL;
static char                  *window   = NULL;

/****************************************************************************/

int
main(
#if NeedFunctionPrototype
     int     argc,
     char    **argv)
#else
     argc,   argv)
      int     argc;
      char    **argv;
#endif
{
      int    *xdata;
      int    *ydata;
      int    *windata;
      char   buf[100];
      struct sm_data_buf           *pspec = &spec_data;
      
/*  check to see which options have been set in the command line */      
      read_args(argc,argv);

/* read data from spectrum file */
      if ( read_from_pipe(spectrum) != 0) {
	    fprintf(stderr,"projecting: failed in read from spectrum file %s\n",spectrum);
	    exit(1);
      }
      
/* check it is a 2d spectrum data set */
      if (strncasecmp(pshm->com_str,"DISPLAY 2",9) != 0 && strncasecmp(pshm->com_str,"WINDOW",6) != 0 ) {
	    fprintf(stderr,"slicing: specified spectrum %s is not a 2D spectrum\n",spectrum);
	    exit(1);
      }

/* make a copy of the spectrum data */
      (void) memcpy((PTR_t) pspec, (PTR_t) pshm, sizeof(struct sm_data_buf));

/* read data from window file */
      if ( read_from_pipe(window) != 0) {
	    fprintf(stderr,"projecting: failed in read from window file %s\n",window);
	    exit(1);
      }
      
/* check it is a 2d spectrum data set */
/*      if (strncasecmp(pshm->com_str,"WINDOW",6) != 0) {
	    fprintf(stderr,"slicing: specified window %s is not a 2D spectrum\n",window);
	    exit(1);
      } */

/* check that spectrum and window are same size */
      if (pspec->size != pshm->size) {
	    if ( (windata = resize_window(pshm->array, pspec->size, pshm->size)) == NULL) {
		  fprintf(stderr,"projecting: window resize failure\n");
		  exit(2);
	    }
      }
      else {
	    windata = pshm->array;
      }
      
/* allocate memory for x & y projected spectra */
      xdata = (int *) calloc(pspec->size, sizeof(int));
      ydata = (int *) calloc(pspec->size, sizeof(int));
      if (xdata == NULL || ydata == NULL) {
	    fprintf(stderr,"projecting: memory allocation failure\n");
	    exit(3);
      }

/* calculate projected spectra */
      project(pspec->array, windata, xdata, ydata, pspec->size);

/* write out X slice to temp file*/
      strcpy(pshm->com_str,"DISPLAY 1");
      strncpy(pshm->name,"Proj onto Y" /*X projected*/,NAME_SIZE);
      pshm->more = 1;
      if (write_slice(".SORT_pipe0", pshm->array, pspec->size, xdata) != 0) {
	    fprintf(stderr,"projecting: error writing out x projection\n");
	    exit(4);
      }

/* write out Y slice to temp file */      
      strncpy(pshm->name,"Proj onto X" /*Y projected*/,NAME_SIZE);
      pshm->more = 0;
      if (write_slice(".SORT_pipe1", pshm->array, pspec->size, ydata) != 0) {
	    fprintf(stderr,"projecting: error writing out y projection\n");
	    exit(4);
      }

/* signal 1d display process to show slices */
      if ( (int) pspec->pid_1d <= 0 || kill(pspec->pid_1d, SIGUSR1) == -1) {
#ifdef SVR4
	    pid_t pid = lookup_proc("sort_xvgr");
#else
	    int   pid = lookup_proc("sort_xvgr");
#endif
	    if ((int) pid > 0) {
		  if (kill(pid, SIGUSR1) == -1) {
			perror("projecting-kill");
			fprintf(stderr,"projecting: failed to signal 1D display process pid = %d\n",
				 (int) pspec->pid_1d);
			exit(5);
		  }
	    }
	    else {
		  fprintf(stderr,"projecting: failed to signal 1D display process pid = %d\n",
			  (int) pspec->pid_1d);
		  exit(5);
	    }
      }
      exit(0);
}
     
void
read_args(
#if NeedFunctionPrototype
     int		argc,
     char		**argv)
#else
      argc, argv)
      int		argc;
      char		**argv;
#endif
{
     int i;
     for(i=1; argv[i] != NULL; i++) {
	   if (strncmp(argv[i],"-dbx",4) == 0) {
		 if (argv[i+1] != NULL) {
		       DBX_val = atoi(argv[i+1]);
		       fprintf(stderr,"projecting: diagnostics level set to %d\n",DBX_val);
		       i += 1;
		 }
	   }
	   else if (strncmp(argv[i],"-spectrum",8) == 0) {
		 if (argv[i+1] != NULL) {
		       spectrum = argv[i+1];
		       if (DBX_val > 0)
			     fprintf(stderr,"projecting: spectrum file = %s\n",spectrum);
		       i++;		       
		 }
		 else {
		       fprintf(stderr,"projecting: no spectrum file specified\n");
		       exit(-1);
		 }
	   }
	   else if (strncmp(argv[i],"-window",6) == 0) {
		 if (argv[i+1] != NULL) {
		       window = argv[i+1];
		       if (DBX_val > 0)
			     fprintf(stderr,"projecting: window file = %s\n",window);
		       i++;		       
		 }
		 else {
		       fprintf(stderr,"projecting: no window file specified\n");
		       exit(-1);
		 }
	   }
     }
     if (spectrum == NULL || window == NULL) {
	   fprintf(stderr,"projecting: spectrum or window file not specified\n");
	   exit(-1);
     }
     return;
}

int
*resize_window(
#if NeedFunctionPrototype
	    int   *data,
	    int   newsize,
	    int   oldsize)
#else
      data, newsize, oldsize)
      int   *data;
      int   newsize;
      int   oldsize;
#endif
{
      int *ptr;
      register int i,j, size;
      
      if ( (ptr = (int *) calloc(newsize*newsize, sizeof(int))) == NULL) {
	    fprintf(stderr,"slicing memory allcoation error\n");
	    return(NULL);
      }
      size = (newsize > oldsize) ? oldsize : newsize; /* minimium */
      for(i=0; i<size ; i++) {
	    for(j=0; j<size; j++) {
		  ptr[j + i*newsize] = data[j + i*oldsize];
	    }
      }
      return(ptr);
}

/*
 *  take specdata AND it with windata and
 *  project the resultant out onto the X and Y planes 
 *  assume square spectra with arrays of form data[0] = (x=0,y=0).
 *  data[1] = (x=1,y=0).... data[size] = (x=0,y=1) etc
 */
void
project(
#if NeedFunctionPrototype
	int *specdata,
	int *windata,
	int *x,
	int *y,
	int size)
#else
      specdata, windata, x, y, size)
      int *specdata;
      int *windata;
      int *x;
      int *y;
      int size;
#endif
{
      register i,j,point;
      for(i=0; i<size ; i++) {
	    for(j=0; j<size; j++) {
		  point = j + i*size;
		  if (specdata[point] && windata[point]) {
			x[j] += specdata[point];
			y[i] += specdata[point];
		  }
	    }
      }
      return;
}
     
int
write_slice(
#if NeedFunctionPrototype
	    char  *out_file,
	    int   *out_data,
	    int   size,
	    int   *in_data)
#else
      out_file, out_data, size, in_data)
      char  *out_file;
      int   *out_data;
      int   size;
      int   *in_data;
#endif
{
      (void) memcpy(out_data, in_data, sizeof(int)*size);
      if ( write_to_pipe(out_file,size) == SORT_FAIL)
	    return(-1);
      return(0);
}

/*
 *  try and find process with name = proc_name and belonging
 *  to process group of this process
 */

#ifdef SVR4

#include <dirent.h>
#include <sys/types.h>
#include <sys/signal.h>
#include <sys/fault.h> 
#include <sys/syscall.h>
#include <sys/procfs.h>
#include <sys/stat.h>
#include <fcntl.h>

pid_t lookup_proc(
#if NeedFunctionPrototype
		  char *proc_name)
#else
      proc_name)
      char *proc_name;
#endif               /* if NeedFunctionPrototype */
{
      pid_t          psid;
      DIR	     *dirp;
      struct dirent  *dp;
      int            fd;
      prpsinfo_t     proc_info;
      char           buf[BUFSIZ];
      
/* get process group id */      
      if ( (psid = getsid((pid_t)0)) == -1) {
	    perror("projecting-getsid");
	    return(-1);
      }

/* open /proc directory for reading and loop through all enteries */
      if ((dirp = opendir("/proc")) == NULL) {
	    perror("projecting-opendir");
	    return(-1);
      }

/* cycle through systems processes */      
      for (dp=readdir(dirp); dp != NULL; dp=readdir(dirp)) {
	    sprintf(buf,"/proc/%s",dp->d_name);
	    if (dp->d_name[0] == '.')
		  continue;
	    if (access(buf, R_OK) == 0) {
		  if ( (fd = open(buf,  O_RDONLY)) == -1) {
			perror("projecting-open");
			break;
		  }
		  if ( ioctl(fd, PIOCPSINFO, (void *) &proc_info) == -1) {
			perror("projecting-ioctl");
			(void) close(fd);
			break;
		  }
		  if (proc_info.pr_sid == psid) {
			int len = strlen(proc_name);
			len = (len > PRFNSZ) ? PRFNSZ : len;
			if (strncmp(proc_name, proc_info.pr_fname, len) == 0) {
			      (void) close(fd);
			      return(proc_info.pr_pid);
			}
		  }
		  (void) close(fd);
	    }
	    else {
		  if (errno != EACCES)
			perror("projecting-access");
	    }
      }
      closedir(dirp);
      return(-1);
}

#else               /* ifdef SVR4 */
      
int   lookup_proc(
#if NeedFunctionPrototype
      char *proc_name)
#else
      proc_name)
      char *proc_name;
#endif              /* if NeedFunctionPrototype */
{
      return(-1);
}

#endif              /* ifdef SVR4 */

