/* libmole---functions for broadcasting EbyE data blocks in EGTS protocol
   over a private (usually!) Ethernet or FDDI to workstations running sort-spy

   Uses the DLPI interface and must run with root priviledge
   works with /dev/le and /dev/bf but not with the Network Peripherals
   fddi interface

   mostly hacked from sort-spy.c

   Tue Mar 21 13:44:27 GMT 1995, DB

*/

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <netdb.h>
#include <unistd.h>

#include <sys/types.h>
#include <sys/fcntl.h>
#include <sys/ioctl.h>
#include <sys/stropts.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/ethernet.h>
#include <sys/dlpi.h>

#include <net/if.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/udp.h>
#include <netinet/ip_icmp.h>

#include "spy.h"

#define DEBUG_OUT    (debug & 1)
#define DEBUG_MISSED (debug & 2)
#define DEBUG_PUT    (debug & 4)
#define DEBUG_READ   (debug & 8)
#define DEBUG_FILTER (debug & 16)
#define DEBUG_SPY    (debug & 32)
#define DEBUG_INFO   (debug & 64)

#define then
 
typedef struct {
   int fd;                                     /* file descriptor for device */
   int max_sdu;                  /* max network packet len allowed by device */
   int sequence;                              /* fragment sequence numbering */
   struct in_addr sourceAddress;      /* source IP address for filter module */
   } MOLE_STREAM;


#define MAX_STREAMS 4

static MOLE_STREAM streams[MAX_STREAMS];
static char message[128];
static int debug=0;

/*****************************************************************************/
static int putmsgack (int stream, struct strbuf *ctlptr, 
      struct strbuf *dataptr, int putflags) {

   struct strbuf ctlbuf;
   struct strbuf databuf;
   int getflags;
   union DL_primitives rcvbuf;
   int i;

   if (putmsg (streams[stream].fd, ctlptr, dataptr, putflags))
      return ERROR;
   if (DEBUG_PUT)
      printf ("putmsgack phase 1 done OK\n");
   ctlbuf.maxlen = sizeof (union DL_primitives);
   ctlbuf.len = 0;
   ctlbuf.buf = (char *) &rcvbuf;
   getflags = RS_HIPRI;
   if (getmsg (streams[stream].fd, &ctlbuf, NULL, &getflags)<0)
      return ERROR;
   if (DEBUG_PUT)
      printf ("putmsgack phase 2 done OK\n");
   if (ctlbuf.len < sizeof (long)) {
      errno = EPROTO;
      return ERROR;
      }
   if (DEBUG_PUT)
      printf ("putmsgack phase 3 done OK\n");
   switch (rcvbuf.dl_primitive) {
      case DL_OK_ACK:
         if (DEBUG_PUT)
            printf ("putmsgack phase 4 done OK\n");
         return OK;

      case DL_ERROR_ACK:
         if (DEBUG_PUT)
            printf ("putmsgack phase 5 \n");
         if (ctlbuf.len < sizeof (dl_error_ack_t)) {    
            if (DEBUG_PUT)
               printf ("putmsgack phase 8 \n");
            errno = EPROTO;
            }
         else
            errno = rcvbuf.error_ack.dl_unix_errno;
         if (DEBUG_PUT)
            printf ("putmsgack phase 9 %x \n",rcvbuf.error_ack.dl_errno);
         return ERROR;

      case DL_BIND_ACK:
         if (DEBUG_PUT)
            printf ("putmsgack phase 6 \n");
         return OK;

      case DL_INFO_ACK:
         if (DEBUG_PUT)
            printf ("putmsgack phase 6 \n");
         streams[stream].max_sdu=rcvbuf.info_ack.dl_max_sdu;
         if (DEBUG_INFO) {
            for (i=0; i<sizeof(dl_info_ack_t)/4; i++)
                printf ("%08x ", *((unsigned int*)&rcvbuf+i));
            printf ("\n");
	    }
         return OK;

      default:
         if (DEBUG_PUT)
            printf ("putmsgack phase 7 \n");
         errno = EPROTO;
         return ERROR;
      }
   }

/*****************************************************************************/
static int putmsgnoack (int stream, struct strbuf *ctlptr, 
        struct strbuf *dataptr, int putflags) {

   if (putmsg (streams[stream].fd, ctlptr, dataptr, putflags)) {
      if (DEBUG_PUT) 
         printf ("putmsgnoack - putmsg()=%d, errno=%d\n", ERROR, errno);
      return ERROR;
      }
   if (DEBUG_PUT)
      printf ("putmsgnoack phase 1 done OK\n");
   return OK;
   }
/*****************************************************************************/
int sortMoleOpen (int stream, char* hostname, char* interface, int devnum) {

   struct hostent *hostEntry;
   struct strbuf strbuf;
   union DL_primitives ctlbuf;
   int i;

   if (stream<1 || stream>MAX_STREAMS) then {
      sprintf (message, "sortMoleOpen: stream number (%d) not in range 1..%d",
         stream, MAX_STREAMS);
      return ERROR;
      }
   if (streams[stream].fd!=0) then {
      sprintf (message, "sortMoleOpen: stream number (%d) not open", stream);
      return ERROR;
      }
   hostEntry = gethostbyname (hostname);
   if (hostEntry == NULL) {
      sprintf (message, "sortMoleOpen: couldn't find hosts database entry for %s", hostname);
      return ERROR;
      }
   memcpy ((char*)&streams[stream].sourceAddress, *hostEntry->h_addr_list, 4);

   i = open (interface, O_RDWR);
   if (i == ERROR) {
      sprintf (message, "sortMoleOpen: failed to open %s", interface);
      return ERROR;
      }
   streams[stream].fd=i;

   /* set up discard message mode */

   if (ioctl (streams[stream].fd, I_SRDOPT, RMSGD) == ERROR) {
      sprintf (message, "sortMoleOpen: failed to enter discard message mode");
      sortMoleClose (stream);
      return ERROR;
      }

   /*  attach to interface    */

   strbuf.buf = (char *) &ctlbuf;
   strbuf.len = DL_ATTACH_REQ_SIZE;
   ctlbuf.dl_primitive = DL_ATTACH_REQ;
   ctlbuf.attach_req.dl_ppa = devnum;

   if (putmsgack (stream, &strbuf, NULL, RS_HIPRI)) {
      sprintf (message, "sortMoleOpen: failed to attach to ppa %d", devnum);
      sortMoleClose (stream);
      return ERROR;
      }

   /* bind */

   strbuf.len = DL_BIND_REQ_SIZE;
   ctlbuf.dl_primitive = DL_BIND_REQ;
   ctlbuf.bind_req.dl_sap = ETHERTYPE_IP;
   ctlbuf.bind_req.dl_max_conind = 0;
   ctlbuf.bind_req.dl_service_mode = DL_CLDLS;
   ctlbuf.bind_req.dl_conn_mgmt = 0;
   ctlbuf.bind_req.dl_xidtest_flg = 0;

   if (putmsgack (stream, &strbuf, NULL, 0)) {
      sprintf (message, "sortMoleOpen: failed to bind");
      sortMoleClose (stream);
      return ERROR;
      }

/* get info */

   strbuf.len = DL_INFO_REQ_SIZE;
   ctlbuf.dl_primitive = DL_INFO_REQ;

   if (putmsgack (stream, &strbuf, NULL, 0)) {
      sprintf (message, "sortMoleOpen: failed to get interface info");
      sortMoleClose (stream);
      return ERROR;
      }

   streams[stream].sequence=0;
   return OK;
   }
/*****************************************************************************/
int sortMoleWrite (int stream, char* data, int len) {

   struct {
      dl_unitdata_req_t txctl;
      char destaddr[6];
      unsigned short txtype; 
      } txctlbuf;
   struct strbuf ctlbuf;
   struct strbuf outbuf;
   DATAGRAM datagram;
   unsigned int broadcast_ip_addr=0xffffffff;
   int fraglen;
   int maxfraglen;
   int offset;
   int i;

   if (stream<1 || stream>MAX_STREAMS) then {
      sprintf (message, "sortMoleWrite: stream number %d not in range 1..%d",
         stream, MAX_STREAMS);
      return ERROR;
      }
   if (streams[stream].fd==0) then {
      sprintf (message, "sortMoleWrite: stream %d not open", stream);
      return ERROR;
      }

   maxfraglen=streams[stream].max_sdu-(datagram.data-(char*)&datagram);
   offset=0;
   while (len>0) {
      fraglen=(len<maxfraglen ? len : maxfraglen);   
      memcpy ((char*)datagram.data, data, fraglen);
      datagram.hdr.stream = stream;
      datagram.hdr.capability[0] = 0;
      datagram.hdr.capability[1] = 0;
      datagram.hdr.options = (fraglen==len) ? OPT_END : 0;
      datagram.hdr.sequence = streams[stream].sequence;
      datagram.hdr.offset = offset;
      datagram.hdr.length = fraglen;
      datagram.rpc.id = 0;
      datagram.rpc.direction = 0;
      datagram.rpc.version = RPC_VERSION;
      datagram.rpc.program = EGTS_RPC_PROG;
      datagram.rpc.progVersion = EGTS_VERSION;
      datagram.rpc.procedure = EGTS_RPC_PROC;
      datagram.rpc.authorisation[0] = 0;
      datagram.rpc.authorisation[1] = 0;
      datagram.rpc.authorisation[2] = 0;
      datagram.rpc.authorisation[3] = 0;
      datagram.udp.uh_sport = EGEB_UDP_PORT;
      datagram.udp.uh_dport = EGTS_UDP_PORT;
      datagram.udp.uh_ulen = datagram.hdr.length+sizeof(DATA_HDR)+
         sizeof(RPC_HDR)+sizeof(struct udphdr);
      datagram.udp.uh_sum = 0;
      datagram.ip.ip_hl=sizeof(struct ip)/4;
      datagram.ip.ip_v=4;
      datagram.ip.ip_tos=0;
      datagram.ip.ip_len=datagram.udp.uh_ulen+sizeof(struct ip);
      datagram.ip.ip_id=0;
      datagram.ip.ip_off=IP_DF;
      datagram.ip.ip_ttl=0xff;
      datagram.ip.ip_p = IPPROTO_UDP;
      datagram.ip.ip_sum=0;
      datagram.ip.ip_src=stream[streams].sourceAddress;
      datagram.ip.ip_dst=*(struct in_addr*)&broadcast_ip_addr;
   
   
      ctlbuf.len = sizeof(txctlbuf);
      ctlbuf.buf = (char *) &txctlbuf;
      txctlbuf.txctl.dl_primitive = DL_UNITDATA_REQ;
      txctlbuf.txctl.dl_dest_addr_length = 8;
      txctlbuf.txctl.dl_dest_addr_offset = sizeof (dl_unitdata_req_t);
      for (i=0; i<6; i++) txctlbuf.destaddr[i]=0xff;
      txctlbuf.txtype=ETHERTYPE_IP; 
   
      outbuf.len=datagram.ip.ip_len;
      outbuf.buf=(char*)&datagram;
   
      if (putmsgnoack (stream, &ctlbuf, &outbuf, 0)) {
         sprintf (message, "sortMoleWrite: failed to send");
         return ERROR;
         }
      len-=fraglen;
      data+=fraglen;
      offset+=fraglen;
      streams[stream].sequence++;
      }
   return OK;
   }
/*****************************************************************************/
int sortMoleClose (int stream) {
   
   if (stream<1 || stream>MAX_STREAMS) then {
      sprintf (message, "sortMoleClose: stream number (%d) not in range 1..%d",
         stream, MAX_STREAMS);
      return ERROR;
      }
   if (streams[stream].fd!=0) then
      close (streams[stream].fd);
   streams[stream].fd=0;
   return OK;
   }
/*****************************************************************************/
char* sortMoleError () {
   return message;
   }
