/******************************************************************************
* libspy.c   sort block access routines
*       These functions can be used by user programs
*       to access data blocks put in shared files by
*       the sort-spy demon task
*/

/*
Tue May  2 11:00:00 BST 1995, DB
added sortSpySignal() function and used fd[stream]!=0 to mark open stream

Thu Mar  9 11:27:09 GMT 1995, DB
changed the ring handling protocol---see the README
*/

#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>

#include <sys/file.h>
#include <sys/types.h>
#include <sys/mman.h>

#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/udp.h>

#include  "spy.h"

int   fd[MAX_STREAM+1] = {0, 0, 0, 0, 0};
FSTR* fstr[MAX_STREAM+1];

struct flock   lock = {F_WRLCK, 0, 0, 0, 0};
struct flock unlock = {F_UNLCK, 0, 0, 0, 0};

/******************************************************************************
* sortSpyOpenFile    make a connection to the shared file for that stream
*       check that the stream is valid
*       open and lock the file
*       mmap the file
*       clear the userPid field (don't want signals initially)
*       return OK or ERROR
*/
int sortSpyOpenFile (int stream, char* fileroot) {
   char fname[128];
   int blocksize;
   int ringsize;
   int ifd;

   if (stream < 1 || stream > MAX_STREAM) {
      perror ("sortSpyOpenFile - stream number out of range");
      return ERROR;
      }
   if (fd[stream] != 0) {
      perror ("sortSpyOpenFile - stream already open");
      return ERROR;
      }
   sprintf (fname, "%s-%d", fileroot, stream);
   ifd = open (fname, O_RDWR, 0);
   if (ifd == ERROR) {
      perror ("sortSpyOpenFile - failed to open shared file");
      return ERROR;
      }
/**************************************do we really need this?
   if (fcntl (ifd, F_SETLK, &lock)) {
      if (errno == EWOULDBLOCK)
         perror ("sortSpyOpenFile - stream already open");
      else
         perror ("sortSpyOpenFile - failed to lock shared file");
      return ERROR;
      }
**************************************************************/
   fstr[stream] = (FSTR *) mmap (NULL, sizeof (FSTR),
         PROT_READ | PROT_WRITE, MAP_SHARED, ifd, 0);
   if ((int) fstr[stream] == ERROR) {
      perror ("sortSpyOpenFile - failed to mmap shared file header");
      return ERROR;
      }
   fstr[stream]->userPid = 0;              /* start without signals */
   blocksize=fstr[stream]->blocksize;
   ringsize=fstr[stream]->ringsize;
   munmap ((caddr_t)fstr[stream], sizeof (FSTR));
   fstr[stream] = (FSTR *) mmap (NULL, sizeof (FSTR)+blocksize*ringsize,
         PROT_READ | PROT_WRITE, MAP_SHARED, ifd, 0);
   if ((int) fstr[stream] == ERROR) {
      perror ("sortSpyOpenFile - failed to mmap shared file");
      return ERROR;
      }
   fd[stream] = ifd;                                     /* mark stream open */
   return OK;
   }
/******************************************************************************
* sortSpyOpen    make a connection to the shared file for that stream
* as for sortSpyOpen, but filename defaulted
* provided for compatibility with earlier version
*/

int sortSpyOpen (int stream) {
   return sortSpyOpenFile (stream, DEFAULT_FILEROOT);
   }

/******************************************************************************
* sortSpySignal   tell the spy process whether or not to use synchronising
* signals
*       check that the stream is valid and opened
*       set or clear the userPid field
*       return OK or ERROR
*/

int sortSpySignal (int stream, int bool) {

   if (stream < 1 || stream > MAX_STREAM) {
      perror ("sortSpySignal - stream number out of range");
      return ERROR;
      }
   if (fd[stream] == 0) {
      perror ("sortSpySignal - stream not open");
      return ERROR;
      }
   fstr[stream]->userPid = bool ? getpid () : 0;
   return OK;
   }
/******************************************************************************
* sortSpyClose       close a connection to an the shared file for that stream
*       check that the stream is valid and opened
*       clear the userPid field
*       close the file
*       return OK or ERROR
*/
int sortSpyClose (int stream) {
   if (stream < 1 || stream > MAX_STREAM) {
      perror ("sortSpyClose - stream number out of range");
      return ERROR;
      }
   if (fd[stream] == 0) {
      perror ("sortSpyClose - stream not open");
      return ERROR;
      }
   fstr[stream]->userPid = 0;                     /* stop the spy signalling */
/**************************************do we really need this?
   if (fcntl (fd[stream], F_SETLK, &unlock)) {
      perror ("sortSpyClose - failed to unlock shared file");
      }
***************************************************************/
   munmap ((caddr_t)fstr[stream], 
      sizeof (FSTR)+fstr[stream]->blocksize*fstr[stream]->ringsize);
   close (fd[stream]);
   fd[stream] = 0;                                     /* mark stream closed */
   return OK;
   }
/******************************************************************************
* sortSpyReadWithSeq   get a block of data for a stream
*          check that the stream is valid and opened
*          check that there is any data in the queue
*          increment the queue read index 
*          return length of data block and its sequence number or ERROR
*/

int sortSpyReadWithSeq (int stream, char *data, int length, int* seq) {
   int rdIndex;
   FSTR* f;
   if (stream < 1 || stream > MAX_STREAM) {
      perror ("sortSpyReadWithSeq - stream number out of range");
      return ERROR;
      }
   if (fd[stream] == 0) {
      perror ("sortSpyReadWithSeq - stream not open");
      return ERROR;
      }
   f=fstr[stream];
   if (f->wrCount == 0) return 0;       /* then no data */

   if (length <= 0)                /* check length valid */
      return length;
   
   rdIndex=(f->wrIndex-f->wrCount+f->ringsize) % f->ringsize;
   if (length > f->length[rdIndex])
      length = f->length[rdIndex];
                        /* fill data block */
   memcpy (data, (char*)f+f->buffer[rdIndex], length);

   length = f->length[rdIndex];    /* save block length */
   *seq = f->sequence[rdIndex];
   f->wrCount--;         /* decrement unread count */

   return length;            /* return actual length of data block */
   }
/******************************************************************************
* sortSpyRead   get a block of data for that stream
* as for sortSpyReadWithSeq, but ignoring the sequence numbering
* provided for compatibility with earlier version
*/

int sortSpyRead (int stream, char *data, int length) {
   int seq;
   return sortSpyReadWithSeq (stream, data, length, &seq);
   }
/*****************************************************************************/
