/*
 * notifier functions for sunsort
 * last modified
 */

#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <memory.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>

#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/mman.h>
#include <sys/ioctl.h>
#include <sys/filio.h>

#include <xview/xview.h>
#include <xview/notify.h>
#include <xview/panel.h>
#include <xview/textsw.h>
#include <xview/xv_xrect.h>

#include "gui_extern.h"

extern int stop_interupt;

/***********************************************************************************************/
/*
 *  notify process that server has finished sorting data
 */
Notify_value
graf_SIGINT_handler(
#if NeedFunctionPrototype
		    Notify_client	client,
		    int			sig,
		    int			mode)
#else
      client,sig,mode)
      Notify_client		client;
      int			sig;
      int			mode;
#endif
{
/* this signal is sent by the user to stop sorting */
	if ( DBX_val >= 2)
	      fprintf(stderr,"sort_gui: graf_SIGINT_handler SIGINT received for client %d\n",client);

/* if batch processing allow for possibility of further sorting */	
	if (batch_mode == SORT_TRUE)
	      return NOTIFY_DONE;
	
	Button_Inactive(FALSE);
/* clear left footer */
	xv_set(sort_gui_window1->window1,FRAME_LEFT_FOOTER,"",NULL);

	Sorting_data = SORT_FALSE;
 
/* undisplay sort statistics frame */		
	xv_set(sort_gui_window1->control_stats,XV_SHOW,FALSE,NULL);
		
/* un-busy popups command frames  */
	Make_popups_busy(FALSE);
	ring_bell();
	
	return NOTIFY_DONE;
}
/***********************************************************************************************/
/*
 *  notify process that server has finished batch processing
 */
Notify_value
graf_SIGUSR1_handler(
#if NeedFunctionPrototype
		    Notify_client	client,
		    int			sig,
		    int			mode)
#else
      client,sig,mode)
      Notify_client		client;
      int			sig;
      int			mode;
#endif
{
/* this signal is sent by the user to stop sorting */
	if ( DBX_val >= 2)
	      fprintf(stderr,"sort_gui: graf_SIGUSR1_handler SIGUSR1 received for client %d\n",client);

	Button_Inactive(FALSE);
/* clear left footer */
	xv_set(sort_gui_window1->window1,FRAME_LEFT_FOOTER,"",NULL);

	Sorting_data = SORT_FALSE;
	batch_mode = SORT_FALSE;
 
/* undisplay sort statistics frame */		
	xv_set(sort_gui_window1->control_stats,XV_SHOW,FALSE,NULL);
		
/* un-busy popups command frames  */
	Make_popups_busy(FALSE);
	ring_bell();
	
/* immediatley update information displayed on windows */
	refresh_gui_func(0);
	
	return NOTIFY_DONE;
}
/***********************************************************************************************/
/*
 *  notify process that status may data needs updating
 *  ie ... display lists
 */
Notify_value
graf_SIGUSR2_handler(
#if NeedFunctionPrototype
		     Notify_client	        client,
		     int			sig,
		     int			mode)
#else
      client,sig,mode)
      Notify_client		client;
      int			sig;
      int			mode;
#endif
{
/* this signal is sent by the user to update display list */
      if ( DBX_val >= 2) 
	    fprintf(stderr,"sort_gui: graf_SIGUSR2_handler SIGUSR2 received for client %d\n",client);

/* set up spectra lists */
	if (read_spectra_list(SORT_TRUE) == SORT_FAIL) {
	      fprintf(stderr,"sort_gui: failed to read display lists correctly\n");
	      exit(1);
	}
      return NOTIFY_DONE;
}

/***********************************************************************************************/
/*
 *  handle problems with communications over the server pipe
 */
Notify_value
graf_SIGPIPE_handler(
#if NeedFunctionPrototype
		    Notify_client	client,
		    int			sig,
		    int			mode)
#else
      client,sig,mode)
      Notify_client		client;
      int			sig;
      int			mode;
#endif
{
      char *ctty = ctermid(NULL);
      if ( DBX_val >= 2)
	    fprintf(stderr,"sort_gui: graf_SIGPIPE_handler SIGPIPE received for client %d\n",client);

      noserver = SORT_TRUE;
      if (ctty != NULL) {
	    /* reset i/o to controlling terminal */
	    int    fd;
	    if ( (fd = open(ctty,O_RDONLY)) != -1)
		  (void) dup2(fd, STDIN_FILENO);
	    if ( (fd = open(ctty,O_WRONLY)) != -1)
		  (void) dup2(fd, STDOUT_FILENO);
      }
      errwin(sort_gui_window1->window1,"sunsort server process died ?");	
      return NOTIFY_DONE;
}

/*******************************************exit handler***************************************/

Notify_value
destroy_func(
#if NeedFunctionPrototype
	     Notify_client 	client,
	     Destroy_status 	status)
#else
      client,status)
      Notify_client 	client;
      Destroy_status 	status;
#endif
{
	if (status == DESTROY_CHECKING) {
		switch( yesno(sort_gui_window1->window1,"Are you sure you want to Exit ?",
			      "no","yes")) {
			case SORT_FALSE:
				fputs("Quitting ...\n",stderr);
				(void) write(1,"exit\n\n",6);
				return NOTIFY_DONE;
			case SORT_TRUE:
			default:
				fputs("quit cancelled\n",stderr);
				notify_veto_destroy(client);
				return NOTIFY_DONE;
		}
	}
	else if (status == DESTROY_CLEANUP) {
	      if ( DBX_val >= 2) fputs("cleaning up ...\n",stderr);
	}
	else if (status == DESTROY_SAVE_YOURSELF) {
	      if ( DBX_val >= 2) fputs("save yourself\n",stderr);
	}
	else {
	      if ( DBX_val >= 2) fputs("process death\n",stderr);
	}
	
/* make sure sort server is terminating ! */
	if (noserver != SORT_TRUE && sort_pid > 1)
	      (void) kill(sort_pid, SIGTERM);
	
	exit(0);
}

/***********************************************************************************************/

Notify_value
read_note(
#if NeedFunctionPrototype
		Notify_client client, int fd)
#else
      client, fd)
      Notify_client client;
      int          fd;
#endif
{
      int bytes = 0;
      char buf[BUFSIZ+1];

      /*
       *  remove data from input stream ... mostly this will be unwanted garbage
       */
      while ( ioctl(fd, FIONREAD, &bytes) != -1 && bytes != 0) {
	    if ( (bytes = read(fd, buf, (bytes > BUFSIZ) ? BUFSIZ : bytes)) >= 0)
		  buf[bytes] = '\0';

	    if (DBX_val >= 2) {
		  fprintf(stderr,"sort_gui: %d bytes read <",bytes);
		  (void) fwrite(buf, bytes, 1, stderr);
		  fprintf(stderr,">\n");
	    }
      }
      return(NOTIFY_DONE);
}

/***********************************************************************************************/
static enum notify_value
check_status(
#if NeedFunctionPrototype
	   int sig)
#else
      sig, code, scp, addr)
      int sig;
      int code;
      struct sigcontext *scp;
      char *addr;
#endif
{
      if (DBX_val >= 10)
	    fprintf(stderr,"sort_gui: received check_status signal\n");

      if (noserver != SORT_TRUE) {
	    if (sort_pid != getppid()) {
		  char *ctty = ctermid(NULL);
		  noserver = SORT_TRUE;
		  if (ctty != NULL) {
			/* reset i/o to controlling terminal */
			int    fd;
			if ( (fd = open(ctty,O_RDONLY)) != -1)
			      (void) dup2(fd, STDIN_FILENO);
			if ( (fd = open(ctty,O_WRONLY)) != -1)
			      (void) dup2(fd, STDOUT_FILENO);
		  }
		  errwin(sort_gui_window1->window1,"sunsort server process died ?");
	    }
      }

/* check whether data in gui window needs updating and if so update it */
      refresh_gui_func(0);

/* if sorting refresh sort statistics */      
      if (Sorting_data == SORT_TRUE) {
	    (void) display_sort_stats();
      }

      return(NOTIFY_DONE);
}

void
setup_intervaltimer(
#if NeedFunctionPrototype
		int bool)
#else
      bool)
      int   bool;
#endif
{
      struct itimerval value;
      
      if (bool == SORT_TRUE) {
	    value.it_value.tv_usec = value.it_interval.tv_usec = 0;
	    value.it_value.tv_sec = value.it_interval.tv_sec = 1;
	    (void) notify_set_itimer_func(notify_client1, check_status, ITIMER_REAL, &value, NULL);
      }
      else {
	    (void) notify_set_itimer_func(notify_client1, NOTIFY_FUNC_NULL, ITIMER_REAL, NULL, NULL);
      }
      return;
}


/***********************************************************************************************/
