/*
 *   filter rountines for writing out EbyE events
 *   first version started on 7/2/96
 *
 */
#include <stdio.h>
#include <memory.h>
#include <stdlib.h>
#include <unistd.h>

#include <sys/types.h>
#include <sys/time.h>

#include "sort_thread.h"
#include "rdtape.h"

#ifdef SVR4
#define NeedFunctionPrototype 1
#else
#define NeedFunctionPrototype 0
#endif

#define GET_BLOCK  0
#define EOD_BLOCK  1

static short     *febye;               /* pointer to start of write data block */
static short     *fptr;                /* pointer to current position in write block */
static short     *f_high_water_mark;   /* pointer to HWM position in write block */
static filtdat_t *filt_data;           /* format info for writing */
static long      out_records;
static long      out_events;
static int       wrt_comment;

/*
 * filter routines for placing data on sunsort write stream
 * each decoder will call:
 *      init_filt   -- before starting to decode events
 *      write_filt  -- to make a request to put an event on the o/p stream
 *      finish_filt -- after finishing decoding (unless error occured already)
 */

int
init_filt(
#if NeedFunctionPrototype
	  int filter, filtdat_t *fdat)
#else
      filter, fdat)
      int       filter;
      filtdat_t *fdat;
#endif
{
      /* zero key parameters */
      febye = NULL;
      out_records = 0;
      out_events = 0;
      adcs_.wrtevt = 0;
      wrt_comment = 1;
      
      /* return immediatley if filter stream not open */
      if (! filter)
	    return(0);

      /* check that valid fdat parameter passed */
      if (fdat == NULL) {
	    fprintf(stderr,"init_filt function passed null data pointer\n");
	    return(-1);
      }

      /* set filter parameters */
      filt_data = fdat;
      
      if ((febye = write_block(0, GET_BLOCK)) == NULL) {
	    fprintf(stderr,"fatal error during request for write buffer\n");
	    return(-1);
      }
      f_high_water_mark = febye + filt_data->hwm;
      
      /* perform beginning of data block functions */
      fptr = filt_data->filt_head(febye);
      
      return(0);
}


int
finish_filt(
#if NeedFunctionPrototype
	    void
#endif
	    )
{
      
      fprintf(stderr,"\nfinished writing after issuing %ld write requests\n",
	      out_records+1);
      fprintf(stderr,"representing %ld events out in %s format\n\n",
	      out_events, filt_data->format);

      if (febye != NULL) {
	    /* perform end of data block functions */
	    filt_data->filt_tail(febye, fptr);

	    if (write_block(filt_data->reclen, EOD_BLOCK) != NULL)
		  return(0);
      }
      
      fprintf(stderr,"*** ERROR was detected during writing ! ***\n");
      return(-1);
}


/*
 * assumes event length is expressed in short ints
 */
int
write_filt(
#if NeedFunctionPrototype
	  short *event, int event_len)
#else
      event, event_len)
      short *event;
      int   event_len;
#endif
{

/* write out buffer if present event takes us beyond HWM */
      if ( fptr+event_len >= f_high_water_mark ) {

	    /* perform end of data block functions */
	    filt_data->filt_tail(febye, fptr);
	    
	    if ( (febye = write_block(filt_data->reclen, GET_BLOCK)) == NULL) {
		  fprintf(stderr,"write to filter file gave error ... STOPPING  filtering\n");
		  return(-1);
	    }
	    f_high_water_mark = febye + filt_data->hwm;
	    out_records++;
	    
	    /* perform beginning of data block functions */
	    fptr = filt_data->filt_head(febye);
      }

      /* add present event to data block */
      (void) memcpy(fptr, event, sizeof(short)*event_len);
      fptr += event_len;
      adcs_.wrtevt = 0;
      out_events++;
      
      return(0);
}

/*
 *  user callable routine to place events on write stream
 *  only support native formats (i.e. NO VAX formats !)
 */
int
wrtevt_(
#if NeedFunctionPrototype
	  int *size, short *array)
#else
      size, array)
      int *size;
      short *array;
#endif
{

      /* check that the output data steam is still OK */
      if (febye == NULL) {
	    if (wrt_comment) {
		  fprintf(stderr,"write data stream not connected or had major error\n");
		  fprintf(stderr,"*** IGNORING all further calls to wrtevt ***\n");
		  wrt_comment = 0;
	    }
	    return(-1);
      }

      /* call encoder specified by user */
      if (filt_data->write_func != NULL)
	    return( filt_data->write_func(*size, array) );

      return(-1);
}
