/*
 *  decoding routine to unpack event by event data and 
 *  pass this on to user routines
 *  *** Data style == EUROGAM ***
 *
 *  started on new version 8/8/95
 *  modified 8/2/95
 *  Changed mask to 0x00001fff to allow starburst to write debug
 *  information in 0x0000E000 and still allow data unpacking.
 *  Much improved error logging and recovery 1/97..2/97
 */

/*
 * This file contains routines common to all Eurogam style decoders.
 * Currently this is Megha and DEMON.
 */


#include <stdio.h>
#include <memory.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>

#include <sys/types.h>
#include <sys/time.h>

#include "sort_thread.h"
#include "rdtape.h"
#include "rdtape_eg.h"

#define DELIMITER_MASK     0xffff
#define END_TOKEN          0xffff0000
#define DATA_MASK          0x00000fff
#ifndef NOS_EG_GROUPS
#define NOS_EG_GROUPS      0x100       /* groups range from 0x0 - 0xff */
#endif
#define MAX_EG_GLEN        0x3f
#define EG_ADDR_SIZE       0x3fff

#define EG_TAPE_1          0x4d45
#define EG_TAPE_2          0x4748
#define EG_TAPE_3          0x4144
#define EG_TAPE_4          0x4154

/*
#define DUMP_RECORDS
#define PRINT_BLOCKS
*/
#define BAD_ONLY

#ifdef DUMP_RECORDS
#include <fcntl.h>
#define DUMP_NTH 1
static int dump_record_count = 1000;
static int dump_nth_count = DUMP_NTH;
static int dump_state = 2;
static int dump_fd;
#define DUMP_FILE "record.dump"
#endif

struct eg_word_struct {
      unsigned short type : 2;
      unsigned short addr :14;
      unsigned short data;
};

static struct group_data groups[NOS_EG_GROUPS];

static short *addr2adc[EG_ADDR_SIZE];
static short adc2addr[NOS_ADC_WORDS];

int
rdtape_eg(
#if NeedFunctionPrototype
	  int filter, int style)
#else
      filter, style)
      int filter;
      int style;
#endif
{
    register short *ebye;                     /* pointer to input buffer */
    register short event_len;
    
    register int *eptr, *header_ptr, *event_end;
    
    time_t start_time, time_now;
    int bytes, record = 0;
    int adc_count = 0;
    long in_events = 0;
    int bad, badlong = 0, badrecords = 0, recovered = 0, badfirst = 0;
    int badlength = 0, badword = 0, badshort = 0, recoverlong = 0;
    int recovershort = 0, total, badeob = 0, badtrunc = 0, bad16 = 0;
    
    register int group, hwords;
    register struct eg_word_struct *wptr;
    register int adc_bytes;
    int *end_ptr;

#ifdef DUMP_RECORDS
    if (dump_state == 2)
    {
	if ((dump_fd = open(DUMP_FILE, O_WRONLY | O_CREAT | O_EXCL, 0666)) < 0)
	{
	    perror(DUMP_FILE);
	    dump_state = 0;
	}
	else
	    dump_state = 1;
    }
#endif
    /* initialize event sorting */
    (void) memset((char *)addr2adc, 0, EG_ADDR_SIZE);
    (void) memset((char *) adcs_.bitpattern, 0, sizeof(adcs_.bitpattern));
    adc_bytes = adcnum_.nos_adcs * sizeof(short);
      
    /* call user's initialization function */      
    (void) init_();
      
    /* set start time */
    start_time = time(NULL);
      
    /* sort requested number of records or until end of file etc */
    for (record=1; ;record++) {
	if ( (ebye = read_block(&bytes)) == NULL) 
	    break;

	end_ptr = ((int *) ebye) + (bytes>>2);

#ifdef PRINT_BLOCKS
	fprintf(stderr, "Block starts as follows:\n");
    {
	int i;
	for(i=0; i<256; i++)
	{
	    fprintf(stderr, "%04x", ebye[i] & 0xffff);
	    if ((i & 15) == 15)
		putc('\n', stderr);
	    else
		putc(' ', stderr);
	}
    }
#endif
	/* check start of record header to see if it a tape block*/
	if (memcmp("MEGHADAT", (char *) ebye, 8) == 0 ||
	    memcmp("EGEVENTD", (char *) ebye, 8) == 0) {
	    if ( ebye[12] != (short) DELIMITER_MASK) {
		fprintf(stderr,"First event header is not a start token "
			"... trying next record\n");
		badfirst++;
#ifdef DUMP_RECORDS
		bad = 1;
		goto EVENT_LOOP_END;
#else
		continue;
#endif
	    }
	    eptr = (int *) &ebye[12];
	}
	/* if not a tape block then just treat as before */
	else {
	    eptr = (int *) &ebye[0];
	    if ( ebye[0] != (short) DELIMITER_MASK) {
		fprintf(stderr,"First event header is not a start token "
			"... trying next record. ");
		fprintf(stderr,"First 6 words were (HSM order): "
			"%04x %04x %04x %04x %04x %04x\n",
			ebye[1] & 0xffff, ebye[0] & 0xffff, ebye[3] & 0xffff,
			ebye[2] & 0xffff, ebye[5] & 0xffff, ebye[4] & 0xffff);
		badfirst++;
#ifdef DUMP_RECORDS
		bad = 1;
		goto EVENT_LOOP_END;
#else
		continue;
#endif
	    }
	}
	    
	bad = 0;

	/* search for end of block indication (header == END_TOKEN) */
	while( *eptr != (int) END_TOKEN) {
	    /* extract fields from event header */
	    header_ptr = eptr++;
	    switch(style)
	    {
	    case STYLE_MEGHA:
		event_len = (*header_ptr & 0x0000ffff) >> 1;
		break;
	    default:
		event_len = (*header_ptr & 0x0000ffff) >> 2;
		break;
	    }

	    /* check that event length is sensible */
	    if((event_len < 0) || (event_len > 4096)) {
		fprintf(stderr,"Event length (%d  %04x) out of range ... "
			"skipping  rest of record %d\n", event_len,
			event_len, record);
		badlength++;
#ifdef DUMP_RECORDS
		bad = 1;
#endif
		break;
	    }
	    event_end = header_ptr + event_len;

	    if (event_end >= end_ptr)
	    {
		fprintf(stderr, "Last event in block truncated - "
			"ignoring it.\n");
		badtrunc++;
#ifdef DUMP_RECORDS
		bad = 1;
#endif
		break;
	    }
	    
	    /* reset adcs_common block */
	    (void) memset((char *) adcs_.adcs, -1, adc_bytes);
            (void) memset((char *) groups, 0, sizeof(groups));
	    adc_count = 0;
		  
	    /* unpack individual event ** EVENT LOOP ** */		  
	    while ( (*eptr & (int) END_TOKEN) != (int) END_TOKEN)  {
		if (eptr >= event_end) {
		    int *sptr = eptr;
		    
		    do {
			eptr++;
		    } while(eptr < end_ptr-1 &&
			    (*eptr & (int) END_TOKEN) != (int) END_TOKEN);
		    if ((*eptr & (int) END_TOKEN) != (int) END_TOKEN)
		    {
			int *ep, i;

			fprintf(stderr, "End of block at short word %d "
				"without end of block marker in record %d\n",
				((short *) sptr) - ebye, record);
			fprintf(stderr,"Words surrounding error were:\n    ");
			for(ep = eptr - 10, i = 4;
			    ep < eptr+30 && ep < end_ptr; ep++)
			{
			    if (i > 69)
			    {
				fprintf(stderr, "\n    ");
				i = 4;
			    }
			    fprintf(stderr, "%08x%c", *ep,
				    (ep == eptr-1) ? '<' : ' ');
			    i += 9;
			}
			fprintf(stderr, "\n");
			badeob++;
			goto EVENT_LOOP_END;
		    }
		    if (!bad)
		    {
			badrecords++;
			bad = 1;
		    }
		    badlong++;
		    if ((eptr-header_ptr)*4 > 1024)
		    {
			fprintf(stderr, "Expected event length %d, no start "
                                "of event marker within 1024 (next at %d). "
                                "Not recovering.\n");
				goto EVENT_LOOP_END;
		    }
		    fprintf(stderr, "Expected event length %d, actually %d, "
			    "error %d. Recovery succeeded.\n", event_len*4,
			    (eptr-header_ptr)*4,
			    (eptr-header_ptr-event_len)*4);
		    recoverlong++;
		    goto EVENT_LOOP_CONTINUE;
		}
		
		/* event type defines tells us how much data is to follow */
		wptr = (struct eg_word_struct *) eptr++;
		switch(style)
		{
		case STYLE_MEGHA:
		    switch ((*(eptr-1)) >> 29) {
		    case 1:           /* simple parameter word found */
			if (addr2adc[wptr->addr & 0x1fff])
			    *addr2adc[wptr->addr & 0x1fff] =
				wptr->data & 0x1fff;
			adc_count++;
			break;
		    case 2:           /* group format parameter(s) found */
			group = wptr->addr & 0x00ff;
			hwords = (wptr->addr >> 8) & 0x003f; 
                        groups[group].group_len = hwords;
                        groups[group].group_ptr = &wptr->data;
			eptr += (hwords / 2);
			break;
		    default:           /* invalid word format found  */
			if ((*eptr & 0xffff) == 0xffff ||
			    (*eptr & 0xe000) == 0x2000)
			{
			    fprintf(stderr, "16 bit data skew in HSM (urk).\n");
			    bad16++;
			}
			else
			{    
			    int *ep, i;
			    
			    fprintf(stderr,"invalid word format found ... "
				    "skipping rest of record %d\n", record);
			    fprintf(stderr,"Words surrounding error were:\n    ");
			    for(ep = eptr - 10, i = 4;
				ep < eptr+30 && ep < end_ptr; ep++)
			    {
				if (i > 69)
				{
				    fprintf(stderr, "\n    ");
				    i = 4;
				}
				fprintf(stderr, "%08x%c", *ep,
					(ep == eptr-1) ? '<' : ' ');
				i += 9;
			    }
			    fprintf(stderr, "\n");
			    badword++;
			}		    
#ifdef DUMP_RECORDS
			bad = 1;
#endif
			goto EVENT_LOOP_END;
		    }
		    break;
		default:
		    switch (wptr->type) {
		    case 0:           /* simple parameter word found */
			if (addr2adc[wptr->addr])
			    *addr2adc[wptr->addr] = wptr->data;
			adc_count++;
			break;
		    case 1:           /* group format parameter(s) found */
			group = wptr->addr & 0x00ff;
			hwords = (wptr->addr >> 8) & 0x003f; 
			groups[group].group_len = hwords;
			groups[group].group_ptr = &wptr->data;
			eptr += (hwords / 2);
			break;
		    default:           /* invalid word format found  */
		    {
			int *ep, i;
			
			fprintf(stderr,"invalid word format found ... "
				"skipping rest of record %d\n", record);
			fprintf(stderr,"Words surrounding error were:\n    ");
			for(ep = eptr - 10, i = 4;
			    ep < eptr+30 && ep < end_ptr; ep++)
			{
			    if (i > 69)
			    {
				fprintf(stderr, "\n    ");
				i = 4;
			    }
			    fprintf(stderr, "%08x%c", *ep,
				    (ep == eptr-1) ? '<' : ' ');
			    i += 9;
			}
		    }
			fprintf(stderr, "\n");
			badword++;
#ifdef DUMP_RECORDS
			bad = 1;
#endif
			goto EVENT_LOOP_END;
		    }
		    break;
		}
	    }
	    
	    /* check that we are past end of event */
	    if (eptr != event_end)
	    {
		if (!bad)
		{
		    badrecords++;
		    bad = 1;
		}
		badshort++;
		recovershort++;
		fprintf(stderr, "Expected event length %d, actually %d, "
			"error %d. Recovery succeeded.\n", event_len*2,
			(eptr-header_ptr)*2, (eptr-header_ptr-event_len)*2);
		continue;
	    }
	    
	    /* provide access to various information to user routine */
	    adcs_.triggernos = adc_count;
	    adcs_.event = ++in_events;
	    adcs_.record = record;
            adcs_.wrtevt = 0;
	    if (bad)
		recovered++;
	    
	    /* call user provided event sorting routine */
	    (void) sortin_();
	    
	    /* write out event if wrtevt set to true */
	    if (filter && adcs_.wrtevt) {
		if ( write_filt((short *)header_ptr, 2*event_len) )
		    filter = 0;
	    }
	EVENT_LOOP_CONTINUE:
	    ;
	    
	}    /* loop on ebye buffer until end token */
	    
	/* come to here to jump out of faulty record */		  
    EVENT_LOOP_END:
	bytes = 0;

#ifdef DUMP_RECORDS
	if (dump_state == 1
#ifdef BAD_ONLY
	    && bad
#endif
	    )
	{
	    if (!--dump_nth_count)
	    {
		int i;

		write(dump_fd, ebye, 16384);
		if (!--dump_record_count)
		{
		    close(dump_fd);
		    dump_state = 0;
		}
		else
		    dump_nth_count = DUMP_NTH;
	    }
	}
#endif	    
	
    }   /* loop over requested number of records or until eof */
    
    /* lets note the time at end */
    time_now = time(NULL);
    
    /* call user provided clearup routine  */
    (void) finish_();
      
    /* output statistics */
    record--;
    fprintf(stderr, "\n*** sort statistics ***\n");
    fprintf(stderr, "\nsorted %ld events in %d seconds\n"
	    "Average sort rate = %g events per second\n",
	    in_events, (int) (time_now-start_time),
	    (double) in_events / (time_now-start_time));
    fprintf(stderr, "Records with missing start token       : %d (%8.4f%%)\n",
	    badfirst, 100.0*badfirst/record);
    fprintf(stderr, "Records with event length out of range : %d (%8.4f%%)\n",
	    badlength, 100.0*badlength/record);
    fprintf(stderr, "Records with invalid word format       : %d (%8.4f%%)\n",
	    badword, 100.0*badword/record);
    fprintf(stderr, "Records with incorrect event length    : %d (%8.4f%%)\n",
	    badrecords, 100.0*badrecords/record);
    fprintf(stderr, "Records without end of block marker    : %d (%8.4f%%)\n",
	    badeob, 100.0*badeob/record);
    fprintf(stderr, "Records containing truncated events    : %d (%8.4f%%)\n",
	    badtrunc, 100.0*badtrunc/record);
    fprintf(stderr, "Records containing 16 bit skew         : %d (%8.4f%%)\n",
	    bad16, 100.0*bad16/record);
    bad = badfirst + badlength + badword + badeob + badtrunc + bad16 +
	(badshort+badlong-recovershort-recoverlong);
    total = badrecords - (badshort+badlong-recovershort-recoverlong);
    fprintf(stderr, "Total records in stream: %d; good: %d (%8.4f%%);\n"
	"      recoverable: %d (%8.4f%%); bad: %d (%8.4f%%)\n", record,
	    (record-bad-total), 100.0*(record-bad-total)/record,
	    total, 100.0*total/record, bad, 100.0*bad/record);
    bad = badshort+badlong;
    total = in_events+bad;
    fprintf(stderr, "Events with word count too short       : %d (%8.4f%%); "
	    "recoveries: %d\n", badshort,
	    100.0*badshort/total, recovershort);
    fprintf(stderr, "Events with word count too long        : %d (%8.4f%%); "
	    "recoveries: %d\n", badlong,
	    100.0*badlong/total, recoverlong);
    fprintf(stderr, "Events recovered by recovery procedures: %d (%8.4f%%)\n",
	    recovered, 100.0*recovered/total);
    fprintf(stderr, "Total events in stream: %d; good: %ld (%8.4f%%);\n"
	"    bad: %d (%8.4f%%)\n", total, in_events, 100.0*in_events/total,
	    bad, 100.0*bad/total);
    fprintf(stderr,"\nfinished sorting data after reading %d records\n",
	    record);
      
    return(filter);
}

/*
 *  fortran callable routine to provide user access to groups data
 *  return 0 for invalid group number
 *         -length if data_len too small
 *         length otherwise
 */
void
egroup_(
#if NeedFunctionPrototype
	  const int *group_nos, unsigned short data[], int *data_len)
#else
      group_nos, data, data_len)
      int            *group_nos;
      unsigned short data[];
      int            *data_len;
#endif
{
      register int i, length;

      /* check that group number is valid */
      if (*group_nos < 0 || *group_nos >= NOS_EG_GROUPS) {
	    *data_len = 0;
	    return;
      }

      /* check length of users array */
      length = groups[*group_nos].group_len;
      if (length > *data_len) {
	    length = *data_len;
	    *data_len = -length;
      }
      else {
	    *data_len = length;
      }

      /* fill users data array with group data */      
      for (i=0; i<length; i++)
	    data[i] = groups[*group_nos].group_ptr[i];

      return;
}

struct group_data *egroup(int group_nos)
{
    if (group_nos < 0 || group_nos >= NOS_EG_GROUPS)
	return NULL;

      return(&groups[group_nos]);
}

/*
 *  routine to map hardware addresses to sort adc number
 *
 *   expect file with syntax
 *   adc1    @01f
 *   adc2    @22e
 *   ..etc   the addreses are in hex
 *
 */
int
eginit(
#if NeedFunctionPrototype
	char *filename)
#else
      filename)
      char *filename;
#endif
{     
      FILE  *rfp;
      char  buf[BUFSIZ], *ptr;
      int line = 0;
      int nos, addr;

      /* get filename of mapping file */
      if (sscanf(filename,"%s",buf) != 1) {
	    fprintf(stderr,"sort: could not decode name of " 
		    "eurogam address mapping file in routine eginit\n");
	    return -1;
      }

      /* open the mapping file */
      if ( (rfp = fopen(buf,"r")) == NULL) {
	    fprintf(stderr,"sort: could not open eurogam " 
		    "address mapping file in routine eginit\n");
	    return -1;
      }

      /* read file and assign mappings */
      while ( fgets(buf,BUFSIZ,rfp)) {
	    line++;

	    for (ptr=buf; *ptr; ptr++)
		  if (isupper((int) *ptr)) tolower((int) *ptr);

	    /* decode line */
	    if (sscanf(buf,"adc%d @%x", &nos, (unsigned int *) &addr) != 2) {
		  fprintf(stderr,"ignored line number %d: %s", line, buf);
		  continue;
	    }

	    /* check that adc nos is in allowed range
	       makesort ensures that adcnum_.nos_adcs <= NOS_ADC_WORDS */
	    if (nos <= 0 || nos > adcnum_.nos_adcs ) {
		  fprintf(stderr,"adc numbber %d out of range, "
			  "ignored line number %d: %s", nos, line, buf);
		  continue;
	    }
	    
	    /* map hardware address into adcs array for use in sortin_()
	    and store inverse mapping for use with write_eg */ 
	    if (addr >= 0 && addr < EG_ADDR_SIZE) {
		  addr2adc[addr] = &adcs_.adcs[nos-1];
		  adc2addr[nos-1] = (short) addr;
	    }
      }

      (void) fclose(rfp);
      return 0;
}

/* fortran callable version of eginit */

void eginit_(char *filename, int *error)
{
    *error = eginit(filename);
}

/*
 *  user callable write event function
 *  write data to filter stream in Eurogam style format
 *
 *  first draft 15/2/95
 */

static int fbuf[NOS_ADC_WORDS+1];

int
write_eg(
#if NeedFunctionPrototype
	  int size, short *array, int style)
#else
      size, array, style)
      int size;
      short *array;
      int style;
#endif
{

      register int i, j;
      register short group, glen;
      register int event_len = 1;
      register short *sptr  = (short *) &fbuf[1];

      (void) memset(fbuf, 0, sizeof(fbuf));
      
      /* ensure that we have an event length within acceptable ranges */
      if (size <= 0) {
	    fprintf(stderr,"you called writefn with an invalid "
		           "value of %d\n",size);
	    return(-2);
      }

      /* pack event */
      for(i=0; i<size; i++) {
	    if (event_len >= NOS_ADC_WORDS) {
		  fprintf(stderr,"EGAM format supports maximum event "
			         "length of %d words\n", NOS_ADC_WORDS);
		  return(-3);
	    }
	    /* pack simple parameter word if corresponding address exists */
	    if (array[i] > 0) {
		  if (adc2addr[i] > 0) {
			*sptr++ = adc2addr[i];
			*sptr++ = array[i];
			event_len++;
		  }
		  else {
			/* check group number */
			group = array[i];
			if (group < 0 || group >= NOS_EG_GROUPS) 
			      continue;
			
			/* check group length */
			glen = -array[i+1];
			if (glen <= 0 || glen >= MAX_EG_GLEN) {
			      fprintf(stderr,"invalid group length => "
				      "%d for EGAM format\n",glen);
			      return(-4);
			}
			if (sptr + glen >= (short *) &fbuf[NOS_ADC_WORDS]) {
			      fprintf(stderr,"group length => %d too large "
				      "for event stack\n",glen);
			      return(-4);
			}
			i += 2;
			
			/* move group onto event stack */
			*sptr++ = 0x4000 | group | (glen << 8);
			for(j=0; j<glen; j++)
			      *sptr++ = array[i++];
			
			/* ensure 4 byte alignment */
			if (glen % 2 == 0) 
			      *sptr++ = 0;
			
			/* advance pointers etc appropriatley */
			event_len += (glen/2) + 1;
		  }
	    }

      }

      if (event_len > 1) {
	    /* setup start token */
	  switch(style)
	  {
	  case STYLE_MEGHA:
	    fbuf[0] = (event_len<<1) | END_TOKEN;
	    break;
	  default:
	    fbuf[0] = (event_len<<2) | END_TOKEN;
	    break;
	  }
	    
	    /* add event to output data buffer */
	    if ( write_filt((short *)fbuf, 2*event_len) )
		  return(-3);
      }
      
      return(event_len);
}


/*
 *  functions used in the filtering of data
 *  write any data to end of filtered data block
 */
void eg_filt_tail(
#if NeedFunctionPrototype
	  short *base, short *fptr)
#else
      base, fptr)
      short *base;
      short *fptr;
#endif
{
      *fptr++ =  (short) DELIMITER_MASK;
      *fptr = 0;
      return;
}
/*
 *  write any data to the beginning of filtered data block
 */
short *eg_filt_head(
#if NeedFunctionPrototype
	  short *base)
#else
      base)
      short *base;
#endif
{
      (void) strcpy((char *)base, "EGEVENTD");
      return(base+12);
}

