/*
 * decoding routine to unpack event by event data and pass this on to user
 * routines
 * *** NSF style data ***
 * prev modified 1/12/94
 * prev modified 24/1/95
 *
 * started on new version 27/4/95
 * modified 5/2/96
 * Modified to cope with 32 bit HSM readout 13/3/97 SMS
 */

#include <stdio.h>
#include <memory.h>
#include <stdlib.h>
#include <unistd.h>

#include <sys/types.h>
#include <sys/time.h>

#include "sort_thread.h"
#include "rdtape.h"

/* Daresbury event parameters */
#define INDIRECT_MASK           0x8000
#define TWL_MASK                0x6000
#define TRIGNOS_MASK            0x1f00
#define MORE_MASK               0x80
#define EVENTLEN_MASK           0x7f
#define DL_OUTRECORD_LEN        4080

static unsigned short Emask[] = {
    0x0001, 0x0002, 0x0004, 0x0008, 0x0010, 0x0020, 0x0040, 0x0080,
    0x0100, 0x0200, 0x0400, 0x0800, 0x1000, 0x2000, 0x4000, 0x8000};
static int trig_field[] = {0, 0, 4};

/*
 * event decoder function
 */
int
rdtape_nsf2
#if NeedFunctionPrototype
(int filter)
#else
(filter)
int filter;
#endif
{
    short *bitpattern[NOS_ADCS_GROUPS], *adc[NOS_ADCS_GROUPS];
    
    /* pointers to data within present event */
    register short *ebye, *eptr, *header_ptr, *tw_ptr;
    
    /* event parameter data */
    register short trig_nos, event_len, trig_w_len, pattern;
    
    time_t start_time, time_now;
    long trig_counter[MAX_NOS_TRIGS];
    int bytes, header, record = 0;
    int first, last;
    unsigned int *p;
    int badformat = 0, bad_first = 0, bad_direct = 0, bad_indirect = 0;
    int bad_length = 0, bad_header = 0, bad;
    
    register int i, j;
    register int adc_bytes, bit_bytes;

    /* initialize event sorting */
    for (i=0; i<NOS_ADCS_GROUPS; i++) {
	adc[i] = &adcs_.adcs[i*16];
	bitpattern[i] = &adcs_.bitpattern[7-i];
    }

    (void) memset((char *) trig_counter, 0, sizeof(trig_counter));
    adc_bytes = adcnum_.nos_adcs * sizeof(short);
    bit_bytes = sizeof(adcs_.bitpattern);
      
    /* call user's initialization function */      
    (void) init_();

    /* set start time */
    start_time  = time(NULL);
      
    /* sort requested number of records or until end of file etc */
    for (record=1; ;record++) {
	if ( (eptr = ebye = read_block(&bytes)) == NULL) 
	    break;

	if (memcmp("MEGHADAT", (char *) eptr, 8) == 0)
	{
	    eptr += 12;
	    bytes -= 24;
	}
	if ((eptr[1] & 0x9e00) != 0x9800)
	{
	    bad_first++;
	    fprintf(stderr,
		    "First header word in block unexpected (was 0x%04x)\n",
		    *eptr);
	    if (++badformat == 5)
	    {
		fprintf(stderr, "%d consecutive bad headers. Are you sure "
			"this is NSF format data ?\n", badformat);
		/*break;*/
	    }
	    continue;
	}

	badformat = 5;

	for(i = bytes>>2, p = (unsigned int *) eptr; i--; p++)
	    *p = ((*p & 0xffff0000)>>16) | ((*p & 0x0000ffff)<<16);

	adcs_.record = record;

	/* decode event block */
	while ( (header = *eptr) != 0) {
	    header_ptr = eptr++;
	    /* reset adcs_ common block */
	    (void) memset((char *) adcs_.adcs, -1, adc_bytes);
	    (void) memset((char *) adcs_.bitpattern, 0, bit_bytes);
	    
	    /* decode event header */
	    adcs_.triggernos = trig_nos = ((header & TRIGNOS_MASK) >> 8) + 1;
	    event_len = header & EVENTLEN_MASK;
	    
	    if (header & INDIRECT_MASK) { /* Indirect trigger */
	    MORE_BIT_LOOP:		                  
		trig_w_len = ((header & TWL_MASK) >> 13) + 1;
		tw_ptr = eptr;
		eptr += trig_w_len;
		if (trig_nos < 24 || trig_nos > 26) {
		    bad_indirect++;
		    fprintf(stderr, "conversion bit indicates indirect "
			    "trigger but trigger nos = %d ... skipping rest "
			    "of record %d\n", trig_nos, record);
		    break;
		}
		first = trig_field[trig_nos - 24];
		last = first + trig_w_len;
		for (i=first; i<last; i++) {
		    pattern = *bitpattern[i] = *(tw_ptr++);
		    if (pattern == 0) continue;
		    for (j=0; j<16; j++) {
			if (pattern & Emask[j])
			    adc[i][j] = *(eptr++);
		    }
		}
		if (header & MORE_MASK) {
		    header = *(eptr++);
		    trig_nos = ((header & TRIGNOS_MASK) >> 8) + 1;
		    event_len += (header & EVENTLEN_MASK);
		    goto MORE_BIT_LOOP;
		}
		if (header_ptr+event_len != eptr) {
		    bad_length++;
		    fprintf(stderr,"Event length is inconsistent with number "
			    "of adcs read from event stack\n");
		    fprintf(stderr,"... skipping rest of record %d\n", record);
		    break;
		}
	    }
	    else if (header > 0) {  /* direct trigger */
		if (trig_nos < 1 || trig_nos > 16) {
		    bad_direct++;
		    fprintf(stderr, "conversion bit indicates direct trigger "
			    "but trigger nos = %d ... skipping rest of "
			    "record %d\n", trig_nos, record);
		    break;
		}
		for (i=1; i<= adcnum_.dtrig[0][trig_nos]; i++) {
		    adcs_.adcs[adcnum_.dtrig[i][trig_nos]-1] = *(eptr++);
		}
		if (header_ptr+event_len != eptr) {
		    bad_length++;
		    fprintf(stderr,"Event length is inconsistent with number "
			    "of adcs read from event stack ... skipping rest "
			    "of record %d\n",record);
		    break;
		}
	    }
	    else {
		bad_header++;
		fprintf(stderr,"ERROR decoding event header in record %d ... "
			"skipping rest of record\n", record);
		break;
	    }
	    /* call user provided event sorting routine */
	    trig_counter[trig_nos]++;
            adcs_.wrtevt = 0;
	    (void) sortin_();

	    /* write out event if wrtevt set to true */
	    if (filter && adcs_.wrtevt) {
		if ( write_filt(header_ptr, event_len) )
		    filter = 0;
	    }
	}   /* loop on the ebye buffer until next header == 0 */

    }         /* loop over requested number of records or until eof */

    /* lets note the time at end */
    time_now = time(NULL);
      
    /* call user provided clearup routine  */
    (void) finish_();
      
    /* output statistics */
    fprintf(stderr,"\n*** sort statistics ***\n");
    for(i=1; i<MAX_NOS_TRIGS; i++) {
	if (trig_counter[i] > 0) {
	    fprintf(stderr,"%ld trigger %d events\n", trig_counter[i], i);
	    trig_counter[0] += trig_counter[i];
	}
    }
    fprintf(stderr,"\nRead %ld events in %d seconds\nAverage sort rate = %g "
	    "events per second\n", trig_counter[0],
	    (int) (time_now-start_time),
	    (double) trig_counter[0] / (time_now-start_time));
    record--;
    fprintf(stderr, "Records with bad first header:          %d (%8.4f%%)\n",
	    bad_first, 100.0*bad_first/record);
    fprintf(stderr, "Records with other bad headers:         %d (%8.4f%%)\n",
	    bad_header, 100.0*bad_header/record);
    fprintf(stderr, "Records with bad direct trigger:        %d (%8.4f%%)\n",
	    bad_direct, 100.0*bad_direct/record);
    fprintf(stderr, "Records with bad indirect trigger:      %d (%8.4f%%)\n",
	    bad_indirect, 100.0*bad_indirect/record);
    fprintf(stderr, "Records with inconsistent event length: %d (%8.4f%%)\n",
	    bad_length, 100.0*bad_length/record);
    bad = bad_first + bad_header + bad_direct + bad_indirect + bad_length;
    fprintf(stderr, "Total records in stream: %d; good: %d (%8.4f%%);\n"
	"        bad: %d (%8.4f%%)\n", record,
	    (record-bad), 100.0*(record-bad)/record, bad, 100.0*bad/record);
    fprintf(stderr,"\nfinished sorting data after reading %d records\n",
	    record);

    return(filter);
}


/*
 * user callable write event function
 * write data to filter stream in Charissa NSF style format
 *
 * first draft 6/2/95
 */

#define MAX_NSF_ADCS 128

static unsigned short trig_pattern[] = {0xf800, 0xf900};
static short fbuf[2+NOS_BIT_WORDS+MAX_NSF_ADCS];

int
write_nsf2(
#if NeedFunctionPrototype
	  int size, short *array)
#else
    size, array)
int size;
short *array;
#endif
{
    register int i, j;
    register short event_len;
    register short *tptr = fbuf;
    register short *bptr = fbuf + 1;
    register short *ptr  = fbuf + 5;
    
    int more = (size > 64) ? 1:0;
    int len = more ? 64 : size;
    int step = 0;

    /* ensure that we have an event length within acceptable ranges */
    if (size > MAX_NSF_ADCS || size <= 0) {
	fprintf(stderr, "NSF format supports maximum event length of 128 "
		"words\n");
	fprintf(stderr,"you called writefn with value of %d\n",size);
	return(-2);
    }

    /* pack event */
    do {
	/* clear down bit masks */
	(void) memset((char *)bptr, 0, sizeof(short)*4);
	event_len = 0;
	
	/* pack data into fbuf */
	for(i=j=0; i<len; i++) {
	    if (array[i] > 0) {
		*ptr++ = array[i];
		*bptr |= Emask[j];
		event_len++;
	    }
	    if (++j >= 16) {
		bptr++;
		j = 0;
	    }
	}
	*tptr = (event_len+5) | trig_pattern[step++];
	
	/* determine if there is more data to follow */
	if (more && event_len > 0) {
	    *tptr |= MORE_MASK;
	    tptr = ptr;
	    bptr = ptr + 1;
	    ptr += 5;
	}
	len = size - len;
	
    } while (more--);
    
    /* calculate full event length */
    event_len = ptr - fbuf;
    
    /* add event to output data buffer */
    if ( write_filt(fbuf, event_len) )
	return(-3);
    
    return(event_len);
}

/*
 * functions used in the filtering of data
 * write any data to end of filtered data block
 */
static void
filt_tail(
#if NeedFunctionPrototype
	  short *base, short *fptr)
#else
    base, fptr)
short *base;
short *fptr;
#endif
{
    unsigned int *p;

    fptr[1] = *fptr = 0;

    for(p = (unsigned int *) base; p < (unsigned int *) fptr; p++)
	*p = ((*p & 0xffff0000)>>16) | ((*p & 0x0000ffff)<<16);

    return;
}

/*
 *  write any data to the beginning of filtered data block
 */
static short *
filt_head
#if NeedFunctionPrototype
(short *base)
#else
(base)
short *base;
#endif
{
    memcpy((char *) base, "MEGHADAT", 8);
    return(base+12);
}

/* NSF filter data specification */
static filtdat_t nsf2_filtdat = {
    DL_OUTRECORD_LEN,
    DL_OUTRECORD_LEN/2,
    filt_head,
    filt_tail,
    write_nsf2,
    "Nsf2"
};

/* routine to get pointer to nsf2_filtdat */
filtdat_t *
get_nsf2_filtdat(
#if NeedFunctionPrototype
		void
#endif
		)
{
    return(&nsf2_filtdat);
}

