/*
 * spectrum handling routines
 *
 * prev modified 12/5/95
 * prev modified 23/11/95 C ordering in 2D arrays
 * prev modified 9/1/96 add routines to inc for real data SPGC
 * last modified 13/6/97 expand inc routines for real data, add set and val
 */

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <errno.h>
#include <memory.h>

#include <sys/types.h>
#include <sys/mman.h>

#ifdef SVR4
#define NeedFunctionPrototype 1
#else
#define NeedFunctionPrototype 0
#endif

#include "sort_thread.h"
#include "eg.h"
#include "spectra.h"

extern int DBX_val;

/* function defined in eg.c */
#if NeedFunctionPrototype
extern int *map_to_spectrum(int, char *, int, int);
extern int make_1d(int, int, char *, spec_entry_t *);
extern int make_2d(int, int, char *, spec_entry_t *);
extern int make_var(int, float, char *, spec_entry_t *);
extern spec_entry_t *get_spec_entry(int, int);
#else
extern int *map_to_spectrum();
extern int make_1d(), make_2d(), make_var();
extern spec_entry_t *get_spec_entry();
#endif

int
read_spectra_data
#if NeedFunctionPrototype
(void)
#else
()
#endif
{
    int i;
    spec_entry_t *sp;

    /* set up the 1d spectra */
    for(i=1; i< TAB_SIZE_1D; i++) {
	if ((sp = get_spec_entry(TYPE_1d, i)) == NULL) {
	    fprintf(stderr, "sort: no shared data entry for 1d spectrum %d\n",
		    i);
	    return(SORT_FAIL);
	}
	if (i != sp->nos)
	    continue;
	if (make_1d(i, sp->size, sp->name, sp) == SORT_FAIL)
	    return(SORT_FAIL);
	if(DBX_val >= 7)
	    fprintf(stderr, "sort: 1d spectrum %s nos %d with size %d\n",
		    sp->name, i, sp->size);
    }
    /* setup the 2d spectra */
    for(i=1; i< TAB_SIZE_2D; i++) {
	if ((sp = get_spec_entry(TYPE_2d, i)) == NULL) {
	    fprintf(stderr, "sort: no shared data entry for 2d spectrum %d\n",
		    i);
	    return(SORT_FAIL);
	}
	if (i != sp->nos)
	    continue;
	if (make_2d(i, sp->size, sp->name, sp) == SORT_FAIL)
	    return(SORT_FAIL);
	if(DBX_val >= 7)
	    fprintf(stderr, "sort: 2d spectrum %s nos %d with size %d\n",
		    sp->name, i, sp->size);
    }
    /* set up the variables */
    for(i=1; i< TAB_SIZE_VAR; i++) {
	if ((sp = get_spec_entry(TYPE_var, i)) == NULL) {
	    fprintf(stderr, "sort: no shared data entry for variable %d\n", i);
	    return(SORT_FAIL);
	}
	if (i != sp->nos)
	    continue;
	if (make_var(i, sp->value, sp->name, sp) == SORT_FAIL)
	    return(SORT_FAIL);
	if(DBX_val >= 7)
	    fprintf(stderr, "sort: variable %s nos %d with value %g\n",
		    sp->name, i, sp->value);
    }

    return(0);
}

/*
 * The elements of tab_?d contain pointers to the storage locations
 * of each spectrum structure. ie nos_tab_1d[1] points to the first
 * address location of the 1d spectrum structure for spectrum number
 * 1. Element [0] will point to the last structure to be filled
 * spectrum
 */
static spec_1d_t *tab_1d[TAB_SIZE_1D];
static spec_2d_t *tab_2d[TAB_SIZE_2D];
static var_str_t *tab_var[TAB_SIZE_VAR];

int
make_1d
#if NeedFunctionPrototype
(int spec_nos,
int spec_size,
char *spec_name,
spec_entry_t *spec_entry)
#else
(spec_nos, spec_size, spec_name, spec_entry)
int spec_nos;    /* spectrum number as defined by user */
int spec_size; /* spectrum size in units of (int) */
char *spec_name; /* spectrum name & file name of saved spectrum */
spec_entry_t *spec_entry;
#endif
{
    spec_1d_t *p;

    if (spec_nos <= 0 || spec_nos >= TAB_SIZE_1D) {
	fprintf(stderr, "make_1d: Invalid 1D spectrum number %d '%s'\n",
		spec_nos, spec_name);
	return(SORT_NOS_ERR);
    }

    p = (spec_1d_t *) malloc(sizeof(spec_1d_t));
    if (!p) {
	perror("make_1d");
	return(SORT_FAIL);
    }
    p->next = 0;

    /* enter data into structure and reserve memory for spectrum storage */
    if (tab_1d[spec_nos] == 0) /* each spectrum should have a unique number */
	tab_1d[spec_nos] = p;
    else {
	fprintf(stderr, "make_1d: entry for spectrum number %d already exists"
		" ... ignoring '%s'\n", spec_nos, spec_name);
	free(p);
	return (SORT_NOS_ERR);
    }
    p->nos = spec_nos;
    p->data = map_to_spectrum(1, spec_name, spec_nos, spec_size);
    if (!p->data) {
	perror("make_1d");
	return (SORT_FAIL);
    }
    p->sp = spec_entry;
    p->version = p->sp->version;

    /* make structure element next point to last entry ie to
       decreasing spec_nos */
    p->next = tab_1d[0];
    tab_1d[0] = p;
    return(0);
}

/* Almost Identical to make_1d */
int
make_2d
#if NeedFunctionPrototype
(int spec_nos,
int spec_size,
char *spec_name,
spec_entry_t *spec_entry)
#else
(spec_nos, spec_size, spec_name, spec_entry)
int spec_nos;    /* spectrum number as defined by user */
int spec_size; /* spectrum size in units of (int) */
char *spec_name; /* spectrum name & file name of saved spectrum */
spec_entry_t *spec_entry;
#endif
{
    spec_2d_t *p;

    if (spec_nos <= 0 || spec_nos >= TAB_SIZE_2D) {
	fprintf(stderr,
		"make_2d: Invalid 2D spectrum number %d ... ignoring '%s'\n",
		spec_nos, spec_name);
	return(SORT_NOS_ERR);
    }

    p = (spec_2d_t *) malloc(sizeof(spec_2d_t));
    if (!p) {
	perror("make_2d");
	return(SORT_FAIL);
    }
    p->next=0;

    /* enter data into structure and reserve memory for spectrum storage */
    if (tab_2d[spec_nos] == 0) /* each spectrum should have a unique number */
	tab_2d[spec_nos] = p;
    else {
	fprintf(stderr, "make_2d: entry for spectrum number %d already "
		"exists ... ignoring '%s'\n", spec_nos, spec_name);
	free(p);
	return (SORT_NOS_ERR);
    }
    p->nos = spec_nos;
    p->data = map_to_spectrum(2, spec_name, spec_nos, spec_size);
    if (!p->data) {
	perror("make_2d");
	return (SORT_FAIL);
    }
    p->sp = spec_entry;
    p->win = spec_entry->win;
    p->version = spec_entry->version;

    /* make structure element next point to last entry ie to
       decreasing spec_nos */
    p->next = tab_2d[0];
    tab_2d[0] = p;
    return(0);
}

/* Almost Identical to make_1d */
int
make_var
#if NeedFunctionPrototype
(int var_nos,
float var_value,
char *var_name,
spec_entry_t *var_entry)
#else
(var_nos, var_value, var_name, var_entry)
int var_nos;     /* variable number as defined by user */
float var_value; /* variable value real*4 */
char *var_name;  /* variable name */
spec_entry_t *var_entry;
#endif
{
    var_str_t *p;

    if (var_nos <= 0 || var_nos >= TAB_SIZE_VAR) {
	fprintf(stderr, "make_var: Invalid variable element %d '%s'\n",
		var_nos, var_name);
	return(SORT_NOS_ERR);
    }

    p = (var_str_t *) malloc(sizeof(var_str_t));
    if (!p) {
	perror("make_var");
	return(SORT_FAIL);
    }
    p->next=0;

    /* enter data into structure */
    if (tab_var[var_nos] == 0) /* each variable should have a unique number */
	tab_var[var_nos] = p;
    else {
	fprintf(stderr,
		"make_var: entry for variable element %d already exists "
		"... ignoring '%s'\n", var_nos, var_name);
	free(p);
	return (SORT_NOS_ERR);
    }
    p->nos = var_nos;
    p->sp = var_entry;
    p->version = var_entry->version;

    /* make structure element next point to last entry ie to
       decreasing var_nos */
    p->next = tab_var[0];
    tab_var[0] = p;
    return(0);
}


/*---routines to get pointers to various data tables---*/

#if NeedFunctionPrototype
spec_1d_t **
get_nametab_1d(void)
{
return(tab_1d);
}

spec_2d_t **
get_nametab_2d(void)
{
return(tab_2d);
}

var_str_t **
get_nametab_var(void)
{
return(tab_var);
}
#else
spec_1d_t **
get_nametab_1d()
{
return(tab_1d);
}

spec_2d_t **
get_nametab_2d()
{
return(tab_2d);
}

var_str_t **
get_nametab_var()
{
return(tab_var);
}
#endif

/*
 * Clear and delete local memory associated with spectrum and variables
 * data structures. Clear data in shared memory
 */

int
delete_1d
#if NeedFunctionPrototype
(int spec_nos)
#else
(spec_nos)
int spec_nos; /* spectrum number as defined by user */
#endif
{
    spec_1d_t *p1d;

    /* check to see that specified spectrum number is within bounds */
    if (spec_nos <= 0 || spec_nos >= TAB_SIZE_1D) {
	fprintf(stderr, "delete_1d (sort): Invalid 1D spectrum number %d\n",
		spec_nos);
	return(SORT_NOS_ERR);
    }

    /* check so see if spectrum entry exists */
    if (tab_1d[spec_nos] != 0) /* each spectrum should have a unique number */
	p1d = tab_1d[spec_nos];
    else {
	fprintf(stderr, "delete_1d (sort): entry for spectrum number %d does "
		"not exist ... ignoring\n", spec_nos);
	return (SORT_NOS_ERR);
    }

    /* free memory associated with spectrum data */
    if (unmap_spectrum(p1d->data, p1d->sp->size) == -1) {
	fprintf(stderr, "delete_1d (sort): failed to unmap old data\n");
	return(SORT_FAIL);
    }

    /* correct link list to account for removal of this element */
    if (p1d == tab_1d[0]) {
	tab_1d[0] = p1d->next;
    }
    else {
	spec_1d_t *entry;
	for (entry=tab_1d[0]; entry != 0; entry = entry->next) {
	    if (entry->next == p1d) {
		entry->next = p1d->next;
		break;
	    }
	}
    }

    /* clear local spectrum data structures */
    (void) memset(p1d, 0, sizeof(spec_1d_t));

    /* free entry in spectrum table */
    free(p1d);
    tab_1d[spec_nos] = 0;

    return(0);
}

int
delete_2d
#if NeedFunctionPrototype
(int spec_nos)
#else
(spec_nos)
int spec_nos; /* spectrum number as defined by user */
#endif
{
    spec_2d_t *p2d;

    /* check to see that specified spectrum number is within bounds */
    if (spec_nos <= 0 || spec_nos >= TAB_SIZE_2D) {
	fprintf(stderr, "delete_2d (sort): Invalid 2D spectrum number %d\n",
		spec_nos);
	return(SORT_NOS_ERR);
    }

    /* check so see if spectrum entry exists */
    if (tab_2d[spec_nos] != 0) /* each spectrum should have a unique number */
	p2d = tab_2d[spec_nos];
    else {
	fprintf(stderr, "delete_2d(sort): entry for spectrum number %d does "
		"not exist ... ignoring\n", spec_nos);
	return (SORT_NOS_ERR);
    }

    /* free memory associated with spectrum data */
    if (unmap_spectrum(p2d->data, p2d->sp->size*p2d->sp->size) == -1) {
	fprintf(stderr, "delete_2d(sort): failed to unmap old data\n");
	return(SORT_FAIL);
    }

    /* correct link list to account for removal of this element */
    if (p2d == tab_2d[0]) {
	tab_2d[0] = p2d->next;
	}
    else {
	spec_2d_t *entry;
	for (entry=tab_2d[0]; entry != 0; entry = entry->next) {
	    if (entry->next == p2d) {
		entry->next = p2d->next;
		break;
	    }
	}
    }

    /* clear local spectrum data structure */
    (void) memset(p2d, 0, sizeof(spec_2d_t));

    /* free entry in spectrum table */
    free(p2d);
    tab_2d[spec_nos] = 0;

    return(0);
}

int
delete_var
#if NeedFunctionPrototype
(int var_nos)
#else
(var_nos)
int var_nos; /* spectrum number as defined by user */
#endif
{
    var_str_t *var;

    /* check to see that specified spectrum number is within bounds */
    if (var_nos <= 0 || var_nos >= TAB_SIZE_VAR) {
	fprintf(stderr, "delete_var (sort): Invalid variable number %d\n",
		var_nos);
	return(SORT_NOS_ERR);
    }

    /* check so see if spectrum entry exists */
    if (tab_var[var_nos] != 0) /* each spectrum should have a unique number */
	var = tab_var[var_nos];
    else {
	fprintf(stderr, "delete_var (sort): entry for spectrum number %d "
		"does not exist ... ignoring\n", var_nos);
	return (SORT_NOS_ERR);
    }

    /* correct link list to account for removal of this element */
    if (var == tab_var[0]) {
	tab_var[0] = var->next;
    }
    else {
	var_str_t *entry;
	for (entry=tab_var[0]; entry != 0; entry = entry->next) {
	    if (entry->next == var) {
		entry->next = var->next;
		break;
	    }
	}
    }

    /* clear local spectrum data structure */
    (void) memset(var, 0, sizeof(var_str_t));

    /* free entry in spectrum table */
    free(var);
    tab_var[var_nos] = 0;

    return(0);
}

/****************************************************************************
 *           user callable spectrum incrementing subroutines etc.           *
 ****************************************************************************/

#include <limits.h>

void
inc1d
#if NeedFunctionPrototype
(register int spec_nos,
register int channel)
#else
(spec_nos, channel)
register int spec_nos;
register int channel;
#endif
{
    register spec_1d_t *ptr;

    if (spec_nos <= 0 || spec_nos >= TAB_SIZE_1D) {
	fprintf(stderr, "inc1d to invalid 1d spectrum number %d", spec_nos);
	return;
    }

    if (!(ptr = tab_1d[spec_nos])) {
	fprintf(stderr, "inc1d to undefined 1d spectrum nos %d\n", spec_nos);
	return;
    }

    if (channel < 0 || channel >= ptr->sp->size) {
	if (DBX_val >= 10)
	    fprintf(stderr, "inc1d to channel %d outside spectrum range\n",
		    channel);
	return;
    }
    if ((ptr->data[channel])++ == INT_MAX) {
	fprintf(stderr, "inc1d overflow at channel %d in spectrum %s\n",
		channel, ptr->sp->name);
	return;
    }
    return;
}

void
inc2d
#if NeedFunctionPrototype
(register int spec_nos,
register int x_ch,
register int y_ch)
#else
(spec_nos, x_ch, y_ch)
register int spec_nos;
register int x_ch;
register int y_ch;
#endif
{
    register spec_2d_t *ptr;

    if (spec_nos <= 0 || spec_nos >= TAB_SIZE_2D) {
	fprintf(stderr, "inc2d to invalid 2d spectrum number %d", spec_nos);
	return;
    }

    if (!(ptr = tab_2d[spec_nos])) {
	fprintf(stderr, "inc2d to undefined 2d spectrum nos %d\n", spec_nos);
	return;
    }

    if (x_ch < 0 || x_ch >= ptr->sp->size ||
	y_ch < 0 || y_ch >= ptr->sp->size) {
	if (DBX_val >= 10)
	    fprintf(stderr,
		    "inc2d to x=%d y=%d outside range of spectrum %s\n",
		    x_ch, y_ch, ptr->sp->name);
	return;
    }
    if ((ptr->data[(y_ch) + (x_ch)*(ptr->sp->size)])++ == INT_MAX) {
	fprintf(stderr, "inc2d overflow at x=%d, y=%d in spectrum %s\n",
		x_ch, y_ch, ptr->sp->name);
	return;
    }
    return;
}

void
incv1d
#if NeedFunctionPrototype
(register int spec_nos,
register int channel,
register int value)
#else
(spec_nos, channel, value)
register int spec_nos;
register int channel;
register int value;
#endif
{
    register spec_1d_t *ptr;

    if (spec_nos <= 0 || spec_nos >= TAB_SIZE_1D) {
	fprintf(stderr, "incv1d to invalid 1d spectrum number %d", spec_nos);
	return;
    }

    if (!(ptr = tab_1d[spec_nos])) {
	fprintf(stderr, "incv1d to undefined 1d spectrum nos %d\n", spec_nos);
	return;
    }

    if (channel < 0 || channel >= ptr->sp->size) {
	if (DBX_val >= 10)
	    fprintf(stderr, "incv1d to channel %d outside spectrum range\n",
		    channel);
	return;
    }
    if ((ptr->data[channel]) >= (INT_MAX - value)) {
	fprintf(stderr, "incv1d overflow at channel %d in spectrum %s\n",
		channel, ptr->sp->name);
	return;
    }
    ptr->data[channel] += value;
    return;
}

void
incv2d
#if NeedFunctionPrototype
(register int spec_nos,
register int x_ch,
register int y_ch,
register int value)
#else
(spec_nos, x_ch, y_ch, value)
register int spec_nos;
register int x_ch;
register int y_ch;
register int value;
#endif
{
    register spec_2d_t *ptr;

    if (spec_nos <= 0 || spec_nos >= TAB_SIZE_2D) {
	fprintf(stderr, "incv2d to invalid 2d spectrum number %d", spec_nos);
	return;
    }

    if (!(ptr = tab_2d[spec_nos])) {
	fprintf(stderr, "incv2d to undefined 2d spectrum nos %d\n", spec_nos);
	return;
    }

    if (x_ch < 0 || x_ch >= ptr->sp->size ||
	y_ch < 0 || y_ch >= ptr->sp->size) {
	if (DBX_val >= 10)
	    fprintf(stderr,
		    "incv2d to x=%d y=%d outside range of spectrum %s\n",
		    x_ch, y_ch, ptr->sp->name);
	return;
    }
    if ((ptr->data[(y_ch)+(x_ch)*(ptr->sp->size)]) >= (INT_MAX - value)) {
	fprintf(stderr, "incv2d overflow at x=%d, y=%d in spectrum %s\n",
		x_ch, y_ch, ptr->sp->name);
	return;
    }

    ptr->data[(y_ch)+(x_ch)*(ptr->sp->size)] += value;
    return;
}

int
win2d
#if NeedFunctionPrototype
(register int spec_nos,
register int x_ch,
register int y_ch)
#else
(spec_nos, x_ch, y_ch)
register int spec_nos;
register int x_ch;
register int y_ch;
#endif
{
    register spec_2d_t *ptr;

    if (spec_nos <= 0 || spec_nos >= TAB_SIZE_2D) {
	fprintf(stderr, "win2d to invalid 2d spectrum number %d\n", spec_nos);
	return SORT_FALSE;
    }

    if (!(ptr = tab_2d[spec_nos])) {
	fprintf(stderr, "win2d to undefined 2d spectrum nos %d\n", spec_nos);
	return SORT_FALSE;
    }

    if (!ptr->win) {
	fprintf(stderr, "win2d called with non window 2d spectrum nos %d\n",
		spec_nos);
	return SORT_FALSE;
    }

    if (x_ch < 0 || x_ch >= ptr->sp->size ||
	y_ch < 0 || y_ch >= ptr->sp->size) {
	if (DBX_val >= 10)
	    fprintf(stderr,
		    "win2d to x=%d y=%d outside range of spectrum %s\n",
		    x_ch, y_ch, ptr->sp->name);
	return SORT_FALSE;
    }

    return ((ptr->data[(y_ch)+(x_ch)*(ptr->sp->size)]) > 10) ? SORT_TRUE
	: SORT_FALSE;
}

void
set1d
#if NeedFunctionPrototype
(register int spec_nos,
register int channel,
register int value)
#else
(spec_nos, channel, value)
register int spec_nos;
register int channel;
register int value;
#endif
{
    register spec_1d_t *ptr;

    if (spec_nos <= 0 || spec_nos >= TAB_SIZE_1D) {
	fprintf(stderr, "set1d to invalid 1d spectrum number %d", spec_nos);
	return;
    }

    if (!(ptr = tab_1d[spec_nos])) {
	fprintf(stderr, "set1d to undefined 1d spectrum nos %d\n", spec_nos);
	return;
    }

    if (channel < 0 || channel >= ptr->sp->size) {
	if (DBX_val >= 10)
	    fprintf(stderr, "set1d to channel %d outside spectrum range\n",
		    channel);
	return;
    }
    ptr->data[channel] = value;
    return;
}

void
set2d
#if NeedFunctionPrototype
(register int spec_nos,
register int x_ch,
register int y_ch,
register int value)
#else
(spec_nos, x_ch, y_ch, value)
register int spec_nos;
register int x_ch;
register int y_ch;
register int value;
#endif
{
    register spec_2d_t *ptr;

    if (spec_nos <= 0 || spec_nos >= TAB_SIZE_2D) {
	fprintf(stderr, "set2d to invalid 2d spectrum number %d", spec_nos);
	return;
    }

    if (!(ptr = tab_2d[spec_nos])) {
	fprintf(stderr, "set2d to undefined 2d spectrum nos %d\n", spec_nos);
	return;
    }

    if (x_ch < 0 || x_ch >= ptr->sp->size ||
	y_ch < 0 || y_ch >= ptr->sp->size) {
	if (DBX_val >= 10)
	    fprintf(stderr,
		    "set2d to x=%d y=%d outside range of spectrum %s\n",
		    x_ch, y_ch, ptr->sp->name);
	return;
    }
    ptr->data[(y_ch)+(x_ch)*(ptr->sp->size)] = value;
    return;
}

int
val1d
#if NeedFunctionPrototype
(register int spec_nos,
register int channel)
#else
(spec_nos, channel)
register int spec_nos;
register int channel;
#endif
{
    register spec_1d_t *ptr;

    if (spec_nos <= 0 || spec_nos >= TAB_SIZE_1D) {
	fprintf(stderr, "val1d to invalid 1d spectrum number %d", spec_nos);
	return -1;
    }

    if (!(ptr = tab_1d[spec_nos])) {
	fprintf(stderr, "val1d to undefined 1d spectrum nos %d\n", spec_nos);
	return -1;
    }

    if (channel < 0 || channel >= ptr->sp->size) {
	if (DBX_val >= 10)
	    fprintf(stderr, "val1d to channel %d outside spectrum range\n",
		    channel);
	return -1;
    }
    return ptr->data[channel];
}

int
val2d
#if NeedFunctionPrototype
(register int spec_nos,
register int x_ch,
register int y_ch)
#else
(spec_nos, x_ch, y_ch)
register int spec_nos;
register int x_ch;
register int y_ch;
#endif
{
    register spec_2d_t *ptr;

    if (spec_nos <= 0 || spec_nos >= TAB_SIZE_2D) {
	fprintf(stderr, "val2d to invalid 2d spectrum number %d", spec_nos);
	return -1;
    }

    if (!(ptr = tab_2d[spec_nos])) {
	fprintf(stderr, "val2d to undefined 2d spectrum nos %d\n", spec_nos);
	return -1;
    }

    if (x_ch < 0 || x_ch >= ptr->sp->size ||
	y_ch < 0 || y_ch >= ptr->sp->size) {
	if (DBX_val >= 10)
	    fprintf(stderr,
		    "val2d to x=%d y=%d outside range of spectrum %s\n",
		    x_ch, y_ch, ptr->sp->name);
	return -1;
    }
    return ptr->data[(y_ch) + (x_ch)*(ptr->sp->size)];
}

/* FORTRAN equivalent routines */

void
inc1d_
#if NeedFunctionPrototype
(int *spec_nos,
int *channel)
#else
(spec_nos, channel)
int *spec_nos;
int *channel;
#endif
{
      inc1d(*spec_nos, *channel);
}

void
inc2d_
#if NeedFunctionPrototype
(int *spec_nos,
int *x_ch,
int *y_ch)
#else
(spec_nos, x_ch, y_ch)
int *spec_nos;
int *x_ch;
int *y_ch;
#endif
{
      inc2d(*spec_nos, *x_ch, *y_ch);
}

void
incv1d_
#if NeedFunctionPrototype
(int *spec_nos,
int *channel,
int *value)
#else
(spec_nos, channel, value)
int *spec_nos;
int *channel;
int *value;
#endif
{
      incv1d(*spec_nos, *channel, *value);
}

void
incv2d_
#if NeedFunctionPrototype
(int *spec_nos,
int *x_ch,
int *y_ch,
int *value)
#else
(spec_nos, x_ch, y_ch, value)
int *spec_nos;
int *x_ch;
int *y_ch;
int *value;
#endif
{
      incv2d(*spec_nos, *x_ch, *y_ch, *value);
}

void
win2d_
#if NeedFunctionPrototype
(int *spec_nos,
int *x_ch,
int *y_ch,
int *in_window)
#else
(spec_nos, x_ch, y_ch, in_window)
int *spec_nos;
int *x_ch;
int *y_ch;
int *in_window;
#endif
{
    *in_window = win2d(*spec_nos, *x_ch, *y_ch);
}

void
set1d_
#if NeedFunctionPrototype
(int *spec_nos,
int *channel,
int *value)
#else
(spec_nos, channel, value)
int *spec_nos;
int *channel;
int *value;
#endif
{
      set1d(*spec_nos, *channel, *value);
}

void
set2d_
#if NeedFunctionPrototype
(int *spec_nos,
int *x_ch,
int *y_ch,
int *value)
#else
(spec_nos, x_ch, y_ch, value)
int *spec_nos;
int *x_ch;
int *y_ch;
int *value;
#endif
{
      set2d(*spec_nos, *x_ch, *y_ch, *value);
}

int
val1d_
#if NeedFunctionPrototype
(int *spec_nos,
int *channel)
#else
(spec_nos, channel)
int *spec_nos;
int *channel;
#endif
{
      return val1d(*spec_nos, *channel);
}

int
val2d_
#if NeedFunctionPrototype
(int *spec_nos,
int *x_ch,
int *y_ch)
#else
(spec_nos, x_ch, y_ch)
int *spec_nos;
int *x_ch;
int *y_ch;
#endif
{
      return val2d(*spec_nos, *x_ch, *y_ch);
}

/* equivalance routines for sort-shell */
void
rinc_
#if NeedFunctionPrototype
(int *spec_nos,
int *channel)
#else
(spec_nos, channel)
int *spec_nos;
int *channel;
#endif
{
      inc1d(*spec_nos, *channel);
}

void
rinc2d_
#if NeedFunctionPrototype
(int *spec_nos,
int *x_ch,
int *y_ch)
#else
(spec_nos, x_ch, y_ch)
int *spec_nos;
int *x_ch;
int *y_ch;
#endif
{
      inc2d(*spec_nos, *x_ch, *y_ch);
}

/* FORTRAN routines that accept integer*2 data */

void
sinc1d_
#if NeedFunctionPrototype
(int *spec_nos,
short *channel)
#else
(spec_nos, channel)
int *spec_nos;
short *channel;
#endif
{
      inc1d(*spec_nos, (int) *channel);
}

void
sinc2d_
#if NeedFunctionPrototype
(int *spec_nos,
short *x_ch,
short *y_ch)
#else
(spec_nos, x_ch, y_ch)
int *spec_nos;
short *x_ch;
short *y_ch;
#endif
{
      inc2d(*spec_nos, (int) *x_ch, (int) *y_ch);
}

void
sincv1d_
#if NeedFunctionPrototype
(int *spec_nos,
short *channel,
short *value)
#else
(spec_nos, channel, value)
int *spec_nos;
short *channel;
short *value;
#endif
{
      incv1d(*spec_nos, (int) *channel, *value);
}

void
sincv2d_
#if NeedFunctionPrototype
(int *spec_nos,
short *x_ch,
short *y_ch,
short *value)
#else
(spec_nos, x_ch, y_ch, value)
int *spec_nos;
short *x_ch;
short *y_ch;
short *value;
#endif
{
      incv2d(*spec_nos, (int) *x_ch, (int) *y_ch, *value);
}

void
swin2d_
#if NeedFunctionPrototype
(int *spec_nos,
short *x_ch,
short *y_ch,
int *in_window)
#else
(spec_nos, x_ch, y_ch, in_window)
int *spec_nos;
short *x_ch;
short *y_ch;
int *in_window;
#endif
{
    *in_window = win2d(*spec_nos, (int) *x_ch, (int) *y_ch);
}

void
sset1d_
#if NeedFunctionPrototype
(int *spec_nos,
short *channel,
short *value)
#else
(spec_nos, channel, value)
int *spec_nos;
short *channel;
short *value;
#endif
{
      set1d(*spec_nos, (int) *channel, *value);
}

void
sset2d_
#if NeedFunctionPrototype
(int *spec_nos,
short *x_ch,
short *y_ch,
short *value)
#else
(spec_nos, x_ch, y_ch, value)
int *spec_nos;
short *x_ch;
short *y_ch;
short *value;
#endif
{
      set2d(*spec_nos, (int) *x_ch, (int) *y_ch, *value);
}

int
sval1d_
#if NeedFunctionPrototype
(int *spec_nos,
short *channel)
#else
(spec_nos, channel)
int *spec_nos;
short *channel;
#endif
{
      return val1d(*spec_nos, (int) *channel);
}

int
sval2d_
#if NeedFunctionPrototype
(int *spec_nos,
short *x_ch,
short *y_ch)
#else
(spec_nos, x_ch, y_ch)
int *spec_nos;
short *x_ch;
short *y_ch;
#endif
{
      return val2d(*spec_nos, (int) *x_ch, (int) *y_ch);
}

/* C routines that accept double data */

void
dinc1d
#if NeedFunctionPrototype
(int spec_nos,
double channel)
#else
(spec_nos, channel)
int spec_nos;
double channel;
#endif
{
      inc1d(spec_nos, (int) (channel + 0.5));
}

void
dinc2d
#if NeedFunctionPrototype
(int spec_nos,
double x_ch,
double y_ch)
#else
(spec_nos, x_ch, y_ch)
int spec_nos;
double x_ch;
double y_ch;
#endif
{
      inc2d(spec_nos, (int) (x_ch + 0.5), (int) (y_ch + 0.5));
}

void
dincv1d
#if NeedFunctionPrototype
(int spec_nos,
double channel,
double value)
#else
(spec_nos, channel, value)
int spec_nos;
double channel;
double value;
#endif
{
      incv1d(spec_nos, (int) (channel + 0.5), value);
}

void
dincv2d
#if NeedFunctionPrototype
(int spec_nos,
double x_ch,
double y_ch,
double value)
#else
(spec_nos, x_ch, y_ch, value)
int spec_nos;
double x_ch;
double y_ch;
double value;
#endif
{
      incv2d(spec_nos, (int) (x_ch + 0.5), (int) (y_ch + 0.5), value);
}

int
dwin2d
#if NeedFunctionPrototype
(int spec_nos,
double x_ch,
double y_ch)
#else
(spec_nos, x_ch, y_ch)
int spec_nos;
double x_ch;
double y_ch;
#endif
{
    return win2d(spec_nos, (int) (x_ch + 0.5), (int) (y_ch + 0.5));
}

void
dset1d
#if NeedFunctionPrototype
(int spec_nos,
double channel,
double value)
#else
(spec_nos, channel, value)
int spec_nos;
double channel;
double value;
#endif
{
      set1d(spec_nos, (int) (channel + 0.5), value);
}

void
dset2d
#if NeedFunctionPrototype
(int spec_nos,
double x_ch,
double y_ch,
double value)
#else
(spec_nos, x_ch, y_ch, value)
int spec_nos;
double x_ch;
double y_ch;
double value;
#endif
{
      set2d(spec_nos, (int) (x_ch + 0.5), (int) (y_ch + 0.5), value);
}

int
dval1d
#if NeedFunctionPrototype
(int spec_nos,
double channel)
#else
(spec_nos, channel)
int spec_nos;
double channel;
#endif
{
      return val1d(spec_nos, (int) (channel + 0.5));
}

int
dval2d
#if NeedFunctionPrototype
(int spec_nos,
double x_ch,
double y_ch)
#else
(spec_nos, x_ch, y_ch)
int spec_nos;
double x_ch;
double y_ch;
#endif
{
      return val2d(spec_nos, (int) (x_ch + 0.5), (int) (y_ch + 0.5));
}

/* FORTRAN routines that accept real data */

void
finc1d_
#if NeedFunctionPrototype
(int *spec_nos,
float *channel)
#else
(spec_nos, channel)
int *spec_nos;
float *channel;
#endif
{
      inc1d(*spec_nos, (int) (*channel + 0.5));
}

void
finc2d_
#if NeedFunctionPrototype
(int *spec_nos,
float *x_ch,
float *y_ch)
#else
(spec_nos, x_ch, y_ch)
int *spec_nos;
float *x_ch;
float *y_ch;
#endif
{
      inc2d(*spec_nos, (int) (*x_ch + 0.5), (int) (*y_ch + 0.5));
}

void
fincv1d_
#if NeedFunctionPrototype
(int *spec_nos,
float *channel,
float *value)
#else
(spec_nos, channel, value)
int *spec_nos;
float *channel;
float *value;
#endif
{
      incv1d(*spec_nos, (int) (*channel + 0.5), *value);
}

void
fincv2d_
#if NeedFunctionPrototype
(int *spec_nos,
float *x_ch,
float *y_ch,
float *value)
#else
(spec_nos, x_ch, y_ch, value)
int *spec_nos;
float *x_ch;
float *y_ch;
float *value;
#endif
{
      incv2d(*spec_nos, (int) (*x_ch + 0.5), (int) (*y_ch + 0.5), *value);
}

void
fwin2d_
#if NeedFunctionPrototype
(int *spec_nos,
float *x_ch,
float *y_ch,
int *in_window)
#else
(spec_nos, x_ch, y_ch, in_window)
int *spec_nos;
float *x_ch;
float *y_ch;
int *in_window;
#endif
{
    *in_window = win2d(*spec_nos, (int) (*x_ch + 0.5), (int) (*y_ch + 0.5));
}

void
fset1d_
#if NeedFunctionPrototype
(int *spec_nos,
float *channel,
float *value)
#else
(spec_nos, channel, value)
int *spec_nos;
float *channel;
float *value;
#endif
{
      set1d(*spec_nos, (int) (*channel + 0.5), *value);
}

void
fset2d_
#if NeedFunctionPrototype
(int *spec_nos,
float *x_ch,
float *y_ch,
float *value)
#else
(spec_nos, x_ch, y_ch, value)
int *spec_nos;
float *x_ch;
float *y_ch;
float *value;
#endif
{
      set2d(*spec_nos, (int) (*x_ch + 0.5), (int) (*y_ch + 0.5), *value);
}

int
fval1d_
#if NeedFunctionPrototype
(int *spec_nos,
float *channel)
#else
(spec_nos, channel)
int *spec_nos;
float *channel;
#endif
{
      return val1d(*spec_nos, (int) (*channel + 0.5));
}

int
fval2d_
#if NeedFunctionPrototype
(int *spec_nos,
float *x_ch,
float *y_ch)
#else
(spec_nos, x_ch, y_ch)
int *spec_nos;
float *x_ch;
float *y_ch;
#endif
{
      return val2d(*spec_nos, (int) (*x_ch + 0.5), (int) (*y_ch + 0.5));
}

/* FORTRAN routines that accept double precision data */

void
dinc1d_
#if NeedFunctionPrototype
(int *spec_nos,
double *channel)
#else
(spec_nos, channel)
int *spec_nos;
double *channel;
#endif
{
      inc1d(*spec_nos, (int) (*channel + 0.5));
}

void
dinc2d_
#if NeedFunctionPrototype
(int *spec_nos,
double *x_ch,
double *y_ch)
#else
(spec_nos, x_ch, y_ch)
int *spec_nos;
double *x_ch;
double *y_ch;
#endif
{
      inc2d(*spec_nos, (int) (*x_ch + 0.5), (int) (*y_ch + 0.5));
}

void
dincv1d_
#if NeedFunctionPrototype
(int *spec_nos,
double *channel,
double *value)
#else
(spec_nos, channel, value)
int *spec_nos;
double *channel;
double *value;
#endif
{
      incv1d(*spec_nos, (int) (*channel + 0.5), *value);
}

void
dincv2d_
#if NeedFunctionPrototype
(int *spec_nos,
double *x_ch,
double *y_ch,
double *value)
#else
(spec_nos, x_ch, y_ch, value)
int *spec_nos;
double *x_ch;
double *y_ch;
double *value;
#endif
{
      incv2d(*spec_nos, (int) (*x_ch + 0.5), (int) (*y_ch + 0.5), *value);
}

void
dwin2d_
#if NeedFunctionPrototype
(int *spec_nos,
double *x_ch,
double *y_ch,
int *in_window)
#else
(spec_nos, x_ch, y_ch, in_window)
int *spec_nos;
double *x_ch;
double *y_ch;
int *in_window;
#endif
{
    *in_window = win2d(*spec_nos, (int) (*x_ch + 0.5), (int) (*y_ch + 0.5));
}

void
dset1d_
#if NeedFunctionPrototype
(int *spec_nos,
double *channel,
double *value)
#else
(spec_nos, channel, value)
int *spec_nos;
double *channel;
double *value;
#endif
{
      set1d(*spec_nos, (int) (*channel + 0.5), *value);
}

void
dset2d_
#if NeedFunctionPrototype
(int *spec_nos,
double *x_ch,
double *y_ch,
double *value)
#else
(spec_nos, x_ch, y_ch, value)
int *spec_nos;
double *x_ch;
double *y_ch;
double *value;
#endif
{
      set2d(*spec_nos, (int) (*x_ch + 0.5), (int) (*y_ch + 0.5), *value);
}

int
dval1d_
#if NeedFunctionPrototype
(int *spec_nos,
double *channel)
#else
(spec_nos, channel)
int *spec_nos;
double *channel;
#endif
{
      return val1d(*spec_nos, (int) (*channel + 0.5));
}

int
dval2d_
#if NeedFunctionPrototype
(int *spec_nos,
double *x_ch,
double *y_ch)
#else
(spec_nos, x_ch, y_ch)
int *spec_nos;
double *x_ch;
double *y_ch;
#endif
{
      return val2d(*spec_nos, (int) (*x_ch + 0.5), (int) (*y_ch + 0.5));
}

/*------------------------------------------------------------------------*/

/*
 * note the EG_MY_DATA_OFFSET constant ... memory mapping is very
 * picky about the page boundaries (here the ptr->data points
 * to the start of the data not the start address given by mmap call)
 */
int
msync_1d
#if NeedFunctionPrototype
(spec_1d_t *ptr, int flags)
#else
(ptr, flags)
spec_1d_t *ptr;
int flags;
#endif
{
    register int size = ptr->sp->size*sizeof(int);
    register char *addr = (char *)ptr->data - EG_MY_DATA_OFFSET;

    return(msync((caddr_t)addr, size, flags));
}

int
msync_2d
#if NeedFunctionPrototype
(spec_2d_t *ptr, int flags)
#else
(ptr, flags)
spec_2d_t *ptr;
int flags;
#endif
{
    register int size = ptr->sp->size * ptr->sp->size*sizeof(int);
    register char *addr = (char *)ptr->data - EG_MY_DATA_OFFSET;

    return(msync((caddr_t)addr, size, flags));
}


void
sync_spectra
#if NeedFunctionPrototype
(void)
#else
()
#endif
{
    spec_1d_t *p1d = get_nametab_1d()[0];
    spec_2d_t *p2d = get_nametab_2d()[0];

    while (p1d) {
	(void) msync_1d(p1d, 0);
	p1d = p1d->next;
    }
    while (p2d) {
	(void) msync_2d(p2d, 0);
	p2d = p2d->next;
    }
}

int
delete_spectra
#if NeedFunctionPrototype
(void)
#else
()
#endif
{
    spec_1d_t *p1d;
    spec_2d_t *p2d;
    var_str_t *var;
    int nos;
    int ret_code = 0;

    /* get pointer to 1d spectrum storage table */
    p1d = get_nametab_1d()[0];
    p2d = get_nametab_2d()[0];
    var = get_nametab_var()[0];

    /* delete all the one dimensional spectra */
    while (p1d) {
	nos = p1d->nos;
	p1d = p1d->next;
	if (delete_1d(nos) == SORT_FAIL) {
	    fprintf(stderr, "delete_spectra (sort): failed to delete 1d "
		    "spectrum number %d\n", nos);
	    ret_code += 0x1;
	    break;
	}
    }

    /* delete all the two dimensional spectra */
    while (p2d) {
	nos = p2d->nos;
	p2d = p2d->next;
	if (delete_2d(nos) == SORT_FAIL) {
	    fprintf(stderr, "delete_spectra (sort): failed to delete 2d "
		    "spectrum number %d\n", nos);
	    ret_code += 0x2;
	    break;
	}
    }

    /* delete all the variables */
    while (var) {
	nos = var->nos;
	var = var->next;
	if (delete_var(nos) == SORT_FAIL) {
	    fprintf(stderr, "delete_spectra (sort): failed to delete "
		    "variable nos %d\n", nos);
	    ret_code += 0x4;
	    break;
	}
    }
    return(ret_code);
}


int
remake_spectra
#if NeedFunctionPrototype
(void)
#else
()
#endif
{
    spec_1d_t *p1d;
    spec_2d_t *p2d;
    /* var_str_t *var; */
    spec_entry_t *sp;
    int nos;
    int update;
    int ret_code = 0;

    /* get pointer to 1d spectrum storage table */
    p1d = get_nametab_1d()[0];
    p2d = get_nametab_2d()[0];
    /* var = get_nametab_var()[0]; */
    
    /* delete all the one dimensional spectra */
    while (p1d) {
	nos = p1d->nos;
	update = (p1d->version != p1d->sp->version);
	p1d = p1d->next;
	
	if (update) {
	    if (DBX_val >= 7)
		fprintf(stderr, "sort: remaking 1d spectra number %d\n", nos);
	    if (delete_1d(nos) == SORT_FAIL) {
		fprintf(stderr,
			"sort: failed to delete 1d spectrum number %d\n", nos);
		ret_code += 0x1;
		break;
	    }
	    if ((sp = get_spec_entry(TYPE_1d, nos)) == NULL) {
		fprintf(stderr,
			"sort: no shared data entry for 1d spectrum %d\n",
			nos);
		ret_code += 0x1;
		break;
	    }
	    if (nos != sp->nos)
		continue;
	    if (make_1d(nos, sp->size, sp->name, sp) == SORT_FAIL) {
		ret_code += 0x1;
		break;
	    }
	    if(DBX_val >= 7)
		fprintf(stderr,
			"sort: remade 1d spectrum %s nos %d with size %d\n",
			sp->name, nos, sp->size);
	}
    }
    
    /* delete all the two dimensional spectra */
    while (p2d) {
	nos = p2d->nos;
	update = (p2d->version != p2d->sp->version);
	p2d = p2d->next;
	
	if (update) {
	    if (DBX_val >= 7)
		fprintf(stderr, "sort: remaking 2d spectra number %d\n", nos);
	    if (delete_2d(nos) == SORT_FAIL) {
		fprintf(stderr,
			"sort: failed to delete 2d spectrum number %d\n", nos);
		ret_code += 0x2;
		break;
	    }
	    if ((sp = get_spec_entry(TYPE_2d, nos)) == NULL) {
		fprintf(stderr,
			"sort: no shared data entry for 2d spectrum %d\n",
			nos);
		ret_code += 0x2;
		break;
	    }
	    if (nos != sp->nos)
		continue;
	    if (make_2d(nos, sp->size, sp->name, sp) == SORT_FAIL) {
		ret_code += 0x2;
		break;
	    }
	    if(DBX_val >= 7)
		fprintf(stderr, "sort: remade 2d spectrum %s nos %d with "
			"size %d win = %d\n",
			sp->name, nos, sp->size, sp->win);
	}
    }
    
    return(ret_code);
}
