/*
 *    eurogam spectrum format emulation routines
 *    C ordering adopted for 2D spectum arrays 
 *
 *    prev modified: 19/5/95
 *    last modified: 22/11/95
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#ifdef SVR4
#define NeedFunctionPrototype 1
static void init_default_header(void);
#else
#define NeedFunctionPrototype 0
static void init_default_header();
#endif

/* based upon compatibility library for porting existing NSF fortran 
   applications to EuroGam system */

/* DB, November 1991
   SJAH, April 1994 */

/* the functions sname, edef, ename, eread, ewrite,  emulate the
   similarly named functions in the OS4000 NSFxLIB libraries

   eexpt adds the 'experiment' concept */

#include "sort_thread.h"
#include "eg.h"

#define FUNCTION
#define then
#define doo
#define NONAMES 8
#define MAXNAMELEN 256
#define NULL 0

#define MAXREAD 8000
#define MAXWRITE 8000

char names[NONAMES][MAXNAMELEN];
static init_state = 0;
extern char *tmp_spectrum_dir;

void FUNCTION
eexpt(
#if NeedFunctionPrototype
		    int *ierr)
#else
      ierr)
      int  *ierr;
#endif
{
/* connect to the experiment defined by the EXPERIMENTDIR environment
   variable */

#ifdef EG_ENVIR
   int res;
   char *dir;
   if (init_state == 1) {
	 *ierr = 0;
	 return;
   }
   *ierr= -1;
   dir=(char *)getenv ("EXPERIMENTDIR");
   if (dir==NULL) then return;
   *ierr=EGsetExperimentPath (dir);
   if (*ierr != 0) then return;
   *ierr=EGconnectExperiment ("capabilities");
   init_state = 1;
   return;
#else
   if (init_state == 0) init_default_header();
   *ierr = 0;
   return;
#endif
 }


int FUNCTION
ename(
#if NeedFunctionPrototype
		   int *id,
		   char *name,
		   int namelen)
#else
      id, name, namelen)
      int *id;
      char *name;
      int namelen;
#endif
{
   int i;
   int ierr= -1;
   if (*id==0 && namelen>0) then {
      i=1;
      while (*names[i]!='\0') {i++; if (i>=NONAMES) then return(ierr);}
      *id=i;
      memcpy (names[i],name,namelen);
      names[i][namelen]='\0';
      }
   else if (*id!=0 && namelen==0) then {
      if (*id<=0 || *id >=NONAMES) then return(ierr);
      *names[*id]='\0';
      }      
   else if (*id==0 && namelen==0) then {
      i=1;
      while (i<NONAMES) doo { *names[i]='\0'; i++;}
      }         
   else { return(ierr); }
   return(0);
   }


int  FUNCTION
edef (
#if NeedFunctionPrototype
		    int *id,
		    int base1,
		    int range1,
		    int base2,
		    int range2,
		    int precision,
		    char *name,
		    int namelen)
#else
      id, base1, range1, base2, range2, precision, name, namelen)
      int *id;
      int base1;
      int range1;
      int base2;
      int range2;
      int precision;
      char *name;
      int namelen;
#endif
{
   int i = 1, type;
   int ierr = -1;
   while (*names[i]!='\0') {i++; if (i>=NONAMES) then return(-1);}
   *id=i;
   memcpy (names[i],name,namelen);
   names[i][namelen]='\0';
   type=(precision==2) ? 3 : 5;
   if (range2==0) 
#ifdef EG_ENVIR
	 ierr=EGcreate1dSpectrum (names[*id], base1, range1, 0, type);
   else
         ierr=EGcreate2dSpectrum (names[*id], base1, range1, base2, range2, 0, type);
#else
 	 ierr=SMEGcreate1dSpectrum (names[*id], base1, range1, 0, type);
   else
         ierr=SMEGcreate2dSpectrum (names[*id], base1, range1, base2, range2, 0, type);
#endif
   if (ierr != 0) then {*names[*id]='\0'; *id=0; }
   return ierr;
   }

int  FUNCTION
einq (
#if NeedFunctionPrototype
		    int id,
		    int *base1,
		    int *range1,
		    int *base2,
		    int *range2, 
		    int *precision)
#else
      id, base1, range1, base2, range2, precision)
      int id;
      int *base1;
      int *range1;
      int *base2;
      int *range2; 
      int *precision;
#endif
{
   int bases[8], ranges[8], layouts[2], types[2], dim;
   int ierr=-1;   
   if (id <=0 || id >=NONAMES) then return(ierr);
#ifdef EG_ENVIR
   ierr=EGinquireSpectrum (names[id], &dim, bases, ranges,layouts,types);
#else
   ierr=SMEGinquireSpectrum (names[id], &dim, bases, ranges,layouts,types);
#endif
   if (ierr!=0) then return(ierr);
   *base2=*range2=0;
   *base1=bases[0]; *range1=ranges[0];
   if (dim>1) then {*base2=bases[1]; *range2=ranges[1];}
   *precision=(types[0]==3 ? 2 : 4);
   return(ierr=0);
   }

int  FUNCTION eread (
#if NeedFunctionPrototype
		     int id,
		     int base1,
		     int range1,
		     int base2,
		     int range2, 
		     int *array,
		     int precision)
#else
      id, base1, range1, base2, range2, array, precision)
      int id;
      int base1;
      int range1;
      int base2;
      int range2; 
      int *array;
      int precision;
#endif
{
   int type, ierr;
   type=(precision==2) ? 3 : 5;
   ierr=-1;   
   if (id <=0 || id >=NONAMES) then return(ierr);
   if (range2==0) then
#ifdef EG_ENVIR	 
      ierr=EGread1dSpectrum (names[id], base1, range1, array, type);
   else
      ierr=EGread2dSpectrum (names[id], base1, range1, base2, range2, array, type);
#else
      ierr=SMEGread1dSpectrum (names[id], base1, range1, array, type);
   else
      ierr=SMEGread2dSpectrum (names[id], base1, range1, base2, range2, array, type);
  
#endif
   return(ierr);
   }

int  FUNCTION
ewrit(
#if NeedFunctionPrototype
		      int id,
		      int base1,
		      int range1,
		      int base2,
		      int range2, 
		      int *array,
		      int precision)
#else
      id, base1, range1, base2, range2, array, precision)
      int id;
      int base1;
      int range1;
      int base2;
      int range2; 
      int *array;
      int precision;
#endif
{
   int type, ierr;
   type=(precision==2) ? 3 : 5;
   if (id <=0 || id >=NONAMES) then return(-1);
   if (range2==0) then
#ifdef EG_ENVIR	 
      ierr=EGwrite1dSpectrum (names[id], base1, range1, array, type);
   else 
      ierr=EGwrite2dSpectrum (names[id], base1, range1, base2, range2, array, type);
#else
      ierr=SMEGwrite1dSpectrum (names[id], base1, range1, array, type);
   else 
      ierr=SMEGwrite2dSpectrum (names[id], base1, range1, base2, range2, array, type);
#endif   
   return(ierr);
   }

static SPECTRUM_HDR eg_spec;

static int FUNCTION typetowidth(
#if NeedFunctionPrototype	
				int type)
#else
      type)
      int  type;
#endif
{
   static int widths[]={1,1,2,2,4,4,4};
   return widths[type];
}

static void
init_default_header(
#if NeedFunctionPrototype
			 void
#endif
			 )
{
      int i;
      eg_spec.magic = 412900921;
      eg_spec.version = 1;
      strncpy (eg_spec.creation, "                    ",20);
      strncpy (eg_spec.modification, "                    ",20);
      for ( i=0; i<8; i++) {
	    eg_spec.bases[i] = eg_spec.ranges[i] = -1;
	    eg_spec.annotation[i] = eg_spec.calibration[i] = eg_spec.efficiency[i] = -1;
      }
      for ( i=0; i<32; i++)
	    eg_spec.information[i] = -1;
      eg_spec.data_array2.pointer = -1;
      eg_spec.str_base_addr = 512;
      eg_spec.str_top_addr = 255;
      eg_spec.count_base_addr = EG_MY_DATA_OFFSET;
      init_state = 1;
      return;
}
      
#ifndef EG_ENVIR
/*
 *    for those of us who live in a eurogam free zone
 *    partial inplementation for use in sunsort only
 */

int FUNCTION
SMEGcreate1dSpectrum(
#if NeedFunctionPrototype
				char *name,
				int   base,
				int   range,
				int   layout,
				int   type)
#else
      name, base, range, layout, type)
      char *name;
      int   base;
      int   range;
      int   layout;
      int   type;
#endif
{
      SPECTRUM_HDR sp;
      FILE     *fp, *fz;
      int width = typetowidth(type);
      
      (void) memcpy((char *) &sp, (char *) &eg_spec, sizeof(SPECTRUM_HDR));

      sp.dim=1;
      sp.bases[0]=base;
      sp.ranges[0]=range;
      sp.data_array1.layout = sp.data_array2.layout = layout;
      sp.data_array1.type = sp.data_array2.type = type;
      sp.count_free_addr = range * typetowidth(type);
      sp.count_top_addr = sp.count_free_addr - 1;
      
      if ( (fp = fopen(name,"w")) == NULL) {
	    return(-1);
      }
      if ( fwrite((char *) &sp, sizeof(SPECTRUM_HDR), 1, fp) != 1) {
	    return(-1);
      }
      (void) fseek(fp, sp.count_base_addr, 0);
      /* set all zeros in spectrum initially */
      if ( (fz = fopen("/dev/zero","r")) != NULL) {
	    int bytes = width * range;
	    char *zptr = (char *) mmap(0, (size_t)bytes, PROT_READ, MAP_PRIVATE, fileno(fz), 0);
	    if (zptr != MAP_FAILED) {
		  if (fwrite((char *) zptr, bytes, 1, fp) != 1)
			fprintf(stderr,"Error in SMEGcreate1dSpectrum: mapping to /dev/zero\n");
		  (void) munmap(zptr, bytes);
	    }
	    (void) fclose(fz);
      }
      
      (void) fclose(fp);

      return(0);
}

int FUNCTION
SMEGcreate2dSpectrum(
#if NeedFunctionPrototype
				  char *name,
				  int   base1,
				  int   range1,
				  int   base2,
				  int   range2,
				  int   layout,
				  int   type)
#else
      name, base1, range1, base2, range2, layout, type)
      char *name;
      int   base1;
      int   range1;
      int   base2;
      int   range2;
      int   layout;
      int   type;
#endif
{
      SPECTRUM_HDR sp;
      FILE     *fp, *fz;
      int width = typetowidth(type);
      int size = range1*range2;
      
      (void) memcpy((char *) &sp, (char *) &eg_spec, sizeof(SPECTRUM_HDR));
      
      sp.dim=2;
      sp.bases[0]=base1;
      sp.ranges[0]=range1;
      sp.bases[1]=base2;
      sp.ranges[1]=range2;
      sp.data_array1.layout = sp.data_array2.layout = layout;
      sp.data_array1.type = sp.data_array2.type = type;
      sp.count_free_addr = range1 * width * range2 * width;
      sp.count_top_addr = sp.count_free_addr - 1;

      if ( (fp = fopen(name,"w")) == NULL) {
	    return(-1);
      }
      if ( fwrite((char *) &sp, sizeof(SPECTRUM_HDR), 1, fp) != 1) {
	    (void) fclose(fp);
	    return(-1);
      }
      (void) fseek(fp, sp.count_base_addr, 0);
      /* set all zeros in spectrum initially */
      if ( (fz = fopen("/dev/zero","r")) != NULL) {
	    int bytes = width * size;
	    char *zptr = (char *) mmap(0, (size_t)bytes, PROT_READ, MAP_PRIVATE, fileno(fz), 0);
	    if (zptr != MAP_FAILED) {
		  if (fwrite((char *) zptr, bytes, 1, fp) != 1)
			fprintf(stderr,"Error in SMEGcreate2dSpectrum: mapping to /dev/zero\n");
		  (void) munmap(zptr, bytes);
	    }
	    (void) fclose(fz);
      }      

      (void) fclose(fp);

      return(0);
}


int FUNCTION
SMEGinquireSpectrum (
#if NeedFunctionPrototype	     
		     char *name,
		     int *dim,
		     int *bases,
		     int *ranges,
		     int *layouts,
		     int *types)
#else
      name, dim, bases, ranges, layouts, types)
      char *name;
      int *dim;
      int *bases;
      int *ranges;
      int *layouts;
      int *types;
#endif
{
      SPECTRUM_HDR sp;
      FILE     *fp;
      int      i;

      if ( (fp = fopen(name,"r")) == NULL) {
	    return(-1);
      }
      if ( fread((char *) &sp, sizeof(SPECTRUM_HDR), 1, fp) != 1) {
	    return(-1);
      }
      (void) fclose(fp);

      if (sp.magic != 412900921)
	    return(-1);
      
      *dim=sp.dim;
      for (i=0; i<sp.dim; i++) doo {
	    bases[i]=sp.bases[i];
	    ranges[i]=sp.ranges[i];
      }
      layouts[0]=sp.data_array1.layout;
      types[0]=sp.data_array1.type;
      layouts[1]=sp.data_array2.layout;
      types[1]=sp.data_array2.type;
      return(0);
   }


/*
 *    need to further develop SMEG routines to take into account bases
 */
int FUNCTION
SMEGread2dSpectrum(
#if NeedFunctionPrototype					
				char *name,
				int base1,
				int range1,
				int base2,
				int range2, 
				int *array,
				int type)
#else
      name, base1, range1, base2, range2, array, type)
      char *name;
      int base1;
      int range1;
      int base2;
      int range2; 
      int *array;
      int type;
#endif

{
      FILE  *fp;
      int   counts_base_addr;
      int   range = range1 * range2;
      int   bytes = range * typetowidth(type);

      if ( (fp = fopen(name,"r+")) == NULL) {
	    return(-1);
      }
      (void) fseek(fp, 424, 0);
      if ( fread((char *) &counts_base_addr, sizeof(int), 1, fp) != 1) {
	    (void) fclose(fp);
	    return(-1);
      }
      (void) fseek(fp, counts_base_addr, 0);
      if ( fread((char *) array, bytes, 1, fp) != 1) {
	    (void) fclose(fp);
	    return(-1);
      }
      (void) fclose(fp);

      return(0);

}

int FUNCTION
SMEGwrite2dSpectrum(
#if NeedFunctionPrototype				 
				 char *name,
				 int base1,
				 int range1,
				 int base2,
				 int range2, 
				 int *array,
				 int type)
#else
      name, base1, range1, base2, range2, array, type)
      char *name;
      int base1;
      int range1;
      int base2;
      int range2; 
      int *array;
      int type;
#endif
{
      FILE  *fp;
      int   counts_base_addr;
      int   range = range1 * range2;
      int   bytes = range * typetowidth(type);

      if ( (fp = fopen(name,"r+")) == NULL) {
	    return(-1);
      }
      (void) fseek(fp, 424, 0);
      if ( fread((char *) &counts_base_addr, sizeof(int), 1, fp) != 1) {
	    (void) fclose(fp);
	    return(-1);
      }
      (void) fseek(fp, counts_base_addr, 0);
      if ( fwrite((char *) array, bytes, 1, fp) != 1) {
	    (void) fclose(fp);
	    return(-1);
      }
      (void) fclose(fp);

      return(0);

}

int FUNCTION
SMEGread1dSpectrum(
#if NeedFunctionPrototype	
				char *name,
				int base,
				int range,
				int *array,
				int type)
#else
      name, base, range, array, type)
      char *name;
      int base;
      int range;
      int *array;
      int type;
#endif
{
      FILE  *fp;
      int   counts_base_addr;
      int   bytes = typetowidth(type) * range;

      if ( (fp = fopen(name,"r+")) == NULL) {
	    return(-1);
      }
      (void) fseek(fp, 424, 0);
      if ( fread((char *) &counts_base_addr, sizeof(int), 1, fp) != 1) {
	    (void) fclose(fp);
	    return(-1);
      }
      (void) fseek(fp, counts_base_addr, 0);
      if ( fread((char *) array, bytes, 1, fp) != 1) {
	    (void) fclose(fp);
	    return(-1);
      }
      (void) fclose(fp);

      return(0);

}

int FUNCTION
SMEGwrite1dSpectrum(
#if NeedFunctionPrototype	
				 char *name,
				 int base,
				 int range,
				 int *array,
				 int type)
#else
      name, base, range, array, type)
      char *name;
      int base;
      int range;
      int *array;
      int type;
#endif
{

      FILE  *fp;
      int   counts_base_addr;
      int   bytes = typetowidth(type) * range;

      if ( (fp = fopen(name,"r+")) == NULL) {
	    return(-1);
      }
      (void) fseek(fp, 424, 0);
      if ( fread((char *) &counts_base_addr, sizeof(int), 1, fp) != 1) {
	    (void) fclose(fp);
	    return(-1);
      }
      (void) fseek(fp, counts_base_addr, 0);
      if ( fwrite((char *) array, bytes, 1, fp) != 1) {
	    (void) fclose(fp);
	    return(-1);
      }
      (void) fclose(fp);

      return(0);
}
				
#endif   /* #ifndef EG_ENVIR */

/*
 *   spectrum storage is mapped to files space (eurogam format files)
 *   NB sunsort only ever uses 4 byte integers for its spectra
 */
int *
Map_spectrum(
#if NeedFunctionPrototype
		  int spec_dim, char *name, int nos, int range, char *mode)
#else
      spec_dim, name, nos, range, mode)
      int    spec_dim;    /* 1d == 1, 2d == 2 */
      char   *name;
      int    nos;
      int    range;
      char   *mode;
#endif
{
      FILE *fp;
      SPECTRUM_HDR sp;
      long len = 0;
      char *addr = NULL;
      char file[BUFSIZ];

/* check initial state */
      if (init_state == 0) {
	    init_default_header();
      }

/* set spectrum header to default state */      
      (void) memcpy((char *) &sp, (char *) &eg_spec, sizeof(SPECTRUM_HDR));

/* generate file name and open file */
      sprintf(file,"%s/%d.%dd",tmp_spectrum_dir,nos,spec_dim);
      if ( (fp = fopen(file,mode)) == NULL) {
	    return(NULL);
      }

/* define spectrum header */
      switch (spec_dim) {
	  case 1:
	    sp.dim=1;
	    len=range * sizeof(int);
	    sp.bases[0]=0;
	    sp.ranges[0]=range;
	    break;

	  case 2: /* note 2d spectrum will have x == y and vice versa */
	    sp.dim=2;
	    len=range * range*sizeof(int);
	    sp.bases[0]=sp.bases[1]=0;
	    sp.ranges[0]=sp.ranges[1]=range;
	    break;

	  default:
	    fprintf(stderr,"sort: spectrum type %d unrecognised in map_spectrum call\n",spec_dim);
	    (void) fclose(fp);
	    return(NULL);
      }
      (void) strncpy(sp.spec_name,name,31);
      sp.data_array1.type = sp.data_array2.type = 5;
      sp.count_free_addr = len;
      sp.count_top_addr = sp.count_free_addr - 1;

/* write out header and position file at end of data */      
      if ( fwrite((char *) &sp, sizeof(SPECTRUM_HDR), 1, fp) != 1) {
	    (void) fclose(fp);
	    return(NULL);
      }

      len += eg_spec.count_base_addr;

      if ( ftruncate(fileno(fp), len)) {
	    fprintf(stderr,"sort: ftruncate failed in map_spectrum call\n");
	    (void) fclose(fp);
	    return(NULL);
      }

/* map file to memory address space of process */	  
      addr = (char *) mmap(0, (size_t)len, (PROT_READ|PROT_WRITE), MAP_SHARED, fileno(fp), 0);
      if (addr == (caddr_t)-1) {
	    fprintf(stderr,"sort: mmap failed in map_spectrum call\n");
	    (void) fclose(fp);
	    return(NULL);
      }

/* close off file and set addr to point at beging of data */
      (void) fclose(fp);
      addr+=eg_spec.count_base_addr;
	
      return((int *)addr);
}

int *
map_spectrum(
#if NeedFunctionPrototype
		  int spec_dim, char *name, int nos, int range)
#else
      spec_dim, name, nos, range)
      int    spec_dim;    /* 1d == 1, 2d == 2 */
      char   *name;
      int    nos;
      int    range;
#endif
{
      return(Map_spectrum(spec_dim, name, nos, range, "w+"));
}

int
unmap_spectrum(
#if NeedFunctionPrototype
		  int *data_base, int elements)
#else
      data_base, elements)
      int *data_base;
      int elements;
#endif
{
      caddr_t addr = (caddr_t) data_base - EG_MY_DATA_OFFSET;
      long    len = elements*sizeof(int) + EG_MY_DATA_OFFSET;
      
      if ( munmap(addr, len) == -1) {
	    perror("unmap_spectrum");
	    return(-1);
      }
      return(0);
}

int *
remap_1d_spectrum(
#if NeedFunctionPrototype
		 spec_1d_t *p1d, int newsize)
#else
      p1d, newsize)
      spec_1d_t    *p1d;
      int          newsize;
#endif
{
      int    size = p1d->sp->size;
      
/* deallocate memory associated with spectrum data */
      if (unmap_spectrum(p1d->data, size) == -1) {
	    fprintf(stderr, "remap_1d_spectrum: error unmapping old data\n");
	    return(NULL);
      }

/* remap spectrum to new size */
      return(Map_spectrum(1, p1d->sp->name, p1d->nos, newsize, "r+"));
}

int *
remap_2d_spectrum(
#if NeedFunctionPrototype
		  spec_2d_t *p2d, int newsize)
#else
      p2d, newsize)
      spec_2d_t    *p2d;
      int          newsize;      
#endif
{
      int    size = p2d->sp->size;
      
/* deallocate memory associated with spectrum data */
      if (unmap_spectrum(p2d->data, size*size) == -1) {
	    fprintf(stderr, "remap_2d_spectrum: error unmapping old data\n");
	    return(NULL);
      }
      
/* remap spectrum to new size */
      return(Map_spectrum(2, p2d->sp->name, p2d->nos, newsize, "r+"));
}

int
rm_spectrum_file(
#if NeedFunctionPrototype
		 int nos, int spec_dim)
#else
      nos, spec_dim)
      int nos;
      int spec_dim;
#endif
{
      char file[BUFSIZ];
/* generate file name and open file */
      sprintf(file,"%s/%d.%dd",tmp_spectrum_dir,nos,spec_dim);
      return(unlink(file));
}
