/*
 * prev modified 17/5/93
 * prev modified 28/10/93
 * prev modified 23/11/93
 * prev modified 31/11/93
 * prev modified 16/2/94
 * prev modified 8/12/94
 * prev modified 11/5/95
 * last modified 22/11/95
 * last modified by spgc 11 Jan 96
 */

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <errno.h>
#include <memory.h>
#include <stdlib.h>

#include <sys/types.h>
#include <sys/mman.h>

#ifdef SVR4
#define NeedFunctionPrototype 1
#else
#define NeedFunctionPrototype 0
#endif

#include "sort_thread.h"
#include "eg.h"

extern int        DBX_val;

/* function defined in eg.c */
#if NeedFunctionPrototype
extern int             *map_spectrum(int , char *, int , int);
extern spec_entry_t    *update_shared_list(int, int, float, char *, int);
static  spec_1d_t      *get_1d_spectrum(char *, int, int);
static  spec_2d_t      *get_2d_spectrum(char *, int, int);
extern int             *remap_1d_spectrum(spec_1d_t *p1d, int newsize);
extern int             *remap_2d_spectrum(spec_2d_t *p1d, int newsize);
extern int              msync_1d(spec_1d_t *ptr, int flags);
extern int              msync_2d(spec_2d_t *ptr, int flags);
#else
extern int             *map_spectrum();
extern spec_entry_t    *update_shared_list();
static  spec_1d_t      *get_1d_spectrum();
static  spec_2d_t      *get_2d_spectrum();
extern int             *remap_1d_spectrum();
extern int             *remap_2d_spectrum();
extern int              msync_1d();
extern int              msync_2d();
#endif

int
read_spectra_file(
#if NeedFunctionPrototype
		  char *spectra_proc)
#else
      spectra_proc)
      char *spectra_proc;
#endif

{
	int nos;		/* current spectrum/variable  number */
	float size;		/* spectrum size or variable value */
	short line=0;		/* line number on input file */
	char name[12];		/* spectrum/variable name */
	char buf[BUFSIZ],*p;    /* temp work data */
	FILE *fd;		/* file descriptor */
	int ret_val;		/* function return value */
	char *spectra_file;     /* name of file where spectra info is expected */
	char *type;             
	
#if NeedFunctionPrototype
	int (*func)(int, float, char *);          /* pointer to function that does the work */
	extern int make_1d(int, float, char *),
	           make_2d(int, float, char *),
	           make_var(int, float, char *);
#else
	int (*func)();          
	extern int make_1d(), make_2d(), make_var();
#endif

/* create expected name of spectra file */	
	spectra_file = malloc(strlen(spectra_proc)+6);
	if (spectra_file == NULL) {
	      perror("read_spectra_file");
	      return(SORT_FAIL);
	}
	sprintf(spectra_file,"%s.spec",spectra_proc);
	      
/* open spectra file for processing */	
	if ( (fd=fopen(spectra_file,"r")) == NULL) {
	      perror("read_spectra_file");
	      fprintf(stderr,"server: error opening spectra file %s\n",spectra_file);
	      free(spectra_file);
	      return(SORT_NO_FILE);
	}

	fprintf(stderr,"read_spec: Reading '%s' file\n",spectra_file);

	while(fgets(buf,BUFSIZ,fd)){
		line++;
		if (p=strchr(buf,'*')) {
			switch(*++p) {
				case 'o': case 'O':
					func = make_1d;
					type = "1D spectrum";
					/*fprintf(stderr,"1-D spectra\n");*/
					break;
				
				case 't': case 'T':
					func = make_2d;
					type = "2D spectrum";
					/*fprintf(stderr,"2-D spectra\n");*/
					break;
				
				case 'v': case 'V':
					/*fprintf(stderr,"Variables\n");*/
					type = "Variable";
					func = make_var;
					break;
					
				default:
					fprintf(stderr,"read_spec: unrecognised * option on line %d\n",line);
					break;
			}
			continue;	
		}
			
		if (sscanf(buf,"%d %s %g",&nos,name,&size) != 3) {
		      fprintf(stderr,"read_spec: format error during file read at line %d\n",line);
		      fprintf(stderr,"<<%s>>\n",buf);
		      continue;
		}
		
		if ((ret_val = (*func)(nos,size,name)) == 0) {
/*		      fprintf(stderr,"%s nos %d name %s and size/value %g\n",type,nos,name,size);*/
			continue;	
		}
		else if (ret_val == SORT_FAIL) {
		      break;
		}
	}
	fclose(fd);
	free(spectra_file);
	fprintf(stderr,"read_spec: Done\n");
return(ret_val);
}

/* The elements of tab_?d contain pointers to the storage locations of
   each spectrum structure. ie nos_tab_1d[1] points to the first address location
   of the 1d spectrum structure for spectrum number 1. Element [0] will point to 
   the last structure to be filled spectrum */
static spec_1d_t *tab_1d[TAB_SIZE_1D];
static spec_2d_t *tab_2d[TAB_SIZE_2D];
static var_str_t *tab_var[TAB_SIZE_VAR];

int
make_1d(
#if NeedFunctionPrototype
	int    spec_nos,	
	float  spec_size,	
	char   *spec_name)	
#else
      spec_nos,spec_size,spec_name)
      int spec_nos;		/* spectrum number as defined by user */
      float spec_size;	        /* spectrum size in units of (int) */
      char *spec_name;	        /* spectrum name & file name of saved spectrum */
#endif
{
	spec_1d_t *p;

	if (spec_nos <= 0 || spec_nos >= TAB_SIZE_1D) {
		fprintf(stderr,"make_1d: Invalid 1D spectrum number %d '%s'\n",spec_nos,spec_name);
		return(SORT_NOS_ERR);
	}

/* create memory for spectrum data */
	p = (spec_1d_t *) malloc(sizeof(spec_1d_t));
	if (!p) {
	      perror("make_1d");
	      return(SORT_FAIL);
	}
	p->next=0;

/* enter data into structure and reserve memory for spectrum storage */
	if (tab_1d[spec_nos] == 0)	/* each spectrum should have a unique number */
		tab_1d[spec_nos] = p;
	else {
		fprintf(stderr,
		"make_1d: entry for spectrum number %d already exists ...ignoring '%s'\n",spec_nos,spec_name);
		free(p);
		return (SORT_NOS_ERR);
	}
	p->nos = spec_nos;
	p->data = map_spectrum(1,spec_name,spec_nos,(int) spec_size);
	if (!p->data) {
		perror("make_1d");
		return (SORT_FAIL); 
	}

	p->sp = update_shared_list(TYPE_1d,spec_nos,spec_size,spec_name,0);
	if (!p->sp) {
	      fprintf(stderr,"make_1d: failed to update shared data list\n");
	      return (SORT_FAIL);
	}
	p->version = p->sp->version;
	
/* make structure element next point to last entry ie to decreasing spec_nos */
	p->next = tab_1d[0];      /* 1st time in routine this will be zero */
	tab_1d[0] = p;
	return(0);
}

/* Almost Identical to make_1d */
int
make_2d(
#if NeedFunctionPrototype
	int     spec_nos,	
	float   spec_size,	
	char    *spec_name)	
#else
      spec_nos,spec_size,spec_name)
      int spec_nos;		/* spectrum number as defined by user */
      float spec_size;	        /* spectrum size in units of (int) */
      char *spec_name;	        /* spectrum name & file name of saved spectrum */
#endif
{
	spec_2d_t *p;
	
	if (spec_nos <= 0 || spec_nos >= TAB_SIZE_2D) {
		fprintf(stderr,"make_2d: Invalid 2D spectrum number %d ...ignoring '%s'\n",spec_nos,spec_name);
		return(SORT_NOS_ERR);
	}

/* create memory for spectrum data */
	p = (spec_2d_t *) malloc(sizeof(spec_2d_t));
	if (!p) {
	      perror("make_2d");
	      return(SORT_FAIL);
	}
	p->next=0;

/* enter data into structure and reserve memory for spectrum storage */
	if (tab_2d[spec_nos] == 0)	/* each spectrum should have a unique number */
		tab_2d[spec_nos] = p;
	else {
		fprintf(stderr,
		"make_2d: entry for spectrum number %d already exists..ignoring '%s'\n",spec_nos,spec_name);
		free(p);
		return (SORT_NOS_ERR);
	}
	p->nos = spec_nos;
	p->data = map_spectrum(2,spec_name,spec_nos,(int) spec_size);
	if (!p->data) {
	      perror("make_2d");
	      return (SORT_FAIL);
	}

	if (spec_name[0] == 'w' || spec_name[0] == 'W')		/* window number */
		p->win = 1;
	else
		p->win = 0;

	p->sp = update_shared_list(TYPE_2d,spec_nos,spec_size,spec_name,p->win);
	if (!p->sp) {
	      fprintf(stderr,"make_2d: failed to update shared data list\n");
	      return (SORT_FAIL);
	}
	p->version = p->sp->version;
	
/* make structure element next point to last entry ie to decreasing spec_nos */
	p->next = tab_2d[0];
	tab_2d[0] = p;
	return(0);
}

/* Almost Identical to make_1d */
int
make_var(
#if NeedFunctionPrototype
	 int     var_nos,	
	 float   var_value,	
	 char    *var_name)
#else
      var_nos,var_value,var_name)
      int var_nos;		/* variable number as defined by user */
      float var_value;	        /* variable value real*4 */
      char *var_name;		/* variable name */
#endif
{
	var_str_t *p;

	if (var_nos <= 0 || var_nos >= TAB_SIZE_VAR) {
		fprintf(stderr,"make_var: Invalid variable element %d '%s'\n",var_nos,var_name);
		return(SORT_NOS_ERR);
	}

/* create memory storage for data structure */
	p = (var_str_t *) malloc(sizeof(var_str_t));
	if (!p) {
	      perror("make_var");
	      return(SORT_FAIL);
	}
	p->next=0;

/* enter data into structure */
	if (tab_var[var_nos] == 0)	/* each variable should have a unique number */
		tab_var[var_nos] = p;
	else {
		fprintf(stderr,
		"make_var: entry for variable element %d already exists ...ignoring '%s'\n",var_nos,var_name);
		free(p);
		return (SORT_NOS_ERR);
	}

	p->sp = update_shared_list(TYPE_var,var_nos,var_value,var_name,0);
	if (!p->sp) {
	      fprintf(stderr,"make_1d: failed to update shared data list\n");
	      return (SORT_FAIL);
	}
	p->nos = var_nos;
	p->version = p->sp->version;
	
/* make structure element next point to last entry ie to decreasing var_nos */
	p->next = tab_var[0];
	tab_var[0] = p;
	return(0);
}

	
/*---routines to get pointers to various data tables---*/

#if NeedFunctionPrototype
spec_1d_t **
get_nametab_1d(void)
{
	return(tab_1d);
}

spec_2d_t **
get_nametab_2d(void)
{
	return(tab_2d);
}

var_str_t **
get_nametab_var(void)
{
	return(tab_var);
}
#else
spec_1d_t **
get_nametab_1d()
{
	return(tab_1d);
}

spec_2d_t **
get_nametab_2d()
{
	return(tab_2d);
}

var_str_t **
get_nametab_var()
{
	return(tab_var);
}
#endif

/*---------------------------------------------------*/
void
clear_1d(
#if NeedFunctionPrototype
	 spec_1d_t	*ptr)
#else
      ptr)
      spec_1d_t	        *ptr;
#endif
{
	register int	i;
	register int	size = ptr->sp->size;
	register int	*data = ptr->data;

	for (i=0; i<size; i++)
		*data++ = 0;
	if ( DBX_val >= 5) fprintf(stderr,"cleared 1d spectrum %d %s\n",
				ptr->nos,ptr->sp->name);
	return;
}
void
clear_2d(
#if NeedFunctionPrototype
	 spec_2d_t	*ptr)
#else
      ptr)
      spec_2d_t	        *ptr;
#endif
{
	register int	i;
	register int 	size = ptr->sp->size*ptr->sp->size;
	register int	*data = ptr->data;

	for (i=0; i<size; i++)
		*data++ = 0;
	if ( DBX_val >= 5) fprintf(stderr,"cleared 2d spectrum %d %s\n",
				ptr->nos,ptr->sp->name);
	return;
}

int
Clear_1d_spectra(
#if NeedFunctionPrototype
	 char	*ptr)
#else
      ptr)
      char        *ptr;
#endif
{
      spec_1d_t	*p1d;

/* if ptr == NULL then clear all else clear selected */      
      if (ptr == NULL) {
	    /* get pointer to spectrum storage tables */
	    p1d = get_nametab_1d()[0];
	    
	    /* clear all the one dimensional spectra */
	    while ( p1d ) {
		  clear_1d(p1d);
		  (void) msync_1d(p1d, 0);
		  p1d = p1d->next;
	    }
	    fprintf(stderr,"server: all 1d spectra cleared...\n");
      }
      else {
	    char      *token;
	    char      *str_pos;
	    int       spec_nos;
	    spec_1d_t **pp1d = get_nametab_1d();
	    
	    if (*ptr == '\0')
		  return(SORT_FALSE);
	    
	    /* get list of spectra to clear from string pointed to by ptr */
	    token = strtok_r(ptr," ",&str_pos);
	    fprintf(stderr,"server: 1d spectra");
	    while (token != NULL) {
		  spec_nos = atoi(token);
		  if (spec_nos > 0 && spec_nos < TAB_SIZE_1D) {
			p1d = pp1d[spec_nos];
			if (p1d != NULL) {
			      clear_1d(p1d);
			      fprintf(stderr," %d",spec_nos);
			}
		  }
		  token = strtok_r(NULL," ",&str_pos);
	    }
	    fprintf(stderr," cleared ...\n");
      }
      return(SORT_TRUE);
}

int
Clear_2d_spectra(
#if NeedFunctionPrototype
	 char	*ptr)
#else
      ptr)
      char        *ptr;
#endif
{
      spec_2d_t	*p2d;

/* if ptr == NULL then clear all else clear selected */      
      if (ptr == NULL) {
	    /* get pointer to spectrum storage tables */
	    p2d = get_nametab_2d()[0];
	    
	    /* clear all the one dimensional spectra */
	    while ( p2d ) {
		  if ( p2d->win == 0) { /* don't clear window spectra */
			clear_2d(p2d);
			(void) msync_2d(p2d, 0);
		  }
		  p2d = p2d->next;
	    }
	    fprintf(stderr,"server: all 2d spectra cleared...\n");
      }
      else {
	    char      *token;
	    char      *str_pos;
	    int       spec_nos;
	    spec_2d_t **pp2d = get_nametab_2d();
	    
	    if (*ptr == '\0')
		  return(SORT_FALSE);
	    
	    /* get list of spectra to clear from string pointed to by ptr */
	    token = strtok_r(ptr," ",&str_pos);
	    fprintf(stderr,"server: 2d spectra");
	    while (token != NULL) {
		  spec_nos = atoi(token);
		  if (spec_nos > 0 && spec_nos < TAB_SIZE_2D) {
			p2d = pp2d[spec_nos];
			if (p2d != NULL) {
			      clear_2d(p2d);
			      fprintf(stderr," %d",spec_nos);
			}
		  }
		  token = strtok_r(NULL," ",&str_pos);
	    }
	    fprintf(stderr," cleared ...\n");
      }
      return(SORT_TRUE);
}

/*
 *   note the EG_MY_DATA_OFFSET constant ... memory mapping is very
 *   picky about the page boundaries (here the ptr->data points
 *   to the start of the data not the start address given by mmap call)
 */
int
msync_1d(
#if NeedFunctionPrototype
	 spec_1d_t	*ptr, int  flags)
#else
      ptr, flags)
      spec_1d_t	        *ptr;
      int                flags;
#endif
{
	register int	size = ptr->sp->size*sizeof(int);
	register char   *addr = (char *)ptr->data - EG_MY_DATA_OFFSET;
	return(msync((caddr_t)addr, size, flags));
}
int
msync_2d(
#if NeedFunctionPrototype
	 spec_2d_t	*ptr, int  flags)
#else
      ptr, flags)
      spec_2d_t	        *ptr;
      int                flags;
#endif
{
	register int 	size = ptr->sp->size * ptr->sp->size*sizeof(int);
	register char   *addr = (char *)ptr->data - EG_MY_DATA_OFFSET;
	return(msync((caddr_t)addr, size, flags));
}


/*
 *  Clear and delete local memory associated with spectrum and variables
 *  data structures. Clear data in shared memory
 */
int
delete_1d(
#if NeedFunctionPrototype
	int    spec_nos)
#else
      spec_nos)
      int spec_nos;		/* spectrum number as defined by user */
#endif
{
	spec_1d_t *p1d;

/* check to see that specified spectrum number is within bounds */	
	if (spec_nos <= 0 || spec_nos >= TAB_SIZE_1D) {
		fprintf(stderr,"delete_1d: Invalid 1D spectrum number %d\n",spec_nos);
		return(SORT_NOS_ERR);
	}

/* check so see if spectrum entry exists */	
	if (tab_1d[spec_nos] != 0)	/* each spectrum should have a unique number */
		p1d = tab_1d[spec_nos];
	else {
		fprintf(stderr,
		"delete_1d: entry for spectrum number %d does not exists ...ignoring\n",spec_nos);
		return (SORT_NOS_ERR);
	}

/* free memory associated with spectrum data */
	if (unmap_spectrum(p1d->data, p1d->sp->size) == -1) {
	      fprintf(stderr,"delete_1d: failed to unmap old data\n");
	      return(SORT_FAIL);
	}

/* remove spectrum file */
	if (rm_spectrum_file(spec_nos, 1) == -1) 
	      fprintf(stderr,"delete_1d: error deleting map file for spectrum nos %d\n",p1d->nos);
	      
/* correct link list to account for removal of this element */
	if (p1d == tab_1d[0]) {
	      tab_1d[0] = p1d->next;
	}
	else {
	      spec_1d_t *entry;
	      for (entry=tab_1d[0]; entry != 0; entry = entry->next) {
		    if (entry->next == p1d) {
			  entry->next = p1d->next;
			  break;
		    }
	      }
	}
	
/* clear spectrum data structures */	
	(void) memset(p1d->sp, 0, sizeof(spec_entry_t));
	(void) memset(p1d, 0, sizeof(spec_1d_t));

/* free entry in spectrum table */
	free(p1d);
	tab_1d[spec_nos] = 0;
	
	return(0);
  }



int
delete_2d(
#if NeedFunctionPrototype
	int    spec_nos)
#else
      spec_nos)
      int spec_nos;		/* spectrum number as defined by user */
#endif
{
	spec_2d_t *p2d;

/* check to see that specified spectrum number is within bounds */	
	if (spec_nos <= 0 || spec_nos >= TAB_SIZE_2D) {
		fprintf(stderr,"delete_2d: Invalid 2D spectrum number %d\n",spec_nos);
		return(SORT_NOS_ERR);
	}

/* check so see if spectrum entry exists */	
	if (tab_2d[spec_nos] != 0)	/* each spectrum should have a unique number */
		p2d = tab_2d[spec_nos];
	else {
		fprintf(stderr,
		"delete_2d: entry for spectrum number %d does not exists ...ignoring\n",spec_nos);
		return (SORT_NOS_ERR);
	}

/* free memory associated with spectrum data */
	if (unmap_spectrum(p2d->data, p2d->sp->size*p2d->sp->size) == -1) {
	      fprintf(stderr,"delete_2d: failed to unmap old data\n");
	      return(SORT_FAIL);
	}

/* remove spectrum file */
	if (rm_spectrum_file(spec_nos, 2) == -1) 
	      fprintf(stderr,"delete_2d: error deleting map file for spectrum nos %d\n",spec_nos);
	      
/* correct link list to account for removal of this element */
	if (p2d == tab_2d[0]) {
	      tab_2d[0] = p2d->next;
	}
	else {
	      spec_2d_t *entry;
	      for (entry=tab_2d[0]; entry != 0; entry = entry->next) {
		    if (entry->next == p2d) {
			  entry->next = p2d->next;
			  break;
		    }
	      }
	}
	
/* clear spectrum data structures */	
	(void) memset(p2d->sp, 0, sizeof(spec_entry_t));
	(void) memset(p2d, 0, sizeof(spec_2d_t));

/* free entry in spectrum table */
	free(p2d);
	tab_2d[spec_nos] = 0;
	
	return(0);
  }


int
delete_var(
#if NeedFunctionPrototype
	int    var_nos)
#else
      var_nos)
      int var_nos;		/* spectrum number as defined by user */
#endif
{
	var_str_t *var;

/* check to see that specified spectrum number is within bounds */	
	if (var_nos <= 0 || var_nos >= TAB_SIZE_VAR) {
		fprintf(stderr,"delete_var: Invalid variable number %d\n",var_nos);
		return(SORT_NOS_ERR);
	}

/* check so see if spectrum entry exists */	
	if (tab_var[var_nos] != 0)	/* each spectrum should have a unique number */
		var = tab_var[var_nos];
	else {
		fprintf(stderr,
		"delete_var: entry for spectrum number %d does not exists ...ignoring\n",var_nos);
		return (SORT_NOS_ERR);
	}

/* correct link list to account for removal of this element */
	if (var == tab_var[0]) {
	      tab_var[0] = var->next;
	}
	else {
	      var_str_t *entry;
	      for (entry=tab_var[0]; entry != 0; entry = entry->next) {
		    if (entry->next == var) {
			  entry->next = var->next;
			  break;
		    }
	      }
	}
	
/* clear spectrum data structures */	
	(void) memset(var->sp, 0, sizeof(spec_entry_t));
	(void) memset(var, 0, sizeof(var_str_t));

/* free entry in spectrum table */
	free(var);
	tab_var[var_nos] = 0;
	
	return(0);
  }

int
delete_spectra(
#if NeedFunctionPrototype
	void)
#else
      )
#endif
{
      spec_1d_t	*p1d;
      spec_2d_t	*p2d;
      var_str_t *var;
      int       nos;
      int       ret_code = 0;

/* get pointer to 1d spectrum storage table */
      p1d = get_nametab_1d()[0];
      p2d = get_nametab_2d()[0];
      var = get_nametab_var()[0];
      
      /* delete all the one dimensional spectra */
      while ( p1d ) {
	    nos = p1d->nos;
	    p1d = p1d->next;
	    if (delete_1d(nos) == SORT_FAIL) {
		  fprintf(stderr,"delete_spectra: failed to delete 1d spectrum number %d\n",nos);
		  ret_code += 0x1;
		  break;
	    }
      }

      /* delete  all the two dimensional spectra */
      while ( p2d ) {
	    nos = p2d->nos;
	    p2d = p2d->next;
	    if (delete_2d(nos) == SORT_FAIL) {
		  fprintf(stderr,"delete_spectra: failed to delete 2d spectrum number %d\n",nos);
		  ret_code += 0x2;
		  break;
	    }
      }

      /* delete all the variables */
      while ( var ) {
	    nos = var->nos;
	    var = var->next;
	    if (delete_var(nos) == SORT_FAIL) {
		  fprintf(stderr,"delete_spectra: failed to delete variable nos %d\n",nos);
		  ret_code += 0x4;
		  break;
	    }
      }
      return(ret_code);
}





/*	take a previously saved spectrum file name_nd_nos (eg somename_1d_23),
	decide whether it is a 1d or a 2d spectrum from the value of n and at 
	what specrum id number to use from the value of nos (ie the position in
	tab_1d).
*/
int 
load_spectrum_file(
#if NeedFunctionPrototype
		   char 	*dir_name,
		   char	        *file_name,
		   int          IO_state)
#else
      dir_name,file_name,IO_state)
      char 	*dir_name;
      char	*file_name;
      int       IO_state;
#endif
{
	char		sbuf[MED_SUNSORT_BUFFER+32];
	char		*cp;
	char		name[NAME_SIZE];		/* read in from file name */
	int		nos;				/* read in from file name */
	int		type;				/* read in from file name */
	int		ret_code;
	int		id;				/* for use in SDB format data */
	FILE            *fp;                            /* for use in X-Y format data */
	
#if NeedFunctionPrototype
	extern int	load_SDB_spectrum(int *id, char *name, int type, int nos, int add);
	extern int	load_XY_spectrum(FILE *fp, char *name, int type, int nos, int add);
#else
	extern int	load_SDB_spectrum();
	extern int	load_XY_spectrum();
#endif

	/* decode file name to obtain data on where to store spectrum info */
	strcpy(sbuf,file_name);
        if ((cp = strrchr(sbuf, '_')) != NULL)
            *cp = ' ';
        if ((cp = strrchr(sbuf, '_')) != NULL)
	    *cp = ' ';
/*
	for (cp=sbuf; *cp != '\0'; cp++)
		if ( *cp == '_') *cp = ' ';
*/
	if ( sscanf(sbuf,"%s %dd %d",name,&type,&nos) != 3) {
		fprintf(stderr,"server: error decoding filename %s during load\n",file_name);
		return(SORT_FAIL);
	}

	if ( DBX_val >= 5)
            fprintf(stderr, "server: load spectrum with IO flags = 0x%x\n" 
                    "        name=%s, type=%dd, nos=%d\n", IO_state, name,
                    type, nos);
	
/* generate fully specified path name for spectrum file */
	sprintf(sbuf,"%s/%s",dir_name,file_name);

/* read file and place data into spectrum storage */
	switch(IO_state & FMT_MASK) {
	   case SDB_TYPE:
/* open connection to sdb file */
	      if (sname(&id,sbuf,strlen(sbuf)) ) {
		    fprintf(stderr,"server: failed to open SDB format file %s for reading\n",sbuf);
		    return(SORT_FAIL);
	      }
	      ret_code = load_SDB_spectrum(&id,name,type,nos,IO_state & LOAD_N_ADD);
	      (void) sname(&id,sbuf,0);
	      break;

	   case XY_TYPE:
	      if ( (fp = fopen(sbuf,"r")) == NULL) {
		    fprintf(stderr,"server: failed to open X-Y format file %s for reading\n",sbuf);
		    return(SORT_FAIL);
	      }
	      ret_code = load_XY_spectrum(fp,name,type,nos,IO_state & LOAD_N_ADD);
	      (void) fclose(fp);
	      break;

	   case EG_TYPE:
	      eexpt(&id);
	      if ( id != 0) {
	            fprintf(stderr,"server: failed to connect to EG experiment\n");
		    return(SORT_FAIL);
	      }
	      if (ename(&id,sbuf,strlen(sbuf)) ) {
		    fprintf(stderr,"server: failed to open EG  format file %s for reading\n",sbuf);
		    return(SORT_FAIL);
	      }
	      ret_code = load_EG_spectrum(&id,name,type,nos,IO_state & LOAD_N_ADD);
	      (void) ename(&id,sbuf,0);
	      break;

	   default:
	      fprintf(stderr,"server: unknown file type 0x%x during load file operation",IO_state & FMT_MASK);
	      return(SORT_FAIL);

	}
/* write out full file name */
	fprintf(stderr," %s\n",sbuf);
	return(ret_code);
}
#define MAX(a,b) ((a) > (b)) ? (a) : (b)
int 
load_SDB_spectrum(
#if NeedFunctionPrototype
		  int	*id,
		  char	*name,
		  int	type,
		  int	nos,
		  int	add)
#else
      id, name, type, nos, add)
      int	*id;
      char	*name;
      int	type;
      int	nos;
      int	add;
#endif
{
	spec_1d_t	*p1d;
	spec_2d_t	*p2d;
	int		prec;
	int 		base1,range1,base2,range2;
	int		max;
	int             *data;
	int             x,y;


/* inquire about sdb file parameters */
      if( sinq(*id,&base1,&range1,&base2,&range2,&prec)) {
      	    fprintf(stderr,"server: sinq function call failed on file");
	    return(SORT_FAIL);
      }

/* check that the type specified by the spectrum's filename is consistent with
   what sdb thinks it should be. check that we have the right precision */
      if ( range1 == 0 || (type == 1 && range2 > 0) || (type == 2 && range2 == 0)) {
	    fprintf(stderr,"server: spectrum type inconsistency. (Wrong spectrum type ?)\n");
	    fprintf(stderr,"\t type = %dd, range1 = %d, range2 = %d \n on file",type,range1,range2);
	    return(SORT_NOS_ERR);		/* non fatal skip this file */
      }
      if ( prec != 4) {
	    fprintf(stderr,"server: sunsort accepts int*4 spectra only ... FAILED to load file\n");
	    return(SORT_NOS_ERR);		/* non fatal skip this file */
      }

/* update data storage according to whether we are dealing with 1d or 2d case */
      switch(type) {
	 case 1:				/* 1d case */
	    /* get spectrum pointer and ensure spectrum is big enough */	    
	    p1d = get_1d_spectrum(name, nos, range1);
	    if (p1d == NULL) 
		  return(SORT_FAIL);
	    /* choose where to put data depending upon whether add is true or not */
	    if ( add) {
		  data = (int *) calloc((unsigned) range1, sizeof(int));
		  if (! data) {
			perror("server => load_SDB_file");
			fprintf(stderr,"server: during load from file");
			return(SORT_FAIL);
		  }
		  /* now lets finally read the data back hopefully */
		  if ( sread(*id,base1,range1,0,0,data,4)) 
			fprintf(stderr,"server: SDB spectrum not read correctly from file");
		  /* add data read from disk to data already in memory */
		  for ( x=0; x < range1; x++)
			p1d->data[x] += data[x];
		  free(data);
	    }
	    else {
		  /* now lets finally read the data back hopefully */
		  if ( sread(*id,base1,range1,0,0,p1d->data,4)) 
			fprintf(stderr,"server: SDB spectrum not read correctly from file");
	    }
	    fprintf(stderr,"server: spectrum %s loaded into 1d spectrum nos %d from file",name,nos);
	    break;
		    
	case 2:				/* 2d case */
	    /* get spectrum pointer and ensure spectrum is big enough */
	    max = MAX(range1,range2);
	    p2d = get_2d_spectrum(name, nos, max);
	    if (p2d == NULL) 
		  return(SORT_FAIL);
	    /* generate tmp storage to read spectrum into */	    
	    data = (int *) calloc((unsigned) max*max, sizeof(int));
	    if (data == NULL) {
		  perror("server => load_SDB_file");
		  fprintf(stderr,"server: during load from file");
		  return(SORT_FAIL);
	    }
	    /* now lets finally read the data back hopefully */
	    if ( sread(*id,base1,range1,base2,range2,data,4)) 
		  fprintf(stderr,"server: SDB spectrum not read correctly from file");
	    /* choose where to put data depending upon whether add is true or not */
	    if ( add) {
		  /* add data read from disk to data already in memory */
		  for ( y=0; y < range2; y++)
			for ( x=0; x < range1; x++)
			      p2d->data[x*range2 + y] += data[x + y*range1];
	    }
	    else {
		  /* add data read from disk to data already in memory */
		  for ( y=0; y < range2; y++)
			for ( x=0; x < range1; x++)
			      p2d->data[x*range2 + y] = data[x + y*range1]; 
	    }
	    (void) free(data);
	    fprintf(stderr,"server: spectrum %s loaded into 2d spectrum nos %d from file",name,nos);
	    break;
	    
	default:
	    fprintf(stderr,"server: unrecognised type = %d during load from file",type);
	    return(SORT_FAIL);
	}
      return(0);
}
/* same as for SDB except change of names from sread -> eread etc */
int 
load_EG_spectrum(
#if NeedFunctionPrototype
		  int	*id,
		  char	*name,
		  int	type,
		  int	nos,
		  int	add)
#else
      id, name, type, nos, add)
      int	*id;
      char	*name;
      int	type;
      int	nos;
      int	add;
#endif
{
	spec_1d_t	*p1d;
	spec_2d_t	*p2d;
	int		prec;
	int 		base1,range1,base2,range2;
	int		max;
	int             x,y;

/* inquire about eg file parameters */
      if( einq(*id,&base1,&range1,&base2,&range2,&prec)) {
      	    fprintf(stderr,"server: einq function call failed on file (Wrong spectrum type ?)\n");
	    return(SORT_FAIL);
      }

/* check that the type specified by the spectrum's filename is consistent with
   what eg thinks it should be. check that we have the right precision */
      if ( range1 == 0 || (type == 1 && range2 > 0) || (type == 2 && range2 == 0)) {
	    fprintf(stderr,"server: spectrum type inconsistency. (Wrong spectrum type ?)\n");
	    fprintf(stderr,"\t type = %dd, range1 = %d, range2 = %d \n on file",type,range1,range2);
	    return(SORT_NOS_ERR);		/* non fatal skip this file */
      }
      if ( prec != 4) {
	    fprintf(stderr,"server: sunsort accepts int*4 spectra only ... FAILED to load file\n");
	    return(SORT_NOS_ERR);		/* non fatal skip this file */
      }

/* update data storage according to whether we are dealing with 1d or 2d case*/
      switch(type) {
	 case 1:				/* 1d case */
	    /* get spectrum pointer and ensure spectrum is big enough */	    
	    p1d = get_1d_spectrum(name, nos, range1);
	    if (p1d == NULL) 
		  return(SORT_FAIL);
	    /* choose where to put data depending upon whether add is true or not */
	    if ( add) {
		  int *data = (int *) calloc((unsigned) range1, sizeof(int));
		  if (! data) {
	   		perror("server => load_EG_file");
	   		fprintf(stderr,"server: during load from file");
	   		return(SORT_FAIL);
		  }
		  /* now lets finally read the data back hopefully */
		  if ( eread(*id,base1,range1,0,0,data,4)) 
			fprintf(stderr,"server: EG spectrum not read correctly from file");
		  /* add data read from disk to data already in memory */
		  for ( x=0; x < range1; x++)
			p1d->data[x] += data[x];
		  free(data);
	    }
	    else {
		  /* now lets finally read the data back hopefully */
		  if ( eread(*id,base1,range1,0,0,p1d->data,4)) 
			fprintf(stderr,"server: EG spectrum not read correctly from file");
	    }
	    fprintf(stderr,"server: spectrum %s loaded into 1d spectrum nos %d from file",name,nos);
	    break;
		    
	case 2:				/* 2d case */
	    /* get spectrum pointer and ensure spectrum is big enough */
	    max = MAX(range1,range2);
	    p2d = get_2d_spectrum(name, nos, max);
	    if (p2d == NULL) 
		  return(SORT_FAIL);
	    /* choose where to put data depending upon whether add is true or not */
	    if ( add) {
		  int *data = (int *) calloc((unsigned) max*max, sizeof(int));
		  if (! data) {
	   		perror("server => load_EG_file");
	   		fprintf(stderr,"server: during load from file");
	   		return(SORT_FAIL);
		  }
		  /* now lets finally read the data back hopefully */
		  if ( eread(*id,base1,range1,base2,range2,data,4)) 
			fprintf(stderr,"server: EG spectrum not read correctly from file");
		  /* add data read from disk to data already in memory */
		  for ( y=0; y < range2; y++)
			for ( x=0; x < range1; x++)
			      p2d->data[x + y*range2] += data[x + y*range2];
		  free(data);
	    }
	    else {
		  /* now lets finally read the data back hopefully */
		  if ( eread(*id,base1,range1,base2,range2,p2d->data,4)) 
			fprintf(stderr,"server: EG spectrum not read correctly from file");
	    }
	    fprintf(stderr,"server: spectrum %s loaded into 2d spectrum nos %d from file",name,nos);
	    break;
	    
	default:
	    fprintf(stderr,"server: unrecognised type = %d during load from file",type);
	    return(SORT_FAIL);
	}
      return(0);
}
#define SPEC_SIZE_1D_GUESS 4096
#define SPEC_SIZE_2D_GUESS 128
int
load_XY_spectrum(
#if NeedFunctionPrototype
		 FILE	*fp,
		 char	*name,
		 int	type,
		 int	nos,
		 int	add)
#else
      fp,name,type,nos,add)
      FILE	*fp;
      char	*name;
      int	type;
      int	nos;
	int	add;
#endif
{
	spec_1d_t	*p1d;
	spec_2d_t	*p2d;
	char		sbuf[MED_SUNSORT_BUFFER];
	int		x_ch,y_ch,counts;
	int 		range;
	int		t_range = 0;

/* check spectrum format by reading first line of file which should be a header 
   in the form=>	 #X-Y ?D spectrum named ????? */
	if ( fgets(sbuf,MED_SUNSORT_BUFFER-1,fp) == NULL || strncmp(sbuf,"#X-Y",4) != 0) {
		fprintf(stderr,"server: #X-Y header not found during XY load from file");
		return(SORT_NOS_ERR);
	}

/* update data storage according to whether we are dealing with 1d or 2d case*/
      switch(type) {
	 case 1:				/* 1d case */
/* check that an entry already exists for this spectrum number ... if not make it */
	    if ( (p1d = tab_1d[nos]) == 0) {
		  if (make_1d(nos,(float) SPEC_SIZE_1D_GUESS,name)) {
		  	fprintf(stderr,"server: make_1d failed for 1d spectrum nos %d from file",nos);
		  	return(SORT_FAIL);
		  }
		  p1d = tab_1d[nos];
	    }
	    else {
/* clear spectrum of present data if not adding*/
		  if ( ! add) 
			for( x_ch=0; x_ch < p1d->sp->size; x_ch++)
		    		p1d->data[x_ch] = 0;
	    }
	    range = p1d->sp->size;

/* read in data line by line */	    
	    while( fscanf(fp,"%d %d",&x_ch,&counts) == 2) {
	   	t_range = MAX(t_range,x_ch);
/* check to see whether the disk spectra size is larger than the present allocated storage */
		if ( t_range >= range) {
		      t_range *= 2;
		      p1d->data = (int *) remap_1d_spectrum(p1d, t_range);
		   	if (! p1d->data) {
				fprintf(stderr,"server: load_XY_file remap error during load from");
				return(SORT_FAIL);
		   	}
		   	range = t_range; 
	        }
	        p1d->data[x_ch] += counts;
	    }

/* update spectrum data structure */
	    p1d->sp->size = range;
	    strcpy(p1d->sp->name,name);
	    p1d->version = ++p1d->sp->version;	    
	    fprintf(stderr,"server: spectrum %s loaded into 1d spectrum nos %d from file",name,nos);
	    break;
		    
	case 2:				/* 2d case */
/* check that an entry already exists for this spectrum number ... if not make it */
	    if ( (p2d = tab_2d[nos]) == 0) {
		  if ( make_2d(nos,(float) SPEC_SIZE_2D_GUESS,name)) {	
			fprintf(stderr,"server: make_2d failed for 2d spectrum nos %d from file",nos);
			return(SORT_FAIL);
		  }
		  p2d = tab_2d[nos];
	    }
	    else {
/* clear spectrum of present data */
		if ( ! add)
			for( y_ch=0; y_ch < p2d->sp->size; y_ch++)
				for( x_ch=0; x_ch < p2d->sp->size; x_ch++)
					p2d->data[x_ch*p2d->sp->size + y_ch] = 0;
	    }
	    range = p2d->sp->size;

/* read data in from file line by line */
	    while (fscanf(fp,"%d %d %d",&x_ch,&y_ch,&counts) == 3) {
	    	t_range = MAX(t_range,x_ch);
	    	t_range = MAX(t_range,y_ch);
/* check to see whether the disk spectra size is larger than the present allocated storage */
		if ( t_range >= range ) {
			  int *tptr,x,y;		/* NB declaration of variables */
			  t_range *= 2;                 /* spectrum may well get bigger still */
/* allocate temp memory buffer */
			  tptr = (int *) calloc((unsigned) range*range, sizeof(int));
			  if (! tptr) {
				perror("sort=> load_XY_file");
				fprintf(stderr,"server: during load from file");
				return(SORT_FAIL);
			  }
/* copy data into temp memory buffer */			  
			  (void) memcpy(tptr, p2d->data, range*range*sizeof(int));
/* resize memory spectrum file  */	  
			  p2d->data = (int *) remap_2d_spectrum(p2d, t_range);
			  if (! p2d->data) {
				fprintf(stderr,"server: remap_2d_spectrum error during load from file");
				free(tptr);
				return(SORT_FAIL);
			  }
/* copy data from old storage region into new storage */
			  for(y=0; y < range; y++)
			  	for(x=0; x < range; x++)
			  		p2d->data[y + x*t_range] = tptr[y + x*t_range];
			  range = t_range;
			  free(tptr);
		}
		p2d->data[y_ch + x_ch*range] += counts;
	    }

/* update spectrum data structure */
	    p2d->sp->size = range;
	    strcpy(p2d->sp->name,name);
	    p2d->version = ++p2d->sp->version;
	    if (name[0] == 'w' || name[0] == 'W')		
		  p2d->sp->win = p2d->win = 1;
	    else
		  p2d->sp->win = p2d->win = 0;
	    fprintf(stderr,"server: spectrum %s loaded into 2d spectrum nos %d from file",name,nos);
	    break;
	    
	default:
	    fprintf(stderr,"server: unrecognised type = %d during load from file",type);
	    return(SORT_FAIL);
	}
	return(0);
}


spec_1d_t*
get_1d_spectrum(
#if NeedFunctionPrototype
		char *name, int nos, int range1)
#else
      name, nos, range1)
      char *name;
      int nos;
      int range1;
#endif
{
      spec_1d_t	*p1d;
      
/* check that an entry already exists for this spectrum number ... if not make it */
      if ( (p1d = tab_1d[nos]) == 0) {
	    if (make_1d(nos,(float) range1,name)) {
		  fprintf(stderr,"server: make_1d failed for 1d spectrum nos %d from file",nos);
		  return(NULL);
	    }
	    p1d = tab_1d[nos];
      }
      
/* check to see whether the disk spectra size is different to the present allocated storage */
      if ( range1 > p1d->sp->size) {
	    p1d->data = (int *) remap_1d_spectrum(p1d, range1);
	    if (! p1d->data) {
		  fprintf(stderr,"server: load_file remap error during load from file");
		  return(NULL);
	    } 
      }
      
/* update spectrum data structure */
      p1d->sp->size = range1;
      strcpy(p1d->sp->name,name);
      p1d->version = ++p1d->sp->version;

      return(p1d);
}


spec_2d_t*
get_2d_spectrum(
#if NeedFunctionPrototype
		char *name, int nos, int max)
#else
      name, nos, max)
      char *name;
      int nos;
      int max;
#endif
{
      spec_2d_t	*p2d;
      
/* check that an entry already exists for this spectrum number ... if not make it */
      if ( (p2d = tab_2d[nos]) == 0) {
	    if ( make_2d(nos,(float) max, name)) {
		  fprintf(stderr,"server: make_2d failed for 2d spectrum nos %d from file",nos);
		  return(NULL);
	    }
	    p2d = tab_2d[nos];
      }
      
/* check to see whether the disk spectra size is larger than the present allocated storage */
      if ( max > p2d->sp->size ) {
	    p2d->data = (int *) remap_2d_spectrum(p2d, max);
	    if (! p2d->data) {
		  fprintf(stderr,"server: load_file remap error during load from file");
		  return(NULL);
	    } 
      }

/* update spectrum data structure */
      p2d->sp->size = max;
      strcpy(p2d->sp->name,name);
      p2d->version = ++p2d->sp->version;
      if (name[0] == 'w' || name[0] == 'W')
	    p2d->sp->win = p2d->win = 1;
      else
	    p2d->sp->win = p2d->win = 0;     

      return(p2d);
}
