/*
 *  sunsort main
 *  prev modified 14/2/95
 *  prev modified 27/3/95
 *  last modified 12/5/95
 *
 */


#define SUNSORT_MAIN

#include <stdio.h>
#include <string.h>
#include <dirent.h>
#include <signal.h>
#include <memory.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>

#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/resource.h>

#include <errno.h>

#include "sort_thread.h"
#include "sort_extern.h"
#include "sort_mem.h"
#include "version.h"

#ifndef MAX
#define MAX(x,y) ((int)(x) < (int)(y)) ? (y) : (x)
#endif
#ifndef MIN
#define MIN(x,y) ((int)(x) > (int)(y)) ? (y) : (x)
#endif

/* infomation concerning current sort state */
extern sort_IO_t 	*sort_io_data;
extern dirs_IO_t	*dirs_io_data;
extern unsigned long    *sorted_records;
extern unsigned long    *filtered_records;

/**************************************************************************************/

void
main(
#if NeedFunctionPrototype
     int		argc,
     char		**argv)
#else
      argc, argv)
      int		argc;
      char		**argv;
#endif
{

    fprintf(stderr,"\n-------------------------" VERLINE "\n");
    fprintf(stderr,"Charissa SunSort Version " VERSION "\n");
    fprintf(stderr,"-------------------------" VERLINE "\n");
    fprintf(stderr,"\nSunsort > ");

/*  check to see which options have been set in the command line */      
      read_args(argc,argv);
      if (sunsort_proc == NULL)
	    sunsort_proc = strdup("./sunsort_proc");

/* generate directory name for shared data files */
      if ( (tmp_spectrum_dir = make_tmp_dir()) == NULL) {
	    fprintf(stderr,"server: error occured during creation of tmp data directory...exiting\n");
	    exit(-1);
      }
      
/* create shared data resources */
      if (create_shared_data(tmp_spectrum_dir) == -1) {
	    fprintf(stderr,"server: error occured during shared data creation...exiting\n");
	    exit(-1);
      }
      if (DBX_val >= 1)
	    fprintf(stderr,"server: using temporary directory %s\n",tmp_spectrum_dir);

/*  read in spectra definitions and create storage */
      if ( read_spectra_file(sunsort_proc) == SORT_FAIL) {
	    fprintf(stderr,"server: error occured during the reading of the spectra file ...exiting\n");
	    exit(-1);
      }
      
/*  setup signal handling functions for SIGINT, SIGPIPE, SIGSEGV and SIGTERM */
      if (setup_sighandlers() == -1) {
	    perror("server");
	    fprintf(stderr,"server failed to initialize signal handlers ...exiting\n");
	    exit(-1);
      }
      
/*
     0) start sort process
     1) start gui if nogui != sort_true
     2) start graphics if nograf != sort_true
     3) set up interrupt handlers for gui and graphics
*/

/*  start user sort process */
      if ( (sort_pid = start_sortproc(sunsort_proc, argc, argv)) == -1) {
	    sort_proc_OK = SORT_FALSE;
	    FDSIN = STDIN_FILENO;
	    FDSOUT = STDOUT_FILENO;
      }
      
/*  start graphical user interface process */      
      if (nogui != SORT_TRUE) {
	    if ( (gui_pid = start_gui("sort_gui", argc, argv)) == -1) {
		  nogui = SORT_TRUE;
		  FDIN = STDIN_FILENO;
		  FDOUT = STDOUT_FILENO;
	    }
      }

/*  start display graphics processes */
      if (nograf != SORT_TRUE) {
	    (void) signal(SIGUSR2, sigwin2d_handler);
	    start_graphics(argc, argv);
      }
      
/*  set this process as process group leader */
#ifdef SVR4
      (void) setpgrp();
#else
      (void) setpgrp(getpid(),0);
#endif

/* set default format type to NSF */
      (void) set_format("nsf");
            
/*  
    the main event loop waiting upon user input to perform actions
*/
      
/*  Turn control over to event loop */
      main_event_loop();
      exit(-1);
}

/******************************************************************************/
/* 
 * check whether specified string corresponds to an existing directory 
 * if it does give user the choice to over write existing files
 * if it doesn't then try to make a new directory into which the
 * data will be saved
 *
 */
int
Query_Save_directory(
#if NeedFunctionPrototype
		     dirs_IO_t	*ptr)
#else
      ptr)
      dirs_IO_t	*ptr;
#endif
{
	char		*save_dir = ptr->save_dir;
	int 		ret_code;
	
	if ((ret_code = inquire(save_dir)) == SORT_FALSE ) {
		switch(errno) {
			case ENOENT:
				if (mkdir(save_dir,0755)) {
					perror("dirs_b1_nh SAVE");
					return(-1);
				}
				if (inquire(save_dir) != SORT_dir) return(-1);
				break;
			default:
				perror("dirs_b1_nh SAVE");
				return(-1);
		}
	}
	else if (ret_code != SORT_dir) {
		fprintf(stderr,"server: %s is not a directory\n",save_dir);
		return(-1);
	}
	return(0);
}

/**************************************************************************************************/
void
start_graphics(
#if NeedFunctionPrototype
		int argc, char **argv)
#else
      argc, argv)
      int   argc;
      char  **argv;
#endif
{
      char *my_argv[100];
      int i;
      
      for(i=1; i<argc && i<98; i++)
	    my_argv[i] = argv[i];
      my_argv[i] = "-sort";
      my_argv[i+1] = (char *)0;

/* set global variables keeping track of graphics state */
      graf_proc1_OK = SORT_TRUE;
      graf_proc2_OK = SORT_TRUE;
      nograf = SORT_FALSE;

/* create graphics processes */
      my_argv[0] = "sort_xvgr";
      if ( (graf_1d_pid = do_process("sort_xvgr",my_argv)) == SORT_FAIL) {
	    fprintf(stderr,"server: 1d display process returned error\n");
	    graf_proc1_OK = SORT_FALSE;
      } 
      my_argv[0] = "sort_spec2d";
      if ( (graf_2d_pid = do_process("sort_spec2d",my_argv)) == SORT_FAIL) {
	    fprintf(stderr,"server: 2d display process returned error\n");
	    graf_proc2_OK = SORT_FALSE;
      } 
      return;
}

/******************************************************************************/
/*
 *  this function is used by background threads as well
 *  so must NOT modify any STATE data
 */
int
do_process(
#if NeedFunctionPrototype
	   char           *process,
	   char           **argv)
#else
      process,argv)
      char           *process;
      char           **argv;
#endif
{
      int pid;
      int i;

      if (DBX_val >= 5) {
	    fprintf(stderr,"server: do_process(%s",process);
	    for (i=0; argv[i] != NULL; i++)
		  fprintf(stderr," ,%s",argv[i]);
	    fprintf(stderr,")\n");
      }
      switch(pid = fork1()) {
	    case -1:
	        fprintf(stderr,"server: failed to fork %s process\n",process);
		return(SORT_FAIL);
	    case 0:
		/* clean up on first 32 streams if they are open */
		for (i=3; i<32; i++)
		      (void) close(i);
		execvp(process,argv);
		fprintf(stderr,"server: failed to exec %s process\n",process);
		_exit(-1);
	    default:
		if (( i = waitpid(pid, NULL, WNOHANG)) != pid && i != -1) {
		      return(pid);
		}
	  }
      return(SORT_FAIL);
}

/**************************************************************************************************/

int
do_pipe_process(
#if NeedFunctionPrototype
	   char           *process,
	   char           **my_argv,
	   int            *streams)
#else
      process,my_argv,streams)
      char           *process;
      char           **my_argv;
      int            *streams;
#endif
{
      int pid;
      int i, j;
      int fd[2][2];

/* diagnostics output */      
      if (DBX_val >= 5) {
	    fprintf(stderr,"server: do_pipe_process(%s",process);
	    for (i=0; my_argv[i] != NULL; i++)
		  fprintf(stderr," ,%s",my_argv[i]);
	    fprintf(stderr,")\n");
      }
      
      /* create pipe for communications between server and gui */      
      if (pipe(fd[0]) != 0) {
	    perror("server");
	    return(0);
      }
      if (pipe(fd[1]) != 0) {
	    perror("server");
	    (void) close(fd[0][0]);
	    (void) close(fd[0][1]);
	    return(0);
      }
      
/* start graphical user interface */
      switch(pid = fork1()) {
	  case -1:
	    fprintf(stderr,"server: failed to fork %s process\n",process);
	    for(i=0;i<2;i++) {
		  for(j=0;j<2;j++){
			(void) close(fd[i][j]);
		      }
	    }
	    return(0);
	  case 0:
	    (void) close(fd[1][0]);
	    (void) close(fd[0][1]);
	    /* in the new process assign stream numbers to ends of pipe */
	    (void) dup2(fd[0][0],streams[2]);
	    (void) dup2(fd[1][1],streams[3]);
	    /* clean up on first 32 streams if they are open */
	    for (i=3; i<32; i++)
		  if (i != streams[2] && i != streams[3])
			(void) close(i);
	    /* start new process */
	    execvp(process,my_argv);
	    fprintf(stderr,"server: failed to exec %s process\n",process);
	    _exit(-1);
	  default:
	    streams[0] = fd[1][0];
	    streams[1] = fd[0][1];
	    (void) close(fd[0][0]);
	    (void) close(fd[1][1]);
	    return(pid);
      }
/*      return(-1); */

}

/**************************************************************************************************/
int
start_gui(
#if NeedFunctionPrototype
		char *process, int argc, char **argv)
#else
      process, argc, argv)
      char  *process;
      int   argc;
      char  **argv;
#endif
{
      char *my_argv[100];
      char pid_str[24], dbx_str[24];
      int streams[4];
      int pid,i;

/* copy argument list */
      my_argv[0] = process;
      for(i=1; i<argc && i<91; i++)
	    my_argv[i] = argv[i];
      my_argv[i] = "-sort_pid";
      sprintf(pid_str,"%d",(int) getpid());
      my_argv[i+1] = pid_str;
      my_argv[i+2] = "-sdir";
      my_argv[i+3] = tmp_spectrum_dir;
      my_argv[i+4] = "-dbx";
      sprintf(dbx_str,"%d",DBX_val);      
      my_argv[i+5] = dbx_str;
      my_argv[i+6] = "-sproc";
      my_argv[i+7] = sunsort_proc;
      my_argv[i+8] = (char *)0;

/* set up stream number for new process*/
      streams[2] = STDIN_FILENO;
      streams[3] = STDOUT_FILENO;

/* start new process */      
      pid = do_pipe_process(process, my_argv, streams);

      if (pid > 0) {
	    if (waitpid(pid, NULL, WNOHANG) != pid) {
		  FDIN = streams[0];
		  FDDAT = FDOUT = streams[1];
		  return(pid);
	    }
      }
      
      if (pid != 0) {
	    (void) close(streams[0]);
	    (void) close(streams[1]);
      }

      return(-1);
}
/**************************************************************************************************/
int
start_sortproc(
#if NeedFunctionPrototype
		char *process, int argc, char **argv)
#else
      process, argc, argv)
      char  *process;
      int   argc;
      char  **argv;
#endif
{
      char *my_argv[100];
      char dbx_str[24];
      int streams[4];
      int pid,i;

/* copy argument list */
      my_argv[0] = process;
      for(i=1; i<argc && i<94; i++)
	    my_argv[i] = argv[i];
      my_argv[i] = "-sdir";
      my_argv[i+1] = tmp_spectrum_dir;
      my_argv[i+2] = "-dbx";
      sprintf(dbx_str,"%d",DBX_val);       
      my_argv[i+3] = dbx_str;
      my_argv[i+4] = (char *)0;


/* set up stream number for new process*/
      streams[2] = SORT_SERVER_INFD;
      streams[3] = SORT_SERVER_OUTFD;

/* start new process */      
      pid = do_pipe_process(process, my_argv, streams);

      if (pid > 0) {
	    if (waitpid(pid, NULL, WNOHANG) != pid) {
		  sort_proc_OK = SORT_TRUE;
		  FDSIN = streams[0];
		  FDSOUT = streams[1];
		  return(pid);
	    }
      }
      
      if (pid != 0) {
	    (void) close(streams[0]);
	    (void) close(streams[1]);
      }

      return(-1);
}
/**************************************************************************************************/

char *
make_tmp_dir(
#if NeedFunctionPrototype
		  void
#endif
		  )
{
      char *file;
      if ( (file = tempnam("/tmp",getenv("USER"))) == NULL)
	    return(NULL);
      if ( mkdir(file,0755)) {
	    free(file);
	    return(NULL);
      }
      return(file);
}

/**************************************************************************************************/
int
query_request(
#if NeedFunctionPrototype
		  char			*string)
#else
      string)
      char			*string;
#endif
{
#define REP_LEN 12
      char   reply[REP_LEN], *cptr;

/* query user */      
      fprintf(stderr,"%s (Y/N) ?  ",string);
      if (fgets(reply, REP_LEN, stdin) == NULL)
	    return(SORT_FALSE);

/* remove leading blanks from reply */
      cptr = rm_leading_blanks(reply,REP_LEN);
      
      if (*cptr == 'y' || *cptr == 'Y')
	    return(SORT_TRUE);
      return(SORT_FALSE);
}

/******************************************************************************/
char *
rm_leading_blanks(
#if NeedFunctionPrototype
		  char			*string,
		  int			len)
#else
      string,len)
      char			*string;
      int			len;
#endif
{
	register int	i;

	for (i=0; string[i] == ' ' && i <= len-1 ;i++)
		;
	return (string + i);
}

/**************************************************************************************************/
void
read_args(
#if NeedFunctionPrototype
     int		argc,
     char		**argv)
#else
      argc, argv)
      int		argc;
      char		**argv;
#endif
{
     int i; 
     for(i=1; argv[i] != NULL; i++) {
	   if (strncmp(argv[i],"-nograf",7) == 0) {
		 nograf = SORT_TRUE;
		 fputs("server: nograf option registered\n",stderr);
	   }
	   else if (strncmp(argv[i],"-dbx",4) == 0) {
		 if (argv[i+1] != NULL) {
		       DBX_val = (int) atof(argv[i+1]);
		       fprintf(stderr,"server: diagnostics level set to %d\n",DBX_val);
		 }
	   }
	   else if (strncmp(argv[i],"-nogui",6) == 0) {
		 nogui = SORT_TRUE;
		 fputs("server: nogui option registered\n",stderr);
	   }
	   else if (strncmp(argv[i],"-batch",6) == 0) {
		  if (argv[i+1] != NULL) {
			char buf[BUFSIZ], reply[BUFSIZ];
			(void) strcpy(buf,argv[i+1]);
			if (batch_read(buf,reply) == SORT_TRUE) {
			      nograf = nogui = SORT_TRUE;
			      batch_mode = SORT_TRUE;
			      fputs("server: batch option registered\n",stderr);
			      return;
			}
		  }
	   }
	   else if (strncmp(argv[i],"-sproc",5) == 0) {
		 if (argv[i+1] != NULL) {
		       sunsort_proc = strdup(argv[i+1]);
		       fprintf(stderr,"server: sunsort to load process %s\n",sunsort_proc);
		 }
	   }
     }
     return;
}

/**************************************************************************************************/
void
main_event_loop(
#if NeedFunctionPrototype
	  void
#endif
		)
{
      char     buf[BUFSIZ];
      char     reply[BUFSIZ];
      int      fd;
      int      ret_code;

      for(;;) {
	    /* check for data on input stream */
	    if ( (fd = check_input()) != -1) 
		  ret_code = read_input(fd, buf);
	    else
		  ret_code = SORT_FALSE;

	    /* check status of child processes (if any) */
	    check_children();

	    /* if command has been read ... try and perform it */
	    if (ret_code == SORT_TRUE) {
		  if ( (ret_code = decode_command(buf,reply)) != SORT_FAIL) {
			write_reply(ret_code,reply,fd);
		  }
		  /* end batch processing if error was detected in command from batch stream */
		  if (batch_mode == SORT_TRUE && fd == FDBAT && ret_code != SORT_TRUE ) {
			sprintf(buf,"failed batch command");
			end_batch(buf,reply);
		  }
	    }

      }
}

int
check_input(
#if NeedFunctionPrototype
	  void)
#else
      )
#endif
{
      fd_set          rset;
      struct timeval  timeout;
      int             nfds;
      int             ret_code;

/* set up fd set for call to  select(2) */
      nfds = MAX(FDIN, FDBAT);
      FD_ZERO(&rset);
      FD_SET(STDIN_FILENO,&rset);
      FD_SET(FDIN,&rset);
      if (Sorting_data(NULL) == SORT_FALSE)
	    FD_SET(FDBAT,&rset);
      
/* set up timer */
      timeout.tv_sec = 0;
      timeout.tv_usec = poll_period;

/* call select to see which input streams have data on them */
      ret_code = select(nfds+1, &rset, NULL, NULL, &timeout);
      if (ret_code == 0) {
	    /* timeout */
	    return(-1);
      }
      else if (ret_code > 0) {
	    /* data to read */
	    if (FD_ISSET(FDIN,&rset))
		  return(FDIN);
	    else if (FD_ISSET(STDIN_FILENO,&rset))
		  return(STDIN_FILENO);
	    else if (FD_ISSET(FDBAT,&rset) && Sorting_data(NULL) != SORT_TRUE) 
			return(FDBAT);
      }
      else {
	    /* error condition */
	    if (errno != EINTR) perror("server...(select call)");
      }
      return(-1);
}

int
read_input(
#if NeedFunctionPrototype
	  int fd, char *buf)
#else
      fd, buf)
      int   fd;
      char  *buf;
#endif
{
      int      n;
      int      cptr = 0;

      for(;;) {
	    if ( (n = read(fd, &buf[cptr], 1)) == 1) {
		  if (buf[cptr++] == '\n' || cptr == BUFSIZ) {
			buf[cptr-1] = '\0';
			return(SORT_TRUE);
		  }
	    }
	    else if ( n<0 ) {
		  fprintf(stderr,"server returned error on command read\n");
		  perror("server");
		  return(SORT_FAIL);
	    }
	    else {
		  /* assume that input is dead and reset input to tty */
		  if (batch_mode == SORT_TRUE) {
			end_batch("EOF detected",buf);
		  }
		  else if (nogui == SORT_FALSE) {
			FDIN = STDIN_FILENO;
			FDDAT = FDOUT = STDOUT_FILENO;
		  }
		  else
			fprintf(stderr,"server returned EOF on command read\n");
		  return(SORT_FAIL);
	    }
      }
}

void
write_reply(
#if NeedFunctionPrototype
	  int   ret_code, char *reply, int in_fd)
#else
      ret_code, reply, in_fd)
      int   ret_code;
      char  *reply;
      int   in_fd;
#endif
{
      int fd;
      struct   msg_buf {
	    int      ret_code;
	    int      buf_len;
	    char     reply[BUFSIZ];
      } message;

/* check source of user input ... if from terminal return to terminal */      
      if (in_fd == STDIN_FILENO)
	    fd = STDOUT_FILENO;
      else if (in_fd == FDIN)
	    fd = FDOUT;
      else
	    fd = FDDAT;
      
      message.buf_len = MIN(strlen(reply),BUFSIZ);

/* write out reply ... add status code if replying to other than terminal */      
      if (fd == STDOUT_FILENO) {
	    (void) write(fd,reply,message.buf_len);
      }
      else {
	    message.ret_code = (ret_code == SORT_TRUE) ? 0:1;
	    strncpy(message.reply, reply, message.buf_len);
	    (void) write(fd, (char *)&message, 2*sizeof(int)+message.buf_len);
      }
      return;
}

void
check_children(
#if NeedFunctionPrototype
	  void)
#else
      )
#endif
{
      int   statusp;

/* check sort process */
      if (sort_proc_OK == SORT_TRUE) {
	    if (waitpid(sort_pid,&statusp,WNOHANG) == sort_pid) {
		  sort_proc_OK = SORT_FALSE;
		  if (FDSIN != STDIN_FILENO) {
			(void) close(FDSIN);
			FDSIN = STDIN_FILENO;
		  }
		  if (FDSOUT != STDOUT_FILENO) {
			(void) close(FDSOUT);
			FDSOUT = STDOUT_FILENO;
		  }
		  /* check to see whether there is a sort thread alive */
		  if (Sorting_data(NULL) == SORT_TRUE) {
			stop_sort_threads();
		  }
		  fprintf(stderr,
			  "server: sort process has exited\n");
	    }
      }   

/* check gui */
      if (nogui == SORT_FALSE) {
	    if (waitpid(gui_pid,&statusp,WNOHANG) == gui_pid) {
		  nogui = SORT_TRUE;
		  if (FDIN != STDIN_FILENO) {
			(void) close(FDIN);
			FDIN = STDIN_FILENO;
		  }
		  if (FDOUT != STDOUT_FILENO) {
			(void) close(FDOUT);
			FDDAT = FDOUT = STDOUT_FILENO;
		  }
		  if (statusp != 0) fprintf(stderr,
			  "server: gui process has exited\n");
	    }
      }
      
/* check 1d display process */
      if (graf_proc1_OK == SORT_TRUE) {
	    if (waitpid(graf_1d_pid,&statusp,WNOHANG) == graf_1d_pid) {
		  graf_proc1_OK = SORT_FALSE;
		  fprintf(stderr,
			  "server: 1d display process has exited\n");
	    }
      }
      
/* check 2d display process */
      if (graf_proc2_OK == SORT_TRUE) {
	    if (waitpid(graf_2d_pid,&statusp,WNOHANG)  == graf_2d_pid) {
		  graf_proc2_OK = SORT_FALSE;
		  fprintf(stderr,
			  "server: 2d display process has exited\n");
	    }
      }

      return;
}

/**************************************************************************************************/
/*
 *  list of available commands and corresponding functions called
 */
static struct {
      char *string;
      int  (*func)();
      char *help_string;
} commands[] = {
      {"batch", batch_read, "batch <batch file name>"},
      {"clear", clear_spectra, "clear <all/1d/2d/selected>"},
      {"close", close_sort_file, "close [tape/remote/disk]"},
      {"debugsort",debugsort,"debugsort"},
      {"display1d", display_1d, "display1d <#1d spectrum nos> + up to 3 others"},
      {"display2d", display_2d, "display2d <#2d spectrum nos>"},
      {"editsort", editsort, "editsort <editor> <sort program name>"},
      {"eject", eject_func, "eject [access mode = r/w]"},
      {"end", end_batch, "end [batch]"},
      {"exit", exit_sunsort, "exit"},
      {"help", help_func, "help [command name]"},
      {"killsort", killsort, "killsort"},
      {"load", load_spectra, "load <all/windows/variables/selected> <load directory> <format=sdb/eg/xy> [#add=1]"},
      {"loadsort", loadsort, "loadsort [-keep] [sort module name]"},
      {"loadspec", loadspec, "loadspec <spectra file>"},
      {"makesort", makesort, "makesort <sort source name> [debug] [[-o sort name] other options...]"},
      {"newtape", newWtape, "newtape <volume name> <#drive nos>"},
      {"open", open_sort_file, "open <tape/remote/disk> <sort file name>"},
      {"overlay1d", overlay_1d, "overlay1d <#1d spectrum nos> + up to 3 others"},
      {"printsenv", print_sortenv, "printsenv"},
      {"printvars", print_vars, "printvars"},
      {"refresh", refresh_func, "refresh <#seconds>"},
      {"reset", reset_command, "reset <graphics/gui>"},
      {"save", save_spectra, "save <all/windows/variables/selected> <save directory> <format=sdb/eg/xy> [#overwrite=1]"},
      {"setsenv", set_sortenv, "setsenv <variable name> <new value>"},
      {"setwin", setwindow, "setwin <#window nos> on <#2d spectrum nos>"},
      {"showwin", showwindow, "showwin <#window nos>"},
      {"sleep", sleep_func, "sleep <#seconds>"},
      {"sort", sort_data, "sort <#skip records> <#number of records to sort>"},
      {"status", status_func, "status"},
      {"tapemount", tapemount, "tapemount <volume name or nonansi> <#drive nos> <access mode = r/w>"},
      {"tapemove", tapemove, "tapemove <#nos files>"},
      {"tapeumount", tapeumount, "tapeumount <access mode = r/w>"},
      {"update", update_gui, "update GUI to reflect changes in variables"},
      {"var", change_var, "var <#variable nos> <#new value>"},
      {"vars", print_vars, "vars (same command as printvars)"},
      {"viewwin", viewwindow, "viewwin <#window nos> <inside/outside> <#2d spectrum nos>"},
      {"wclose", close_filt_file, "wclose [tape/disk]"},
      {"wopen", open_filt_file, "wopen <tape/disk> <write file name>"},
      { NULL, NULL, NULL}
};
/**************************************************************************************************/
int
decode_command(
#if NeedFunctionPrototype
	  char *ipbuf, char *opbuf)
#else
      ipbuf, opbuf)
      char     *ipbuf;
      char     *opbuf;      
#endif
{
      char   com_str[BUFSIZ];
      char   *opts;
      char   *charptr;
      int    i;

/* remove any blanks from the beginning of the command line */
      ipbuf = rm_leading_blanks(ipbuf,BUFSIZ);
      
/* begin to decode command line	*/	      
      if (sscanf(ipbuf,"%s",com_str) <= 0) {
	    /* check that command line is not just a CR */      
	    if (ipbuf[0] == '\n' || ipbuf[0] == '\0') {
		  fprintf(stderr,"Sunsort > ");
		  sprintf(opbuf,"wibble > \n");
		  return(SORT_FAIL);
	    }
	    else
		  sprintf(opbuf,"error decoding command string %s\n",ipbuf);
	    return(SORT_FALSE);
      }
      
      opts = ipbuf + strlen(com_str);
/* convert command string to lower case */
      for (charptr = com_str; *charptr != '\0'; charptr++)
	    if (isupper((int) *charptr)) tolower((int) *charptr);
      if (DBX_val >= 5)
	    fprintf(stderr,"server: command string = %s, opts = %s\n",com_str,opts);

/* check that command line is a comment line */      
      if (com_str[0] == '#')
	    return(comment_line(ipbuf+1,opbuf));
      
/* using command call appropriate function to act on it */      
      for(i=0; commands[i].string != NULL ; i++) {
	    if (strcmp(com_str,commands[i].string) == 0) {
		  if (commands[i].func != NULL)
			return((*commands[i].func)(opts, opbuf));
		  else {
			fprintf(stderr,"server: null command operation ... seek help!\n");
			return(SORT_FAIL);
		  }
	    }
      }

      fprintf(stderr,"server: unrecognised command << %s >> passing it to sh\n",com_str);
      if (system(ipbuf) == 127) {
	    fprintf(stderr,"server: could not execute shell\n");
      }
      else
	    fprintf(stderr,"SunSort > ");
      return(SORT_FAIL);
}
/**************************************************************************************************/
/*
 *  display help information
 */
int
help_func(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      int i,j;
      char help_str[BUFSIZ];
      if (DBX_val >= 5)
	    fprintf(stderr,"server: help_func called with arg %s\n",opt_str);

      if (sscanf(opt_str,"%s",help_str) != 1) {
	    fprintf(stderr,"The following commands are available:\n");
	    for(i=j=0; commands[i].string != NULL ; i++) {
		  fprintf(stderr,"%-12s \t",commands[i].string);
		  if (++j >= 4) {
			fprintf(stderr,"\n");
			j = 0;
		  }
	    }
	    if (j != 0) fprintf(stderr,"\n");
      }
      else {
	    char *help_ptr = rm_leading_blanks(help_str,BUFSIZ);
	    for(i=0; commands[i].string != NULL ; i++) {
		  if (strcmp(help_ptr,commands[i].string) == 0) {
			fprintf(stderr,"=> %s\n",commands[i].help_string);
			break;
		  }
	    }
	    if (commands[i].string == NULL) {
		  fprintf(stderr,"help not available for %s\n",help_ptr);
	    }
      }
      sprintf(reply,"\n");
      return(SORT_TRUE);
}

/*
 *  report the state of the sort process
 */
int
status_func(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      sort_IO_t *ptr  = sort_io_data;
      int       state = ptr->IO_state;

      fprintf(stderr,"*********************************************************\n");
/* reading information */      
      if ( (state & R_TAPE_MOUNTED) && !(state & READING_TAPE)) {
	    (void) dtstate(0);
      }
      switch (state & READING_IN) {
	    case READING_DISK:
	    (void) ddstate(0);
	    fprintf(stderr,"Records read in so far = %d\n\n",(int) *sorted_records); 
	    break;
	    case READING_TAPE:
	    (void) dtstate(0);
	    fprintf(stderr,"Records read in so far = %d\n\n",(int) *sorted_records);
	    break;
	    case READING_REMOTE:
	    fprintf(stderr,"Reading from remote source\n");
	    (void) remote_op(STATUS_REMOTE);
	    break;
	    case READING_DEMON:
	    demonstatus();
	    fprintf(stderr,"Records read in so far = %d\n\n",(int) *sorted_records); 
	    break;
	    default:
	    fprintf(stderr,"No read file open\n\n");
	    break;
      }

/* writing information */
      if ( (state & W_TAPE_MOUNTED) && !(state & WRITING_TAPE)) {
	    (void) dtstate(1);
      }
      switch (state & WRITING_OUT) {
	    case WRITING_DISK:
	    (void) ddstate(1);
	    fprintf(stderr,"Records filtered out so far = %d\n\n",(int) *filtered_records);
	    break;
	    case WRITING_TAPE:
	    (void) dtstate(1);
	    fprintf(stderr,"Records filtered out so far = %d\n\n",(int) *filtered_records); 
	    break;
	    default:
	    fprintf(stderr,"No write file open\n\n");
	    break;
      }
      
      fprintf(stderr,"Last save to directory => %s \nLast load from directory => %s\n\n",
	      dirs_io_data->save_dir,dirs_io_data->load_dir);

      if (sort_proc_OK == SORT_TRUE) {
	    fprintf(stderr,"sort process is running\n");
	    fprintf(stderr,"sort process name = %s: pid = %ld\n", sunsort_proc,sort_pid);
	    fprintf(stderr,"sort process to use decoder function named => %s\n\n",ptr->decoder);
      }
      else {
	    fprintf(stderr,"sort process is NOT running\n\n");
      }
      if (nogui != SORT_TRUE) {
	    fprintf(stderr,"gui process is running: pid = %ld\n\n",gui_pid);
      }
      else {
	    fprintf(stderr,"gui process is NOT running\n\n");
      }
      if (graf_proc1_OK == SORT_TRUE) {
	    fprintf(stderr,"1d graphics process is running: pid = %ld\n",graf_1d_pid);
      }
       else {
	    fprintf(stderr,"1d graphics process is NOT running\n");
      }
      if (graf_proc2_OK == SORT_TRUE) {
	    fprintf(stderr,"2d graphics process is running: pid = %ld\n",graf_2d_pid);
      }
      else {
	    fprintf(stderr,"2d graphics process is NOT running\n");
      }      

      fprintf(stderr,"*********************************************************\n");
      sprintf(reply,"\n");

      return(SORT_TRUE);
}

/*
 *   write down pipe connected to sort process
 */
void
write_to_sortproc(
#if NeedFunctionPrototype
	  char *message)
#else
      message)
      char *message
#endif
{
      if (sort_proc_OK != SORT_TRUE) {
	    fprintf(stderr,"server: attempt to communicate with nonexistent sort process\n");
	    return;
      }
       if (write(FDSOUT, message, strlen(message)+1) != (strlen(message)+1))
	    fprintf(stderr,"server: attempt to communicate with sort process failed\n");
      return;
}

/*
 *  load new sort process...
 *                          1) kill old sort process if any
 *                          2) delete old spectra
 *                          3) start new sort process and load new spectra
 *                          4) update gui display lists
 */
int
loadsort(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      char *my_argv[2];
      char *token, *lasts;      
      int  keep_opt = SORT_FALSE;
      
      if (DBX_val >= 5)
	    fprintf(stderr,"server: loadsort function called with args => %s\n",opt_str);
      
/* check that not trying to sort */      
      if (Sorting_data(NULL) != SORT_FALSE) {
	    sprintf(reply,"invalid command while sorting\n");
	    return(SORT_FALSE);
      }

/* check so see if there is a sort process alive already */
      if (sort_proc_OK == SORT_TRUE) {
	    if (killsort(opt_str,reply) != SORT_TRUE) {
		  return(SORT_FAIL);
	    }
      }

/* check to see if a new name has been provided and if we keep the present spectra */      
      adjust_white_space(opt_str);
      token = strtok_r(opt_str, " ", &lasts);
      while (token != NULL) {
	    if (strncmp(token,"-keep",5) == 0) {
		  keep_opt = SORT_TRUE;
	    }
	    else{
		  if (sunsort_proc != NULL)
			free(sunsort_proc);
		  sunsort_proc = strdup(token);
	    }
	    token = strtok_r(NULL, " ", &lasts);
      }      

      if (keep_opt == SORT_FALSE) {
/* delete any existing spectra */
	    if (delete_spectra() != 0) {
		  fprintf(stderr,"server: delete spectra in loadsort call returned error\n");
		  fprintf(stderr,"server: ** You are advised to EXIT sunsort **\n");
	    }

/* create new spectrum information */      
	    if (read_spectra_file(sunsort_proc) == SORT_FAIL) {
		  fprintf(stderr,"server: error occured during the reading of the spectra file\n");
		  sprintf(reply,"error occured during the reading of the spectra file\n");
		  return(SORT_FALSE);
	    }
      
/* refresh gui display list */
	    if (nogui != SORT_TRUE && gui_pid > 0) {
		  kill(gui_pid, SIGUSR2);
	    }
      }
      
/* start new sort process */
      my_argv[0] = sunsort_proc;
      my_argv[1] = (char *)0;
      if ( (sort_pid = start_sortproc(sunsort_proc, 1, my_argv)) == -1) {
	    sort_proc_OK = SORT_FALSE;
	    FDSIN = STDIN_FILENO;
	    FDSOUT = STDOUT_FILENO;
	    sprintf(reply,"failed to load new sort process %s\n",sunsort_proc);
	    return(SORT_FALSE);
      }

      sprintf(reply,"loadsort started %s\n",sunsort_proc);

      if(keep_opt == SORT_FALSE)
          fprintf(stderr, "server: spectra erased.\n");
      else
          fprintf(stderr, "server: spectra preserved.\n");

      return(SORT_TRUE);
}

/*
 *  run makesort to generate new sort program executable
 */
int
makesort(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      char *my_argv[100];
      int  statusp;
      char *token, *lasts;
      int i;
      
      if (DBX_val >= 5)
	    fprintf(stderr,"server: makesort function called with args => %s\n",opt_str);

/* check that not trying to sort */      
      if (Sorting_data(NULL) != SORT_FALSE) {
	    sprintf(reply,"invalid command while sorting\n");
	    return(SORT_FALSE);
      }
      
/* check so see if there is a sort process alive already */
/* this isn't needed but is cleaner if makesort ends up replacing the disk file currently loaded */
      if (sort_proc_OK == SORT_TRUE) {
	    if (killsort(opt_str,reply) != SORT_TRUE) {
		  return(SORT_FAIL);
	    }
      }
      
/* setup argument vector */
      my_argv[0] = "makesort";
      adjust_white_space(opt_str);
      my_argv[1] = token = strtok_r(opt_str, " ", &lasts);
      for (i=2; token != NULL || i<99; i++) {
	    my_argv[i] = token = strtok_r(NULL, " ", &lasts);
      }
      my_argv[i] = (char *)0;

/* perform makesort */      
      if ( (i = do_process(my_argv[0], my_argv)) == -1) {
	    sprintf(reply,"error during start of makesort process\n");
	    return(SORT_FALSE);
      }
      if (waitpid(i, &statusp, 0) != i) {
	    perror("server makesort");
	    sprintf(reply,"wait for makesort process returned before process exited\n");
	    return(SORT_FALSE);
      }

      if (DBX_val >= 1)
	    fprintf(stderr,"server: makesort returned with status code = %d\n",statusp);

/* warn of non zero exit code */      
      if (statusp != 0) {
	    sprintf(reply,"makesort returned non zero exit code %d\n",statusp);
	    return(SORT_FALSE);
      }

      sprintf(reply,"makesort finished ok\n");
      return(SORT_TRUE);
}

/*
 *  kill off current sort process
 */
int
killsort(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      int   statusp;
      int   reset = SORT_FALSE;
      char  buf[BUFSIZ];
      
      if (DBX_val >= 5)
	    fprintf(stderr,"server: killsort function called with args => %s\n",opt_str);

      if (sscanf(opt_str,"%s",buf) == 1) {
	    if (strncmp(buf,"reset",5) == 0)
		  reset = SORT_TRUE;
      }

/* check to see if there is a sort process running .. if so kill it */      
      if (sort_proc_OK == SORT_TRUE) {
	    if (sort_pid > 0) {
		  kill(sort_pid, SIGKILL);
		  (void) waitpid(sort_pid,&statusp,0);
	    }
	    sort_proc_OK = SORT_FALSE;
	    if (FDSIN != STDIN_FILENO) {
		  (void) close(FDSIN);
		  FDSIN = STDIN_FILENO;
	    }
	    if (FDSOUT != STDOUT_FILENO) {
		  (void) close(FDSOUT);
		  FDSOUT = STDOUT_FILENO;
	    }
      }
      else {
	    if (reset == SORT_TRUE)
		  (void) reset_all_buffers();
	    sprintf(reply,"no sort process running!\n");
	    return(SORT_FALSE);
      }

/* check to see whether we are sorting ... if so reset buffers etc */      
      if (Sorting_data(NULL) == SORT_TRUE) {
	    stop_sort_threads();
      }
      else if (reset == SORT_TRUE) {
	    (void) reset_all_buffers();
      }
      
      sprintf(reply,"killsort suceeded\n");
      return(SORT_TRUE);
}

/*
 *  start debugger on current sort process
 */
int
debugsort(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      char   *data;
#ifdef SVR4
      char  *debugger = "debugger";
      extern void *background_thread(void *data);
#else
      char  *debugger = "dbxtool";
      extern void *background_thread();
#endif
      
      if (DBX_val >= 5)
	    fprintf(stderr,"server: debugsort function called with args => %s\n",opt_str);
      
/* check that sort process is alive */
      if (sort_proc_OK != SORT_TRUE) {
	    sprintf(reply,"no sort process running ... perform loadsort first\n");
	    return(SORT_FALSE);
      }

/* allocate memory for argument to debug_thread */      
      data = (char *) malloc(strlen(debugger)+strlen(sunsort_proc)+24);
      if (data == NULL) {
	    sprintf(reply,"server: memory allocation failed in debugsort call\n");
	    return(SORT_FALSE);
      }
      (void) sprintf(data,"%s %s %ld",debugger, sunsort_proc, sort_pid);

/* start a new thread to fork debugger */      
      if (thr_create(NULL, NULL, background_thread, (void *) data, THR_DETACHED, NULL )) {
	    sprintf(reply,"error while trying to start debug thread\n");
	    return(SORT_FALSE);
      }

      sprintf(reply,"debugsort started ok\n");
      return(SORT_TRUE);
}

/*
 *  edit specified sort file
 */
int
editsort(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      char *data;
 #ifdef SVR4
      extern void *background_thread(void *data);
#else
      extern void *background_thread();
#endif     
      if (DBX_val >= 5)
	    fprintf(stderr,"server: editsort function called with args => %s\n",opt_str);

/* allocate memory for argument to debug_thread */      
      data = (char *) malloc(strlen(opt_str));
      if (data == NULL) {
	    sprintf(reply,"server: memory allocation failed in editsort call\n");
	    return(SORT_FALSE);
      }
      (void) strcpy(data, opt_str);
      
/* start a new thread to fork editor */      
      if (thr_create(NULL, NULL, background_thread, (void *) data, THR_DETACHED, NULL )) {
	    sprintf(reply,"error while trying to start edit thread\n");
	    return(SORT_FALSE);
      }

      sprintf(reply,"editsort started ok\n");
      return(SORT_TRUE);
}

/*
 *  reinitialize the spectra and variables held in sunsort
 */
int
loadspec(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      char buf[BUFSIZ];
      
      if (DBX_val >= 5)
	    fprintf(stderr,"server: loadspec function called with args => %s\n",opt_str);

/* check that not trying to sort */      
      if (Sorting_data(NULL) != SORT_FALSE) {
	    sprintf(reply,"invalid command while sorting\n");
	    return(SORT_FALSE);
      }

/* check to see if spectrum file really exists */
      if (sscanf(opt_str,"%s",buf) == 1) {
	    char buf2[BUFSIZ];
	    (void) sprintf(buf2,"%s.spec",buf);
	    if (inquire(buf2) != SORT_reg) {
		  sprintf(reply,"%s does not exist or is not a regular file\n",buf2);
		  return(SORT_FALSE);
	    }
      }
      else {
	    sprintf(reply,"invalid options string in loadspec call\n");
	    return(SORT_FALSE);
      }
      
/* delete any existing spectra */
      if (delete_spectra() != 0) {
	    fprintf(stderr,"server: delete spectra in loadspec call returned error\n");
	    fprintf(stderr,"server: ** You are advised to EXIT sunsort **\n");
      }

/* create new spectrum information */      
      if (read_spectra_file(buf) == SORT_FAIL) {
	    fprintf(stderr,"server: error occured during the reading of the spectra file\n");
	    sprintf(reply,"error occured during the reading of the spectra file\n");
	    return(SORT_FALSE);
      }
      
/* refresh gui display list */
      if (nogui != SORT_TRUE && gui_pid > 0) {
	    kill(gui_pid, SIGUSR2);
      }
      
/* refresh info in sort process */ 
      if (sort_proc_OK == SORT_TRUE) {
	    write_to_sortproc("reload spectra");
      }
      
      sprintf(reply,"loadspec finished OK\n");
      return(SORT_TRUE);
}
