/*
 *   signal handling routines
 *   prev modified   25/1/95
 *   prev modified   24/3/95
 *   last modified   15/5/95
 *
 */

#include <stdio.h>
#include <string.h>
#include <dirent.h>
#include <signal.h>
#include <memory.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>

#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/resource.h>

#include <errno.h>

#include "sort_extern.h"
#include "sort_mem.h"

#ifdef SVR4
#define NeedFunctionPrototype 1
extern int *remap_2d_spectrum(spec_2d_t *p2d, int newsize);
#else
#define NeedFunctionPrototype 0
extern int *remap_2d_spectrum();
#endif

static char      inbuf[BUFSIZ], outbuf[BUFSIZ];

static void
sigint_handler(
#if NeedFunctionPrototype
	   int sig)
#else
      sig, code, scp, addr)
      int sig;
      int code;
      struct sigcontext *scp;
      char *addr;
#endif
{
      if (DBX_val >= 1)
	    fprintf(stderr,"server: SIGINT receieved\n");
      (void) Stop_interupt(SORT_SET, SORT_TRUE);
      if (batch_mode == SORT_TRUE) {
	    strcpy(inbuf,"end batch");
	    end_batch(inbuf, outbuf);
      }
      return;
}

static void
sigusr1_handler(
#if NeedFunctionPrototype
	   int sig)
#else
      sig, code, scp, addr)
      int sig;
      int code;
      struct sigcontext *scp;
      char *addr;
#endif
{
      if (DBX_val >= 1)
	    fprintf(stderr,"server: SIGUSR1 receieved ... software end of data in sort process\n");
      (void) Stop_interupt(SORT_SET, SORT_TRUE);
      return;
}

static void
sigpipe_handler(
#if NeedFunctionPrototype
	   int sig)
#else
      sig, code, scp, addr)
      int sig;
      int code;
      struct sigcontext *scp;
      char *addr;
#endif
{
      fprintf(stderr,"server: SIGPIPE receieved\n");
      return;
}

void
sigwin2d_handler(
#if NeedFunctionPrototype
	   int sig)
#else
      sig, code, scp, addr)
      int sig;
      int code;
      struct sigcontext *scp;
      char *addr;
#endif
{
	struct sm_data_buf	stack;	
	spec_2d_t		*p2d;

      if (DBX_val >= 1)
	    fprintf(stderr,"server: SIGUSR2 receieved\n");

/* non ipc system */
	if ( read_from_pipe(".SORT_pipe_W") == SORT_FAIL) 
		fprintf(stderr,"server: error during reading of data from graf process\n");

/* copy spectrum data from shared memory segment into stack */
	(void) memcpy((char *) &stack, (char *) pshm, sizeof(struct sm_data_buf));
	if ( DBX_val >= 1) fprintf(stderr,"server: sigusr2 handler for win = %d\n",stack.win2d);

/* get pointer to window data structure */
	p2d = get_nametab_2d()[stack.win2d];
	if (! p2d) {
		fprintf(stderr,"server: undefined 2D window %d\n",stack.win2d);
		return;
	}
	
/* check that the sizes of the new window and its memory storage are compatible */
	if (stack.size != p2d->sp->size) {
		p2d->data = (int *) remap_2d_spectrum(p2d, stack.size );
		if (! p2d->data) {
			perror("sig2_handler =>new window");
			fprintf(stderr,
				"server: window %d ERASED due to error...seek help\n",stack.win2d);
			(void) delete_2d(p2d->nos);
			if (sort_proc_OK == SORT_TRUE) {
			      write_to_sortproc("check spectra");
			}
			return;
		}
		if (sort_proc_OK == SORT_TRUE) {
		      write_to_sortproc("check spectra");
		}
	}
	
/* ok got this far so lets copy the window into the spectrum storage */
	(void) memcpy((char *) p2d->data, (char *) stack.array,
		 sizeof(int)*p2d->sp->size*p2d->sp->size);
	(void) signal(SIGUSR2, sigwin2d_handler);
      return;
}

void
sigusr2_handler(
#if NeedFunctionPrototype
	   int sig)
#else
      sig, code, scp, addr)
      int sig;
      int code;
      struct sigcontext *scp;
      char *addr;
#endif
{
      if (DBX_val >= 1)
	    fprintf(stderr,"server: SIGUSR2 receieved\n");
      (void) signal(SIGUSR2, sigwin2d_handler);
      return;
}

static void
sigsegv_handler(
#if NeedFunctionPrototype
	   int sig)
#else
      sig, code, scp, addr)
      int sig;
      int code;
      struct sigcontext *scp;
      char *addr;
#endif
{
      fprintf(stderr,"server: SIGSEGV receieved\n");
/* try and kill off other child processes */      
      kill(-getpid(),SIGTERM);
      abort();
      return;
}

static void
sigbus_handler(
#if NeedFunctionPrototype
	   int sig)
#else
      sig, code, scp, addr)
      int sig;
      int code;
      struct sigcontext *scp;
      char *addr;
#endif
{
      fprintf(stderr,"server: SIGBUS receieved\n");
/* try and kill off other child processes */      
      kill(-getpid(),SIGTERM);
      abort();
      return;
}

static void
sigterm_handler(
#if NeedFunctionPrototype
	   int sig)
#else
      sig, code, scp, addr)
      int sig;
      int code;
      struct sigcontext *scp;
      char *addr;
#endif
{
      if (DBX_val >= 1)
	    fprintf(stderr,"server: SIGTERM received\n");

      (void) exit_sunsort(inbuf, outbuf);
      
      exit(1);
}

/************************************************************************************************/
int
masksig(
#if NeedFunctionPrototype
		int how, int signal)
#else
      how, signal)
      int   how;
      int signal;
#endif
{
      sigset_t  sigset;

      if (sigemptyset(&sigset))
	    return(-1);
      if (sigaddset(&sigset,signal))
	    return(-1);
      if (sigprocmask(how,&sigset,NULL))
	    return(-1);
      return(0);
}

/************************************************************************************************/
#define MAX_SPEC_DISPLAY 4
extern int displayed_spectrum[MAX_SPEC_DISPLAY+2];

static void
sigtimer_handler(
#if NeedFunctionPrototype
	   int sig)
#else
      sig, code, scp, addr)
      int sig;
      int code;
      struct sigcontext *scp;
      char *addr;
#endif
{
      if (DBX_val >= 5)
	    fprintf(stderr,"server: SIGALRM receieved\n");

/* if refresh option enabled, update displayed spectra */      
      if (refresh_time != REFRESH_OFF) {
	    int more, loop;

	    inbuf[0] = '\0';
	    if (DBX_val >= 5)
		  fputs("server: refresh displayed spectra event received\n",stderr);
	    /* refresh 1d spectra */
	    if ( (more = displayed_spectrum[MAX_SPEC_DISPLAY]) != 0) {
		  for (loop = 0; loop < more; loop++) {
			sprintf(inbuf,"%s %d ",inbuf, displayed_spectrum[loop]);
		  }
		  (void) display_1d(inbuf,outbuf);
	    }
	    /* refresh 2d spectra */
	    if (displayed_spectrum[MAX_SPEC_DISPLAY+1] != 0) {
		  sprintf(inbuf,"%d\n",displayed_spectrum[MAX_SPEC_DISPLAY+1]);
		  (void) display_2d(inbuf,outbuf);
	    }
      }

      return;
}

void
setup_intervaltimer(
#if NeedFunctionPrototype
		int bool)
#else
      bool)
      int   bool;
#endif
{
      struct itimerval value;
      struct sigaction newvec;

      (void) sigemptyset(&newvec.sa_mask);
      
      if (bool == SORT_TRUE) {
	    newvec.sa_handler = sigtimer_handler;
#ifdef SVR4
	    newvec.sa_flags = SA_RESTART;
#else
	    newvec.sa_flags = 0;
#endif
	    (void) sigaction(SIGALRM,&newvec,NULL);
	    
	    value.it_value.tv_usec = value.it_interval.tv_usec = 0;
	    value.it_value.tv_sec = value.it_interval.tv_sec = refresh_time;
	    if (setitimer(ITIMER_REAL,&value,NULL))
		  fprintf(stderr,"server: interval timer failed to setup\n");
      }
      else {
	    value.it_value.tv_usec = value.it_interval.tv_usec = 0;
	    value.it_value.tv_sec = value.it_interval.tv_sec = 0;
	    if (setitimer(ITIMER_REAL,&value,NULL))
		  fprintf(stderr,"server: interval timer failed to setup\n");
#ifdef SVR4
	    (void) sigset(SIGALRM, SIG_IGN);
#else
	    (void) signal(SIGALRM, SIG_IGN);
#endif
      }
      return;
}

/************************************************************************************************/

int
setup_sighandlers(
#if NeedFunctionPrototype
		  void)
#else
      )
#endif
{
      struct sigaction newvec;

      (void) sigemptyset(&newvec.sa_mask);

/* setup handler for SIGPIPE */      
#ifdef SVR4	    
      newvec.sa_flags = 0;
#else
      newvec.sa_flags = SA_INTERRUPT;
#endif
      newvec.sa_handler = sigpipe_handler;
      if (sigaction(SIGPIPE,&newvec,NULL) == -1)
	    return(-1);

/* setup handler for SIGTERM */
      newvec.sa_handler = sigterm_handler;
      if (sigaction(SIGTERM,&newvec,NULL) == -1)
	    return(-1);

/* setup handler for SIGSEGV */
      newvec.sa_handler = sigsegv_handler;
      if (sigaction(SIGSEGV,&newvec,NULL) == -1)
	    return(-1);
      
/* setup handler for SIGBUS */
      newvec.sa_handler = sigbus_handler;
      if (sigaction(SIGBUS,&newvec,NULL) == -1)
	    return(-1);      

/* setup handler for SIGINT ^C */      
#ifdef SVR4	    
      newvec.sa_flags = SA_RESTART;
#else
      newvec.sa_flags = 0;
#endif
      (void) sigaddset(&newvec.sa_mask, SIGALRM);
      newvec.sa_handler = sigint_handler;
      if (sigaction(SIGINT,&newvec,NULL) == -1)
	    return(-1);

/* setup handler for SIGUSR1 */
      newvec.sa_handler = sigusr1_handler;
      if (sigaction(SIGUSR1,&newvec,NULL) == -1)
	    return(-1);

      return(0);
}
      
