/* 
 * buffit curve fitting and integration.
 *
 */

#include <stdio.h>
#include <math.h>
#include <unistd.h>
#include <xview/xview.h>
#include <xview/canvas.h>
#include <xview/panel.h>
#include <xview/font.h>
#include "globals.h"
#include "../buffit_dir/buffitio.h"
#include "alerts.h"
#include "compwin1.h"
#include "compwin2.h"
#include "draw.h"
#include "events.h"
#include "files.h"
#include "graphutils.h"
#include "main.h"
#include "objutils.h"
#include "plotone.h"
#include "setutils.h"
#include "symwin.h"
#include "utils.h"

void drawgraph2();		/* calls plotone(), called any time the graph
				 * needs to be re-drawn */
void clear_graph_viewport();

void create_int_frame();
void create_buffit_frame();
void create_peakf_frame();
void create_mstick_frame();
void do_int_proc();
double do_sum(), do_int();
Notify_value sigchldcatcher_compwin1();

/*
 * generic quit proc for all of the frames defined below
 */
#define FRAME_KEY        101

/* ARGSUSED */
void generic_done_proc(item, event)
    Panel_item item;
    Event *event;
{
    Frame frame = (Frame) xv_get(item, XV_KEY_DATA, FRAME_KEY);

    xv_set(frame, WIN_SHOW, FALSE, 0);
}

/* numerical integration */
static Frame int_frame = (Frame) 0;
static Panel int_panel;
static Panel_item toggle_set_int_item;
static Panel_item int_sum_item;
static Panel_item int_cent_item;
static Panel_item int_lim1_item;
static Panel_item int_lim2_item;

void create_int_frame()
{

    if (int_frame) {
	xv_set(int_frame, WIN_SHOW, TRUE, FRAME_CLOSED, FALSE, NULL);
	return;
    }
    int_frame = (Frame) xv_create(main_frame, FRAME,
				  FRAME_LABEL, "Integration",
				  NULL);
    int_panel = (Panel) xv_create(int_frame, PANEL,
				  XV_HELP_DATA, "xvgr:int_panel",
				  PANEL_LAYOUT, PANEL_VERTICAL,
				  NULL);
    define_select_set_panel2(int_panel, toggle_set_int_item, "Select set:");
    xv_set(toggle_set_int_item, PANEL_VALUE_X, xv_col(int_panel, 12), NULL);

    int_lim1_item = xv_create(int_panel, PANEL_TEXT,
			     XV_HELP_DATA, "xvgr:int_limit1",
			     PANEL_LABEL_STRING, "Tag1:",
			     PANEL_VALUE_DISPLAY_LENGTH, 7,
			     PANEL_VALUE_X, xv_col(int_panel, 5),
			     NULL);

    int_lim2_item = xv_create(int_panel, PANEL_TEXT,
			     XV_HELP_DATA, "xvgr:int_limit2",
			     PANEL_LABEL_STRING, "Tag2:",
			     PANEL_VALUE_DISPLAY_LENGTH, 7,
			     PANEL_VALUE_X, xv_col(int_panel, 20),
			     NULL);

    
    int_sum_item = xv_create(int_panel, PANEL_TEXT,
			     XV_HELP_DATA, "xvgr:int_sum",
			     PANEL_LABEL_STRING, "Sum:",
			     PANEL_VALUE_DISPLAY_LENGTH, 15,
			     PANEL_VALUE_X, xv_col(int_panel, 10),
			     NULL);
    int_cent_item = xv_create(int_panel, PANEL_TEXT,
			     XV_HELP_DATA, "xvgr:int_centroid",
			     PANEL_LABEL_STRING, "Centroid:",
			     PANEL_VALUE_DISPLAY_LENGTH, 15,
			     PANEL_VALUE_X, xv_col(int_panel, 10),
			     NULL);

 
    (void) xv_create(int_panel, PANEL_BUTTON,
	      XV_HELP_DATA, "xvgr:int_apply",
	      PANEL_LABEL_WIDTH, 65,
	      PANEL_LABEL_STRING, "Use tags",
	      PANEL_NOTIFY_PROC, do_int_proc,
	      XV_X, xv_col(int_panel, 2),
	      XV_Y, xv_row(int_panel, 4),
	      NULL);

    (void) xv_create(int_panel, PANEL_BUTTON,
	      XV_HELP_DATA, "xvgr:int_apply",
	      PANEL_LABEL_WIDTH, 65,     
	      PANEL_LABEL_STRING, "Set tags",
	      PANEL_NOTIFY_PROC, do_int_proc2,
	      XV_X, xv_col(int_panel, 15),
	      XV_Y, xv_row(int_panel, 4),
	      NULL);

    (void) xv_create(int_panel, PANEL_BUTTON,
		     PANEL_LABEL_STRING, "Done",
		     XV_HELP_DATA, "xvgr:int_done",
		     PANEL_NOTIFY_PROC, generic_done_proc,
		     XV_KEY_DATA, FRAME_KEY, int_frame,
		     XV_X, xv_col(int_panel, 10),
		     XV_Y, xv_row(int_panel, 5),
		     NULL);
    window_fit(int_panel);
    window_fit(int_frame);
    xv_set(int_frame, WIN_SHOW, TRUE, NULL);
}

/*
 * numerical integration ... also called from events.c with item = 0
 */
/*ARGSUSED*/
void do_int_proc(item, event)
    Panel_item item;
    Event *event;
{
    int setno;
    int l1, l2;
    double centroid;
    extern int tag1, tag2;

    setno = (int) xv_get(toggle_set_int_item, PANEL_VALUE);
    if (item == 0) {
	  sprintf(buf,"%d", l1=tag1);
	  xv_set(int_lim1_item, PANEL_VALUE, buf, NULL);
	  sprintf(buf,"%d", l2=tag2);
	  xv_set(int_lim2_item, PANEL_VALUE, buf, NULL);
    } else {
	  l1 = atoi((char *) xv_get(int_lim1_item, PANEL_VALUE));
	  l2 = atoi((char *) xv_get(int_lim2_item, PANEL_VALUE));
    }
    if (g[cg].p[setno].spec == SPECTRUM_DATA) {
	  sprintf(buf, "%10.1f", do_sum(setno, l1, l2, &centroid));
	  xv_set(int_sum_item, PANEL_VALUE, buf, NULL);
	  sprintf(buf, "%g", centroid);
	  xv_set(int_cent_item, PANEL_VALUE, buf, NULL);
    } else {
	  sprintf(buf, "%5.2f", do_int(setno, l1, l2, &centroid));
	  xv_set(int_sum_item, PANEL_VALUE, buf, NULL);
	  sprintf(buf, "%g", centroid);
	  xv_set(int_cent_item, PANEL_VALUE, buf, NULL);
    }
}

/*
	compute mean and standard dev
*/
void stasum(x, n, xbar, sd, flag)
    double x[], *xbar, *sd;
    int n, flag;
{
    int i;

    *xbar = 0;
    *sd = 0;
    if (n < 1)
	return;
    for (i = 0; i < n; i++)
	*xbar = (*xbar) + x[i];
    *xbar = (*xbar) / n;
    for (i = 0; i < n; i++)
	*sd = (*sd) + (x[i] - *xbar) * (x[i] - *xbar);
    if (n - flag)
	*sd = sqrt(*sd / (n - flag));
    else {
	errwin("compmean: (n-flag)==0");
	*sd = 0;
    }
}
/*
 * trapezoidal rule
 */
double trapint(x, y, resx, resy, n)
    double x[], y[], resx[], resy[];
int n;

{
    int i;
    double h;
    double sum = 0.0;

    for (i = 1; i < n; i++) {
	h = (x[i] - x[i - 1]);
	if (resx != NULL) {
	    resx[i - 1] = (x[i - 1] + x[i]) * 0.5;
	}
	sum = sum + h * (y[i - 1] + y[i]) * 0.5;
	if (resy != NULL) {
	    resy[i - 1] = sum;
	}
    }
    return sum;
}
double
do_sum(setno,l1,l2,centroid)
      int setno;
      int l1, l2;
      double *centroid;
{
      double xmin,xmax,ymin,ymax;
      int    i, tmp;
      double *x, *y;
      double sum = 0.0, censum = 0.0;
      if (! isactive(cg,setno)) {
	    errwin("Set not active");
	    return(0.0);
      }
      getsetminmax(cg,setno,&xmin,&xmax,&ymin,&ymax);
      x = getx(cg,setno);
      y = gety(cg,setno);
      if (l1 > l2) iswap(&l1,&l2);
      for(i = l1; i<= l2; i++) {
	    tmp = get_chan_content(i,x,y,xmin,xmax);
	    sum += tmp;
	    censum += tmp*i;
      }
      *centroid = censum/sum;
      return(sum);
}
/*
 * numerical integration ... assume  x[0] < x[1] < x[2] etc
 */
double do_int(setno, l1, l2, centroid)
    int setno;
    int l1, l2;
    double *centroid;
{
    int i;
    double xmin,xmax,ymin,ymax;
    double base, height, rect, area;
    double *x, *y;
    double sum = 0.0, centrdsum = 0.0;
    if (!isactive(cg, setno)) {
	errwin("Set not active");
	return 0.0;
    }
    getsetminmax(cg,setno,&xmin,&xmax,&ymin,&ymax);
    x = getx(cg,setno);
    y = gety(cg,setno);
    if (l1 > l2) iswap(&l1,&l2);
    if (l1 < xmin) l1 = xmin;
    if (l2 > xmax) l2 = xmax;
    if (l1 > xmax || l2 < xmin) return(0.0);
    for (i=0; x[i] < l1; i++)
	  ;
    if ( x[i] != l1) {
	  double y_l1 = y[i-1] + (y[i] - y[i-1]) * (l1 - x[i-1])/(x[i] - x[i-1]);
	  base = x[i] - l1;
	  rect = base * MIN(y_l1,y[i]);
	  height = fabs(y[i] - y_l1);
	  sum = area = rect + 0.5*base*height;
	  centrdsum = 0.5 * area  * (x[i] + l1);
    }
    while (x[i+1] < l2) {
	  base = x[i+1] - x[i];
	  rect = base * MIN(y[i],y[i+1]);
	  height = fabs(y[i+1] - y[i]);
	  area = rect + 0.5*base*height;
	  sum += area;
	  centrdsum += 0.5 * area * (x[i+1] + x[i]);
	  i++;
    }
    {
	  double y_l2 = y[i] + (y[i+1] - y[i]) * (l2 - x[i])/(x[i+1] - x[i]);
	  base = l2 - x[i];
	  rect = base * MIN(y[i],y_l2);
	  height = fabs(y_l2 - y[i]);
	  area = rect + 0.5*base*height;
	  sum += area;
	  centrdsum += 0.5 * area * (x[i] + l2);
    }
    *centroid = (sum > 0) ? centrdsum / sum : 0.0;
    return(sum); 
}

#define MAXPEAKS 10
#define GAUSS_FIT  0
#define SKEWG_FIT  1
#define LORTZ_FIT  2
#define FIX_WID  0x1
#define FIX_POS  0x2
#define SAME_WID 0x4
#include <signal.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <xview/notify.h>

/* window items for buffit fit frame */
static Frame buffit_frame = (Frame) 0;
static Panel buffit_panel;
static Panel_item buffit_set_item;
static Panel_item buffit_fittype_item;
static Panel_item buffit_fitopt_item;
static Panel_item buffit_npparm_item;
static Panel_item buffit_nbparm_item;
static Panel_item buffit_bkshape_item;
static Panel_item buffit_initial_width_item;
static Panel_item buffit_initial_widthlo_item;
static Panel_item buffit_info_item;
static Panel_item buffit_cbox1_item;
static Panel_item buffit_limits_item[2];
static Panel_item buffit_channel_item[MAXPEAKS];

static void do_buffit_tags();
static void do_buffit_proc();
static void redraw_buffit_tags();
static int buffit_check_tags();
static void buffit_apply_fit();

/* current tags limit and centroid positions (entered last by mouse) */
static int   peak_pos[MAXPEAKS+2];
static int   nos_peaks;

void create_buffit_frame()
{
    int i, width;
    Panel_item button[4];

    if (buffit_frame) {
	xv_set(buffit_frame, WIN_SHOW, TRUE, FRAME_CLOSED, FALSE, NULL);
	return;
    }
    buffit_frame = (Frame)
	xv_create(main_frame, FRAME,
		  FRAME_LABEL, "Buffit Curve fitting",
		  NULL);
    buffit_panel = (Panel)
	xv_create(buffit_frame, PANEL,
		  XV_HELP_DATA, "xvgr:buffit_panel",
		  PANEL_LAYOUT, PANEL_VERTICAL,
		  NULL);
    define_select_set_panel2(buffit_panel, buffit_set_item, "Select set:");
    xv_set(buffit_set_item, PANEL_VALUE_X, xv_col(buffit_panel, 18), NULL);
    buffit_fittype_item = (Panel_item)
	xv_create(buffit_panel, PANEL_CHOICE,
		  PANEL_LABEL_STRING, "Fit type:",
		  PANEL_CHOICE_NCOLS, 3,
		  PANEL_CHOICE_STRINGS,
		  "Gaussian",
		  "Skew Gauss",
		  "Lorentzian",
		  0,
		  PANEL_VALUE_X, xv_col(buffit_panel, 18),
		  NULL);
    buffit_fitopt_item = (Panel_item)
	xv_create(buffit_panel, PANEL_TOGGLE,
		  PANEL_LABEL_STRING, "Options:",
		  PANEL_CHOICE_NCOLS, 3,
		  PANEL_CHOICE_STRINGS,
		  "Fix width",
		  "Fix Position",
		  "Same Width",
		  NULL,
		  PANEL_VALUE_X, xv_col(buffit_panel, 18),
		  NULL);
    buffit_npparm_item = (Panel_item)
	xv_create(buffit_panel, PANEL_NUMERIC_TEXT,
		  XV_HELP_DATA, "xvgr:buffit_npparm",
		  PANEL_LABEL_STRING, "Number of peaks:",
		  PANEL_VALUE_DISPLAY_LENGTH, 4,
		  PANEL_MIN_VALUE, 1,
		  PANEL_MAX_VALUE, MAXPEAKS,
		  PANEL_VALUE_X, xv_col(buffit_panel, 18),
		  NULL);
    buffit_nbparm_item = (Panel_item)
	xv_create(buffit_panel, PANEL_NUMERIC_TEXT,
		  XV_HELP_DATA, "xvgr:buffit_nbparm",
		  PANEL_LABEL_STRING, "Background order:",
		  PANEL_VALUE_DISPLAY_LENGTH, 4,
		  PANEL_MIN_VALUE, 0,
		  PANEL_MAX_VALUE, BK_ORDER,
		  PANEL_VALUE_X, xv_col(buffit_panel, 18),
		  NULL);
    buffit_bkshape_item = (Panel_item)
	xv_create(buffit_panel, PANEL_CHOICE,
		  PANEL_LABEL_STRING, "Background Shape:",
		  PANEL_CHOICE_NCOLS, 3,
		  PANEL_CHOICE_STRINGS,
		  "Polynomial",
		  "Const+Exp",
		  "Const+WS",
		  0,
		  PANEL_VALUE_X, xv_col(buffit_panel, 18),
		  NULL);
    (void) xv_create(buffit_panel, PANEL_MESSAGE,
		     PANEL_LABEL_STRING, "Initial guess",
		     XV_X, xv_col(buffit_panel, 8),
		     NULL);
    buffit_initial_width_item  = (Panel_item)
	xv_create(buffit_panel, PANEL_TEXT,
		  XV_HELP_DATA, "xvgr:buffit_initparm",
		  PANEL_LABEL_STRING, "width:",
		  PANEL_VALUE_DISPLAY_LENGTH, 15,
		  PANEL_VALUE, "1.0",
		  PANEL_VALUE_X, xv_col(buffit_panel, 8),
		  0);
    buffit_initial_widthlo_item = (Panel_item)
	xv_create(buffit_panel, PANEL_TEXT,
		  XV_HELP_DATA, "xvgr:buffit_initparm",
		  PANEL_LABEL_STRING, "widthlo:",
		  PANEL_VALUE_DISPLAY_LENGTH, 15,
		  PANEL_VALUE, "1.0",
		  PANEL_VALUE_X, xv_col(buffit_panel, 32),
		  XV_Y, xv_get(buffit_initial_width_item, XV_Y),
		  0);
    buffit_cbox1_item = (Panel_item)
	xv_create(buffit_panel, PANEL_CHECK_BOX,
		  PANEL_CHOICE_NCOLS, 1,
		  PANEL_LAYOUT, PANEL_HORIZONTAL,
		  PANEL_CHOICE_STRINGS,
		  "use values shown on frame",
		  0,
		  XV_X, xv_col(buffit_panel, 8),
		  NULL);

    buffit_limits_item[0]  = (Panel_item)
	xv_create(buffit_panel, PANEL_TEXT,
		  XV_HELP_DATA, "xvgr:buffit_limit",
		  PANEL_LABEL_STRING, "limit 1:",
		  PANEL_VALUE_DISPLAY_LENGTH, 15,
		  PANEL_VALUE_X, xv_col(buffit_panel, 8),
		  0);
    buffit_limits_item[1]  = (Panel_item)
	xv_create(buffit_panel, PANEL_TEXT,
		  XV_HELP_DATA, "xvgr:buffit_limit",
		  PANEL_LABEL_STRING, "limit 2:",
		  PANEL_VALUE_DISPLAY_LENGTH, 15,
		  PANEL_VALUE_X, xv_col(buffit_panel, 32),
		  XV_Y, xv_get(buffit_limits_item[0], XV_Y),
		  0);
    for(i=0; i < MAXPEAKS; i += 2) {
	sprintf(buf,"Peak %d:",i+1);
	buffit_channel_item[i]  = (Panel_item)
	    xv_create(buffit_panel, PANEL_TEXT,
		      XV_HELP_DATA, "xvgr:buffit_initparm",
		      PANEL_LABEL_STRING, buf,
		      PANEL_VALUE_DISPLAY_LENGTH, 15,
		      PANEL_VALUE_X, xv_col(buffit_panel, 8),
		      0);
	sprintf(buf,"Peak %d:",i+2);
	buffit_channel_item[i+1]  = (Panel_item)
	    xv_create(buffit_panel, PANEL_TEXT,
		      XV_HELP_DATA, "xvgr:buffit_initparm",
		      PANEL_LABEL_STRING, buf,
		      PANEL_VALUE_DISPLAY_LENGTH, 15,
		      PANEL_VALUE_X, xv_col(buffit_panel, 32),
		      XV_Y, xv_get(buffit_channel_item[i], XV_Y),
		      0);
    }
    buffit_info_item = (Panel_item)
	xv_create(buffit_panel, PANEL_MESSAGE,
		  PANEL_LABEL_STRING, "Info: ",
		  XV_X, xv_col(buffit_panel, 1),
		  NULL);
    button[0] = (Panel_item)
	xv_create(buffit_panel, PANEL_BUTTON,
		  XV_HELP_DATA, "xvgr:buffit_apply",
		  PANEL_LABEL_STRING, "Apply",
		  PANEL_NOTIFY_PROC, do_buffit_proc,   
		  XV_X, xv_col(buffit_panel, 3),
		  0);
    i = xv_get(button[0], XV_Y);
    button[1] = (Panel_item)
	xv_create(buffit_panel, PANEL_BUTTON,
		  XV_HELP_DATA, "xvgr:buffit_tags",
		  PANEL_LABEL_STRING, "Set tags",
		  PANEL_NOTIFY_PROC, do_buffit_tags,   
		  XV_X, xv_col(buffit_panel, 12),
		  XV_Y, i,
		  0);
    button[2] = (Panel_item)
	xv_create(buffit_panel, PANEL_BUTTON,
		  XV_HELP_DATA, "xvgr:buffit_tags",
		  PANEL_LABEL_STRING, "Update",
		  PANEL_NOTIFY_PROC, redraw_buffit_tags,   
		  XV_X, xv_col(buffit_panel, 22),
		  XV_Y, i,
		  0);
    button[3] = (Panel_item)
	xv_create(buffit_panel, PANEL_BUTTON,
		  XV_HELP_DATA, "xvgr:buffit_done",
		  PANEL_LABEL_STRING, "Done",
		  PANEL_NOTIFY_PROC, generic_done_proc,
		  XV_KEY_DATA, FRAME_KEY, buffit_frame,
		  XV_X, xv_col(buffit_panel, 34),
		  XV_Y, i,
		  NULL);
    window_fit(buffit_panel);
    window_fit(buffit_frame);

    width = xv_get(buffit_panel, XV_WIDTH) - 36;
    for(i=0; i<4; i++)
	width -= xv_get(button[i], XV_WIDTH);

    xv_set(button[0], XV_X, width/2, NULL);
    for(i=1; i<4; i++)
	xv_set(button[i], XV_X,
	       xv_get(button[i-1], XV_X) + xv_get(button[i-1], XV_WIDTH) + 12,
	       NULL);

    xv_set(buffit_frame, WIN_SHOW, TRUE, NULL);
}

static int intcompare(i,j)
      int *i, *j;
{
      return(*i - *j);
}

/*
 *   function called by set_action() in events.c after the
 *   fit tags have been set. Also called by redraw_buffit_tags()
 *   with nos_tags_unset == -1 
 */
void
update_buffit_tags(nos_tags_unset)
      int  nos_tags_unset;
{
      int i;
      if (nos_peaks - nos_tags_unset <= 0) return;
      if (nos_tags_unset != -1) qsort(peak_pos,nos_peaks+2,sizeof(int),
#ifdef __STDC__
				      (int (*)(const void *, const void *))
#endif
				      intcompare);
      sprintf(buf,"%d",peak_pos[0]);
      xv_set(buffit_limits_item[0],PANEL_VALUE,buf,NULL);
      sprintf(buf,"%d",peak_pos[nos_peaks+1]);
      xv_set(buffit_limits_item[1],PANEL_VALUE,buf,NULL);
      for (i=1; i <= nos_peaks; i++) {
	    sprintf(buf,"%d",peak_pos[i]);
	    xv_set(buffit_channel_item[i-1],PANEL_VALUE,buf,NULL);
      }
      for (i=nos_peaks+1; i < MAXPEAKS+1; i++)
	    xv_set(buffit_channel_item[i-1],PANEL_VALUE,"",NULL);
      return;
}

/*
 *  clear peak_pos array and call set_cfit_tags in events.c
 *  to set new values in this array
 */
static void
do_buffit_tags()
{
      int   i;
      nos_peaks = (int) xv_get(buffit_npparm_item,PANEL_VALUE);
      for (i=0; i < MAXPEAKS+2; i++)
	    peak_pos[i] = 0;
      set_cfit_tags(nos_peaks+2,peak_pos,update_buffit_tags);
      return;
}

static Notify_client  buffit_nclient = (Notify_client) 103;

/*
 * write out data from frame and spectra to tmp file and then run buffit process
 */
static void
do_buffit_proc()
{
    int fit_type, fit_opt, bgd_order, setno;
    double width, widthlo;
    double xmin, xmax, ymin, ymax, *ydata, *xdata, *xp;
    buffit_in_t buffin;
    char  *my_argv[4], *out_file;
    int i;
    FILE *fp;
    
    /* check that tags in frame are consistent with those entered by the
       mouse */      
    if ( buffit_check_tags() == 0)
	return;

    /* fill argument vector */
    my_argv[0] = "sort_buffit";
    my_argv[1] = "-spectrum";
    my_argv[2] = (out_file = tempnam(NULL,"SORT_")); 
    my_argv[3] = (char *)0;

    /* get other fit parameters */      
    fit_type = (int) xv_get(buffit_fittype_item,PANEL_VALUE);
    fit_opt = (int) xv_get(buffit_fitopt_item,PANEL_VALUE);
    bgd_order = (int) xv_get(buffit_nbparm_item,PANEL_VALUE);
    strcpy(buf,(char *) xv_get(buffit_initial_width_item,PANEL_VALUE));
    width = atof(buf);
    if (width <= 0.0) {
	errwin("Value of width guess <= 0");
	return;
    }
    strcpy(buf,(char *) xv_get(buffit_initial_widthlo_item,PANEL_VALUE));
    widthlo = atof(buf);
    setno = (int) xv_get(buffit_set_item,PANEL_VALUE);
    if ( ! isactive_set(cg,setno)) {
	errwin("Selected set not active !");
	return;
    }
    
    (void) memset((char *)&buffin, 0, sizeof(buffit_in_t));
    
    if (fit_opt & FIX_WID) buffin.ifixwd = 1;
    if (fit_opt & FIX_POS) buffin.ifixcn = 1;
    buffin.npeak = nos_peaks;
    buffin.nback = bgd_order;
    buffin.width = width;
    buffin.widlo = widthlo;
    
    switch(fit_type) {
    case GAUSS_FIT:
	break;
    case SKEWG_FIT:
	if (widthlo <= 0.0) {
	    errwin("Value of widthlo guess <= 0");
	    return;
	}
	buffin.iskew = 1;
	break;
    case LORTZ_FIT:
	buffin.lorentz = 1;
	break;
    default:
	errwin("Unrecognised fit type");
	return;
    }

    buffin.ibkshape = (int) xv_get(buffit_bkshape_item, PANEL_VALUE) +
	BACKGROUND_POLY;

    if (!(fit_opt & SAME_WID)) {
	buffin.idiffwd = 1;
	for (i=0; i<FIT_DIM; i++) {
	    buffin.fwhmin[i] = width;
	    buffin.fwloin[i] = widthlo;
	}
    }
    if ( (buffin.idiffwd && buffin.ifixwd) && buffin.ifixcn) {
	errwin("combination of options not implemented");
	return;
    }

    getsetminmax(cg,setno,&xmin,&xmax,&ymin,&ymax);

    if (peak_pos[0] >= xmax) {
	errwin("lower limit exceeds max channel of spectrum!");
	return;
    }

    if (peak_pos[nos_peaks+1] <= xmin) {
	errwin("upper limit less than min channel of spectrum!");
	return;
    }


    /* Check set has linear x spacing */
    xdata = getx(cg, setno);
    ydata = gety(cg, setno);
    buffin.nchan = getsetlength(cg, setno);

    if (buffin.nchan < 2)
    {
	errwin("Set has less than 2 points.");
	return;
    }

    xmin = xmax = xdata[1] - xdata[0];
    for(i = buffin.nchan-2, xp = xdata+1; i--; xp++)
    {
	if ((ymin = xp[1] - *xp) < xmin)
	    xmin = ymin;
	if (ymin > xmax)
	    xmax = ymin;
    }

    if (xmax == 0 || xmin == 0)
    {
	errwin("X spacing between points is 0 at at least one point.");
	return;
    }

    ymin = fabs(xmax-xmin);
    
    /* Allow a little slop on the linearity */
    if (ymin/fabs(xmin) > 1e-3 || ymin/fabs(xmax) > 1e-3)
    {
	errwin("Spacing between points is non-linear.");
	return;
    }

    buffin.gradient = (xmin + xmax)/2;
    buffin.offset = xdata[0];
    buffin.dxplot = 0.1;

    buffin.ichan = (peak_pos[0]-buffin.offset)/buffin.gradient+0.5;
    buffin.jchan = (peak_pos[nos_peaks+1]-buffin.offset)/buffin.gradient+1.5;

    for (i=1; i<=nos_peaks; i++) 
	buffin.centin[i-1] = (peak_pos[i]-buffin.offset)/buffin.gradient+1;
    
    if ((fp = fopen(out_file,"w")) == NULL) {
	errwin("failed to open buffit read file");
	return;
    }
    
    if (fwrite((char *) &buffin, sizeof(buffit_in_t), 1, fp) != 1) {
	errwin("Write to buffit input file failed");
	fclose(fp);
	return;
    } 
    
    if (fwrite((char *) ydata, sizeof(double), buffin.nchan, fp)
	!= buffin.nchan) {
	errwin("Write to buffit input file failed");
	fclose(fp);
	return;
    } 

    (void) fclose(fp);
      
    xv_set(buffit_info_item,PANEL_LABEL_STRING,"fit in progress",NULL);
    /* start buffit process */
    do_process("sort_buffit", my_argv, buffit_nclient,
	       sigchldcatcher_compwin1);
    
    if (out_file != NULL)
	free(out_file);
}

/* check that tags in frame are consistent with those entered by the mouse */
static int
buffit_check_tags()
{
      int i, tmp, use_typedin_values;
      char msg1[128], msg2[128];
      use_typedin_values = (int) xv_get(buffit_cbox1_item,PANEL_VALUE);
      tmp = (int) xv_get(buffit_npparm_item,PANEL_VALUE);
      if (use_typedin_values) {
	    nos_peaks = tmp;
      } else if (tmp != nos_peaks) {
	    strcpy(msg1,"Number of peaks differs from number");
	    strcpy(msg2,"of centroids entered with mouse");
	    if (yesno(msg1,msg2,"Continue","Abort") == 0) 
		  return(0);
	    nos_peaks = tmp;
      }
      for (i=0; i<nos_peaks; i++) {
	    strcpy(buf,(char *) xv_get(buffit_channel_item[i],PANEL_VALUE));
	    tmp = atoi(buf);
	    if (use_typedin_values) {
		  peak_pos[i+1] = tmp;
	    } else if (tmp != peak_pos[i+1]) {
		  sprintf(msg1,"Tag value entered by mouse = %d",peak_pos[i+1]);
		  sprintf(msg2," using value from frame = %d",tmp);
		  if (yesno(msg1,msg2,"Continue","Abort") == 0) 
			return(0);
		  peak_pos[i+1] = tmp;
	    }
      }
      strcpy(buf,(char *) xv_get(buffit_limits_item[0],PANEL_VALUE));
      tmp = atoi(buf);
      if (use_typedin_values) {
	     peak_pos[0] = tmp;
      } else if (tmp != peak_pos[0]) {
	    sprintf(msg1,"Limit 1 value entered by mouse = %d",peak_pos[0]);
	    sprintf(msg2," using value from frame = %d",tmp);
	    if (yesno(msg1,msg2,"Continue","Abort") == 0) 
		  return(0);
	    peak_pos[0] = tmp;
      }
      strcpy(buf,(char *) xv_get(buffit_limits_item[1],PANEL_VALUE));
      tmp = atoi(buf);
      if (use_typedin_values) {
	     peak_pos[nos_peaks+1] = tmp;
      } else if (tmp != peak_pos[nos_peaks+1]) {
	    sprintf(msg1,"Limit 2 value entered by mouse = %d",peak_pos[nos_peaks+1]);
	    sprintf(msg2," using value from frame = %d",tmp);
	    if (yesno(msg1,msg2,"Continue","Abort") == 0) 
		  return(0);
	    peak_pos[nos_peaks+1] = tmp;
      }
      for (i=1; i<=nos_peaks; i++) {
	    if (! MIN(peak_pos[0],peak_pos[i])) {
		  errwin("Limit 1 > centroid guess !");
		  return(0);
	    }
	    if (! MAX(peak_pos[nos_peaks+1],peak_pos[i])) {
		  errwin("Limit 2 < centroid guess !");
		  return(0);
	    }
      }
      return(1);
}
/*
 *
 *
 */
static void
redraw_buffit_tags()
{
      int use_typedin_values, tmp;
      int x, y, i;

      clear_graph_viewport(cg);
      drawgraph2(cg);
      use_typedin_values = (int) xv_get(buffit_cbox1_item,PANEL_VALUE);

      tmp = (int) xv_get(buffit_npparm_item,PANEL_VALUE);
      if (use_typedin_values) {
	    nos_peaks = tmp;
      }
      if (world2device((double) tmp, 0.0, &x, &y) == 0) return;
      draw_tmp_tag(cg,x,y);
      for (i=0; i<nos_peaks; i++) {
	    strcpy(buf,(char *) xv_get(buffit_channel_item[i],PANEL_VALUE));
	    tmp = atoi(buf);
	    if (use_typedin_values) {
		  peak_pos[i+1] = tmp;
	    }
	    if (world2device((double) tmp, 0.0, &x, &y) == 0) return;
	    draw_tmp_tag(cg,x,y);
      }
      strcpy(buf,(char *) xv_get(buffit_limits_item[0],PANEL_VALUE));
      tmp = atoi(buf);
      if (use_typedin_values) {
	     peak_pos[0] = tmp;
      }
      if (world2device((double) tmp, 0.0, &x, &y) == 0) return;
      draw_tmp_tag(cg,x,y);
      
      strcpy(buf,(char *) xv_get(buffit_limits_item[1],PANEL_VALUE));
      tmp = atoi(buf);
      if (use_typedin_values) {
	     peak_pos[nos_peaks+1] = tmp;
      }
      if (world2device((double) tmp, 0.0, &x, &y) == 0) return;
      draw_tmp_tag(cg,x,y);
      if (! use_typedin_values) {
	    update_buffit_tags(-1);
	    xv_set(buffit_npparm_item,PANEL_VALUE, nos_peaks, NULL);
      }
      return;
}

static Frame buffit_popup = (Frame) 0;
static Panel buffit_popup_panel;
static Panel_item buffit_cent_label;
static Panel_item buffit_fwhm_label;
static Panel_item buffit_cent_item[MAXPEAKS];
static Panel_item buffit_area_item[MAXPEAKS];
static Panel_item buffit_fwhm_item[MAXPEAKS];
static Panel_item buffit_bkgrd_item;
static Panel_item buffit_chisq_item;
static int buffit_popup_centre, buffit_popup_left, buffit_popup_right;

void create_buffit_popup()
{
    int i, y;
    Panel_item tmp, area_label, button;
    
    if (buffit_popup) {
	xv_set(buffit_popup, WIN_SHOW, TRUE, FRAME_CLOSED, FALSE, NULL);
	return;
    }
    buffit_popup = (Frame)
	xv_create(buffit_frame, FRAME,
		  XV_LABEL, "Buffit Curve fitting results",
		  NULL);
    buffit_popup_panel = (Panel)
	xv_create(buffit_popup, PANEL,
		  PANEL_LAYOUT, PANEL_VERTICAL,
		  WIN_BORDER, FALSE,
		  NULL);

    buffit_cent_label = (Panel_item)
	xv_create(buffit_popup_panel, PANEL_MESSAGE,
		  PANEL_LABEL_STRING, "Centroid",
		  PANEL_LABEL_BOLD, TRUE,
		  XV_X, xv_col(buffit_popup_panel, 10),
		  NULL);
    y = xv_get(buffit_cent_label, XV_Y);
    area_label = (Panel_item)
	xv_create(buffit_popup_panel, PANEL_MESSAGE,
		  PANEL_LABEL_STRING, "Area",
		  PANEL_LABEL_BOLD, TRUE,
		  XV_X, xv_col(buffit_popup_panel, 35),
		  XV_Y, y,
		  NULL);
    buffit_fwhm_label = (Panel_item)
	xv_create(buffit_popup_panel, PANEL_MESSAGE,
		  PANEL_LABEL_STRING, "FWHM",
		  PANEL_LABEL_BOLD, TRUE,
		  XV_X, xv_col(buffit_popup_panel, 60),
		  XV_Y, y,
		  NULL);
    buffit_popup_right = xv_col(buffit_popup_panel, 67);
    
    for(i=0; i < MAXPEAKS; i++) {
	sprintf(buf,"Peak %d:",i+1);
	tmp = (Panel_item)
	    xv_create(buffit_popup_panel, PANEL_MESSAGE,
		      PANEL_LABEL_STRING, buf,
		      PANEL_LABEL_BOLD, TRUE,
		      XV_X, xv_col(buffit_popup_panel,1),
		      0);
	y = xv_get(tmp, XV_Y);
	buffit_cent_item[i] = (Panel_item)
	    xv_create(buffit_popup_panel, PANEL_MESSAGE,
		      PANEL_LABEL_STRING, "NIL",
		      XV_X, xv_col(buffit_popup_panel,4),
		      XV_Y, y,
		      0);
	buffit_area_item[i] = (Panel_item)
	    xv_create(buffit_popup_panel, PANEL_MESSAGE,
		      PANEL_LABEL_STRING, "NIL",
		      XV_X, xv_col(buffit_popup_panel,30),
		      XV_Y, y,
		      0);
	buffit_fwhm_item[i] = (Panel_item)
	    xv_create(buffit_popup_panel, PANEL_MESSAGE,
		      PANEL_LABEL_STRING, "NIL",
		      XV_X, xv_col(buffit_popup_panel,53),
		      XV_Y, y,
		      0);
	xv_set(buffit_fwhm_item[i], XV_X,
	       buffit_popup_right - xv_get(buffit_fwhm_item[i], XV_WIDTH),
	       NULL);
    }
    buffit_popup_left = xv_get(tmp, XV_X) + xv_get(tmp, XV_WIDTH) + 4;
    
    buffit_bkgrd_item = (Panel_item)
	xv_create(buffit_popup_panel, PANEL_MESSAGE,
		  PANEL_LABEL_STRING, "Background area: ",
		  XV_X, xv_col(buffit_popup_panel, 20),
		  NULL);
    buffit_chisq_item = (Panel_item)
	xv_create(buffit_popup_panel, PANEL_MESSAGE,
		  PANEL_LABEL_STRING, "Chi-squared: ",
		  XV_X, xv_col(buffit_popup_panel, 20),
		  NULL);
    button = (Panel_item)
	xv_create(buffit_popup_panel, PANEL_BUTTON,
		  PANEL_LABEL_STRING, "Done",
		  PANEL_NOTIFY_PROC, generic_done_proc,
		  XV_KEY_DATA, FRAME_KEY, buffit_popup,
		  XV_X, xv_col(buffit_popup_panel, 30),
		  NULL);
    
    window_fit(buffit_popup_panel);
    window_fit(buffit_popup); 

    buffit_popup_centre = xv_get(buffit_popup_panel, XV_WIDTH)/2;
    xv_set(button, XV_X,
	   buffit_popup_centre - xv_get(button, XV_WIDTH)/2, NULL);
    xv_set(buffit_bkgrd_item, XV_X,
	   buffit_popup_centre - xv_get(buffit_bkgrd_item, XV_WIDTH), NULL);
    xv_set(buffit_chisq_item, XV_X,
	   buffit_popup_centre - xv_get(buffit_chisq_item, XV_WIDTH), NULL);
    buffit_popup_centre = (buffit_popup_left + buffit_popup_right)/2;
    xv_set(area_label, XV_X,
	   buffit_popup_centre - xv_get(area_label, XV_WIDTH)/2, NULL);
    for(i=0; i < MAXPEAKS; i++)
	xv_set(buffit_area_item[i], XV_X,
	       buffit_popup_centre - xv_get(buffit_area_item[i], XV_WIDTH)/2,
	       NULL);
    xv_set(buffit_popup, WIN_SHOW, TRUE, NULL);
}

static void
buffit_apply_fit(pid)
int  pid;
{
    buffit_out_t  bout;
    FILE *fp;
    char in_file[L_tmpnam+24];
    int i, j, fit_set, posl, posr;
    unsigned nelem;
    double *x, *y, *xp;
    static const int plotcolour[3] = {5, 3, 4};
    
    xv_set(buffit_info_item,PANEL_LABEL_STRING,"fit completed",NULL);
    /* read buffit output data */

    sprintf(in_file,"%sBUFFIT_%d",P_tmpdir,pid);
    if ( (fp = fopen(in_file,"r")) == NULL) {
	errwin("error opening buffit results file");
	return;
    }
    if (fread((char *) &bout, sizeof(buffit_out_t), 1, fp) != 1) {
	errwin("error reading buffit output file");
	fclose(fp);
	return;
    }
      
    /* update popup containing results of fit */
    create_buffit_popup();
    for (i=0; i<bout.npeak; i++) {
	sprintf(buf, "%.2f +",
		(bout.centr[i]-1)*bout.gradient + bout.offset);
	xv_set(buffit_cent_item[i], PANEL_LABEL_STRING, buf, NULL);
	sprintf(buf, "- %.2f", bout.fwerr[i]*bout.gradient);
	xv_set(buffit_fwhm_item[i], PANEL_LABEL_STRING, buf, NULL);
    }
    posl = xv_get(buffit_cent_item[0], XV_WIDTH);
    for(i=1; i<bout.npeak; i++)
	if ((j = xv_get(buffit_cent_item[i], XV_WIDTH)) > posl)
	    posl = j;
    posl += buffit_popup_left;
    xv_set(buffit_cent_label, XV_X,
	   posl - xv_get(buffit_cent_label, XV_WIDTH)/2,
	   NULL);
    posr = xv_get(buffit_fwhm_item[0], XV_WIDTH);
    for(i=1; i<bout.npeak; i++)
	if ((j = xv_get(buffit_fwhm_item[i], XV_WIDTH)) > posr)
	    posr = j;
    posr = buffit_popup_right - posr;
    xv_set(buffit_fwhm_label, XV_X,
	   posr - xv_get(buffit_fwhm_label, XV_WIDTH)/2, NULL);
    for (i=0; i<bout.npeak; i++) {
	xv_set(buffit_cent_item[i], XV_X,
	       posl - xv_get(buffit_cent_item[i], XV_WIDTH), NULL);
	sprintf(buf, "%.2f +/- %.2f",
		(bout.centr[i]-1)*bout.gradient + bout.offset,
		bout.cenerr[i]*bout.gradient);
	xv_set(buffit_cent_item[i], PANEL_LABEL_STRING, buf, NULL);
	sprintf(buf, "%.2f +/", bout.area[i]);
	xv_set(buffit_area_item[i], PANEL_LABEL_STRING, buf, NULL);
	xv_set(buffit_area_item[i], XV_X,
	       buffit_popup_centre - xv_get(buffit_area_item[i], XV_WIDTH),
	       NULL);
	sprintf(buf, "%.2f +/- %.2f", bout.area[i], bout.arerr[i]);
	xv_set(buffit_area_item[i], PANEL_LABEL_STRING, buf, NULL);
	j = xv_get(buffit_fwhm_item[i], XV_WIDTH);
	sprintf(buf, "%.2f +/- %.2f", bout.fwhm[i]*bout.gradient,
		bout.fwerr[i]*bout.gradient);
	xv_set(buffit_fwhm_item[i], PANEL_LABEL_STRING, buf, NULL);
	xv_set(buffit_fwhm_item[i], XV_X,
	       posr - xv_get(buffit_fwhm_item[i], XV_WIDTH) + j, NULL);
    }
    for (; i<MAXPEAKS; i++) {
	xv_set(buffit_cent_item[i], PANEL_LABEL_STRING, "not fitted",
	       XV_X, buffit_popup_left, NULL);
	xv_set(buffit_cent_item[i], XV_X,
	       posl - xv_get(buffit_cent_item[i], XV_WIDTH)/2, NULL);
	xv_set(buffit_area_item[i], PANEL_LABEL_STRING, "not fitted", NULL);
	xv_set(buffit_area_item[i], XV_X,
	       buffit_popup_centre - xv_get(buffit_area_item[i], XV_WIDTH)/2,
	       NULL);
	xv_set(buffit_fwhm_item[i], PANEL_LABEL_STRING, "not fitted", NULL);
	xv_set(buffit_fwhm_item[i], XV_X,
	       posr - xv_get(buffit_fwhm_item[i], XV_WIDTH)/2, NULL);
    }
    sprintf(buf,"Background area: %.2f +/- %.2f", bout.bkarea, bout.bkaerr);
    xv_set(buffit_bkgrd_item,PANEL_LABEL_STRING,buf,NULL);
    sprintf(buf,"Chi-squared: %.2f",bout.chisq);
    xv_set(buffit_chisq_item,PANEL_LABEL_STRING,buf,NULL);

    /* kill last sets containing last fits ... if any */
    for (i=0; i < g[cg].maxplot; i++) {
	if (isactive_set(cg,i) && g[cg].p[i].spec == BUFFIT_DATA)
	    softkillset(cg,i);
    }
    
    /* read in fit curve etc for display */      
    nelem = bout.jplot - bout.iplot;

    for(j=3; j--; )
    {
	x = (double *) calloc(nelem, sizeof(double));
	y = (double *) calloc(nelem, sizeof(double));
	
	if (x == NULL || y == NULL) {
	    errwin("Insufficient memory for fit spectrum");
	    cxfree(x);
	    cxfree(y);
	    break;
	}
     
	if (fread((char *) y, sizeof(double), nelem, fp) != nelem) {
	    errwin("error reading buffit output file");
	    cxfree(x);
	    cxfree(y);
	    break;
	}

	if ((fit_set = nextset(cg)) == -1) {
	    errwin("No sets free for fit spectrum");
	    cxfree(x);
	    cxfree(y);
	    break;
	}

	for(xp = x, i=bout.iplot; i<bout.jplot; i++)
	    *xp++ = (i*bout.dxplot-1)*bout.gradient + bout.offset;
	
	activateset(cg, fit_set);
	settype(cg, fit_set, XY);
	set_spectype(cg, fit_set, BUFFIT_DATA);
	setplotcolor(cg, fit_set, plotcolour[j]);
	setcol(cg, x, fit_set, (int) nelem, 0);
	setcol(cg, y, fit_set, (int) nelem, 1);
	updatesetminmax(cg, fit_set);
    }

    clear_graph_viewport(cg);
    drawgraph2(cg);
    (void) fclose(fp);
    (void) unlink(in_file);
}

/*
 *    used in setting region of interest for various calc options
 *
 */
Panel_item  *limits_ptr;     /* pointer to array of Panel_items holding limits */
int         range_tags[2];

void
update_range_tags(nos_tags_unset)
      int nos_tags_unset;
{
      if (nos_tags_unset >= 1) return;
      if (range_tags[0] > range_tags[1])
	    iswap(&range_tags[0], &range_tags[1]);
      sprintf(buf,"%d",range_tags[0]);
      xv_set(limits_ptr[0],PANEL_VALUE,buf,NULL);
      sprintf(buf,"%d",range_tags[1]);
      xv_set(limits_ptr[1],PANEL_VALUE,buf,NULL);
      return;
}

int check_limits(limitl,limitu,xmin,xmax)
      double    *limitl, *limitu;
      double    xmin,   xmax;
{
      if (*limitl > *limitu) fswap(limitl, limitu);
      if (*limitl < xmin) {
	    *limitl = xmin;
      } else if (*limitl >= xmax) {
	    errwin("lower limit exceeds max channel of spectrum!");
	    return(-1);
      }
      if (*limitu > xmax) {
	    *limitu = xmax;
      } else if (*limitu <= xmin) {
	    errwin("upper limit less than min channel of spectrum!");
	    return(-1);
      }
      return (int) (*limitu - *limitl + 1);
}

/***************************************************************************************/

/* window items for peakfit frame */
static Frame peakf_frame = (Frame) 0;
static Panel peakf_panel;
static Panel_item peakf_set_item;
static Panel_item peakf_fwhm_item;
static Panel_item peakf_accept_item;
static Panel_item peakf_info_item;
static Panel_item peakf_limits_item[2];


static void do_peakf_tags();
void do_peakf_proc();

void create_peakf_frame()
{
    if (peakf_frame) {
	xv_set(peakf_frame, WIN_SHOW, TRUE, FRAME_CLOSED, FALSE, NULL);
	return;
    }
    peakf_frame = (Frame) xv_create(main_frame, FRAME,
				     FRAME_LABEL, "Peak find",
				     NULL);
    peakf_panel = (Panel) xv_create(peakf_frame, PANEL,
				     XV_HELP_DATA, "xvgr:peakf_panel",
				     PANEL_LAYOUT, PANEL_VERTICAL,
				     NULL);
    define_select_set_panel2(peakf_panel, peakf_set_item, "Select set:");
    xv_set(peakf_set_item, PANEL_VALUE_X, xv_col(peakf_panel, 20), NULL);
    peakf_fwhm_item = (Panel_item) xv_create(peakf_panel, PANEL_NUMERIC_TEXT,
						XV_HELP_DATA, "xvgr:peakf_npparm",
						PANEL_LABEL_STRING, "fwhm (channels):",
						PANEL_VALUE_DISPLAY_LENGTH, 4,
						PANEL_MIN_VALUE, 1,
						PANEL_MAX_VALUE, 8192,
						PANEL_VALUE_X, xv_col(peakf_panel, 24),
						XV_Y, xv_row(peakf_panel, 1),
						NULL);
    peakf_accept_item = (Panel_item) xv_create(peakf_panel, PANEL_NUMERIC_TEXT,
					       XV_HELP_DATA, "xvgr:peakf_nbparm",
					       PANEL_LABEL_STRING, "acceptence (%):",
					       PANEL_VALUE_DISPLAY_LENGTH, 4,
					       PANEL_MIN_VALUE, 1,
					       PANEL_MAX_VALUE, 100,
					       PANEL_VALUE, 10,
					       PANEL_VALUE_X, xv_col(peakf_panel, 24),
					       XV_Y, xv_row(peakf_panel, 2),
					       NULL);

    peakf_limits_item[0]  = (Panel_item) xv_create(peakf_panel, PANEL_TEXT,
						     XV_HELP_DATA, "xvgr:peakf_limit",
						     PANEL_LABEL_STRING, "limit 1:",
						     PANEL_VALUE_DISPLAY_LENGTH, 15,
						     XV_X, xv_col(peakf_panel, 1),
						     XV_Y, xv_row(peakf_panel, 4),
						     PANEL_VALUE, "1",
						     0);
    peakf_limits_item[1]  = (Panel_item) xv_create(peakf_panel, PANEL_TEXT,
						    XV_HELP_DATA, "xvgr:peakf_limit",
						    PANEL_LABEL_STRING, "limit 2:",
						    PANEL_VALUE_DISPLAY_LENGTH, 15,
						    XV_X, xv_col(peakf_panel, 25),
						    XV_Y, xv_row(peakf_panel, 4),
						    PANEL_VALUE, "8192",
						    0);

    peakf_info_item = (Panel_item) xv_create(peakf_panel, PANEL_MESSAGE,
					      PANEL_LABEL_STRING, "Info: ",
					      XV_X, xv_col(peakf_panel, 1),
					      XV_Y, xv_row(peakf_panel, 5),
					      NULL);
    (void) xv_create(peakf_panel, PANEL_BUTTON,
		     XV_HELP_DATA, "xvgr:peakf_apply",
		     PANEL_LABEL_STRING, "Apply",
  		     PANEL_NOTIFY_PROC, do_peakf_proc,   
		     XV_X, xv_col(peakf_panel, 6),
		     XV_Y, xv_row(peakf_panel, 6),
		     0);
    (void) xv_create(peakf_panel, PANEL_BUTTON,
		     XV_HELP_DATA, "xvgr:peakf_tags",
		     PANEL_LABEL_STRING, "Set tags",
  		     PANEL_NOTIFY_PROC, do_peakf_tags,   
		     XV_X, xv_col(peakf_panel, 15),
		     XV_Y, xv_row(peakf_panel, 6),
		     0);
    (void) xv_create(peakf_panel, PANEL_BUTTON,
		     XV_HELP_DATA, "xvgr:peakf_done",
		     PANEL_LABEL_STRING, "Done",
		     PANEL_NOTIFY_PROC, generic_done_proc,
		     XV_KEY_DATA, FRAME_KEY, peakf_frame,
		     XV_X, xv_col(peakf_panel, 25),
		     XV_Y, xv_row(peakf_panel, 6),
		     NULL);
    window_fit(peakf_panel);
    window_fit(peakf_frame);
    xv_set(peakf_frame, WIN_SHOW, TRUE, NULL);
}


static void
do_peakf_tags()
{
      limits_ptr = peakf_limits_item;
      set_cfit_tags(2,range_tags,update_range_tags);
      return;
}

static Notify_client  peakf_nclient = (Notify_client) 104;
void
do_peakf_proc()
{
      int fwhm, accept, setno;
      double *ydata, limitl, limitu; 
      int setlen;
      char *my_argv[15], *out_file, strbuf[4][128];
      FILE *fp;
      
      fwhm = (int) xv_get(peakf_fwhm_item,PANEL_VALUE);
      accept = (int) xv_get(peakf_accept_item,PANEL_VALUE);
      setno = (int) xv_get(peakf_set_item,PANEL_VALUE);
      if ( ! isactive_set(cg,setno)) {
	    errwin("Selected set not active !");
	    return;
      }

/* check limits are appropriate and get pointers to data */
      ydata = gety(cg, setno);
      if ( check_set_parameters(setno, peakf_limits_item, &limitl, &limitu) < 0)
	    return;
   
/* fill argument vector my_argv */
      my_argv[0] = "sort_peakfind";
      my_argv[1] = "-spectrum";
      my_argv[2] = (out_file = tempnam(NULL,"SORT_"));
      my_argv[3] = "-title";
      if (strlen(g[cg].labs.stitle.s) != 0)
	    my_argv[4] = g[cg].labs.stitle.s;
      else
	    my_argv[4] = out_file;
      my_argv[5] = "-fwhm";
      (void) sprintf((my_argv[6] = strbuf[0]),"%d",fwhm);
      my_argv[7] = "-accept";
      (void) sprintf((my_argv[8] = strbuf[1]),"%d",accept);
      my_argv[9] = "-limit";
      (void) sprintf((my_argv[10] = strbuf[2]),"%d", (int)limitl);
      (void) sprintf((my_argv[11] = strbuf[3]),"%d", (int)limitu);
      my_argv[12] = (char *)0;
      
/* write out spectrum data to data file */      
      if ( (fp = fopen(out_file,"w")) == NULL) {
	    errwin("failed to open peak find data file");
	    return;
      }
      setlen = getsetlength(cg,setno);
      if ( fwrite((char *) ydata, sizeof(double), setlen, fp) != setlen) {
	    errwin("Write to peak find data file failed");
	    fclose(fp);
	    return;
      } 
      (void) fclose(fp);
      
      xv_set(peakf_info_item,PANEL_LABEL_STRING,"peak find in progress",NULL);
/* start peak find process */
      do_process("sort_peakfind", my_argv, peakf_nclient, sigchldcatcher_compwin1);
      if (out_file != NULL) free(out_file);
      return;
}


static void
peakf_apply_tags(pid)
      int  pid;
{
      FILE *fp;
      int nos, tmp_line_color;
      double channel;
      char str_buf[L_tmpnam+24], chan_buf[64], area_buf[64];
      
      sprintf(str_buf,"%sPEAKF_%d",P_tmpdir,pid);
      if ( (fp = fopen(str_buf,"r")) == NULL) {
	    errwin("failed to open peak find output file");
	    return;
      }
      tmp_line_color = line_color;
      line_color = 4;
      while ( fscanf(fp,"%d %s %s",&nos, chan_buf, area_buf) == 3){
	    int tagno;
	    world w;
	    channel = atof(chan_buf);
	    if ((tagno = next_tag()) >= 0) {
		  tags[tagno].gno = cg;
		  get_graph_world(cg,&w);
		  tags[tagno].line_no = define_tag_line(tagno,channel,w.yg1,w.yg2);
		  if (tags[tagno].line_no == -1)
			tags[tagno].active = OFF;
	    } 
      }
      (void) fclose(fp);
      (void) unlink(str_buf);
      line_color = tmp_line_color;
      xv_set(peakf_info_item,PANEL_LABEL_STRING,"peak find completed",NULL);
      return;
}



/***************************************************************************************/

/* window items for mstick frame */
static Frame mstick_frame = (Frame) 0;
static Panel mstick_panel;
static Panel_item mstick_set_item;
static Panel_item mstick_fwhm_item;
static Panel_item mstick_accept_item;
static Panel_item mstick_area_item;
static Panel_item mstick_peak1max_item;
static Panel_item mstick_peak2max_item;
static Panel_item mstick_info_item;
static Panel_item mstick_limits_item[2];


static void do_mstick_tags();
void do_mstick_proc();

/* items for linear regression */
static Panel_item toggle_degree_item;	/* degree of fit */
static Panel_item toggle_resid_item;	/* load residuals, fitted values or
					 * nothing */

void create_mstick_frame()
{
    if (mstick_frame) {
	xv_set(mstick_frame, WIN_SHOW, TRUE, FRAME_CLOSED, FALSE, NULL);
	return;
    }
    mstick_frame = (Frame) xv_create(main_frame, FRAME,
				     FRAME_LABEL, "Matchsticks",
				     NULL);
    mstick_panel = (Panel) xv_create(mstick_frame, PANEL,
				     XV_HELP_DATA, "xvgr:mstick_panel",
				     PANEL_LAYOUT, PANEL_VERTICAL,
				     NULL);
    define_select_set_panel2(mstick_panel, mstick_set_item, "Select set:");
    xv_set(mstick_set_item, PANEL_VALUE_X, xv_col(mstick_panel, 20), NULL);
    mstick_fwhm_item = (Panel_item) xv_create(mstick_panel, PANEL_NUMERIC_TEXT,
						XV_HELP_DATA, "xvgr:mstick_npparm",
						PANEL_LABEL_STRING, "fwhm (channels):",
						PANEL_VALUE_DISPLAY_LENGTH, 4,
						PANEL_MIN_VALUE, 1,
						PANEL_MAX_VALUE, 8192,
					        PANEL_VALUE, 30,
						PANEL_VALUE_X, xv_col(mstick_panel, 24),
						XV_Y, xv_row(mstick_panel, 1),
						NULL);
    mstick_accept_item = (Panel_item) xv_create(mstick_panel, PANEL_NUMERIC_TEXT,
					       XV_HELP_DATA, "xvgr:mstick_nbparm",
					       PANEL_LABEL_STRING, "acceptence (%):",
					       PANEL_VALUE_DISPLAY_LENGTH, 4,
					       PANEL_MIN_VALUE, 1,
					       PANEL_MAX_VALUE, 100,
					       PANEL_VALUE, 10,
					       PANEL_VALUE_X, xv_col(mstick_panel, 24),
					       XV_Y, xv_row(mstick_panel, 2),
					       NULL);
    mstick_area_item = (Panel_item) xv_create(mstick_panel, PANEL_NUMERIC_TEXT,
					       XV_HELP_DATA, "xvgr:mstick_nbparm",
					       PANEL_LABEL_STRING, "Min peak area:",
					       PANEL_VALUE_DISPLAY_LENGTH, 8,
					       PANEL_MIN_VALUE, 1,
					       PANEL_MAX_VALUE, 10000000,
					       PANEL_VALUE, 10000,
					       PANEL_VALUE_X, xv_col(mstick_panel, 24),
					       XV_Y, xv_row(mstick_panel, 3),
					       NULL);
    mstick_peak1max_item = (Panel_item) xv_create(mstick_panel, PANEL_NUMERIC_TEXT,
					       XV_HELP_DATA, "xvgr:mstick_nbparm",
					       PANEL_LABEL_STRING, "Peak 1 <",
					       PANEL_VALUE_DISPLAY_LENGTH, 4,
					       PANEL_MIN_VALUE, 1,
					       PANEL_MAX_VALUE, 8192,
					       PANEL_VALUE, 600,
					       PANEL_VALUE_X, xv_col(mstick_panel, 10),
					       XV_Y, xv_row(mstick_panel, 4),
					       NULL);
    mstick_peak2max_item = (Panel_item) xv_create(mstick_panel, PANEL_NUMERIC_TEXT,
					       XV_HELP_DATA, "xvgr:mstick_nbparm",
					       PANEL_LABEL_STRING, "Peak 2 < ",
					       PANEL_VALUE_DISPLAY_LENGTH, 4,
					       PANEL_MIN_VALUE, 1,
					       PANEL_MAX_VALUE, 8192,
					       PANEL_VALUE, 1200,
					       PANEL_VALUE_X, xv_col(mstick_panel, 30),
					       XV_Y, xv_row(mstick_panel, 4),
					       NULL);

    mstick_limits_item[0]  = (Panel_item) xv_create(mstick_panel, PANEL_TEXT,
						    XV_HELP_DATA, "xvgr:mstick_limit",
						    PANEL_LABEL_STRING, "limit 1:",
						    PANEL_VALUE_DISPLAY_LENGTH, 15,
						    XV_X, xv_col(mstick_panel, 1),
						    XV_Y, xv_row(mstick_panel, 7),
						    PANEL_VALUE, "1",
						    0);
    mstick_limits_item[1]  = (Panel_item) xv_create(mstick_panel, PANEL_TEXT,
						    XV_HELP_DATA, "xvgr:mstick_limit",
						    PANEL_LABEL_STRING, "limit 2:",
						    PANEL_VALUE_DISPLAY_LENGTH, 15,
						    XV_X, xv_col(mstick_panel, 25),
						    XV_Y, xv_row(mstick_panel, 7),
						    PANEL_VALUE, "8192",
						    0);

    mstick_info_item = (Panel_item) xv_create(mstick_panel, PANEL_MESSAGE,
					      PANEL_LABEL_STRING, "Info: ",
					      XV_X, xv_col(mstick_panel, 1),
					      XV_Y, xv_row(mstick_panel, 8),
					      NULL);
    (void) xv_create(mstick_panel, PANEL_BUTTON,
		     XV_HELP_DATA, "xvgr:mstick_apply",
		     PANEL_LABEL_STRING, "Apply",
  		     PANEL_NOTIFY_PROC, do_mstick_proc,
		     XV_X, xv_col(mstick_panel, 6),
		     XV_Y, xv_row(mstick_panel, 9),
		     0);
    (void) xv_create(mstick_panel, PANEL_BUTTON,
		     XV_HELP_DATA, "xvgr:mstick_tags",
		     PANEL_LABEL_STRING, "Set tags",
  		     PANEL_NOTIFY_PROC, do_mstick_tags,   
		     XV_X, xv_col(mstick_panel, 15),
		     XV_Y, xv_row(mstick_panel, 9),
		     0);
    (void) xv_create(mstick_panel, PANEL_BUTTON,
		     XV_HELP_DATA, "xvgr:mstick_done",
		     PANEL_LABEL_STRING, "Done",
		     PANEL_NOTIFY_PROC, generic_done_proc,
		     XV_KEY_DATA, FRAME_KEY, mstick_frame,
		     XV_X, xv_col(mstick_panel, 25),
		     XV_Y, xv_row(mstick_panel, 9),
		     NULL);

    toggle_degree_item = xv_create(mstick_panel, PANEL_CYCLE,
				   XV_HELP_DATA, "xvgr:reg_degree",
				   PANEL_LABEL_STRING, "Regression:",
				   PANEL_CHOICE_STRINGS,
				   "Linear",
				   "Quadratic",
				   "Cubic",
				   "4th degree",
				   "5th degree", 
				   "6th degree", 
				   "7th degree", 
				   "8th degree", 
				   "9th degree", 
				   "10th degree", 
				    NULL,
				   PANEL_VALUE_X, xv_col(mstick_panel, 15),
                                   PANEL_VALUE_Y, xv_row(mstick_panel, 5),
				   NULL);
    toggle_resid_item = xv_create(mstick_panel, PANEL_CYCLE,
				  XV_HELP_DATA, "xvgr:reg_load",
				  PANEL_LABEL_STRING, "Load:",
				  PANEL_CHOICE_STRINGS,
				  "Fitted values",
				  "Residuals", NULL,
				  PANEL_VALUE_X, xv_col(mstick_panel, 15),
                                   PANEL_VALUE_Y, xv_row(mstick_panel, 6),
				  NULL);

    window_fit(mstick_panel);
    window_fit(mstick_frame);
    xv_set(mstick_frame, WIN_SHOW, TRUE, NULL);
}

static void
do_mstick_tags()
{
      limits_ptr = mstick_limits_item;
      set_cfit_tags(2,range_tags,update_range_tags);
      return;
}

static Notify_client  mstick_nclient = (Notify_client) 105;
void
do_mstick_proc()
{
      int fwhm, accept, setno;
      double *ydata, limitl, limitu; 
      int setlen;
      char *my_argv[15], *out_file,  strbuf[4][128];
      FILE *fp;

      fwhm = (int) xv_get(mstick_fwhm_item,PANEL_VALUE);
      accept = (int) xv_get(mstick_accept_item,PANEL_VALUE);
      setno = (int) xv_get(mstick_set_item,PANEL_VALUE);
      if ( ! isactive_set(cg,setno)) {
	    errwin("Selected set not active !");
	    return;
      }

/* check limits are appropriate and get pointers to data */
      ydata = gety(cg, setno);
      if ( check_set_parameters(setno, mstick_limits_item, &limitl, &limitu) < 0)
	    return;
   
/* fill argument vector my_argv */
      my_argv[0] = "sort_peakfind";
      my_argv[1] = "-spectrum";
      my_argv[2] = (out_file = tempnam(NULL,"SORT_"));
      my_argv[3] = "-title";
      if (strlen(g[cg].labs.stitle.s) != 0)
	    my_argv[4] = g[cg].labs.stitle.s;
      else
	    my_argv[4] = out_file;
      my_argv[5] = "-fwhm";
      (void) sprintf((my_argv[6] = strbuf[0]),"%d",fwhm);
      my_argv[7] = "-accept";
      (void) sprintf((my_argv[8] = strbuf[1]),"%d",accept);
      my_argv[9] = "-limit";
      (void) sprintf((my_argv[10] = strbuf[2]),"%d", (int)limitl);
      (void) sprintf((my_argv[11] = strbuf[3]),"%d", (int)limitu);
      my_argv[12] = (char *)0;
      
/* write out spectrum data to data file */      
      if ( (fp = fopen(out_file,"w")) == NULL) {
	    errwin("failed to open peak find data file");
	    return;
      }
      setlen = getsetlength(cg,setno);
      if ( fwrite((char *) ydata, sizeof(double), setlen, fp) != setlen) {
	    errwin("Write to peak find data file failed");
	    fclose(fp);
	    return;
      } 
      (void) fclose(fp);
      
      xv_set(mstick_info_item,PANEL_LABEL_STRING,"Matchstick peak find in progress",NULL);
/* start peak find process */
      do_process("sort_peakfind", my_argv, mstick_nclient, sigchldcatcher_compwin1);
      if (out_file != NULL) free(out_file);
      return;
}

/*Fit to Matchstick centroids*/    
#define CALC_REG     6
Notify_client  regmstick_nclient = (Notify_client) 106;

void mstick_reg_fit(pid)
      int  pid;
{
      FILE *ifp, *ofp;
      char str_buf[L_tmpnam+24], chan_buf[64];
      char pkfile_buf[L_tmpnam+24], area_buf[64];
      int nos, set, setlen, ideg, iresid, peakno;
      char command[BUFSIZ];
      char *c_ptr = command;
      double *ydata, *xdata, limitl, limitu;
      double *regxdata, *regydata, reglimitl, reglimitu;
      char *my_argv[15], *reg_file;
      double channel, area, areamin, chann1max, chann2max;
      double minx,maxx,miny,maxy;

      sprintf(str_buf,"%sPEAKF_%d",P_tmpdir,pid);
      xv_set(mstick_info_item,PANEL_LABEL_STRING,"Matchstick peak find Completed",NULL);
      fprintf(stderr,"\n");

/*Check that found peaks are sensible and
  write ``good matchstick peaks'' to file */

      sprintf(pkfile_buf,"%sPEAKO_%d",P_tmpdir,pid);
      if ( (ofp = fopen(pkfile_buf,"w")) == NULL) {
	    errwin("failed to open peak data write file");
	    return;
      }
      if ( (ifp = fopen(str_buf,"r")) == NULL) {
	    errwin("failed to open peak data read file");
	    return;
      }

      areamin = (double) xv_get(mstick_area_item,PANEL_VALUE);
      chann1max = (double) xv_get(mstick_peak1max_item,PANEL_VALUE);
      chann2max = (double) xv_get(mstick_peak2max_item,PANEL_VALUE);
      peakno=0;

      while ( fscanf(ifp,"%d %s %s",&nos, chan_buf, area_buf) == 3){
	    channel = atof(chan_buf);
	    area = atof(area_buf);

            if(area<areamin){
              fprintf(stderr,"Discarded peak %d channel %g: too small area= %g\n",nos,channel,area);
              continue;
            }


            peakno++;

            if(peakno==1 && channel > chann1max){
               fprintf(stderr, "Missing peak1\n");
               peakno++;
            }
            if(peakno==2 && channel > chann2max){
               fprintf(stderr, "Missing peak2\n");
               peakno++;
            }

            fprintf(ofp,"%g %d\n",channel,peakno);
      }
      
      (void) fclose(ifp);
      (void) fclose(ofp);

      if(peakno<2){
         errwin("Not enough peaks found for a fit!");
         return;
      }

/*load ''good'' centroid data into next available set*/
      if ((set = nextset(cg)) == -1) {
          errwin("failed to open a new set");
	  return;
      }
      
    getdata(cg,pkfile_buf,DISK,XY);

    if ( (setlen = check_set_len(set,3)) < 0)
  	   return;
    if ( check_set_parameters(set, mstick_limits_item, &limitl, &limitu) < 0)
	   return;

    clear_graph_viewport(cg);

    set_prop(cg, SET,
	     SETNUM, set,
	     SYMBOL, TYPE, 2,
	     SYMBOL, FILL, 1,
	     SYMBOL, SIZE, 0.8,
	     SYMBOL, CHAR, 0,
	     SKIP, 0,
	     LINESTYLE, 0,
	     LINEWIDTH, 0,
	     COLOR, 2,
	     FILL, TYPE, 0,
	     FILL, WITH, 0,
	     FILL, COLOR, 2,
	     FILL, PATTERN, 1,
	     0);

             minx= limitl-((limitu-limitl)/10.0);
             maxx= limitu+((limitu-limitl)/10.0);
             miny= 0;
             maxy= peakno+1;
             g[cg].w.xg1 = minx;
	     g[cg].w.xg2 = maxx;
	     g[cg].w.yg1 = miny;
	     g[cg].w.yg2 = maxy;

      (void) unlink(str_buf);
      (void) unlink(pkfile_buf);

/*perform chosen regression type*/

    ideg = (int) xv_get(toggle_degree_item, PANEL_VALUE) + 1;
    iresid = (int) xv_get(toggle_resid_item, PANEL_VALUE);

/* get pointers to data fields */
    xdata = getx(cg, set);
    ydata = gety(cg, set);

/* fill new argument vector my_argv */
      my_argv[0] = "sort_calc_xvgr";
      my_argv[1] = "-spectrum";
      my_argv[2] = (reg_file = tempnam(NULL,"SORT_"));

    reglimitl=(double) 1;
    reglimitu=(double) setlen;
    regxdata=ydata;
    regydata=xdata;

/* setup command string */
    (void) memset(command,0,sizeof(command));
    sprintf(command,"%d %d %d %g %g %c",CALC_REG,setlen,0,reglimitl,reglimitu,'\0');
    c_ptr += strlen(c_ptr) + 1;
    sprintf(c_ptr,"%d %d %c",ideg,iresid,'\0');
    c_ptr += strlen(c_ptr) + 1;
    sprintf(c_ptr,"%d deg fit of set %d %c", ideg, set, '\0');

/* write out relevant data */

    compwin2_outdat(reg_file,command,sizeof(command),regxdata,regydata,setlen);
    
    fprintf(stderr,"\nRegression fit to set %d with x and y flipped i.e.\n",set);
    fprintf(stderr,"Dependent variable=x, Independent variable=y\n");
    xv_set(mstick_info_item,PANEL_LABEL_STRING,"Performing regression fit...",set,NULL);
/* start peak find process */
    do_process("sort_calc_xvgr",my_argv, regmstick_nclient, sigchldcatcher_compwin2);
    if (reg_file != NULL) free(reg_file);


/*COMPARE FIT WITH PEAK CENTROIDS LOOKING FOR DIFFERENCES HERE*/



    return;
}


/***************************************************************************************/

void
do_process(process,argv,nclient,sigchldcatcher)
      char *process;
      char **argv;
      Notify_client  nclient;
      Notify_value (*sigchldcatcher)();
{
      int pid;

      switch(pid = fork()) {
	    case -1:
	        sprintf(buf,"failed to fork %s process\n",process);
		errwin(buf);
		return;
	    case 0:
		execvp(process,argv);
		fprintf(stderr,"failed to exec %s process\n",process);
		_exit(-1);
	    default:
		notify_set_wait3_func(nclient, sigchldcatcher, pid);
		return;
	  }
}

/*ARGSUSED*/
Notify_value
sigchldcatcher_compwin1(nclient, pid, status, rusage)
      Notify_client   nclient;
      int             pid;
#ifdef SVR4
      int             *status;
#else
      union wait      *status;
#endif
      struct rusage   *rusage;
{
      if (WIFEXITED(*status)) {
	    if (WEXITSTATUS(*status)) {
		  sprintf(buf,"calculation1 returned non zero exit status");
		  errwin(buf);
		  return NOTIFY_DONE;
	    } 
	    if (nclient == buffit_nclient)
		  buffit_apply_fit(pid);
	    else if (nclient == peakf_nclient)
		  peakf_apply_tags(pid);
            else if(nclient == mstick_nclient)
                  mstick_reg_fit(pid);
            return NOTIFY_DONE;
      }
      sprintf(buf,"SIGCHLD from child process %d, id %d not handled",
	      pid, nclient);
      errwin(buf);
      return NOTIFY_IGNORED;
}
