/*
 *  slicing routine for sunsort.
 *  using given 2d spectrum produce two 1d spectra with X=constant
 *  and with y=constant .... assume square 2d spectrum.
 *  arguments -spectrum <2d data file> -limit <X> <Y>
 *
 *  last modified   16/1/95
 */

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <stdlib.h>
#include <memory.h>

#include <sys/param.h>
#include <sys/types.h>

#include <errno.h>

#include "sort_def.h"
#include "sort_mem.h"

/********************** global variables and symbols **************************/

int     DBX_val;
char    *spectrum = NULL;
int     X_point, Y_point;

extern   void read_args();
extern   int  xslice(), yslice();
extern   int  write_slice();
#ifdef SVR4
extern   pid_t  lookup_proc();
#define NeedFunctionPrototype 1
#else
extern   int    lookup_proc();
#endif

/****************************************************************************/

int
main(
#if NeedFunctionPrototype
     int     argc,
     char    **argv)
#else
      argc, argv)
      int     argc;
      char    **argv;
#endif
{
      int    *xdata;
      int    *ydata;
      char   buf[100];

/*  check to see which options have been set in the command line */      
      read_args(argc,argv);

/* read data from spectrum file */
      if ( read_from_pipe(spectrum) != 0) {
	    fprintf(stderr,"slicing: failed in read from spectrum file %s\n",spectrum);
	    exit(1);
      }

/* check it is a 2d spectrum data set */
      if (strncasecmp(pshm->com_str,"DISPLAY 2",9) != 0 && strncasecmp(pshm->com_str,"WINDOW",6) != 0) {
	    fprintf(stderr,"slicing: specified spectrum data is not a 2D spectrum\n");
	    exit(1);
      }

/* allocate memory for x & y slice spectra */
      xdata = (int *) calloc(pshm->size, sizeof(int));
      ydata = (int *) calloc(pshm->size, sizeof(int));
      if (xdata == NULL || ydata == NULL) {
	    fprintf(stderr,"slicing: memory allocation failure\n");
	    exit(2);
      }

/* calculate slice spectra */
      if (xslice(pshm->array, pshm->size, X_point, xdata) != 0) {
	    fprintf(stderr,"slicing: error calculating xslice for x=%d\n",X_point);
	    exit(3);
      }
      if (yslice(pshm->array, pshm->size, Y_point, ydata) != 0) {
	    fprintf(stderr,"slicing: error calculating yslice for y=%d\n",X_point);
	    exit(3);
      }

/* write out X slice to temp file*/
      strcpy(pshm->com_str,"DISPLAY 1");
      sprintf(buf, "X=%d", X_point);
      strncpy(pshm->name,buf,NAME_SIZE);
      pshm->more = 1;
      if (write_slice(".SORT_pipe0", pshm->array, pshm->size, xdata) != 0) {
	    fprintf(stderr,"slicing: error writing out xslice\n");
	    exit(4);
      }

/* write out Y slice to temp file */      
      sprintf(buf, "Y=%d", Y_point);
      strncpy(pshm->name,buf,NAME_SIZE);
      pshm->more = 0;
      if (write_slice(".SORT_pipe1", pshm->array, pshm->size, ydata) != 0) {
	    fprintf(stderr,"slicing: error writing out yslice\n");
	    exit(4);
      }

/* signal 1d display process to show slices */
      if ( (int) pshm->pid_1d <= 0 || kill(pshm->pid_1d, SIGUSR1) == -1) {
#ifdef SVR4
	    pid_t pid = lookup_proc("sort_xvgr");
#else
	    int   pid = lookup_proc("sort_xvgr");
#endif
	    if ((int) pid > 0) {
		  if (kill(pid, SIGUSR1) == -1) {
			perror("slicing-kill");
			fprintf(stderr,"slicing: failed to signal 1D display process pid = %d\n",
				(int) pshm->pid_1d);
			exit(5);
		  }
	    }
	    else {
		  fprintf(stderr,"slicing: failed to signal 1D display process pid = %d\n",
			  (int) pshm->pid_1d);
		  exit(5);
	    }
      }
      return(0);
}
     
void
read_args(
#if NeedFunctionPrototype
     int		argc,
     char		**argv)
#else
      argc, argv)
      int		argc;
      char		**argv;
#endif
{
     int i;
     for(i=1; argv[i] != NULL; i++) {
	   if (strncmp(argv[i],"-dbx",4) == 0) {
		 if (argv[i+1] != NULL) {
		       DBX_val = atoi(argv[i+1]);
		       fprintf(stderr,"slicing: diagnostics level set to %d\n",DBX_val);
		       i += 1;
		 }
	   }
	   else if (strncmp(argv[i],"-spectrum",8) == 0) {
		 if (argv[i+1] != NULL) {
		       spectrum = argv[i+1];
		       if (DBX_val > 0)
			     fprintf(stderr,"slicing: spectrum file = %s\n",spectrum);
		       i++;		       
		 }
		 else {
		       fprintf(stderr,"slicing: no spectrum file specified\n");
		       exit(-1);
		 }
	   }
	   else if (strncmp(argv[i],"-limit",5) == 0) {
		 if (argv[i+1] != NULL && argv[i+2] != NULL) {
		       X_point = atoi(argv[i+1]);
		       Y_point = atoi(argv[i+2]);
		       if (DBX_val > 0)
			     fprintf(stderr,"slicing: limits ars set to X = %d, Y = %d\n", X_point, Y_point);
		       i+= 2;
		 }
		 else {
		       fprintf(stderr,"slicing: X and Y limits not specified\n");
		       exit(-1);
		 }
	   }
     }
     if (spectrum == NULL) {
	   fprintf(stderr,"slicing: no spectrum file specified\n");
	   exit(-1);
     }
     return;
}

/*
 * calculate spectrum for line X=constant=point
 * assume spectrum starts from point (0,0)
 */
int
yslice(
#if NeedFunctionPrototype
	int  *data,
	int  size,
	int  point,
	int  *slice)
#else
        data, size, point, slice)
        int  *data;
        int  size;
        int  point;
        int  *slice;
#endif
{
      register int i;
      if (point < 0 || point > size)
	    return(-1);
      for(i=0; i<size; i++) {
	    slice[i] = data[point + i*size];
      }
      return(0);
}
/*
 * calculate spectrum for line Y=constant=point
 * assume spectrum starts from point (0,0)
 */
int
xslice(
#if NeedFunctionPrototype
	int  *data,
	int  size,
	int  point,
	int  *slice)
#else
        data, size, point, slice)
        int  *data;
        int  size;
        int  point;
        int  *slice;
#endif
{
      register int i;
      if (point < 0 || point > size)
	    return(-1);
      for(i=0; i<size; i++) {
	    slice[i] = data[point*size + i];
      }
      return(0);
}
      
int
write_slice(
#if NeedFunctionPrototype
	    char  *out_file,
	    int   *out_data,
	    int   size,
	    int   *in_data)
#else
      out_file, out_data, size, in_data)
      char  *out_file;
      int   *out_data;
      int   size;
      int   *in_data;
#endif
{
      (void) memcpy(out_data, in_data, sizeof(int)*size);
      if ( write_to_pipe(out_file,size) == SORT_FAIL)
	    return(-1);
      return(0);
}

/*
 *  try and find process with name = proc_name and belonging
 *  to process group of this process
 */

#ifdef SVR4

#include <dirent.h>
#include <sys/types.h>
#include <sys/signal.h>
#include <sys/fault.h> 
#include <sys/syscall.h>
#include <sys/procfs.h>
#include <sys/stat.h>
#include <fcntl.h>

pid_t lookup_proc(
#if NeedFunctionPrototype
		  char *proc_name)
#else
      proc_name)
      char *proc_name;
#endif               /* if NeedFunctionPrototype */
{
      pid_t          psid;
      DIR	     *dirp;
      struct dirent  *dp;
      int            fd;
      prpsinfo_t     proc_info;
      char           buf[BUFSIZ];
      
/* get process group id */      
      if ( (psid = getsid((pid_t)0)) == -1) {
	    perror("slicing-getsid");
	    return(-1);
      }

/* open /proc directory for reading and loop through all enteries */
      if ((dirp = opendir("/proc")) == NULL) {
	    perror("slicing-opendir");
	    return(-1);
      }

/* cycle through systems processes */      
      for (dp=readdir(dirp); dp != NULL; dp=readdir(dirp)) {
	    sprintf(buf,"/proc/%s",dp->d_name);
	    if (dp->d_name[0] == '.')
		  continue;
	    if (access(buf, R_OK) == 0) {
		  if ( (fd = open(buf,  O_RDONLY)) == -1) {
			perror("slicing-open");
			break;
		  }
		  if ( ioctl(fd, PIOCPSINFO, (void *) &proc_info) == -1) {
			perror("slicing-ioctl");
			(void) close(fd);
			break;
		  }
		  if (proc_info.pr_sid == psid) {
			int len = strlen(proc_name);
			len = (len > PRFNSZ) ? PRFNSZ : len;
			if (strncmp(proc_name, proc_info.pr_fname, len) == 0) {
			      (void) close(fd);
			      return(proc_info.pr_pid);
			}
		  }
		  (void) close(fd);
	    }
	    else {
		  if (errno != EACCES)
			perror("slicing-access");
	    }
      }
      closedir(dirp);
      return(-1);
}

#else               /* ifdef SVR4 */
      
int   lookup_proc(
#if NeedFunctionPrototype
      char *proc_name)
#else
      proc_name)
      char *proc_name;
#endif              /* if NeedFunctionPrototype */
{
      return(-1);
}

#endif              /* ifdef SVR4 */

