/*
 * ''cal''      Calibration                                Steve Chappell
 *
 * Subroutines to calibrate detector hits in terms of the 
 * event array
 *                                          
 * Updates
 *   24th Apr 1998
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include ".sunsort_initadc.h"
#include <subs.h>
#include "cal.h"

struct xstats xstats;               /*Extra statistics*/
int ndets, ndetsjunk;

static int k;	                    /* Particle index */

static void do_pulser(struct detectors *dp)
{
    while(dp != NULL)
    {
	if (dp->adcnum > 384)
	    inc2d(SP2_PULSERT, (int) (dp->value/8 + 0.5), dp->adcnum - 384);
	else if (dp->adcnum > 0)
	    inc2d(SP2_PULSERA, (int) (dp->value/32 + 0.5), dp->adcnum);
	do_pulser(dp->hitlist.head);
	dp = dp->hitlist.next;
    }
}

/*
 * EbyE calibration
 */
int cal(void)
{
    int i;
    struct detectors *dp;
    static int pulvetos[] = { PULVETOS };
    
    /*adclist_dump();*/           /*Dump adclist*/
    ev_zeroheader();              /*Reset event header*/
    plothits();                   /*Plot Channel hit patterns*/
    markhitslist(adclist_.adclist, adcs_.adcs);	  /*mark hits on detector
						    structure*/
    /*Classify event and generate stats*/
    switch (detector_event(CHECK_PROC, expt))
    {
    case DET_ABORTSORT:
	return EV_ABORTSORT;
    case DET_SKIPEVENT:
	return EV_SKIPEVENT;
    }
    
    /*event_dump(expt, 0); */          /*Output event info*/
    
    for (i=0; i<=ndets; i++)
	if (evhead.cmult[i] > 0)
	    inc2d(SP2_CMULT, evhead.cmult[i], i);

    if (evhead.cmult[0] > PULVETO){   /*Possible Pulser Event*/
	for(i = 0; i < 320; i += 32)  /*check each crate*/
	{
	    int j, c;
	
	    for(j = 1; j < 32; j++)
		if (ADC(i + j) > 0)
		    c++;
	    if (c >= pulvetos[i/32])  /*Highly probable pulser event*/
	    {
		do_pulser(expt);

		evhead.type = EV_PULSER;
		return EV_PULSER;
	    }
	}
    }
    if (evhead.cmult[0] > MULTMAX){  /*Process Max Multiplicity Event*/
	evhead.type = EV_MULTMAX;
	return EV_MULTMAX;
    }
    else{                            /*Calibrate and process Strip Hits*/
	k = 0;
	switch (detector_event(HIT_PROC, expt))
	{
	case DET_ABORTSORT:
	    return EV_ABORTSORT;
	case DET_SKIPEVENT:
	    return EV_SKIPEVENT;
	}
	
	for(i = 0; i <= ndets; i++)
	    if (evhead.pmult[i] > 0)   /*single particle multiplicities*/
		inc2d(SP2_PMULT, evhead.pmult[i], i);

	evhead.n[0] = evhead.n[1] = k; /*fill out header with singles*/
    }

    return EV_OK;
    
}


/*
 *  User functions called by detector_event for each detector type
 */

int strip_event(int proc, struct detectors *dp)
{
    switch(proc)
    {
    case CHECK_PROC:
	switch(dp->hits)    /*How many of the 4 signals making up this strip
			      fired ?*/
	{
	case 1:		    /*1, 2, 3 - partial hit - bad*/
	case 2:
	case 3:
	{
	    struct detectors *siga, *sigt;

	    for(siga = dp->children; siga != NULL; siga = sigt->next)
	    {
		sigt = siga->next;

		if (siga->hits != 0)
		{
		    if (sigt->hits == 0)
			xstats.badt++;
		    else
		    {
			evhead.cmult[dp->parent->number]++;
			evhead.cmult[0]++;
			xstats.chanok++;
			xstats.detpartial++;
			CHANSPECS(siga, sigt);
		    }
		}
		else if(sigt->hits != 0)
		    xstats.bada++;
	    }

	    return DET_OK;
	}
	case 4:	    /*4 - good hit*/
	{
	    struct detectors *siga, *sigt;

	    xstats.chanok += 2;
	    xstats.detelok++;
	    evhead.cmult[dp->parent->number] += 2;
	    evhead.cmult[0] += 2;
	   
	    siga = dp->children;
	    sigt = siga->next;
	    CHANSPECS(siga, sigt);

	    siga = sigt->next;
	    sigt = siga->next;
	    CHANSPECS(siga, sigt);
	    return DET_OK;
	}
	default:    /*paranoia*/
	    fprintf(stderr, "cal: strip_event entered with hits = %d.\n",
		    dp->hits);
	    return DET_OK;
	}
    case HIT_PROC: /*Calibrate and process strip hits*/
    {
	struct stripdata *sp;
	struct pssddata *pp;
	struct detectors *siga, *sigat, *sigb, *sigbt;
	double a, b, e, xd, yd, pfac, mrec, pcm, vcm, pcm_z;
	int tdiff, tabs, dt;
	struct xy detpos;
	struct hit_coord hitvec;

	if (dp->hits != 4)
	    return DET_OK;
	
    	/*Veto non-linear adc226 -> d8s2 and missing adc250 -> d8s10*/
      	if(dp->parent->number == 8 && dp->number == 2 ||
	   dp->parent->number == 6 && dp->number == 16 ||
	   dp->parent->number == 5 && dp->number == 2)

	    return DET_OK; 
	
	sp = (struct stripdata *) dp->user;
	pp = (struct pssddata *) dp->parent->user;
	dt = dp->parent->number;

	siga = dp->children;
	sigat = siga->next;
	sigb = sigat->next;
	sigbt = sigb->next;

	a = siga->value;
	b = sigb->value;
	e = sp->gcal[0]*a+sp->gcal[1]*b;
	xd = (0.5*(b-a)/(a+b)+0.5)*PSCALE;
	yd = dp->number*32 - 16;
	tdiff = sigat->value - sigbt->value + 1000;
	tabs = (sigat->value + sigbt->value)/2;

	inc2d(SP2_TDIFF, tdiff/TSCALE, dt*16 + dp->number - 16);
	inc2d(SP2_TABS, tabs/20, dt*16 + dp->number - 16);
	/*Calibration spectra*/
#ifdef WRTSP_EVX
	dinc2d((dt-1)*16 + 100 + dp->number, xd, e/ESCALE);
	dinc2d(SP2_CAL, xd, e/ESCALE);
#elif WRTSP_EVY
	dinc2d((dt-1)*16 + 100 + dp->number, yd, e/ESCALE);
	dinc2d(SP2_CAL, yd, e/ESCALE);
#elif WRTSP_EVT
	dinc2d((dt-1)*16 + 100 + dp->number, tdiff/TSCALE, e/ESCALE);
	dinc2d(SP2_CAL, tdiff/TSCALE, e/ESCALE);
#elif WRTSP_XVT
	dinc2d((dt-1)*16 + 100 + dp->number, tdiff/TSCALE, xd);
	dinc2d(SP2_CAL, tdiff/TSCALE, xd);
#elif WRTSP_YVT
	dinc2d((dt-1)*16 + 100 + dp->number, tdiff/TSCALE, yd);
	dinc2d(SP2_CAL, tdiff/TSCALE, yd);
#elif WRTSP_XY
	dinc2d((dt-1)*16 + 100 + dp->number, xd, yd);
	dinc2d(SP2_CAL, xd, yd);
#elif WRTSP_XVTDIFF
	dinc2d((dt-1)*16 + 100 + dp->number, xd, tdiff/5-100);
#endif
	
	/*test if hit parameters are sensible*/
	if (e <= MIN_ENERGY || e >= MAX_ENERGY ||
	    xd < sp->xlim[0] || xd > sp->xlim[1] ||
	    tdiff < MIN_T || tdiff > MAX_T)
	    return DET_OK;
	
	/* Calculate true coordinates from xd and dp->number */
	detpos.x = (xd - sp->xlim[0])/(sp->xlim[1] - sp->xlim[0]) - 0.5;
	detpos.y = dp->number * 0.0625 - .46875;
	coord(dt, &detpos, &hitvec);
	
	/*Fill out Event Structure etc. assuming 2-body scattering*/
	evhead.pmult[dt]++;
	evhead.pmult[0]++;
	k++ /*particle index*/;
	
	ev[k].e = e;
	evhead.etot += ev[k].e;
#ifdef EV_TABS
	ev[k].t = tabs;
#else
	ev[k].t = tdiff;
#endif
	ev[k].m = reac.mass[0];
	ev[k].det = dt;
	ev[k].seg = dp->number;
	ev[k].fold = 1;
	ev[k].p = pfac = sqrt(2*ev[k].m*ev[k].e);
	pfac /= hitvec.r;
	ev[k].px = hitvec.x*pfac;
	ev[k].py = hitvec.y*pfac;
	ev[k].pz = hitvec.z*pfac;
	ev[k].th = hitvec.th;
	ev[k].phi = hitvec.phi;
	vcm=(sqrt(2.0*reac.e1*reac.mass[1]))/(reac.mass[1]+reac.mass[2]);
	pcm_z=ev[k].pz-ev[k].m*vcm;
	pcm=sqrt(ev[k].px*ev[k].px+ev[k].py*ev[k].py+pcm_z*pcm_z);
	ev[k].thcm=(acos(pcm_z/pcm))/RAD;
	ev[k].q = 0.0;
	mrec=reac.mass[1]+reac.mass[2]-ev[k].m;
	if(mrec > 0)
	    ev[k].q2=((ev[k].m+mrec)*ev[k].e - 
		      (mrec-reac.mass[1])*reac.e1   - 
		      2.0*(sqrt(reac.mass[1]*ev[k].m*reac.e1*ev[k].e)) *
		      cos(ev[k].th*RAD)
		      )/mrec;
	else
	    ev[k].q2=0.0;
	ev[k].hi = 0;
	ev[k].li = 0;
	dinc2d(SP2_THPHI, ev[k].th*THSCALE,ev[k].phi);
	
	/*Calibration spectra*/

#ifdef WRTSP_EVTH
	dinc2d((dt-1)*16 + 100 + dp->number, ev[k].th*THSCALE, ev[k].e/ESCALE);
	dinc2d(SP2_CAL, ev[k].th*THSCALE, ev[k].e/ESCALE);
#elif WRTSP_Q2VTH
	dinc2d((dt-1)*16 + 100 + dp->number, ev[k].th*THSCALE,
	       ev[k].q2/Q2SCALE +170);
	dinc2d(SP2_CAL, ev[k].th*THSCALE, ev[k].q2/Q2SCALE +170);
#elif WRTSP_Q2VTHCM
	dinc2d((dt-1)*16 + 100 + dp->number, ev[k].thcm*THSCALE,
	       ev[k].q2/Q2SCALE +128);
	dinc2d(SP2_CAL, ev[k].thcm*THSCALE, ev[k].q2/Q2SCALE +128);
#endif
	
	dinc2d(SP2_EVX, ev[k].th*THSCALE, ev[k].e/ESCALE);

	
	return DET_OK;
    }
    default:
	return DET_OK;
    }
}

int monitor_event(int proc, struct detectors *dp)
{
    switch(proc)
    {
    case CHECK_PROC:
	switch(dp->hits)    /*How many of the 2 signals making up this
			      detector fired ?*/
	{
	case 1:		/*partial hit*/
	    if (dp->children->hits == 0)
		xstats.bada++;
	    else
		xstats.badt++;
	    return DET_OK;
	case 2:		/*good hit*/
	{
	    struct detectors *siga, *sigt;

	    xstats.chanok++;
	    xstats.detelok++;

	    siga = dp->children;
	    sigt = siga->next;
	    CHANSPECS(siga, sigt);

	    evhead.cmult[dp->number]++;
	    evhead.cmult[0]++;

	    dinc1d(SP_MONITOR, siga->value);

	    return DET_OK;
	}
	default:    /*paranoia*/
	    fprintf(stderr, "cal: Monitor_event entered with hits = %d.\n",
		    dp->hits);
	    return DET_OK;
	}
    default:
	return DET_OK;
    }
}
	    
int junksig_event(int proc, struct detectors *dp)
{
    switch(proc)
    {
    case CHECK_PROC:
	switch(dp->hits)    /*How many of the 2 signals making up this
			       detector fired ?*/
	{
	case 1:		/*partial hit*/
	    xstats.badjunk++;
	    return DET_OK;
	case 2:		/*good hit*/
	{
	    struct detectors *siga, *sigt;

	    xstats.goodjunk++;

	    siga = dp->children;
	    sigt = siga->next;
	    CHANSPECS(siga, sigt);

	    return DET_OK;
	}
	default:    /*paranoia*/
	    fprintf(stderr, "cal: Junk_event entered with hits = %d.\n",
		    dp->hits);
	    return DET_OK;
	}
    default:
	return DET_OK;
    }
}


/*
 * write out the extra statistics
 */
void write_xstats(void)
{
    fprintf(stderr,"\n*** xtra statistics ***\n\n");
    fprintf(stderr,"Channels with both energy and time : %d\n"
	    "Channels with amplitude but no time   : %d\n"
	    "Channels with time but no amplitude   : %d\n"
	    "Elements completely hit               : %d\n"
	    "Elements partially hit                : %d\n"
	    "Good junk                             : %d\n"
	    "Bad junk                              : %d\n",
	    xstats.chanok, xstats.badt, xstats.bada, xstats.detelok,
	    xstats.detpartial, xstats.goodjunk, xstats.badjunk);
}


/*
 *Routine to plot channel hit patterns i.e. Tartan plots
 */
void plothits(void)
{
    short *sp, *sp2, a, b, spec;
    
    for(sp = adclist_.adclist; (a = *sp) != -1; sp++){
	if(ADC(a) > 0){
	    inc1d(SP_CHANNSAT, a);
	    spec = (a < 385) ? SP2_AHIT : (a -= 384, SP2_THIT);
	    inc2d(spec, a, a);
	    for(sp2 = sp + 1; (b = *sp2) != -1; sp2++){
		if ((spec == SP2_THIT) != (b > 384))
		    continue;
		if (ADC(b) > 0) {
		    inc2d(spec, a, b = ((b > 384) ? b - 384 : b));
		    inc2d(spec, b, a);
		}
	    }
	}
	else
	    inc1d(SP_ZEROAT, a);
    }
    
    
    return;
}
