/*
 * ''calset''   Calibration Setup                           Steve Chappell
 *
 * Subroutines to set up calibration
 *                                          
 * Updates
 *   11th Jan 1998
 */

#include <stdio.h>
#include <stdlib.h>
#include ".sunsort_initadc.h"
#include <subs.h>
#include "cal.h"

int eginit(char *);           /*eurogam format map file initialiser*/

struct detectors *expt = NULL;
int strip_init(struct detectors *dp);
int ampsig_init(struct detectors *dp);

static FILE *gcalfp, *tcalfp, *xlimfp;

/*
 * calset called from init
 */
int calset(void)
{
    struct detectors *dp;
    FILE *fp;
    int i;

    memset (&xstats, 0, sizeof(xstats));       /*zero extra statistics*/

/*initialise eurogam address map*/
    if (eginit(CALIB_DIR "egmap.cal") != 0){
	if(eginit("egmap.dat") != 0){
	    fprintf(stderr, "calset: Error reading \"%s\" and \"%s\".\n",
		    CALIB_DIR "egmap.cal", "egmap.dat");
	    return(1);
	}    
	else
	fprintf(stderr, "calset: Error reading \"%s\", "
		"using default \"%s\".\n",CALIB_DIR "egmap.cal", "egmap.dat");
    }
    else
	fprintf(stderr, "calset: Using address map \"%s\".\n",
		CALIB_DIR "egmap.cal");


/*next generation stuff*/

    if ((expt = loadexperiment(CALIB_DIR "expt.cal")) == NULL)
    {
	fprintf(stderr, "calset: Unable to load experiment structure file.\n");
	return 1;
    }
    fprintf(stderr,"calset: Experiment structure file loaded.\n");

    if (loadadcs(expt, CALIB_DIR "map.cal",384) < 0)
    {
	fprintf(stderr, "calset: Unable to load ADC numbers.\n");
	return 1;
    }
    printf("calset: ADC numbers loaded.\n");

    if (initmarkhits(expt, MAXCHAN)<0)
    {
	fprintf(stderr, "calset: Unable to initialise markhits.\n");
	return 1;
    }
    fprintf(stderr,"calset: markhits initialised.\n");

    if (loadgainoffset(CALIB_DIR "go.cal") < 0)
	return 1;
    if (loadthresholds(CALIB_DIR "thresh.cal") < 0)
	return 1;

    /* Select the functions to be called for the different types of
       detectors by detector_init and detector_event */

    setdetprocs("junksig", NULL, NULL, NULL, junksig_event);
    setdetprocs("monitor", NULL, NULL, NULL, monitor_event);
    setdetprocs("strip", NULL, strip_init, NULL, strip_event);
    setdetprocs("signala", NULL, ampsig_init, NULL, NULL);
    setdetprocs("signalb", NULL, ampsig_init, NULL, NULL);
    setdetprocs("signalc", NULL, ampsig_init, NULL, NULL);
    setdetprocs("signald", NULL, ampsig_init, NULL, NULL);
    setdetprocs("signale", NULL, ampsig_init, NULL, NULL);

    /* Now perform the initialisation */

    if ((gcalfp = fopen(CALIB_DIR "gsig.cal", "r")) != NULL)
	fprintf(stderr, "calset: Reading file %s.\n", CALIB_DIR "gsig.cal");
    else
	fprintf(stderr, "calset: No file %s, using default values.\n",
		CALIB_DIR "gsig.cal");

    if ((tcalfp = fopen(CALIB_DIR "tsig.cal", "r")) != NULL)
	fprintf(stderr, "calset: Reading file %s.\n", CALIB_DIR "tsig.cal");
    else
	fprintf(stderr, "calset: No file %s, using default values.\n",
		CALIB_DIR "tsig.cal");

    if ((xlimfp = fopen(CALIB_DIR "xlim.cal", "r")) != NULL)
	fprintf(stderr, "calset: Reading file %s.\n", CALIB_DIR "xlim.cal");
    else
	fprintf(stderr, "calset: No file %s, using default values.\n",
		CALIB_DIR "xlim.cal");


    if (detector_init(expt) == DET_ABORTSORT)
	return 1;

    ndets = 0;		/* Number of non-junk detectors */
    ndetsjunk = 0;	/* Total number of detectors */
    for(dp = expt->children; dp != NULL; dp = dp->next)
    {
	ndetsjunk++;
	if (strncmp(dp->info->name, "junk", 4)) /* if name is not junk... */
	{
	    ndets++; /* Then this is a non-junk detector */
	    if (ndets != ndetsjunk)
	    {
		fprintf(stderr, "calset: Put junk detectors at end of "
			"experiment definition.\n");
		return 1;
	    }
	}
    }

    if (gcalfp != NULL)
	fclose(gcalfp);
    if (tcalfp != NULL)
	fclose(tcalfp);
    if (xlimfp != NULL)
	fclose(xlimfp);

    if (coord_init(ndets))
    {
	fprintf(stderr, "calset: Error initialising coordinates.\n");
	return 1;
    }
 
    if ((fp = fopen(CALIB_DIR "detpos.cal", "r")) == NULL)
    {
	fprintf(stderr, "calset: Can't find detector setup file \""
		CALIB_DIR "detpos.cal\".\n");
	return 1;
    }
    fprintf(stderr, "calset: Reading file " CALIB_DIR "detpos.cal.\n");

    for(dp = expt->children, i = 1; i <= ndets; i++, dp=dp->next)
	if(strcmp(dp->info->name, "monitor") && coord_det(i, fp))
	{
	    fclose(fp);
	    return 1;
	}

    fclose(fp);

    /*expt_dump(expt, 0);*/              /*Output the experiment set-up*/
    if(ev_init(NP_MAX)) return(1);       /*initialise event structure*/
    if (reac_init()) return 1;           /*reaction definition*/

    return(0);
}


/*
 *  User functions called by detector_init for each detector type
 */

int ampsig_init(struct detectors *dp)
{
    if (dp->threshmax > 8191)
	dp->threshmax = 8191;
}

int strip_init(struct detectors *dp)
{
    struct stripdata *sp;
    char buffer[SUBS_BUFSIZE];

    if (dp->user == NULL)
	if ((dp->user = malloc(sizeof(struct stripdata))) == NULL)
	{
	    fprintf(stderr, "calset: Out of memory allocation strip data.\n");
	    return DET_ABORTSORT;
	}

    sp = (struct stripdata *) dp->user;

    if (gcalfp == NULL)
	sp->gcal[0] = sp->gcal[1] = 1.0;
    else
    {
	if (mygetline(buffer, SUBS_BUFSIZE, gcalfp) == NULL)
	{
	    fprintf(stderr, "Unexpected end of gcal file.\n");
	    return DET_ABORTSORT;
	}
	if (sscanf(buffer, "%f %f", sp->gcal, sp->gcal + 1) != 2)
	{
	    fprintf(stderr, "Expected 2 x gcal, got \"%s\".\n", buffer);
	    return DET_ABORTSORT;
	}
    }

    if (tcalfp == NULL)
	sp->toff[0] = sp->toff[1] = 0.0;
    else
    {
	if (mygetline(buffer, SUBS_BUFSIZE, tcalfp) == NULL)
	{
	    fprintf(stderr, "Unexpected end of tcal file.\n");
	    return DET_ABORTSORT;
	}
	if (sscanf(buffer, "%f %f", sp->toff, sp->toff + 1) != 2)
	{
	    fprintf(stderr, "Expected 2 x toff, got \"%s\".\n", buffer);
	    return DET_ABORTSORT;
	}
    }

    if (xlimfp == NULL)
    {
	sp->xlim[0] = 0;
	sp->xlim[1] = PSCALE;
    }
    else
    {
	if (mygetline(buffer, SUBS_BUFSIZE, xlimfp) == NULL)
	{
	    fprintf(stderr, "Unexpected end of xlim file.\n");
	    return DET_ABORTSORT;
	}
	if (sscanf(buffer, "%f %f", sp->xlim, sp->xlim + 1) != 2)
	{
	    fprintf(stderr, "Expected 2 x xlim, got \"%s\".\n", buffer);
	    return DET_ABORTSORT;
	}
    }

    return DET_OK;
}
