/* 
 * ``coord.c''  Routines for Coordinate Reconstruction     SPGC
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "subs.h"

struct det_coord *dp = NULL;

/*
 * Initialise detector coordinates
 */
int coord_init(int n){
    /*create space for dp[1 to n detectors] structure array*/
    
    if (dp != NULL)
	free(dp);
    
    if ((dp = (struct det_coord *) 
	 calloc(n+1, sizeof(struct det_coord))) == NULL)
    {
	fprintf(stderr, "Unable to allocate memory"
		" for detector coord structure.\n");
	return(1);
    }
    return(0);
}

static int coord_readxyz(FILE *fp, struct xyz *r)
{
    char buf[SUBS_BUFSIZE];

    if (mygetline(buf, SUBS_BUFSIZE, fp) == NULL)
    {
	fprintf(stderr, "Unexpected end of coords file.\n");
	return -1;
    }
    if(sscanf(buf, " %f %f %f", &(r->x), &(r->y), &(r->z)) != 3) {
	fprintf(stderr,"Error! Expected coords xyz got \"%s\"\n",buf);	
	return(-1);
    }
    
    return 0;
}
    

/*
 * Process Detector coordinate file
 */
int coord_det(int i, FILE *fp)
{
    struct xy xydat, *cp = &xydat;
    struct hit_coord hit_coord, *hp = &hit_coord;
    
    if (coord_readxyz(fp, &(dp[i].ra)) || coord_readxyz(fp, &(dp[i].rb)) ||
	coord_readxyz(fp, &(dp[i].rc)) || coord_readxyz(fp, &(dp[i].rd)))
	return -1;

    /*calculate vectors & detector centre coordinates*/
    cp->x = 0.0;
    cp->y = 0.0;
    coord(i,cp,hp);
    dp[i].r0.x = hp->x;
    dp[i].r0.y = hp->y;
    dp[i].r0.z = hp->z;
    dp[i].Xp.x = dp[i].rb.x - dp[i].ra.x;
    dp[i].Xp.y = dp[i].rb.y - dp[i].ra.y;
    dp[i].Xp.z = dp[i].rb.z - dp[i].ra.z;
    dp[i].Yp.x = dp[i].rd.x - dp[i].ra.x;
    dp[i].Yp.y = dp[i].rd.y - dp[i].ra.y;
    dp[i].Yp.z = dp[i].rd.z - dp[i].ra.z;   
    dp[i].n.x = dp[i].Xp.y*dp[i].Yp.z - dp[i].Xp.z*dp[i].Yp.y;
    dp[i].n.y = dp[i].Xp.z*dp[i].Yp.x - dp[i].Xp.x*dp[i].Yp.z;
    dp[i].n.z = dp[i].Xp.x*dp[i].Yp.y - dp[i].Xp.y*dp[i].Yp.x;	
    dp[i].Xp2 =  dp[i].Xp.x*dp[i].Xp.x + dp[i].Xp.y*dp[i].Xp.y +
	dp[i].Xp.z*dp[i].Xp.z;
    dp[i].Yp2 =  dp[i].Yp.x*dp[i].Yp.x + dp[i].Yp.y*dp[i].Yp.y + 
	dp[i].Yp.z*dp[i].Yp.z;  
    dp[i].Xp0 = dp[i].r0.x*dp[i].Xp.x + dp[i].r0.y*dp[i].Xp.y + 
	dp[i].r0.z*dp[i].Xp.z;
    dp[i].Yp0 = dp[i].r0.x*dp[i].Yp.x + dp[i].r0.y*dp[i].Yp.y + 
	dp[i].r0.z*dp[i].Yp.z;
    
    return(0);
}
/*
 * Generate coordinates for ancillary detector
 */
int coord_andet(int i, FILE *fp)
{
    char buf[SUBS_BUFSIZE];
    float thx,thy,l,h,r;
    
    if (mygetline(buf, SUBS_BUFSIZE, fp) == NULL)
    {
	fprintf(stderr, "Unexpected end of coords file.\n");
	return -1;
    }
    if( sscanf(buf, " %f %f %f %f %f",&thx,&thy,&l,&h,&r) !=5){
	fprintf(stderr,"Error! Expected "
		"'thx thy l h r'\n"
		"got \"%s\".\n", buf);
	return(-1);
    }
    
    dp[i].r0.x = r*cos(thy*RAD)*sin(thx*RAD);
    dp[i].r0.y = r*sin(thy*RAD);
    dp[i].r0.z = r*cos(thy*RAD)*cos(thx*RAD);
    
    dp[i].Xp.x = l*cos(thx*RAD);
    dp[i].Xp.y = 0;
    dp[i].Xp.z = -l*sin(thx*RAD);
    dp[i].Yp.x = 0;
    dp[i].Yp.y = h*cos(thy*RAD);
    dp[i].Yp.z = h*sin(thy*RAD);
    
    dp[i].ra.x = dp[i].r0.x - dp[i].Xp.x/2 - dp[i].Yp.x/2;
    dp[i].ra.y = dp[i].r0.y - dp[i].Xp.y/2 - dp[i].Yp.y/2;
    dp[i].ra.z = dp[i].r0.z - dp[i].Xp.z/2 - dp[i].Yp.z/2;
    dp[i].rb.x = dp[i].r0.x + dp[i].Xp.x/2 - dp[i].Yp.x/2;
    dp[i].rb.y = dp[i].r0.y + dp[i].Xp.y/2 - dp[i].Yp.y/2;
    dp[i].rb.z = dp[i].r0.z + dp[i].Xp.z/2 - dp[i].Yp.z/2;
    dp[i].rc.x = dp[i].r0.x + dp[i].Xp.x/2 + dp[i].Yp.x/2;
    dp[i].rc.y = dp[i].r0.y + dp[i].Xp.y/2 + dp[i].Yp.y/2;
    dp[i].rc.z = dp[i].r0.z + dp[i].Xp.z/2 + dp[i].Yp.z/2;
    dp[i].rd.x = dp[i].r0.x - dp[i].Xp.x/2 + dp[i].Yp.x/2;
    dp[i].rd.y = dp[i].r0.y - dp[i].Xp.y/2 + dp[i].Yp.y/2;
    dp[i].rd.z = dp[i].r0.z - dp[i].Xp.z/2 + dp[i].Yp.z/2;   

    dp[i].n.x = dp[i].Xp.y*dp[i].Yp.z - dp[i].Xp.z*dp[i].Yp.y;
    dp[i].n.y = dp[i].Xp.z*dp[i].Yp.x - dp[i].Xp.x*dp[i].Yp.z;
    dp[i].n.z = dp[i].Xp.x*dp[i].Yp.y - dp[i].Xp.y*dp[i].Yp.x;	
    dp[i].Xp2 =  dp[i].Xp.x*dp[i].Xp.x + dp[i].Xp.y*dp[i].Xp.y +
	dp[i].Xp.z*dp[i].Xp.z;
    dp[i].Yp2 =  dp[i].Yp.x*dp[i].Yp.x + dp[i].Yp.y*dp[i].Yp.y + 
	dp[i].Yp.z*dp[i].Yp.z;
    dp[i].Xp0 = dp[i].r0.x*dp[i].Xp.x + dp[i].r0.y*dp[i].Xp.y + 
	dp[i].r0.z*dp[i].Xp.z;
    dp[i].Yp0 = dp[i].r0.x*dp[i].Yp.x + dp[i].r0.y*dp[i].Yp.y + 
	dp[i].r0.z*dp[i].Yp.z;

    return(0);
}


/*
 * Calculate event coordinates for a hit on a detector
 */
int coord(int dt, struct xy *cp, struct hit_coord *hp)
{    
    /*detector to lab coordinate transformation*/
    
    hp->x=(0.5-cp->x-cp->y)*dp[dt].ra.x + cp->x*dp[dt].rb.x + 
	0.5*dp[dt].rc.x + cp->y*dp[dt].rd.x;
    hp->y=(0.5-cp->x-cp->y)*dp[dt].ra.y + cp->x*dp[dt].rb.y + 
	0.5*dp[dt].rc.y + cp->y*dp[dt].rd.y;
    hp->z=(0.5-cp->x-cp->y)*dp[dt].ra.z + cp->x*dp[dt].rb.z + 
	0.5*dp[dt].rc.z + cp->y*dp[dt].rd.z;
    
    hp->r=sqrt(hp->x*hp->x+hp->y*hp->y+hp->z*hp->z);
    hp->th=(acos(hp->z/hp->r))/RAD;
    if((hp->phi=(atan2(hp->y,hp->x))/RAD) < 0) 
	hp->phi+=360;
    
    return(0);
    
}


/*
 * Return which detector was hit for a th,phi trajectory
 */
int ndet_hit(int ndet, float th, float phi)
{
    int i;
    for(i=1; i <ndet+1; i++){
	if(det_hit(i, th, phi))
	    return(i);
    }
    return(0);
}
/*
 * Check if hit a particular detector
 */
int det_hit(int i, float th, float phi)
{
    struct xyz rhat,r;
    float gamma,gamma1,gamma2,xd,yd;
    
    rhat.x = sin(th*RAD)*cos(phi*RAD);
    rhat.y = sin(th*RAD)*sin(phi*RAD);
    rhat.z = cos(th*RAD);
    
    gamma1 = (dp[i].r0.x*dp[i].n.x + dp[i].r0.y*dp[i].n.y + 
	      dp[i].r0.z*dp[i].n.z);
    gamma2= (rhat.x*dp[i].n.x + rhat.y*dp[i].n.y + rhat.z*dp[i].n.z);
    
    if ((gamma1 < 0) != (gamma2 < 0))
	return(0);
    
    gamma = gamma1/gamma2;
    
    r.x = gamma*rhat.x;
    r.y = gamma*rhat.y;
    r.z = gamma*rhat.z;
    
    xd = (r.x*dp[i].Xp.x + r.y*dp[i].Xp.y +r.z*dp[i].Xp.z - dp[i].Xp0)/
	dp[i].Xp2;
    yd = (r.x*dp[i].Yp.x + r.y*dp[i].Yp.y +r.z*dp[i].Yp.z - dp[i].Yp0)/
	dp[i].Yp2;
    
    if (fabs(xd) <= 0.5 && fabs(yd) <= 0.5)
	return(i);
    else
	return(0);
}

/*
 * Returns pointer to a bin in th,phi containing hit info
 */
int *gethitbin(float theta, float phi)
{
    int it, ip;

    it = theta*HITBINSIZE;
    ip = phi*HITBINSIZE;
    
    if (it == 180*HITBINSIZE)
	it = 180*HITBINSIZE-1;

    if (ip == 360*HITBINSIZE)
	ip = 360*HITBINSIZE-1;

    return(hitbins + it*360*HITBINSIZE + ip);
}
