/*
 * ''corr''      Angular Correlations                      Steve Chappell
 *
 *   General Subroutine to reconstruct angular correlations for
 *   decay of a resonant particle into two breakup fragments
 *   given the position of the virtual resonant particle
 *   in the event array
 *   rp* -> Heavy Ion (hi) + Light Ion (li)
 
 * Notes
 
 *   In Polar Co-ords  (hor,vert,beam)=(x,y,z)
 *   In Fixed Axial Co-ords          (xa,ya,za)
 *   In Basel Co-ords                (xb,yb,zb)
 
 * Updates
 *   18th Dec 97: Angles converted to positive range 0 -> 360
 *                Basel.phis 0->180
 *                Bug Fix in axial->Basel transform
 */

#include <stdio.h>
#include <math.h>
#include "subs.h"

struct coords pspace, *psp = &pspace;

int corr(int vrp)
{
    
    float vcm,pcm;
    float p_li_azy,p_hi_azy,p_vrp_azy,dths,tha_li,tha_hi;

    int li,hi;

    /*fill out phase space structure for polar co-ord frame*/    
    /*(Nb.Particles hi,li in lab, vrp in c.m. frame)*/
    hi=ev[vrp].hi;
    li=ev[vrp].li; 
    
    psp->polar.p_li.x=ev[li].px;
    psp->polar.p_li.y=ev[li].py;
    psp->polar.p_li.z=ev[li].pz;

    psp->polar.p_hi.x=ev[hi].px;
    psp->polar.p_hi.y=ev[hi].py;
    psp->polar.p_hi.z=ev[hi].pz;

    /*Phase Space Analysis ...*/
    vcm=(sqrt(2.0*reac.e1*reac.mass[1]))/(reac.mass[1]+reac.mass[2]);

    psp->polar.p_vrp.x=ev[vrp].px;
    psp->polar.p_vrp.y=ev[vrp].py;
    psp->polar.p_vrp.z=ev[vrp].pz-ev[vrp].m*vcm;
    pcm=sqrt(psp->polar.p_vrp.x*psp->polar.p_vrp.x+
	     psp->polar.p_vrp.y*psp->polar.p_vrp.y+
	     psp->polar.p_vrp.z*psp->polar.p_vrp.z);

    psp->polar.vrel.x=psp->polar.p_li.x/ev[li].m-psp->polar.p_hi.x/ev[hi].m;
    psp->polar.vrel.y=psp->polar.p_li.y/ev[li].m-psp->polar.p_hi.y/ev[hi].m;
    psp->polar.vrel.z=psp->polar.p_li.z/ev[li].m-psp->polar.p_hi.z/ev[hi].m;

    psp->polar.ths=(acos(psp->polar.p_vrp.z/pcm))/RAD;

    psp->polar.phis=((psp->polar.p_vrp.y >= 0) ? 1 : -1)*
	             (acos(psp->polar.p_vrp.x/
                     (sqrt(psp->polar.p_vrp.x*psp->polar.p_vrp.x+
			   psp->polar.p_vrp.y*psp->polar.p_vrp.y))))/RAD
			       +((psp->polar.p_vrp.y >= 0) ? 0 : 360);
    psp->polar.psi=(acos(psp->polar.vrel.z/
			 (sqrt(psp->polar.vrel.x*psp->polar.vrel.x+
			       psp->polar.vrel.y*psp->polar.vrel.y+
			       psp->polar.vrel.z*psp->polar.vrel.z))))/RAD;
    psp->polar.chi=((psp->polar.vrel.y >= 0) ? 1 : -1)*
	(acos(psp->polar.vrel.x/
	      (sqrt(psp->polar.vrel.x*psp->polar.vrel.x+
		    psp->polar.vrel.y*psp->polar.vrel.y))))/RAD
			+((psp->polar.vrel.y >= 0) ? 0 : 360);

    /*
     * FIXED ``AXIAL'' Co-ordinates (xa,ya,za)=(beam,horiz,vert)
     * Transformation from Polar Co-ord frame (x,y,z) ---> (za,xa,ya)
     */
    
    psp->axial.p_li.x=psp->polar.p_li.z;
    psp->axial.p_li.y=psp->polar.p_li.x;
    psp->axial.p_li.z=psp->polar.p_li.y;
    p_li_azy=sqrt(psp->axial.p_li.y*psp->axial.p_li.y + 
		  psp->axial.p_li.z*psp->axial.p_li.z);   
    tha_li = ((psp->axial.p_li.z >= 0) ? 1 : -1)*
	(acos(psp->axial.p_li.y/p_li_azy))/RAD
	    +((psp->axial.p_li.z >= 0) ? 0 : 360);

    psp->axial.p_hi.x=psp->polar.p_hi.z;
    psp->axial.p_hi.y=psp->polar.p_hi.x;
    psp->axial.p_hi.z=psp->polar.p_hi.y;
    p_hi_azy=sqrt(psp->axial.p_hi.y*psp->axial.p_hi.y + 
		  psp->axial.p_hi.z*psp->axial.p_hi.z);    
    tha_hi = ((psp->axial.p_hi.z >= 0) ? 1 : -1)*
	(acos(psp->axial.p_hi.y/p_hi_azy))/RAD
	    +((psp->axial.p_hi.z >= 0) ? 0 : 360);
    
    psp->axial.p_vrp.x=psp->polar.p_vrp.z;
    psp->axial.p_vrp.y=psp->polar.p_vrp.x;
    psp->axial.p_vrp.z=psp->polar.p_vrp.y;

    p_vrp_azy=sqrt(psp->axial.p_vrp.y*psp->axial.p_vrp.y + 
		  psp->axial.p_vrp.z*psp->axial.p_vrp.z);
    dths = ((psp->axial.p_vrp.z >= 0) ? 1 : -1)*
	(acos(psp->axial.p_vrp.y/p_vrp_azy))/RAD
	    +((psp->axial.p_vrp.z >= 0) ? 0 : 360); 
    
    psp->axial.vrel.x=psp->axial.p_li.x/ev[li].m-psp->axial.p_hi.x/ev[hi].m;
    psp->axial.vrel.y=psp->axial.p_li.y/ev[li].m-psp->axial.p_hi.y/ev[hi].m;
    psp->axial.vrel.z=psp->axial.p_li.z/ev[li].m-psp->axial.p_hi.z/ev[hi].m;
    
    psp->axial.ths=(acos(psp->axial.p_vrp.z/pcm))/RAD;

    if((psp->axial.phis=(atan2(psp->axial.p_vrp.y,psp->axial.p_vrp.x))/RAD)<0)
	psp->axial.phis+=360;
    
    psp->axial.psi=(acos(psp->axial.vrel.z/
			 (sqrt(psp->axial.vrel.x*psp->axial.vrel.x+
			       psp->axial.vrel.y*psp->axial.vrel.y+
			       psp->axial.vrel.z*psp->axial.vrel.z))))/RAD;
    if((psp->axial.chi=(atan2(psp->axial.vrel.y,psp->axial.vrel.x))/RAD)<0)
	psp->axial.chi+=360;
    
    /*
     * FLOATING ``BASEL'' Co-ords (xb,yb,zb) => (ths_b=90,phis_b,psi_b,chi_b)
     * ebye Rotn about xa to define trajectory in the reac plane (xb,yb)
     */
    
    psp->basel.p_li.x=psp->axial.p_li.x;
    psp->basel.p_li.y=p_li_azy*(cos((tha_li-dths)*RAD));
    psp->basel.p_li.z=p_li_azy*(sin((tha_li-dths)*RAD));
    
    psp->basel.p_hi.x=psp->axial.p_hi.x;
    psp->basel.p_hi.y=p_hi_azy*(cos((tha_hi-dths)*RAD));
    psp->basel.p_hi.z=p_hi_azy*(sin((tha_hi-dths)*RAD));
    
    psp->basel.p_vrp.x=psp->axial.p_vrp.x;
    psp->basel.p_vrp.y=p_vrp_azy;
    psp->basel.p_vrp.z=0;
    
    psp->basel.vrel.x=psp->basel.p_li.x/ev[li].m-psp->basel.p_hi.x/ev[hi].m;
    psp->basel.vrel.y=psp->basel.p_li.y/ev[li].m-psp->basel.p_hi.y/ev[hi].m;
    psp->basel.vrel.z=psp->basel.p_li.z/ev[li].m-psp->basel.p_hi.z/ev[hi].m;
    
    psp->basel.ths=(acos(psp->basel.p_vrp.z/pcm))/RAD;
    psp->basel.phis=(acos(psp->basel.p_vrp.x/
			  (sqrt(psp->basel.p_vrp.x*psp->basel.p_vrp.x+
				psp->basel.p_vrp.y*psp->basel.p_vrp.y))))/RAD; 
    psp->basel.psi=(acos(psp->basel.vrel.z/
			 (sqrt(psp->basel.vrel.x*psp->basel.vrel.x+
			       psp->basel.vrel.y*psp->basel.vrel.y+
			       psp->basel.vrel.z*psp->basel.vrel.z))))/RAD;
    
    if((psp->basel.chi=(atan2(psp->basel.vrel.y,psp->basel.vrel.x))/RAD)<0)
	psp->basel.chi+=360;
    
    return(0);

}
