/*
  Non-specific FORTRAN interface routines.

  Access from C is done directly with pointers.

  A global variable - experiment_ - holds the experiment as seen by the
  FORTRAN interface. Another global variable - detector_ - holds the
  current position inside the structure where immediate FORTRAN calls
  take effect.

  FORTRAN routines infixed by `this' take effect at the position
  held in detector_, those without the `this' expect a global position
  relative to the top of the experiment.

  The current pointer may be moved with detectorparent, detectornext
  and detectorchildren. These move the pointer and return 1 on success,
  else they return 0 and leave the pointer as is.
 */

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include "structure.h"

/* Pointer to experiment used by FORTRAN calls */

struct detectors *experiment_ = NULL, *detector_ = NULL;

#define MAXDEPTH 64

static struct detectors *save[MAXDEPTH];
static int savedepth = 0;

/* Routine to change absolute reference into pointer */

struct detectors *vgetchild(char *reference, va_list ap)
{
    int len, n;
    struct detectors *child, *dp = experiment_;

    if (dp == NULL)
	return(NULL);

    while(*reference != '\0')
    {
	if (*reference == '%')
	{
	    n = *(va_arg(ap, int *));
	    for(child = dp->children; child != NULL && --n;
		child = child->next)
		;
	    if (child == NULL)
		return(NULL);
	    reference++;
	}
	else
	{
	    for(child = dp->children; child != NULL; child = child->next)
	    {
		if (!strncasecmp(reference, child->info->name,
				 len = strlen(child->info->name))
		    && (reference[len] == '.' || reference[len] == '\0'))
		    break;
	    }
	    if (child == NULL)
		return(NULL);
	    reference += len;
	}
	dp = child;
	if (*reference == '.')
	    reference++;
    }

    return(dp);
}

/* Load an experiment and make it the default for FORTRAN calls */

int loadexperiment_(char *filename)
{
    return (detector_ = experiment_ = loadexperiment(filename)) != NULL;
}

/* FORTRAN call to get the name of a specified detector element */

char *getname_(char *reference, ...)
{
    struct detectors *child;

    va_list ap;
    va_start(ap, reference);

    child = vgetchild(reference, ap);

    va_end(ap);

    return(child->info->name);
}

char *getthisname_()
{
    return((detector_ == NULL) ? "" : detector_->info->name);
}

int getnumber_(char *reference, ...)
{
    struct detectors *child;

    va_list ap;
    va_start(ap, reference);

    child = vgetchild(reference, ap);

    va_end(ap);

    return(child->number);
}

int getthisnumber_()
{
    return((detector_ == NULL) ? -1 : detector_->number);
}

void detectortop_()
{
    detector_ = experiment_;
}

int detectorparent_()
{
    struct detectors *dp;

    if (detector_ == NULL || (dp = detector_->parent) == NULL)
	return 0;

    detector_ = dp;
    return 1;
}

int detectornext_()
{
    struct detectors *dp;

    if (detector_ == NULL || (dp = detector_->next) == NULL)
	return 0;

    detector_ = dp;
    return 1;
}

int detectorchildren_()
{
    struct detectors *dp;

    if (detector_ == NULL || (dp = detector_->children) == NULL)
	return 0;

    detector_ = dp;
    return 1;
}

int detectorsave_()
{
    if (savedepth == MAXDEPTH)
	return 0;
    
    save[savedepth++] = detector_;
    return 1;
}

int detectorrestore_()
{
    if (savedepth == 0)
	return 0;

    detector_ = save[--savedepth];
    return 1;
}

void detectorforget_()
{
    savedepth = 0;
}
