/*
 * ''nrp''      N Fold Resonant Particles                  Steve Chappell
 *
 * General Subroutine to Generate resonant particles 
 * from N breakup products and add the particles to the
 * the ev array.

 * Notes

 * Includes subroutine ''ncr'' to calculate r combinations from n

 * Reconstruct Pairs from two singles
 * Reconstruct Triples from Pairs plus singles
 * Reconstruct Quads from Triples plus singles
 * etc etc
 * Result is ``heavy ion'' and ``light ion''  point back up the tree 
 * to the single particle numbers and DO NOT correspond to the 
 * sequential breakup into HI and LI.

 * In the event array.... If N=nhits  nCr combinations of each type
 * Singles  1              --> N
 * Doubles  N +1           --> N +1 +NC2-1
 * Triples  N +1 NC2-1 +1  --> N +1 +NC2-1 +1 +NC3-1
 * Quads

 * Updates
 *   24th Jun 97: Translation to C
 *   17th Jul 97: Streamline code to handle any value of N
 *   9th Jun 98:  Streamline code to generate up to the Nfold rp.
 */

#include <stdio.h>
#include <math.h>
#include "subs.h"

static void combine(int previ, int *vrp, int lastvrp, int nfold)
{
    int i;
    
    for(i=previ+1; i <= evhead.n[1]; i++)
    {
	if (lastvrp > 0){
	    rp(lastvrp, i, *vrp);
	    /*fprintf(stderr,"(%d %d)=%d\n",lastvrp,i,*vrp);*/
	}
	if(ev[*vrp].fold < nfold)
	    combine(i, vrp+1, *vrp, nfold);
	(*vrp)++;
    }
}

int nrp(int fold)
{

    int vrp[N1_MAX+1];
    int i;
    

    /*make sure we don't try and build an Nfold combination 
      for too few singles*/
    if(fold > evhead.n[1])
	fold = evhead.n[1];
    
    if(fold < 2)
	return 0;
    
    if(fold > N1_MAX){
	fprintf(stderr,"nrp: Not building rp combinations\n"
		"...Cannot fit nrp(%d) for %d particles in ev array.\n" 
		"NP_MAX=%d\n" ,fold,evhead.n[1],NP_MAX);
	return(1);
    }

/*
 * Put Number of different Nfold combinations
 * Plus Pointers to ``start'' and ``Next'' locations 
 * of Nfold particles in the Event Header
 */
    
    vrp[1]=evhead.p[1];
    
    for(i=2; i <= fold; i++)
    {
	evhead.n[i] = ncr(evhead.n[1],i);
	evhead.p[i] = evhead.p[i-1] + evhead.n[i-1];
	evhead.n[0] = evhead.n[0] + evhead.n[i];
	vrp[i] = evhead.p[i];
	if(evhead.n[0] > NP_MAX){
	    fprintf(stderr,"nrp: Not building rp combinations\n"
		    "...Cannot fit nrp(%d) for %d particles in ev array.\n" 
		    "NP_MAX=%d\n" ,fold,evhead.n[1],NP_MAX);    
	    return(1);
	}
    }
    
    /* Build the combinations of virtual fold particles*/
    combine(0, vrp+1, 0, fold);
    
    return(0);
}


/*
 *''ncr''  Calculate r combinations out of n               Steve Chappell
 */
int ncr(int n_ncr, int r_ncr)
{
    int i_ncr;
    int value=1;
    if(n_ncr < 0 || r_ncr < 0 || n_ncr < r_ncr){
        fprintf(stderr,"ncr: Error! n=%d r=%d\n",n_ncr,r_ncr);
	return(0);
    }
    for(i_ncr=1; i_ncr<=r_ncr; i_ncr++)
	value=(n_ncr-i_ncr+1)*value/i_ncr;
    /*fprintf(stderr,"ncr: n=%d r=%d = %d\n",n_ncr,r_ncr,value);*/
    return value;
}

