/*
 *  Channel Routines                                   Steve Chappell
 *
 *  Procedures to set up and calibrate each data channel
 *
 * Updates
 *   21st October 1997: Tidy up*
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include "channel.h"
#include "rdtape.h"
#include "subs.h"

int eginit(char *);           /*eurogam format map file initialiser*/

static struct channel_data {  /*parameters associated with each channel*/
    float gain;
    float offset;
    short min_threshold;
    short max_threshold;
    char hit;
    unsigned int set;
    float value;
} *channel_data = NULL;


/*
 *  Initialise channel calibrations
 */
static int cal_maxchans = 0;
int channel_init(int maxchans)
{
    FILE *fp;
    int i, min, max;
    struct channel_data *cp;
    char buffer[SUBS_BUFSIZE];
    float g, o;

    if (channel_data != NULL)
	free(channel_data);
    if ((channel_data = (struct channel_data *)
	 calloc(maxchans, sizeof(struct channel_data))) == NULL)
    {
	fprintf(stderr, "Unable to allocate memory for channel data.\n");
	return(1);
    }

    cal_maxchans = maxchans;

    if (eginit(CALIB_DIR "egmap.cal") != 0){
	if(eginit("egmap.dat") != 0){
	    fprintf(stderr, "Error reading \"%s\" and \"%s\".\n",
		    CALIB_DIR "egmap.cal", "egmap.dat");
	    return(1);
	}    
	else
	fprintf(stderr, "Error reading \"%s\", using default \"%s\".\n",
		CALIB_DIR "egmap.cal", "egmap.dat");
    }
    else
	fprintf(stderr, "Using address map \"%s\".\n",CALIB_DIR "egmap.cal");
    
    /*default settings*/
    for(i=maxchans, cp = channel_data; i--; cp++)
    {
	cp->gain = 1.0;
	cp->offset = 0.0;
	cp->min_threshold = 0;
	cp->max_threshold = 8192;
    }


    if ((fp = fopen(CALIB_DIR "go.cal", "r")) != NULL)
    {
	fprintf(stderr, "Reading file \"%s\".\n",CALIB_DIR "go.cal");
	while (mygetline(buffer, SUBS_BUFSIZE, fp) != NULL)
	{
	    if (sscanf(buffer, " %d %f %f", &i, &g, &o) != 3)
		fprintf(stderr, "Expected 'adc gain offset', got \"%s\".\n",
			buffer);
	    else
	    {
		channel_data[i-1].gain = g;
		channel_data[i-1].offset = o;
	    }
	}
    }
    else
    {
	fprintf(stderr, "No file \"%s\", using default settings.\n",
		CALIB_DIR "go.cal");
    }
    
    if ((fp = fopen(CALIB_DIR "thrsh.cal", "r")) != NULL)
    {
	fprintf(stderr, "Reading file \"%s\".\n", CALIB_DIR "thrsh.cal");
	while (mygetline(buffer, SUBS_BUFSIZE, fp) != NULL)
	{
	    if (sscanf(buffer, " %d %d %d", &i, &min, &max) != 3)
		fprintf(stderr, "Expected 'adc min max', got \"%s\".\n",
			buffer);
	    else
	    {
		channel_data[i-1].min_threshold = min;
		channel_data[i-1].max_threshold = max;
	    }
	}
    }
    else
    {
	fprintf(stderr, "No file \"%s\", using default settings.\n",
		CALIB_DIR "thrsh.cal");
    }

    return(0);
}

/*
 * channel event handler
 *      loops through all channels
 */
static unsigned int set = 0;
void channel_event()
{
    int i;
    struct channel_data *cp;

    if (set == 0)
	for(i = cal_maxchans, cp = channel_data; i--; cp++)
	    cp->set = 0xffffffff;
    if ((set += 2) > 0xfffffff0)
	set = 0;
}

/*
 *  return whether there's an in-range hit on this channel (non-zero
 *  means true)
 */
int channel_hit(int chan)
{
    struct channel_data *cp = channel_data + chan - 1;

    if ((cp->set & 0xfffffffe) != set)
    {
	int a = adcs_.adcs[chan-1];

	if (a > cp->min_threshold && a < cp->max_threshold)
	{
	    cp->hit = 1;
	    cp->set = set | 1;
	}
	else
	{
	    cp->hit = 0;
	    cp->value = 0;
	    cp->set = set;
	}
    }
    return cp->hit;
}

/*
 * return the calibrated value for a channel (amplitude style)
 */
float channel_a(int chan)
{
    struct channel_data *cp = channel_data + chan - 1;

    if (cp->set != set)
    {
	if (channel_hit(chan))
	    cp->value = (adcs_.adcs[chan-1] - cp->offset)*cp->gain;
	cp->set = set;
    }
    
    return cp->value;
}

/*
 * return the calibrated value for a channel (time style)
 */
float channel_t(int chan)
{
    return (float) adcs_.adcs[chan-1];
}
