/* 
 * ''scanga''   Event Simulation Routines               Steve Chappell

 *   ``RPS/Monte Carlo'' routines to generate pseudo data
 *   for heavy ion breakup reactions  using Charissa detection systems

 *   Uses Lab Fixed Co-ords (X,Y,Z) = (Hor,Vert,Beam)

 * Updates
 *   12th Dec97 :Linked to SunSort!
 *   17th Dec97 :Convert to Lab fixed co-ord convention. 
 *              :Angles converted to positive range 0->360.
 *              :Code added to generate isotropic event distribution.
 *   11th Jan98 :Gaussian random number generator smears detected energies
 *   7th May98  :Writes hits to the event array
 *   1st Jun98  :Basel correlation function included
 *   19th Jun98 :2nd partial wave in basel corr fn added.
 */

#include <stdio.h> 
#include <stdlib.h> 
#include <math.h>
#include <string.h>
#include <sys/time.h>
#include "../subs.h"

struct sim sim;
struct ev_struct *evsim = NULL;

static int bkup[SIM_NPMAX],hitsave[SIM_NPMAX];
static int nhits;
static float vcm,vrel[SIM_NPMAX];
static int rorder[SIM_NPMAX];
static void grorder(int);

/*
 * Initialise Scanga
 */
int scanga_init(char *filename)
{
    FILE *fp;
    int i,j;
    float th,phi,tmp;
    struct timeval tp;
    
    fprintf(stderr,"\nScanga: Initialising...\n\n");

    /*seed random number generator*/
    gettimeofday(&tp, NULL);
    srand48(tp.tv_sec + tp.tv_usec);

    /*initialise simulated event structure*/
    if(evsim_init(SIM_NPMAX)) return(-1);
    if(ev_init(NP_MAX)) return 1;       /*initialise event structure*/
    if (reac_init()) return 1;           /*reaction definition*/

    /*Read Input File*/
    if(scanga_input(filename)) return(-1);

    /*Generate detector coverage*/
    if(sim.det_sim){
	
	/*This code tests for detector hits on the fly*/
	/*The alternative is to always setup the array 
	 *gethitbin(th,phi) by running the code here*/
	
	memset(hitbins, sizeof(hitbins), 0);
	
	if(sim.thabs > 0) fprintf(stderr,"\nAbsorber shields up to "
				  "Theta = %6.2f deg\n",sim.thabs);
	
	/*Check detector coverage if requested*/
	if(sim.cov){
	    if((fp = fopen("sim_cov.out","w"))!= NULL){
		fprintf(stderr, "Writing complete th,phi coverage to file"
			" \"%s\".\n","sim_cov.out");
	    }
	    for(i=1; i<sim.ndet+1; i++){
		fprintf(stderr, "det %d: ",i);
		switch(sim.det_type[i]){
		case DTYPE_S:
		    fprintf(stderr,"(S)trip Array detector\n");
		    break;
		case DTYPE_a:
		    fprintf(stderr,"(a)ncillary detector\n");
		    break;
		default:
		    fprintf(stderr,"Scanga: Error initialising"
			    " detector coverage\n");
		    return(-1);
		}
		fprintf(fp,"\n");
		for(th=1/(2.0*HITBINSIZE); th<180; th += 1.0/HITBINSIZE){
		    if(th < sim.thabs)
			continue;
		    for(phi=1/(2.0*HITBINSIZE); 
			phi<360; phi += 1.0/HITBINSIZE)
		    {
			if(*gethitbin(th, phi) = det_hit(i,th,phi))
			    fprintf(fp, "%f %f\n", th,phi);
		    }
		}
	    }
	    fclose(fp);
	}
    }
    

    /*Kinematics Set-up*/
    fprintf(stderr,"\nSimulating Reaction of form :\n\n"
	    "m1(E1) + m2 -> m3*(Ex3) + m4*(Ex4)\n"
	    "Q1\n\n"
	    "m3*(Ex3) -> m5* + m6     :qb3\n"
	    "m4*(Ex4) -> m7* + m8     :qb4\n"
	    "m5*(Ex5) -> m9  + m10    :qb5\n"
	    "m7*(Ex7) -> m11 + m12    :qb7\n");
    
    /*determine resonant particle breakups*/
    /*zero arrays*/
    for(i=1;i<13;i++){
	sim.fstate[i] = sim.hit[i] = bkup[i] = 0;
    }
    if(evsim[5].m != 0 && evsim[6].m != 0){
	evsim[3].hi=5;
	evsim[3].li=6;
	bkup[3]=1;
    }
    if(evsim[7].m != 0 && evsim[8].m != 0){
	evsim[4].hi=7;
	evsim[4].li=8;
	bkup[4]=1;
    }
    if(bkup[3] != 0 && evsim[9].m != 0 && evsim[10].m != 0){
	evsim[5].hi=9;
	evsim[5].li=10;
	bkup[5]=1;
    }
    if(bkup[4] != 0 && evsim[11].m != 0 && evsim[12].m != 0){
	evsim[7].hi=11;
	evsim[7].li=12;
	bkup[7]=1;
    }

    /*Check that input for detection simulation makes at least some sense
     *Not Bomb proof. Relies upon user intelligence a little*/
    
    if(sim.det_sim){
	for( i=3 ; i< 13;i++){
	    if(sim.coinc[i]== 1 && evsim[i].m <= 0.000001){
		fprintf(stderr,"Canny detect massless particles!\n");
		return(-1);
	    }
	}
	if(sim.coinc[3] == 1 && (sim.coinc[5] == 1 || sim.coinc[6] == 1)){
	    fprintf(stderr,"Canny detect ejectile AND ejectile breakup!\n");
	    return(-1);
	}
	if(sim.coinc[4] == 1 && (sim.coinc[7] == 1 || sim.coinc[8] == 1)){
	    fprintf(stderr,"Canny detect recoil AND recoil breakup!\n");
	    return(-1);
	}
	if(sim.coinc[5] == 1 && (sim.coinc[9] == 1 || sim.coinc[10] == 1)){
	    fprintf(stderr,"Canny detect 5 AND particle 5 breakup!\n");
	    return(-1);
	}
	if(sim.coinc[7] == 1 && (sim.coinc[11] == 1 || sim.coinc[12]== 1)){
	    fprintf(stderr,"Canny detect 7 AND particle 7 breakup!\n");
	    return(-1);
	}

	
	/*Check if final stage breakup detection wanted*/
	if(bkup[5] && (sim.coinc[9] == 0 && sim.coinc[10] == 0)){
	    bkup[5]=0;
	    fprintf(stderr,"No simulation of part5 daughters required!\n");
	}
	if(bkup[7] && (sim.coinc[11] == 0 && sim.coinc[12] == 0)){
	    bkup[7]=0;
	    fprintf(stderr,"No simulation of part7 daughters required!\n");
	}
	/*Check if ejectile breakup detection wanted*/
	if(bkup[3] && 
	   ((bkup[5] == 0  && sim.coinc[5] == 0 && sim.coinc[6] == 0) ||
	    (bkup[5] && sim.coinc[9] == 0 && sim.coinc[10] == 0))){
	    bkup[3] = 0;	
	    fprintf(stderr,"No simulation of ejectile daughters required!\n");
	}
	/*Check if recoil breakup detection wanted*/
	if(bkup[4] && 
	   ((bkup[7] == 0  && sim.coinc[7] == 0 && sim.coinc[8] == 0) ||
	    (bkup[7] && sim.coinc[11] == 0 && sim.coinc[12] == 0))){
	    bkup[4] = 0;	
	    fprintf(stderr,"No simulation of recoil daughters required!\n");
	}
    }
    
    
    /*Recap on final state particles and deduced reaction*/

    fprintf(stderr,"\nDeduced Reaction for Simulation:\n\n");
    fprintf(stderr,"%5.1f (%6.2f MeV) + %5.1f -> "
	    "%5.1f*[%6.2f ] + %5.1f*[%6.2f ]\n"
	    "Q1 =%7.2f \n\n",
	    evsim[1].m,sim.e1,evsim[2].m,evsim[3].m,
	    sim.ex[3],evsim[4].m,sim.ex[4],sim.q1);
    sim.fstate[3] = sim.fstate[4] = 1;
    if(bkup[3]){
	sim.fstate[3] = 0;
	sim.fstate[5] = sim.fstate[6] = 1;	
	fprintf(stderr,"%5.1f* (%6.2f ) -> "
		"%5.1f* + %5.1f  :qb3 = %7.2f \n",
		evsim[3].m,sim.ex[3],evsim[5].m,evsim[6].m,sim.qb[3]);
    }
    if(bkup[4]){
	sim.fstate[4] = 0;
	sim.fstate[7] = sim.fstate[8] = 1;
	fprintf(stderr,"%5.1f* (%6.2f ) -> "
		    "%5.1f* + %5.1f  :qb4 = %7.2f \n",
		    evsim[4].m,sim.ex[4],evsim[7].m,evsim[8].m,sim.qb[4]);
    }
    if(bkup[5]){
	sim.fstate[5] = 0;
	sim.fstate[9] = sim.fstate[10] = 1;
	fprintf(stderr,"%5.1f* (%6.2f ) -> "
		    "%5.1f  + %5.1f  :qb5 = %7.2f \n",
		    evsim[5].m,sim.ex[5],evsim[9].m,evsim[10].m,sim.qb[5]);
    }
    if(bkup[7]){
	sim.fstate[7] = 0;
	sim.fstate[11] = sim.fstate[12] = 1;
	fprintf(stderr,"%5.1f* (%6.2f ) -> "
		"%5.1f  + %5.1f  :qb7 = %7.2f \n",
		evsim[7].m,sim.ex[7],evsim[11].m,
		evsim[12].m,sim.qb[7]);
    }

    fprintf(stderr,"\n=>Final state (3-12): ");
    nhits = 0;
    for(i=3;i<13;i++){
	
	if(sim.fstate[i] < sim.coinc[i]){
	    fprintf(stderr,"\nError! fstate < coinc requirement\n");
	    return(-1);
	}
	fprintf(stderr," %d",sim.fstate[i]);
	if(sim.coinc[i])
	    fprintf(stderr,".");
	
	/*Nb: Want to ``see'' all particles when 
	  no detector simulation required*/
	if(sim.fstate[i]){
	    sim.hit[i] = 1;
	    nhits++;
	    hitsave[nhits] = i; /*remember simulated particle index*/
	    evsim[i].det = 0;
	}
    }
    fprintf(stderr,"\n[. means detection required ]\n");
    
    
    /*Mcarlo Set-up*/
    if(sim.effcalc == 0){
	sim.effmin=0.0;
	sim.effmax=0.0;
	sim.effstep=1.0;
	fprintf(stderr,"\nScanga: Generating Pseudo Data ...\n\n");
    }
    else{
	fprintf(stderr,"\nScanga: Generating Pseudo Data ...\n"
		"and efficiencies...\n\n");
	;
    }
    
    /*End of Set-up*/
    return(0);
}

/*
 * Initialise kinematics and coordinates
 */
int scanga_kin_init(void)
{
    int i;
    float mtot,ecmi,ecmf,erel,vreltot;
  
    /* m2(m1,m3*)m4* */
    mtot=evsim[1].m+evsim[2].m;
    ecmi=evsim[2].m*sim.e1/mtot;
    vcm=(sqrt(2*evsim[1].m*sim.e1))/mtot;
    /* m12*->m3*+m4* */
    ecmf=ecmi-sim.ex[3]-sim.ex[4]+sim.q1;
    if(ecmf <= 0){
	fprintf(stderr,"-ve ecmf! ...Skipping Calc.\n");
	/*continue*/ return(-1);
    }

    /*first stage of reaction*/
    bkup[2] = 1;
    evsim[2].hi = 3;
    evsim[2].li = 4;
    evsim[2].vx = evsim[2].vy = 0;
    evsim[2].vz = vcm;
    vreltot=sqrt(2*ecmf*mtot/(evsim[3].m*evsim[4].m));
    vrel[3]=vreltot*evsim[4].m/mtot;
    vrel[4]=vreltot*evsim[3].m/mtot;
    
    /*subsequent stages*/
    /*vrp->hi+li*/
    for(i=3;i<8;i++){
	if(bkup[i]){
	    erel=sim.ex[i]+sim.qb[i];
	    if(erel <= 0){
		fprintf(stderr,"Scanga:-ve erel! ...Skipping Calc.\n");
		/*continue*/ return(-1);
	    }
	    vreltot=sqrt(2*erel*evsim[i].m/
			 (evsim[evsim[i].hi].m*evsim[evsim[i].li].m));
	    vrel[evsim[i].hi]=vreltot*evsim[evsim[i].li].m/evsim[i].m;
	    vrel[evsim[i].li]=vreltot*evsim[evsim[i].hi].m/evsim[i].m;
	}
    }
    
    return(0);
}


/*
 * Scanga Event by event Simulation
 */   
int scanga_event(void)
{
    int i,j,k,n,hi,li;
    float vrelx[SIM_NPMAX],vrely[SIM_NPMAX],vrelz[SIM_NPMAX];
    float  ths,phs,arg,f,phis,chi[2],psi[2],d,dp,dths, x ,y;
    float  pfac, mrec, pcm, vcm, pcm_z, tmp, tmp_norm, norm_d, norm_dp;
   
    ev_zeroheader();              /*Reset event header*/
    
    for(i=2;i<8;i++){
	if(bkup[i]){
	    hi=evsim[i].hi;
	    li=evsim[i].li;
	    
	    switch(sim.bkup_coords){
		
	    case 'b': /* 
		       *   Breakup using Basel correlation function
		       *   Chappell+Rae PRC 53(1996)2879
		       *   assumes m1+m2 -> m3* + m4*,  m3* -> m5+m6 , 
		       *   [m4*->m7+m8] ...etc
		       *    Nb preserving lab coords (X,Y,Z) = (Hor,Vert,Beam)
		       */
		if(i==2){
		    do{
			do{
			    phis=180.0*drand48();
			} while(drand48() > sin(phis*RAD));
			for(k=0;k<2;k++){
			    do{
				psi[k]=180.0*drand48();
			    } while(drand48() > sin(psi[k]*RAD));
			    chi[k]=360.0*drand48();
			}
			arg=0.0;
			for(k=0;k<sim.b_nl;k++){
			    
			    /*d matrix for m=0 p147 Brink and Satchler*/
			    d=dp=1.0;
			    norm_dp=norm_d=1.0;
			    tmp = cos(0.5*psi[0]*RAD)*sin(0.5*psi[0]*RAD);
			    tmp_norm = cos(0.5*90.0*RAD)*sin(0.5*90.0*RAD);
			    for(n=0;n<sim.b_k[k];n++){
				d *= tmp;
				norm_d *= tmp_norm;
			    }
			    d /= norm_d;

			    tmp = cos(0.5*psi[1]*RAD)*sin(0.5*psi[1]*RAD);
			    for(n=0;n<sim.b_kp[k];n++){
				dp *= tmp;
				norm_dp *= tmp_norm;
			    }
			    dp /= norm_dp;

			    arg += sim.b_al[k]*(sin((sim.b_l[k]+0.5)*phis*RAD +
						    sim.b_k[k]*chi[0]*RAD + 
						    sim.b_kp[k]*chi[1]*RAD + 
						    0.25*M_PI))*d*dp;
			}
			f=(arg*arg)/sin(phis*RAD);
		    } while (drand48() > f);
		    evsim[hi].thcm=phis;
		    evsim[li].thcm=180.0-phis;
		    dths=360.0*drand48(); /*smear over cylindrical geometry*/
		    vrelx[hi]=+vrel[hi]*sin(phis*RAD)*cos(dths*RAD);
		    vrely[hi]=-vrel[hi]*sin(phis*RAD)*sin(dths*RAD);
		    vrelz[hi]=+vrel[hi]*cos(phis*RAD);
		    break;
		}
		else if(i==3 || i==4){
		    j=i-3;
		    hi=evsim[i].hi;
		    li=evsim[i].li;
		    x = -vrel[hi]*sin(chi[j]*RAD)*sin(psi[j]*RAD);
		    y = -vrel[hi]*cos(psi[j]*RAD);
		    vrelz[hi]= -vrel[hi]*cos(chi[j]*RAD)*sin(psi[j]*RAD);
		    vrelx[hi] = x*cos(dths*RAD) + y*sin(dths*RAD);
		    vrely[hi] = -x*sin(dths*RAD) + y*cos(dths*RAD);
		    break;
		}
		
	    default: /*
		      *    Randomize Over Phase Space using 
		      *    Isotropic breakups only
		      */
		
		if(i==2){ 
		    ths=180.0*drand48();
		    evsim[hi].thcm=ths;
		    evsim[li].thcm=180-ths;
		}
		else
		    do{
			ths=180.0*drand48();
		    }while (drand48() > sin(ths*RAD));
		
		phs=360.0*drand48();
		vrelx[hi]= +vrel[hi]*sin(ths*RAD)*cos(phs*RAD);
		vrely[hi]= +vrel[hi]*sin(ths*RAD)*sin(phs*RAD);
		vrelz[hi]= +vrel[hi]*cos(ths*RAD);
	    }

	    vrelx[li]=-vrelx[hi]*evsim[hi].m/evsim[li].m;
	    vrely[li]=-vrely[hi]*evsim[hi].m/evsim[li].m;
	    vrelz[li]=-vrelz[hi]*evsim[hi].m/evsim[li].m;
	    evsim[hi].vx=vrelx[hi] + evsim[i].vx;
	    evsim[hi].vy=vrely[hi] + evsim[i].vy;
	    evsim[hi].vz=vrelz[hi] + evsim[i].vz;
	    evsim[hi].v=sqrt(evsim[hi].vx*evsim[hi].vx+
			     evsim[hi].vy*evsim[hi].vy+
			     evsim[hi].vz*evsim[hi].vz);
	    evsim[li].vx=vrelx[li] + evsim[i].vx;
	    evsim[li].vy=vrely[li] + evsim[i].vy;
	    evsim[li].vz=vrelz[li] + evsim[i].vz;
	    evsim[li].v=sqrt(evsim[li].vx*evsim[li].vx+
			     evsim[li].vy*evsim[li].vy+
			     evsim[li].vz*evsim[li].vz);
	    evsim[hi].px=evsim[hi].m*evsim[hi].vx;
	    evsim[hi].py=evsim[hi].m*evsim[hi].vy;
	    evsim[hi].pz=evsim[hi].m*evsim[hi].vz;
	    evsim[hi].p=evsim[hi].m*evsim[hi].v;
	    evsim[li].px=evsim[li].m*evsim[li].vx;
	    evsim[li].py=evsim[li].m*evsim[li].vy;
	    evsim[li].pz=evsim[li].m*evsim[li].vz;
	    evsim[li].p=evsim[li].m*evsim[li].v;
	    evsim[hi].e=0.5*evsim[hi].m*evsim[hi].v*evsim[hi].v;
	    evsim[li].e=0.5*evsim[li].m*evsim[li].v*evsim[li].v;
	    evsim[hi].th=(acos(evsim[hi].vz/evsim[hi].v))/RAD;
	    evsim[li].th=(acos(evsim[li].vz/evsim[li].v))/RAD;
	    if((evsim[hi].phi=(atan2(evsim[hi].vy,evsim[hi].vx))/RAD)<0)
		evsim[hi].phi += 360;
	    if((evsim[li].phi=(atan2(evsim[li].vy,evsim[li].vx))/RAD)<0)
		evsim[li].phi += 360;
	}
    }
    
    /*==== Simulate Particle Detection============*/
    
    if(sim.det_sim){	    
	/*Check which final state particles hit a detector
	  and are at trajectories greater than extent of absorber. 
	  Compare detected hit pattern to coincidence requirement*/
	
	nhits=0;
	/*no hit pattern included here yet*/
	for(i=3; i<13; i++){
	    sim.hit[i] = 0;
	    if(sim.fstate[i] &&
	       evsim[i].e >= sim.ethrsh[i] &&
	       evsim[i].th > sim.thabs &&
	       (evsim[i].det=
		ndet_hit(sim.ndet,evsim[i].th,evsim[i].phi))){	
		/*Minimum coincidence requirement*/
		sim.hit[i] = 1;
		nhits++;
		hitsave[nhits] = i; /*remember simulated particle index*/
		
	    }
	    if(sim.hit[i] < sim.coinc[i]) return(0); /*detection failed*/
	}
	if(nhits == 0) return(0); /*detection failed*/
	
    }      /*End detection simulation*/    
    
    
    /*=== Event Good if it made it this far! ===*/
    
    /*load event header*/
    evhead.n[0]=evhead.n[1]=evhead.pmult[0]=nhits;
    
    /*load event array in random sequence*/
    grorder(nhits);
    for(j=1; j<nhits+1; j++){
	
	k=rorder[j];
	i=hitsave[j];

	ev[k].e = evsim[i].e;
	/*smear final state particle energy and set dispersion*/
	if(sim.eres > 0)
	    ev[k].e = reac.edisp*grand(ev[k].e,sim.eres);
	else
	    ev[k].e = reac.edisp*ev[k].e;
	evhead.etot += ev[k].e;
	ev[k].m = reac.mass[0];
	ev[k].det = evsim[i].det;
	ev[k].seg = 0;
	ev[k].fold = 1;
	ev[k].p = pfac = sqrt(2*ev[k].m*ev[k].e);
	pfac /= evsim[i].v;
	ev[k].px = evsim[i].vx*pfac;
	ev[k].py = evsim[i].vy*pfac;
	ev[k].pz = evsim[i].vz*pfac;
	ev[k].th = evsim[i].th;
	ev[k].phi = evsim[i].phi;
	vcm=(sqrt(2.0*reac.e1*reac.mass[1]))/
	    (reac.mass[1]+reac.mass[2]);
	pcm_z=ev[k].pz-ev[k].m*vcm;
	pcm=sqrt(ev[k].px*ev[k].px+ev[k].py*ev[k].py+pcm_z*pcm_z);
	ev[k].thcm=(acos(pcm_z/pcm))/RAD;
	ev[k].q = 0.0;
	mrec=reac.mass[1]+reac.mass[2]-ev[k].m;
	if(mrec > 0)
	    ev[k].q2=((ev[k].m+mrec)*ev[k].e - 
		      (mrec-reac.mass[1])*reac.e1   - 
		      2.0*(sqrt(reac.mass[1]*ev[k].m*reac.e1*ev[k].e))
		      *cos(ev[k].th*RAD))/mrec;
	else
	    ev[k].q2=0.0;
	ev[k].hi = 0;
	ev[k].li = 0;
    }
    
    return(nhits);
    
}


/*
 * Finish Scanga
 */
int scanga_finish(void)
{
    fprintf(stderr,"\nScanga: Finished\n\n");
    return(0);
}


/*
 * Initialise detector coordinates
 */
int evsim_init(int n)
{
    if (evsim != NULL)
	free(evsim);
    
    if ((evsim = (struct ev_struct *) 
	 calloc(n+1, sizeof(struct ev_struct))) == NULL)
    {
	fprintf(stderr,
		"Unable to allocate memory for simulated event structure.\n");
	return(1);
    }
    
    return(0);
}

/*
 * write simulated event to file or event structure
 */
int wrt_evsim(int n, FILE *fpw)
{
    int i,ntest=0;

    if(n < 1){
	fprintf(stderr,"wrt_evsim: No hit in detection system!\n");
	return(0);
    }
    
    fprintf(fpw,"%d\n",n);
    
    for(i=3; i<13; i++)
	if(sim.hit[i] > 0){
	    ntest++;
	    fprintf(fpw,"%8.3f %8.3f %8.3f\n",
		    evsim[i].e,evsim[i].th,evsim[i].phi);
	    if(ntest>n){
		fprintf(stderr,"wrt_evsim: Error! Too many hits written\n");
		fprintf(stderr,"nhits = %d ntest = %d\n",n,ntest);
		return(0);
	    }
	}
    return(0);
}

/*
 * read simulated event from file
 */
int read_evsim(void)
{
    return(0);
}


/* Random number generator with gaussian distribution
 * f(x) = 1/(s *sqrt(2*pi)) * exp(- (x-c)**2/(2*s**2))
 * with centroid c and standard deviation s
 * an inverse Chebyshev polynomial expansion is used
 * to evaulate x  = f**-1 (u) where u is the random deviate
 * see chapter 26 of Abramowitz and Stegun eqn 26.2.23
 */

#define c0 2.515517
#define c1 0.802853
#define c2 0.010328
#define d1 1.432788
#define d2 0.189269
#define d3 0.001308
#define FWHMFACT .42466090014400952136 /* (1/(2*sqrt(2*log(2))) */

float grand(double c, double fwhm)
{
    double t,tsq,n;
     
    /* assume c=0.0, s=1.0 and get random probability , 0 < t <= 0.5*/
    
    t=drand48();
    if (t >= 0.5)
    {
	n = 1;
	t -= 0.5;
    }
    else
	n = 0;

    /*compute inverse by eqn 26.2.23*/
    if (t < 1.0e-30)
    {
	t = 11.7539400024;
	tsq = 138.1550558;
    }
    else
    {
	tsq = -log(t*t);
	t=sqrt(tsq);
    }

    t -= (c0+c1*t+c2*tsq)/(1.0+d1*t+(d2+d3*t)*tsq);
    
    /*randomize x in positive and negative direction*/
    if (n)
	t = -t;
    
    /*correct for centroid and standard deviation*/
    return t*fwhm*FWHMFACT + c;
}

/*
 * random shuffle of n numbers
 */
void grorder(int m)
{
    int i,j,k;
    
    for(j=1;j<m+1;j++)  /*initialise*/
	rorder[j]=j;
    
    for(j=m; j>1; j--){
	k=j*drand48()+1;
	i=rorder[k];
	rorder[k]=rorder[j];
	rorder[j]=i;
    }
}
