/*
 *  decoding routine to unpack event by event data and pass this on to user routines
 *  *** NSCL style data ***
 *
 *  started on new version 24/7/95
 *  modified 9/2/96
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include <sys/types.h>
#include <sys/time.h>

#include "sort_def.h"
#include "rdtape.h"

#define SWAPSHORT(s) ( ((s&0x00ff) << 8) + ((s&0xff00) >> 8) )

#define NSCL_DATA          1
#define NSCL_SCALER        2
#define NSCL_SCALER_PEEK   3
#define NSCL_START         11
#define NSCL_STOP          12
#define NSCL_PAUSE         13
#define NSCL_RESUME        14
#define NSCL_LINKLOST      15
#define NSCL_FIRSTUSR      32
#define NSCL_LASTUSR       64

#define NSCL_CONTROLBUF_LEN  129
#define NSCL_BUFHEADER_LEN   32
#define NSCL_BUFHEADER_WORDS 16
#define NSCL_RECORD_COUNT    8192
#define NSCL_RECORD_HWM      8158    /* NSCL_RECORD_COUNT - NSCL_BUFHEADER_LEN - 2 */
#define NSCL_NOS_SCALERS     64

static void
SWAPLONG(
#if NeedFunctionPrototype
	  short *ptr, int nos)
#else
      ptr, nos)
      short *ptr;
      int nos;
#endif
{
      short s1, s2;
      while (nos >0) {
	    s1 = SWAPSHORT(*ptr);
	    s2 = SWAPSHORT(*(ptr+1));
	    *ptr++ = s2;
	    *ptr++ = s1;
	    nos--;
      }
      return;
}

static short out_events, out_words;

int
rdtape_nscl
#if NeedFunctionPrototype
(int filter)
#else
(filter)
int filter;
#endif
{
    register short *ebye;                     /* pointer to input buffer */
    
    register short *eptr, *header_ptr, *adclist;
    register short event_len, buf_type;
    register short nos_words, nos_events;
    
    register int i, adc_bytes;
    
    time_t start_time, time_now;
    int scalers[NSCL_NOS_SCALERS], *sptr;
    int bytes, record = 0;
    long in_counter = 0;
    
/* initialize event sorting */
      (void) memset((char *) scalers, 0, NSCL_NOS_SCALERS*sizeof(int));
      (void) memset((char *) adcs_.bitpattern, 0, sizeof(adcs_.bitpattern));
      adc_bytes = adcnum_.nos_adcs * sizeof(short);
      
/* call user's initialization function */      
      if (init())
      {
	  fprintf(stderr, initfailmess);
	  return filter;
      }
      
/* set start time */
      start_time = time(NULL);
      
/* sort requested number of records or until end of file etc */
      for (record=1; ; record++) {
	  if ( (eptr = ebye = read_block(&bytes)) == NULL) 
	      break;
	  
	  /* read buffer header */
	  nos_words = SWAPSHORT( *eptr);
	  buf_type = SWAPSHORT( *(eptr+1));
	  nos_events = SWAPSHORT( *(eptr+6));
	  
	  /* check that data structure is reasonably self consistent */
	  if ( nos_words*sizeof(short) > bytes ||
	       nos_words < NSCL_BUFHEADER_WORDS) {
	      fprintf(stderr,"faulty record structure, header gives %d words, "
		      "read returned %d words\n",
		      nos_words, (bytes/2));
	      fprintf(stderr,"... stopping sorting at record %d\n",record);
	      break;
	  }
	  
	  /* make buf_type an absolute value */
	  if (buf_type < 0)
	      buf_type = -buf_type;
	  
	  /* test to see whether we have a data buffer if not continue or
	     break from loop */
	  if (buf_type == NSCL_DATA) {
	      adcs_.bitpattern[7] = SWAPSHORT( *(eptr+7)); /* nos_LAMS */
	      adcs_.bitpattern[6] = SWAPSHORT( *(eptr+9)); /* nos_bitregs */
	  }
	  else if (buf_type == NSCL_SCALER || buf_type == NSCL_SCALER_PEEK) {
	      if (nos_events < NSCL_NOS_SCALERS) {
		  eptr = &ebye[26];
		  SWAPLONG(eptr, nos_events);
		  /*LINTED*/
		  sptr = (int *) eptr;
		  /* update the scalers */
		  for (i=0; i<nos_events; i++) {
		      scalers[i] += *sptr++;
		  }
	      }
	      else {
		  fprintf(stderr,"** Number of scalers defined in sort "
			  "decoder routine NOT large enough **\n");
		  fprintf(stderr,"scaler buffer requires atleast %d defined "
			  "... stopping sorting\n",nos_events);
		  break;
	      }
	      continue;
	  }
	  else if (buf_type == NSCL_START) {
	      short runnos = SWAPSHORT( *(eptr+3));
	      fprintf(stderr,"NSCL start buffer for run nos %d\n",runnos);
	      continue;
	  }
	  else if (buf_type == NSCL_STOP) {
	      short runnos = SWAPSHORT( *(eptr+3));
	      fprintf(stderr,"NSCL stop buffer for run %d\n",runnos);
	      break;
	  }
	  else {
	      if (buf_type <= 0 || buf_type > NSCL_LASTUSR) {
		  fprintf(stderr,"NSCL buffer header gave unrecognised type "
			  "%d\n",buf_type);
		  buf_type = NSCL_STOP;
		  break;
	      }
	      continue;
	  }
	  
	  /* skip buf header data */
	  eptr = ebye + NSCL_BUFHEADER_WORDS;
	  
	  /* start decoding event block */
	  while ( nos_events > 0) {
	      adcs_.triggernos = event_len = SWAPSHORT(*eptr);
	      header_ptr = eptr++;
	      
	      /* check that we are no going over the end of the buffer */
	      if ( (nos_words -= event_len) < NSCL_BUFHEADER_WORDS) {
		  fprintf(stderr,"New event would take exceed buffer word "
			  "count\n");
		  fprintf(stderr,"... skipping rest of record %d\n",record);
		  break;
	      }
	      
	      /* reset adcs_ common block */
	      (void) memset((char *) adcs_.adcs, -1, adc_bytes);
	      adclist = adclist_.adclist;

	      /* pack adc array */
	      if (event_len <= adcnum_.nos_adcs) {
		  for (i=1; i<event_len; i++) {
		      adcs_.adcs[i-1] = (short) SWAPSHORT(*eptr);
		      *adclist++ = (short) i;
		      eptr++;
		  }
	      }
	      else {
		  fprintf(stderr,"Event length %d greater than maximum "
			  "allowed length %d\n",
			  event_len, adcnum_.nos_adcs);
		  eptr = header_ptr + event_len;
	      }
	      
	      /* check that we have unpacked event correctly */
	      if (header_ptr+event_len != eptr) {
		  fprintf(stderr,"Event length is inconsistent with number "
			  "of adcs read from event stack\n");
		  fprintf(stderr,"... skipping rest of record %d\n",record);
		  break;
	      }
	      
	      /* call user provided event sorting routine */
	      adcs_.event = ++in_counter;
	      adcs_.record = record;
	      adcs_.wrtevt = 0;
	      *adclist = -1;
	      
	      if (sortin())
	      {
		  fprintf(stderr, sortinfailmess);
		  goto ABORT_SORT;
	      }
	      
	      /* write out event if wrtevt set to true */
	      if (filter && adcs_.wrtevt) {
		  if ( write_filt(header_ptr, event_len) )
		      filter = 0;
		  /* increment counters ... must be after write_filt call */
		  out_words += event_len;
		  out_events++;
	      }
	      
	      nos_events--;
	  }   /* loop on the ebye buffer until no more events */
      }         /* loop over requested number of records or until eof */
      
ABORT_SORT:
      
/* lets note the time at end */
      time_now = time(NULL);
      
/* call user provided clearup routine  */
      if (finish())
	  fprintf(stderr, finishfailmess);
      
/* output statistics */
      fprintf(stderr,"\n*** sort statistics ***\n");
      for (i=0; i < NSCL_NOS_SCALERS; i++) {
	  if (scalers[i] != 0) 
	      fprintf(stderr,"SCALER number %d = \t %d\n",i,scalers[i]);
      }
      fprintf(stderr,"\nread %ld events in %d seconds\n"
	      "Average sort rate = %g events per second\n\n",
	      in_counter, (int) (time_now-start_time),
	      (double) in_counter / (time_now-start_time));
      
      fprintf(stderr,"\nfinished sorting data after reading %d records\n",
	      record-1);

      return filter;
}


/*
 *  functions used in the filtering of data
 *  write any data to end of filtered data block
 */
static void
filt_tail(
#if NeedFunctionPrototype
	  short *base, short *fptr)
#else
      base, fptr)
      short *base;
      short *fptr;
#endif
{
      *fptr = 0;
      base[0] = out_words + NSCL_BUFHEADER_WORDS;
      base[0] = SWAPSHORT(base[0]);
      base[1] = 0x0100;               /* SWAPSHORT(NSCL_DATA) */
      base[6] = SWAPSHORT(out_events);
      return;
}
/*
 *  write any data to the beginning of filtered data block
 */
static short *
filt_head(
#if NeedFunctionPrototype
	  short *base)
#else
      base)
      short *base;
#endif
{
      out_events = out_words = 0;
      return(base+NSCL_BUFHEADER_WORDS);
}

/* NSF filter data specification */
static filtdat_t nscl_filtdat = {
      NSCL_RECORD_COUNT,
      NSCL_RECORD_HWM/2,
      filt_head,
      filt_tail,
      NULL,
      "NSCL"
};

/* routine to get pointer to nsf_filtdat */
filtdat_t *
get_nscl_filtdat(
#if NeedFunctionPrototype
		void
#endif
		)
{
      return(&nscl_filtdat);
}
