/*
 * decoding routine to unpack event by event data and pass this on to user
 * routines
 * 
 * *** NSCL block style ***
 * *** Event style as for JTM experiment at ANL Feb 1998 ***
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include <sys/types.h>
#include <sys/time.h>

#include "sort_def.h"
#include "rdtape.h"

#define NSCL_DATA          1
#define NSCL_SCALER        2
#define NSCL_SCALER_PEEK   3
#define NSCL_START         11
#define NSCL_STOP          12
#define NSCL_PAUSE         13
#define NSCL_RESUME        14
#define NSCL_LINKLOST      15
#define NSCL_FIRSTUSR      32
#define NSCL_LASTUSR       64

#define NSCL_CONTROLBUF_LEN  129
#define NSCL_BUFHEADER_LEN   32
#define NSCL_BUFHEADER_WORDS 16
#define NSCL_RECORD_COUNT    8192
#define NSCL_RECORD_HWM      8158    /* NSCL_RECORD_COUNT - NSCL_BUFHEADER_LEN - 2 */
#define NSCL_NOS_SCALERS     64

#define ID_ID_MASK      0xc000
#define ID_LENGTH_MASK  0x3e00
#define ID_BASE_MASK    0x01ff
#define ID_LENGTH_SHIFT 9
#define ID_SIMPLE       0x0000
#define ID_INDEXED      0x4000

static short out_events, out_words;

static void event_dump(short *header_ptr, short *end_ptr, short *eptr)
{
    int i;
    unsigned short *ep;
    
    fprintf(stderr, "Event was: \n");
    for(i = 0, ep = (unsigned short *) header_ptr;
	ep < (unsigned short *) end_ptr; ep++)
    {
	fprintf(stderr, "%04x", *ep);
	i += 4;
	if (i > 78)
	{
	    putc('\n', stderr);
	    i = 0;
	}
	else
	{
	    if (ep == (unsigned short *) eptr)
		putc('<', stderr);
	    else
		if (ep+1 == (unsigned short *) eptr)
		    putc('>', stderr);
		else
		    putc(' ', stderr);
	    i++;
	}
    }
    putc('\n', stderr);
}

int
rdtape_nscljtm
#if NeedFunctionPrototype
(int filter)
#else
(filter)
int filter;
#endif
{
    register short *eptr, *header_ptr, *end_ptr, *adclist;
    register short event_len, buf_type;
    register short nos_words, nos_events;
    
    register int i, adc_bytes;
    unsigned int *p;
    
    time_t start_time, time_now;
    int scalers[NSCL_NOS_SCALERS], *sptr;
    int bytes, record = 0, base, group_len;
    long in_counter = 0;
    
    /* initialize event sorting */
    (void) memset((char *) scalers, 0, NSCL_NOS_SCALERS*sizeof(int));
    (void) memset((char *) adcs_.bitpattern, 0, sizeof(adcs_.bitpattern));
    adc_bytes = adcnum_.nos_adcs * sizeof(short);
    
    /* call user's initialization function */      
    if (init())
    {
	fprintf(stderr, initfailmess);
	return filter;
    }
      
    /* set start time */
    start_time = time(NULL);
    
    /* sort requested number of records or until end of file etc */
    for (record=1; ; record++)
    {
	if ( (eptr = read_block(&bytes)) == NULL) 
	    break;

	/* Switch whole buffer from little endian shorts to big endian */
	
	/*LINTED*/
	for(i = bytes>>2, p = (unsigned int *) eptr; i--; p++)
	    *p = ((*p & 0xff000000)>>8) | ((*p & 0x00ff0000) << 8) |
		((*p & 0x0000ff00) >> 8) | ((*p & 0x000000ff) <<8);
	
	/* read buffer header */
	nos_words = eptr[0];
	buf_type = eptr[1];
	nos_events = eptr[6];
	
	/* check that data structure is reasonably self consistent */
	if ( nos_words*sizeof(short) > bytes ||
	    nos_words < NSCL_BUFHEADER_WORDS)
	{
	    fprintf(stderr,"faulty record structure, header gives %d words, "
		    "read returned %d words\n", nos_words, (bytes/2));
	    fprintf(stderr,"... skipping record %d\n",record);
	    continue;
	}
		  
	/* make buf_type an absolute value */
	if (buf_type < 0)
	    buf_type = -buf_type;

	/* test to see whether we have a data buffer */
	switch(buf_type)
	{
	case NSCL_DATA:
	    adcs_.bitpattern[7] = eptr[7]; /* nos_LAMS */
	    adcs_.bitpattern[6] = eptr[9]; /* nos_bitregs */
	    break;
	case NSCL_SCALER:
	case NSCL_SCALER_PEEK:
	    if (nos_events < NSCL_NOS_SCALERS) {
		/* update the scalers */
		/*LINTED*/
		for (sptr = scalers, p = (unsigned int *) (eptr + 26);
		     nos_events--; p++) {
		    *sptr++ += (*p >> 16) | (*p << 16);
		}
	    }
	    else {
		fprintf(stderr,"** Number of scalers defined in sort decoder "
			"routine NOT large enough **\nscaler buffer requires "
			"at least %d defined ... stopping sorting\n",
			nos_events);
		goto ABORT_SORT;
	    }
	    goto EVENT_LOOP_CONTINUE;
	case NSCL_START:
	    fprintf(stderr, "NSCL start buffer for run number %d\n", eptr[3]);
	    goto EVENT_LOOP_CONTINUE;
	case NSCL_STOP:
	    fprintf(stderr, "NSCL stop buffer for run number %d\n", eptr[3]);
	    goto EVENT_LOOP_CONTINUE;
	default:
	    fprintf(stderr, "%s buffer type %d ... skipping.\n",
		    (buf_type <= 0 || buf_type > NSCL_LASTUSR) ? "Illegal" :
		    "Unknown", buf_type);
	    goto EVENT_LOOP_CONTINUE;
	}
	
	/* skip buf header data */
	eptr += NSCL_BUFHEADER_WORDS;
	
	/* start decoding event block */
	while (nos_events--)
	{
	    event_len = *eptr;
	    header_ptr = eptr++;
	    
	    /* check that we are not going over the end of the buffer */
	    if ((nos_words -= event_len) < NSCL_BUFHEADER_WORDS)
	    {
		fprintf(stderr, "New event would take exceed buffer word "
			"count\n... skipping rest of record %d\n",record);
		break;
	    }
	    
	    /* reset adcs_ common block */
	    (void) memset((char *) adcs_.adcs, -1, adc_bytes);
	    adclist = adclist_.adclist;
	    
	    /* pack adc array */
	    
	    adcs_.triggernos = *eptr++;
	    for(end_ptr = header_ptr + event_len; eptr < end_ptr; )
	    {
		group_len = ((*eptr & ID_LENGTH_MASK) >> ID_LENGTH_SHIFT)+1;
		base = (*eptr & ID_BASE_MASK);
		if (group_len + 1 > end_ptr - eptr)
		{
		    fprintf(stderr, "Group length inconsistent with number "
			    "of words in event in record %d.", record);
		    event_dump(header_ptr, end_ptr, eptr);
		    fprintf(stderr, "Skipping rest of record.\n");
		    goto EVENT_LOOP_CONTINUE;
		}
		
		switch(*eptr++ & ID_ID_MASK)
		{
		case ID_SIMPLE:
		    memcpy(adcs_.adcs + base, eptr, group_len * sizeof(short));
		    eptr += group_len;
		    while(group_len--)
			*adclist++ = ++base;
		    break;
		case ID_INDEXED:
		    for(; group_len--; eptr++)
		    {
			i = base + ((*((unsigned short *) eptr))>>12);
			adcs_.adcs[i] = *eptr & 0x0fff;
			*adclist++ = i+1;
		    }
		    break;
		default:
		    fprintf(stderr,
			    "Unknown group header %04x in record %d.\n",
			    eptr[-1], record);
		    event_dump(header_ptr, end_ptr, eptr);
		    fprintf(stderr, "Skipping rest of record.\n");
		    goto EVENT_LOOP_CONTINUE;
		}
	    }

	    /* call user provided event sorting routine */
	    adcs_.event = ++in_counter;
	    adcs_.record = record;
	    adcs_.wrtevt = 0;
	    *adclist = -1;

	    if (sortin())
	    {
		fprintf(stderr, sortinfailmess);
		goto ABORT_SORT;
	    }
	    
	    
	    /* write out event if wrtevt set to true */
	    if (filter && adcs_.wrtevt) {
		if ( write_filt(header_ptr, event_len) )
		    filter = 0;
		/* increment counters ... must be after write_filt call */
		out_words += event_len;
		out_events++;
	    }
	    
	}   /* loop on the ebye buffer until no more events */

    EVENT_LOOP_CONTINUE:
	;
    }         /* loop over requested number of records or until eof */
    
 ABORT_SORT:

    /* lets note the time at end */
    time_now = time(NULL);
    
    /* call user provided clearup routine  */
    if (finish())
	fprintf(stderr, finishfailmess);
    
    /* output statistics */
    fprintf(stderr,"\n*** sort statistics ***\n");
    for (i=0, sptr = scalers; i < NSCL_NOS_SCALERS; i++, sptr++) {
	if (*sptr != 0) 
	    fprintf(stderr,"Scaler %2d = %d\n", i, *sptr);
    }
    fprintf(stderr,"\nread %ld events in %d seconds\n"
	    "Average sort rate = %g events per second\n\n",
	    in_counter, (int) (time_now-start_time),
	    (double) in_counter / (time_now-start_time));
    
    fprintf(stderr,"\nfinished sorting data after reading %d records\n",
	    record-1);
    
    return filter;
}


/*
 *  functions used in the filtering of data
 *  write any data to end of filtered data block
 */
static void
filt_tail
#if NeedFunctionPrototype
(short *base, short *fptr)
#else
(base, fptr)
short *base;
short *fptr;
#endif
{
    unsigned int *p;
    
    *fptr = 0;
    base[0] = out_words + NSCL_BUFHEADER_WORDS;
    base[1] = NSCL_DATA;
    base[6] = out_events;
    
    /*LINTED*/
    for(p = (unsigned int *)base; p <= (unsigned int *) fptr; p++)
	*p = ((*p & 0xff0000)>>8) | ((*p & 0x00ff0000) << 8) |
	    ((*p & 0x0000ff00) >> 8) | ((*p & 0x000000ff) <<8);
}

/*
 *  write any data to the beginning of filtered data block
 */
static short *
filt_head
#if NeedFunctionPrototype
(short *base)
#else
(base)
short *base;
#endif
{
    out_events = out_words = 0;
    return base+NSCL_BUFHEADER_WORDS;
}

/* NSF filter data specification */
static filtdat_t nscljtm_filtdat = {
    NSCL_RECORD_COUNT,
    NSCL_RECORD_HWM/2,
    filt_head,
    filt_tail,
    NULL,
    "NSCLJTM"
};

/* routine to get pointer to nsf_filtdat */
filtdat_t *
get_nscljtm_filtdat
#if NeedFunctionPrototype
(void)
#else
()
#endif
{
    return &nscljtm_filtdat;
}
