/*	datatape.c
	routines to read and write 8mm data tapes 
	Steve Mowbray Jan 92

	prev modified 16/6/95   sjah version
	prev modified 27/7/95   sjah version
	last modified 01/2/96   sjah version
	
	read and write ansi tapes in D and F formats
	raw mode tapes ... no formating information
	
***********************************************************************/

#include <stdio.h>
#include <fcntl.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/time.h>	/* time functions		*/
#include <time.h>
#include <sys/mtio.h>	/* mt controls          */

#include "sort_def.h"
#include "datatape.h"
#include "sort_threadint.h"
#include "data_tape.h"

static void fix2gblimit P((int fd));
static int dtread_vr P((int, char *));
static int dtread_fr P((int, char *));
static int dtwrite_vr P((int, char *));
static int dtwrite_fr P((int, char *));
static int newvol P((int fd, char *volume));
static void toupperstring P((char *s));

/* information set in dtopen and dtclose commands */
struct file_data {
      char	filename[80];   /* file name (if any)   */
      int	blen;		/* physical block length */
      int	rlen;		/* record length, <= block length */
      long      blocks;		/* number of blocks written/read */
      int       blocking;       /* TRUE if as multiple records in each block, or FALSE if only one record per block */
      int	hdr34;		/* TRUE if hdr3 and hdr4 records exist */
      int       hold;           /* TRUE if we are to rewind to beginning of READ file on close */
      int       eod;            /* TRUE if READ on file has passed eof mark */
      IOfunc    iofunc;         /* pointer to function used with io on data */
      char      format;         /* set to F (fixed) or D (variable) or U */
};

/* information set in dtmount and dtumount commands */
struct device_info {
    int			fd;	     /* file descriptor for device	     */
    char	        volume[8];   /* volume name                          */
    int                 drive;       /* drive number of device               */
    int		      	vol_mounted; /* true if volume mounted on device     */
    int                 just_mounted;/* true if volume has just been mounted */
    int		      	file_open;   /* true if a file is currently open     */
    int                 nonansi;     /* true if tape not in ansi format      */
    struct file_data	fdata;	     /* info about current file		     */
};


#define MAXWRITEBLK 16384			/* must be <= WRITEBIUFFSIZE */
#define WRITEBUFFSIZE 16384 
#define READBUFFSIZE 16384			/* buff size - must be as large as 	*/
						/* largest possible block on tape  	*/
static char read_buffer[READBUFFSIZE];	        /* data buffer                     	*/
static char write_buffer[WRITEBUFFSIZE];
static int read_buffer_pos;			/* current position in read_buffer	*/
static int write_buffer_pos;			/* current position in write_buffer */
static char rconv[5];				/* used for atoi conversion        	*/
static char wconv[5];				/* used for itoa conversion        	*/
static int verbose = 0;				/* whether to be verbose           	*/

struct ansi_vol1 vol1;	/* volume header structure */
struct ansi_hdr1 hdr1;	/*         file            */
struct ansi_hdr2 hdr2;  /*        header           */
struct ansi_hdr4 hdr4;	/*      structures         */

struct device_info read_dev;	/* info about read device  */
struct device_info write_dev;	/* info about write device */

/*************************************************************************************/
/* do tape operation op (see <sys/mtio.h>) count times */
int
tape_op(
#if NeedFunctionPrototype
	    int	    op,
	    int	    count,
	    int	    fd)
#else
      op, count, fd)	
      int	op;
      int	count;
      int	fd;			/* file descriptor for tape device */
#endif
{
	struct mtop mt;
	mt.mt_op    = op;
	mt.mt_count = count;
	return( ioctl(fd, MTIOCTOP, &mt) );
}
/*************************************************************************************/
/* flag READ_MODE=close read device, WRITE_MODE=close write device */
static int
close_tape_device( 
#if NeedFunctionPrototype
		      int flag, int eject)
#else
      flag, eject)
      int flag;
      int eject;
#endif
{
/* close device associated with exabyte */      
	if( flag == WRITE_MODE || flag == NEWVOL_MODE) {
	      if (eject) {
		    if ( tape_op(MTOFFL, 1, write_dev.fd) < 0)
			  perror("close_tape_device");
	      }
	      if (close( write_dev.fd ) == -1) {
		    perror("close_tape_device");
		    return(TAPE_ERROR);
	      }
	      write_dev.file_open = FALSE;
	      write_dev.vol_mounted = FALSE;
	      return(0);
	}
	else if (flag == READ_MODE) {
	      if (eject) {
		    if ( tape_op(MTOFFL, 1, read_dev.fd) < 0)
			  perror("close_tape_device");
	      }
	      if (close( read_dev.fd ) == -1) {
		    perror("close_tape_device");
		    return(TAPE_ERROR);
	      }
	      read_dev.file_open = FALSE;
	      read_dev.vol_mounted = FALSE;
	      return(0);
	}
	return(FCLOSE_UNKNOWN_MODE);
}
/*************************************************************************************/
/*
 *   open unix device driver file associated with exabyte
 *   return < 0 on error and 0 on sucess
 */
static int
open_tape_device( 
#if NeedFunctionPrototype
		     int	flag,
		     int 	drive)
#else
      flag, drive)	        /* open no-rewind tape device      */
      int	flag;		/* O_WRONLY for writing, O_RDONLY for reading  */
      int       drive;	        /* systems drive number for device to be opened */
#endif
{
	int	fd;
	char    tape_name[24];

/* check that we have a valid flag   */	
	if ( flag != O_RDWR && flag != O_RDONLY) {
	      fprintf(stderr,"datatape - error opening device: Invalid open flag %d\n",flag);
	      return(MOUNT_OPEN_ERROR);
	}
	
/* check that we have a valid drive number */
	if (drive >= 1000) {
	      fprintf(stderr,"datatape - error opening device: Invalid drive number %d\n",drive);
	      return(MOUNT_INVALID_DEVICE);
	}
	
/* set up device name for tape */
#ifdef SVR4
	(void) sprintf(tape_name,"/dev/rmt/%dbn",drive);			
#else
	(void) sprintf(tape_name,"/dev/nrst%d",drive);
#endif
	
/* open device */
        if ( (fd = open(tape_name, flag)) == -1) {
		perror("datatape - error opening device");
		fprintf(stderr, "error during open on device %s\n",tape_name);
		return (MOUNT_OPEN_ERROR);
	}
	
/* update device information structure */	
	if( flag == O_RDONLY ) {
	      read_dev.fd  = fd;
	      read_dev.drive = drive;
	}
	else {
	      write_dev.fd = fd;
	      write_dev.drive = drive;
	}
	
	return(0);	/* no error */
}
/*************************************************************************************/

#if 0 /* this functions is never used */

/* get current file number */
static int
tape_getfile(
#if NeedFunctionPrototype
		 int     fd)
#else
      fd )	     
      int	fd;			/* file descriptor for tape device */
#endif
{
	struct mtget mt;
	ioctl(fd, MTIOCGET, &mt);
	return( (int) mt.mt_fileno );
}

#endif

/*************************************************************************************/
/*
 * move tape by requested number of data files
 * with ANSI tapes assume that they are always at the beginning of new file
 */
int
skip_files(
#if NeedFunctionPrototype
	   int nfiles)
#else
      nfiles )
      int nfiles;          /* number of tape files to move past */
#endif
{
      /* check to see that the read file isn't open */
      if ( read_dev.file_open ) {
	    if( verbose ) fprintf(stderr, "skip_file: read file open\n");
	    return(FOPEN_FILE_OPEN);
      }

      /* check to see if tape is in ansi format or not */
      if ( read_dev.nonansi ) {
	    if( tape_op( MTFSF, nfiles, read_dev.fd ) == -1 ) {
		  perror("skip_file");
		  return(TAPE_ERROR);
	    }
      }
      else {
	    int   errors = 0;

	    /* -ve =>backwards and +ve => forwards */
	    if ( nfiles < 0 ) {
		  /* determine the number of ansi headers */
		  int	headers = read_dev.fdata.hdr34 ? 4 : 2;

		  /* skip nfiles*3,file marks to move nfiles number of files */
		  if( tape_op( MTBSF, nfiles*-3, read_dev.fd ) == -1 ) 
			errors++;

		  /* move to beginning of the ansi header records */
		  if ( tape_op(MTBSR, headers, read_dev.fd) < 0) 
			errors++;
	    }
	    else {
		  /* skip nfiles*3,file marks to move nfiles number of files */
		  if( tape_op( MTFSF, nfiles*3, read_dev.fd ) == -1 ) 
			errors++;
	    }

	    /* if errors now non zero rewind tape to beginning of 1st file */
	    if ( errors ) {
		  perror("skip_file");
		  fprintf(stderr, "rewinding tape on drive %d ...\n", read_dev.drive);
		  if( tape_op(MTREW, 1, read_dev.fd ) < 0 )			 /* rewind */
			if( verbose ) perror( "skip_file" );
		  /* skip vol1 header */
		  if ( read( read_dev.fd, read_buffer, HDR_BLOCK_LEN ) != HDR_BLOCK_LEN) {
			fprintf(stderr,"skip_file: error reading volume header block\n");
			return(TAPE_ERROR);
		  }
	    }
      }
      
      return(0);    
}
/*************************************************************************************/
static int
read_ansi_headers(
#if NeedFunctionPrototype
		      void
#endif
		      )
{
	int	ibytes;
	char	temp[6];
	memset( temp, '\0', 6 );

	/* assume for the moment that either the tape is rewound,         */
	/* or that we're at the start of a file                           */

	ibytes = read(read_dev.fd, read_buffer, READBUFFSIZE); /* read block */
	if( !ibytes )
		return (FOPEN_EOV); /* read 0 bytes, so double file mark(=EOV) */
					  /* (the tape should be positioned after    */
					  /* a filemark or vol1)                     */

	/* check the headers:         */
	/*********** HDR1 *************/

	if (ibytes != HDR_BLOCK_LEN)	/* first length */
		return(FOPEN_FAULTY_STRUCT);
	memcpy( (char *) &hdr1, read_buffer, HDR_BLOCK_LEN );
	if ( strncasecmp( "HDR1", hdr1.header, 4) )
		return (FOPEN_FAULTY_STRUCT);
	memset( read_dev.fdata.filename, '\0', 80 );
	memcpy( read_dev.fdata.filename, hdr1.name, 17 );
	/* check nothing else in hdr1, go on to hdr2        */

	/*********** HDR2 *************/

	ibytes = read(read_dev.fd, read_buffer, READBUFFSIZE);
	if( !ibytes || ibytes != HDR_BLOCK_LEN )
		return (FOPEN_FAULTY_STRUCT);
	memcpy( (char *) &hdr2, read_buffer, HDR_BLOCK_LEN );
	if ( strncasecmp( "HDR2", hdr2.header, 4) )
		return (FOPEN_FAULTY_STRUCT);
	/* check record format is D (variable) or F (fixed) */
	if ( (hdr2.recfmt != REC_VARIABLE) && (hdr2.recfmt != REC_FIXED) )
		return (FOPEN_FAULTY_STRUCT);
	else
	      read_dev.fdata.format = hdr2.recfmt;
	read_dev.fdata.blocking =    (hdr2.blocked_records == 'B') ;
	memcpy( temp, hdr2.blocklen, 5);
	read_dev.fdata.blen =    atoi(temp);
	memcpy( temp, hdr2.reclen, 5);
	read_dev.fdata.rlen   =    atoi(temp);

	/********** HDR3 & 4 ***********/

	/* ignore HDR3, just check its length                 */
	/* if the read gives 0 bytes, we've come to the end   */
	/* of the headers, there is no hdr3 or hdr4 record	*/
	ibytes = read(read_dev.fd, read_buffer, READBUFFSIZE);
	if( ibytes ){
		if( ibytes != HDR_BLOCK_LEN )
			return (FOPEN_FAULTY_STRUCT);
		ibytes = read(read_dev.fd, read_buffer, READBUFFSIZE);
		if( !ibytes || ibytes != HDR_BLOCK_LEN )
			return (FOPEN_FAULTY_STRUCT);
		memcpy( (char *) &hdr4, read_buffer, HDR_BLOCK_LEN );
		memcpy( read_dev.fdata.filename+17, hdr4.name2, 63);
		read_dev.fdata.hdr34 = TRUE;
		ibytes = read(read_dev.fd, read_buffer, READBUFFSIZE);
		if (ibytes)         /* read file mark in the case where hdr34 == TRUE */
		      return(FOPEN_FAULTY_STRUCT);
	}
	else
		read_dev.fdata.hdr34 = FALSE;
	if( verbose ) {
		fprintf(stderr, "file %.*s\n", (int) strcspn( read_dev.fdata.filename, " " ),
			read_dev.fdata.filename);
		fprintf(stderr, "block length %i, maximum record length %i\n", read_dev.fdata.blen,
			read_dev.fdata.rlen);
	}
	return(SUCCESS);

}
/*************************************************************************************/
static int
write_ansi_headers( 
#if NeedFunctionPrototype
		       int flag)
#else
      flag )
      int flag;
#endif
{
	char		temp[8];
	time_t	tp;
	struct tm	*localt;
	tp = time( NULL );
	localt = localtime( &tp );

	/***************** fill in the headers **********************/
	/********************** HDR1 / EOF1 *************************/
	memset( (char *) &hdr1, ' ', 80 );	/* space fill */
	if( flag == HEAD )
		memcpy(hdr1.header, "HDR1",			4);
	else
		memcpy(hdr1.header, "EOF1",			4);
	memcpy(hdr1.name, write_dev.fdata.filename, strlen(write_dev.fdata.filename));
	memcpy(hdr1.fileset, write_dev.volume,	6);
	memcpy(hdr1.volume_num, "0001",			4);
	memcpy(hdr1.file_num, "0001",				4);
	memcpy(hdr1.gen, "0001",				4);
	memcpy(hdr1.genver, "00",				2);
	strftime( hdr1.created, 7, " %y%j", localt );
	memcpy(hdr1.expires, " 00000",			6);
	if( flag == HEAD ){
		memcpy(hdr1.blockcount, "000000", 		6);
	}
	else {
		sprintf( temp, "%6.6li", write_dev.fdata.blocks );
		memcpy( hdr1.blockcount, temp, 		6);
	}
	memcpy(hdr1.tapesys, "DATATAPE-SUN ",		13);

	/********************** HDR2 / EOF2 *************************/
	memset( (char *) &hdr2, ' ', 80 );	/* space fill */
	if( flag == HEAD )
		memcpy(hdr2.header, "HDR2",			4);
	else
		memcpy(hdr2.header, "EOF2",			4);
	memcpy(hdr2.block_offset, "00",			2);
	hdr2.recfmt = write_dev.fdata.format;
	hdr2.blocked_records = 'B';
	sprintf( temp, "%5.5i", write_dev.fdata.blen );
	memcpy( hdr2.blocklen, temp, 5 );
	sprintf( temp, "%5.5i", write_dev.fdata.rlen );
	memcpy( hdr2.reclen, temp, 5 );
	
	/* write the headers, followed by 1 file mark */
	if ( write( write_dev.fd, (char *) &hdr1, HDR_BLOCK_LEN) != HDR_BLOCK_LEN) {
	      perror("write_ansi_headers");
	      return(TAPE_ERROR);
	}
	if ( write( write_dev.fd, (char *) &hdr2, HDR_BLOCK_LEN) != HDR_BLOCK_LEN) {
	      perror("write_ansi_headers");
	      return(TAPE_ERROR);
	}
	if( tape_op( MTWEOF, 1, write_dev.fd ) < 0) {
	      perror("write_ansi_headers");
	      return(TAPE_ERROR);
	}
	return(SUCCESS);
}
/*************************************************************************************/

/*
 *   return a pointer to the function to be used to read from
 *   or write to the data tape
 *
 */
IOfunc
dtiofunc(
#if NeedFunctionPrototype
	 int access_mode)
#else
	  access_mode)
          int access_mode;
#endif
{
      if (access_mode == READ_MODE) {
	    return(read_dev.fdata.iofunc);
      }
      else if (access_mode == WRITE_MODE) {
	    return(write_dev.fdata.iofunc);
      }
      else 
	    fprintf(stderr,"DTIOFUNC: unrecognised access mode\n");
      return(NULL);
}
/*************************************************************************************/

/*
 *   
 *   print out some of the information held in the file_data structure
 *   for either the read or write OPEN tape file
 */
int
dtstate(
#if NeedFunctionPrototype
	int access_mode)
#else
      access_mode)
      int access_mode;
#endif
{
      if (access_mode == READ_MODE) {
	    if( !read_dev.vol_mounted )
		  return(1);
	    fprintf(stderr,"%s exabyte volume %s mounted on drive %d for reading\n",
		    (read_dev.nonansi) ? "NONANSI" : "ANSI", read_dev.volume, read_dev.drive);
	    if( ! read_dev.file_open )
		  return(2);
	    fprintf(stderr,"Reading from file %s with blocklen = %d \nand reclen = %d: format = %c\n",
		    read_dev.fdata.filename, read_dev.fdata.blen,
		    read_dev.fdata.rlen, read_dev.fdata.format);
	    return(0);

      }
      else if (access_mode == WRITE_MODE) {
	    if( !write_dev.vol_mounted )
		  return(1);
	    fprintf(stderr,"%s exabyte volume %s mounted on drive %d for writing\n",
		    (write_dev.nonansi) ? "NONANSI" : "ANSI", write_dev.volume, write_dev.drive);
	    if( ! write_dev.file_open )
		  return(2);
	    fprintf(stderr,"Writing to file %s with blocklen = %d and \nreclen = %d: format = %c\n",
		    write_dev.fdata.filename, write_dev.fdata.blen,
		    write_dev.fdata.rlen, write_dev.fdata.format);
	    return(0);
      }
      else
	    fprintf(stderr,"DTSTATE: unrecognised access mode\n");
      return(-1);
}

/*************************************************************************************/
/* open physical UNIX tape device and setup device info structure */
/* mount an tape for reading or writing  */
int
ansi_mount(
#if NeedFunctionPrototype
	  char	*volume,
	  int	tape,
	  int	mode)
#else
      volume, tape, mode)
      char	*volume;	/* volume name						*/
      int 	tape;	        /* device name e.g. "st1", NULL for default st0 */
      int	mode;		/* READ_MODE (0) or WRITE_MODE (1)  or NEWVOL_MODE (2)*/
#endif
{
      char	volume_temp[6];
      int	iopen;
      int       buffsize;
      int       fd;
      int       ibytes;
      char	*buff;
      int       volume_l = strlen(volume);

      if( volume_l > 6 ) {
	    fprintf(stderr, "dtmount: vol name too long %.*s\n", volume_l, volume);
	    return (MOUNT_INVALID_VOL);	/* vol name too long */
      }
      
      switch( mode ) {
	    case READ_MODE :
		  if( (iopen = open_tape_device( O_RDONLY, tape)) != 0) {
			return(iopen);
		  }
		  buff = read_buffer;
		  buffsize = READBUFFSIZE;
		  fd = read_dev.fd;
		  break;
		  
	   case WRITE_MODE :
		  if( (iopen = open_tape_device( O_RDWR, tape)) != 0) {
			return(iopen);
		  }
		  buff = write_buffer;
		  buffsize = WRITEBUFFSIZE;
		  fd = write_dev.fd;
		  break;
		  
	    case NEWVOL_MODE:
		  if( (iopen = open_tape_device( O_RDWR, tape)) != 0) {
			return(iopen);
		  }
		  buff = write_buffer;
		  buffsize = WRITEBUFFSIZE;
		  fd = write_dev.fd;
		  if( tape_op( MTREW, 1 , fd ) < 0 )  /* rewind tape */
			if( verbose ) perror("dtmount");
		  (void) newvol(fd,volume); 
		  break;
		  
	    default :
		  return (MOUNT_UNKNOWN_MODE);
	    }
      
/* rewind tape */
      if( tape_op( MTREW, 1 , fd ) < 0 )  
	    if( verbose ) perror("dtmount");
	
/* read a block, should be vol1 header */
      ibytes = read( fd, buff, buffsize);
      if( ibytes != HDR_BLOCK_LEN ) {
	    (void) close_tape_device( mode, (mode == READ_MODE) ? TRUE:FALSE);
	    fprintf(stderr, "dtmount: no vol1 header or vol1 header has wrong length\n");
	    return(MOUNT_NOT_ANSI_FORMAT);
      }
      
/* now check volume name etc */
      memcpy( (char *) &vol1, buff, HDR_BLOCK_LEN );
      
      if ( strncasecmp( "VOL1", vol1.header, 4) ){
	    (void) close_tape_device( mode, TRUE);
	    fprintf(stderr, "dtmount: first block not vol1 header\n");
	    return (MOUNT_NOT_ANSI_FORMAT);
      }
      memset(volume_temp,' ',6);
      memcpy(volume_temp, volume, volume_l );
      if ( strncasecmp( volume_temp, vol1.label, 6 ) ){
	    (void) close_tape_device( mode, TRUE);
	    fprintf(stderr, "dtmount: wrong volume %.6s\n", vol1.label);
	    return (MOUNT_WRONG_TAPE); /* wrong volume label */
      }
      
/* setup more of the device info structure */
      if( mode == READ_MODE ) {
	    memcpy( read_dev.volume, vol1.label, 6 );
	    read_dev.vol_mounted  = TRUE;
	    read_dev.just_mounted  = TRUE;
	    read_dev.nonansi = FALSE;
      }
      else {
	    memcpy( write_dev.volume, vol1.label, 6 );
	    write_dev.vol_mounted = TRUE;
	    write_dev.just_mounted = TRUE;
	    write_dev.nonansi = FALSE;
	    write_dev.fdata.eod = FALSE;    /* used to indicate error condition */
      }
      if( verbose ) fprintf(stderr, "volume %.*s mounted\n",volume_l, volume);
      return(SUCCESS);
}

int
raw_mount(
#if NeedFunctionPrototype
	  int	tape,
	  int	mode)
#else
      tape, mode)
      int	tape;	        /* device number */
      int	mode;		/* READ_MODE (0) or WRITE_MODE (1)  or NEWVOL_MODE (2)*/
#endif
{
      int	iopen;

      switch( mode ) {
		case READ_MODE :
			if( (iopen = open_tape_device( O_RDONLY, tape)) != 0) {
			      return(iopen);
			}

			/* rewind tape */
			if( tape_op( MTREW, 1 , read_dev.fd) < 0 )  
			      if( verbose ) perror("dtmount");
			
			/* setup device info structure */
			strncpy( read_dev.volume, "NONANSI", 7 );
			read_dev.vol_mounted  = TRUE;
			read_dev.just_mounted  = TRUE;
			read_dev.nonansi = TRUE;
			break;
			
		case WRITE_MODE :
		case NEWVOL_MODE:
			if( (iopen = open_tape_device( O_RDWR, tape)) != 0) {
			      return(iopen);
			}

			/* rewind tape */
			if( tape_op( MTREW, 1 , write_dev.fd) < 0 )  
			      if( verbose ) perror("dtmount");

			/* setup device info structure */
			strncpy( write_dev.volume, "NONANSI", 7 );
			write_dev.vol_mounted = TRUE;
			write_dev.just_mounted = TRUE;
			write_dev.nonansi = TRUE;
			break;
			
		default :
			return (MOUNT_UNKNOWN_MODE);
	}

	return(SUCCESS);
}

int
dtmount(
#if NeedFunctionPrototype
	  char	*volume,
	  int	tape,
	  int	mode)
#else
      volume, tape, mode)
      char	*volume;	/* volume name						*/
      int	tape;	        /* device number */
      int	mode;		/* READ_MODE (0) or WRITE_MODE (1)  or NEWVOL_MODE (2)*/
#endif
{
      if( verbose ) {
	    fprintf(stderr, "attempting to open volume %s on drive %d for ", volume, tape);
	    if (mode != READ_MODE) 
		  fprintf(stderr, "writing...\n");
	    else
		  fprintf(stderr, "reading...\n");
      }

      if (strcasecmp(volume,"nonansi") == 0)
	    return(raw_mount(tape, mode));
      else 
	    return(ansi_mount(volume, tape, mode));
}

/*************************************************************************************/
int
dtumount( 
#if NeedFunctionPrototype
	   int flag, int eject)
#else
      flag, eject)
      int flag;
      int eject;
#endif
{
	return (close_tape_device(flag, eject));
}

/*************************************************************************************/
/* open a file on the currently mounted volume, */
/* for reading or writing depending on flag == READ_MODE or WRITE_MODE */
/* blocklen and reclen are dummies when reading */
/* reclen is the maximum possible reclen, must be <= blocklen - 4 for ANSI D format*/
/* return 0 on success, <0 on error	*/
int
ansi_open(
#if NeedFunctionPrototype
	   char	*filename,
	   int	flag,
	   int	blocklen,
	   int	reclen,
	   int	option)
#else
      filename, flag, blocklen, reclen, option)
      char	*filename;
      int	flag;
      int	blocklen;
      int       reclen;
      int       option;
#endif
{
      int	ibytes;
      int	ierr;
      char	filename_temp[80];
      int	filename_l = strlen(filename);
      int       block_hlen = (option & ANSI_FIXED_REC) ? 0 : 4;
	

      if (flag == WRITE_MODE) {
	    /* check that we have a tape mounted and have't already got a file open */
	    if ( !write_dev.vol_mounted ) {
		  if( verbose ) fprintf(stderr, "dtopen: volume not mounted\n" );
		  return( FOPEN_VOL_NOT_MOUNTED );
	    }
	    if ( write_dev.file_open ) {
		  if( verbose ) fprintf(stderr, "dtopen: file already open\n" );
		  return( FOPEN_FILE_OPEN );
	    }

	    /* check to see if we have had an error opening a file on this mounted tape */
	    if (write_dev.fdata.eod == TAPE_ERROR) {
		  fprintf(stderr,"dtopen: write tape in error state ... unmounted tape and try again\n");
		  return (TAPE_ERROR);
	    }
	    /* set write_dev.fdata.eod to error initially but reset it to FALSE after a SUCCESSFUL open */
	    write_dev.fdata.eod = TAPE_ERROR;
	    
	    /* check and set block and record lengths */
	    if ( !blocklen ) blocklen = MAXWRITEBLK;
	    if ( !reclen )   reclen = blocklen - block_hlen;
	    if ( reclen > blocklen - block_hlen || reclen < 1) {
		  if( verbose ) fprintf( stderr, "dtopen: invalid record size %i",reclen );
		  return( FOPEN_REC_SIZE );
	    }
	    if ( blocklen < 1024 || blocklen > MAXWRITEBLK ) {
		  if( verbose ) fprintf( stderr, "dtopen: invalid block size %i",blocklen );
		  return( FOPEN_BLOCK_SIZE );
	    }
	    write_dev.fdata.blen = blocklen;
	    write_dev.fdata.rlen = reclen;
	    
	    /* set write function to use depending on specified format */
	    if ( option & ANSI_FIXED_REC) {
		  write_dev.fdata.format = REC_FIXED;
		  write_dev.fdata.iofunc = dtwrite_fr;
		  if (reclen  < blocklen)
			(void) memset(write_buffer + reclen, '^', blocklen - reclen);
	    }
	    else {
		  write_dev.fdata.format = REC_VARIABLE;
		  write_dev.fdata.iofunc = dtwrite_vr;
	    }

	    /** get to end of medium by skipping a file at a time if data files exist  **/
	    if ( write_dev.just_mounted ) {
		  int cont = TRUE;
		  write_dev.just_mounted = FALSE;
		  
		  while (cont != FALSE) {
			ibytes = read( write_dev.fd, write_buffer, HDR_BLOCK_LEN );
			switch ( ibytes ) {
			   case 0:	                     /* file mark, so skip back over it */
			      if( tape_op( MTBSF, 1, write_dev.fd ) < 0 ) {
				    if ( verbose ) perror( "dtopen" );
				    return(TAPE_ERROR);
			      }
			      cont = FALSE;
			      break;
			
			   case -1:	                     /* read error, assume it is a blank tape */
			      if ( verbose ) perror( "dtopen" );
			      cont = FALSE;
			      break;
			
			   case HDR_BLOCK_LEN:	             /* file found */
			      if( tape_op( MTFSF, 3, write_dev.fd ) < 0 ) {
				    if ( verbose ) perror( "dtopen" );
				    return(FOPEN_FAULTY_STRUCT);
			      }
			      break;
			      
			   default:	                    /* shouldn't happen */
			      return(TAPE_ERROR);
			}
		  }
	    }

	    /* update device info structure with new filename */
	    memset(write_dev.fdata.filename, '\0', 80);
	    strncpy(write_dev.fdata.filename, filename, 17);
	    toupperstring(write_dev.fdata.filename);	

	    /* write ansi headers to tape */
	    if( verbose ) fprintf( stderr, "dtopen: writing headers\n" );
	    if (write_ansi_headers( HEAD ) < 0)
		  return(TAPE_ERROR);
	    
	    write_dev.file_open = TRUE;
	    write_dev.fdata.blocks = 0;
	    write_dev.fdata.eod = FALSE;
	    write_buffer_pos = 0;
      }
      else if (flag == READ_MODE) {
	    /* check that we have a tape mounted and have't already got a file open */
	    if( !read_dev.vol_mounted ) {
		  if( verbose ) fprintf(stderr, "dtopen: volume not mounted\n" );
		  return( FOPEN_VOL_NOT_MOUNTED );
	    }
	    if( read_dev.file_open ) {
		  if( verbose ) fprintf(stderr, "dtopen: file already open\n" );
		  return( FOPEN_FILE_OPEN );
	    }
	    read_dev.just_mounted = FALSE;

	    /* rewind tape to beginning if rewind option set */
	    if( option & REWIND_OPT ) {
		  fprintf(stderr, "rewinding tape on drive %d ...\n", read_dev.drive);
		  if( tape_op(MTREW, 1, read_dev.fd ) < 0 )			 /* rewind */
			if( verbose ) perror( "dtopen" );
		  /* skip vol1 header */
		  if ( read( read_dev.fd, read_buffer, HDR_BLOCK_LEN ) != HDR_BLOCK_LEN) {
			fprintf(stderr,"dtopen: error reading volume header block\n");
			return(TAPE_ERROR);
		  }
	    }
	    
	    /* read headers, return error if there is one */
	    while ( !read_dev.file_open ){
		  if (Stop_interupt(0)) {
			fprintf(stderr,"datatape: dtopen interupted during open operation\n");
			return(FOPEN_INTERUPT);
		  }
		  
		  /* read ansi header for data file */
		  if ( (ierr = read_ansi_headers()) < 0) {
			if( verbose ) fprintf(stderr, "dtopen: error reading file headers\n");
			return(ierr);
		  }
		  
		  /* check filename */
		  memset(filename_temp, ' ', 80);
		  strncpy(filename_temp, filename, filename_l);
		  if ( strncasecmp( read_dev.fdata.filename, filename_temp, 17) ){
			/* wrong file, skip to next one */
			fprintf( stderr, "dtopen: wrong file (%.17s), "
				"skipping...\n", read_dev.fdata.filename);
			if( tape_op( MTFSF, 1, read_dev.fd ) < 0 ) {
			      if( verbose ) perror("dtopen");
			      return(TAPE_ERROR);
			}
			/* do a read to see if the trailers are there */
			ibytes = read( read_dev.fd, read_buffer, READBUFFSIZE );
			if ( ibytes <= 0) {
			      /* if we've read another file mark, we've hit the EOM */
			      return( FOPEN_NOT_FOUND );
			}
			else { /* skip to next file mark ... to next file's headers */
			      if( tape_op( MTFSF, 1, read_dev.fd ) < 0 ) {
				    if( verbose ) perror("dtopen");
				    return(TAPE_ERROR);
			      }
			}
		  }
		  else{
			if( verbose ) fprintf( stderr, "dtopen: file opened %s\n", filename_temp);
			read_dev.file_open = TRUE;
			if (read_dev.fdata.format == REC_FIXED)
			      read_dev.fdata.iofunc = dtread_fr;
			else
			      read_dev.fdata.iofunc = dtread_vr;
			strncpy(read_dev.fdata.filename, filename, 17);
			/* check whether want to reopen same file after close */
			read_dev.fdata.hold = option & HOLD_OPT;
			read_dev.fdata.eod = FALSE;
			read_buffer_pos = 0;
			/* NB file's blocklen and reclen set by ansi header */
		  }
	    }
      }
      else {
	    if( verbose ) fprintf(stderr, "dtopen: unknown mode\n" );
	    return(FOPEN_UNKNOWN_MODE);
      }
      return( SUCCESS );
}

/*ARGSUSED*/
int
raw_open(
#if NeedFunctionPrototype
	   char	*filename,
	   int	flag,
	   int	blocklen,
	   int	reclen,
	   int	option)
#else
      filename, flag, blocklen, reclen, option)
      char	*filename;
      int	flag;
      int	blocklen;
      int       reclen;
      int       option;
#endif
{
	
	if (flag == WRITE_MODE ) {
	      /* check that we have a tape mounted and have't already got a file open */
	      if( !write_dev.vol_mounted ) {
		    if( verbose ) fprintf(stderr, "dtopen: volume not mounted\n" );
		    return( FOPEN_VOL_NOT_MOUNTED );
	      }
	      if( write_dev.file_open ) {
		    if( verbose ) fprintf(stderr, "dtopen: file already open\n" );
		    return( FOPEN_FILE_OPEN );
	      }
	      
	      /* check and set block and record lengths */
	      if( !blocklen ) blocklen = MAXWRITEBLK;
	      if( !reclen )   reclen = blocklen;
	      if( reclen > blocklen || reclen < 1) {
		    if( verbose ) fprintf( stderr, "dtopen: invalid record size %i",
					  reclen );
		    return( FOPEN_REC_SIZE );
	      }
	      if( blocklen < 1024 || blocklen > MAXWRITEBLK ) {
		    if( verbose ) fprintf( stderr, "dtopen: invalid block size %i",
					  blocklen );
		    return( FOPEN_BLOCK_SIZE );
	      }

	      /* update file info structure */
	      write_dev.fdata.blen = blocklen;
	      write_dev.fdata.rlen = reclen;
	      write_dev.fdata.format = REC_UNIX;
	      write_dev.fdata.iofunc = dtwrite_fr;
	      
	      /* if just mounted then go to end of medium  */
	      if( write_dev.just_mounted ) {
		    write_dev.just_mounted = FALSE;
		    if( tape_op( MTEOM, 1, write_dev.fd ) < 0 ) {
			  if ( verbose ) perror( "dtopen" );
			  return TAPE_ERROR;
		    }
		    /* backspace one of the file marks ... assume double file mark ?????????? */
		    if( tape_op( MTBSF, 1, write_dev.fd ) < 0 ) {
			  if ( verbose ) perror( "dtopen" );
			  return TAPE_ERROR;
		    }
	      }
	      strcpy(write_dev.fdata.filename,"UNKNOWN");
	      write_dev.file_open = TRUE;
	      write_dev.fdata.blocks = 0;
	      write_buffer_pos = 0;
	}
	else if (flag == READ_MODE ) {
	      /* check that we have a tape mounted and have't already got a file open */
	      if( !read_dev.vol_mounted ) {
		    if( verbose ) fprintf(stderr, "dtopen: volume not mounted\n" );
		    return( FOPEN_VOL_NOT_MOUNTED );
	      }
	      if( read_dev.file_open ) {
		    if( verbose ) fprintf(stderr, "dtopen: file already open\n" );
		    return( FOPEN_FILE_OPEN );
	      }
	      read_dev.just_mounted = FALSE;
	      
	      /* rewind tape to beginning if rewind option set */
	      if( option & REWIND_OPT ) {
		    if( verbose ) fprintf(stderr, "rewinding...\n");
		    if( tape_op(MTREW, 1, read_dev.fd ) < 0 )	{
			  if( verbose ) perror( "dtopen" );
			  return(TAPE_ERROR);
		    }
	      }
	      
	      /* set block length and record length for read tape */
	      if( !blocklen ) blocklen = MAXWRITEBLK;
	      if( !reclen )   reclen = blocklen;
	      if( reclen > blocklen || reclen < 1) {
		    if( verbose ) fprintf( stderr, "dtopen: invalid record size %i",
					  reclen );
		    return( FOPEN_REC_SIZE );
	      }
	      if( blocklen < 512 || blocklen > MAXWRITEBLK ) {
		    if( verbose ) fprintf( stderr, "dtopen: invalid block size %i",
					  blocklen );
		    return( FOPEN_BLOCK_SIZE );
	      }
	      
	      /* update file info structure */
	      strcpy(read_dev.fdata.filename,"UNKNOWN");
	      read_dev.fdata.iofunc = dtread_fr;
	      read_dev.fdata.format = REC_UNIX;
	      read_dev.fdata.blen = blocklen;
	      read_dev.fdata.rlen = reclen;
	      read_dev.fdata.eod = FALSE;
	      read_dev.file_open = TRUE;
	      read_buffer_pos = 0;
	}
	else {
	      if( verbose ) fprintf(stderr, "dtopen: unknown mode\n" );
	      return(FOPEN_UNKNOWN_MODE);
	}

	return( SUCCESS );
}

int
dtopen(
#if NeedFunctionPrototype
	   char	*filename,
	   int	flag,
	   int	blocklen,
	   int	reclen,
	   int	option)
#else
      filename, flag, blocklen, reclen, option)
      char	*filename;
      int	flag;
      int	blocklen;
      int       reclen;
      int       option;
#endif
{
      int nonansi = (flag == WRITE_MODE) ? write_dev.nonansi : read_dev.nonansi;

      if ( nonansi )
	    return(raw_open(filename, flag, blocklen, reclen, option));
      else
	    return(ansi_open(filename, flag, blocklen, reclen, option));
}

/*************************************************************************************/
int
ansi_close(
#if NeedFunctionPrototype
	   int flag)
#else
      flag )
      int flag;	           /* 0 for READ, 1 for WRITE */
#endif
{
	int	headers;	/* number of headers */
	int	i;

	
	if ( flag == READ_MODE) {
	      /* check to see that we actually do have a file open */
	      if ( !read_dev.file_open ) {
		    if( verbose ) fprintf(stderr, "dtclose: no file open for reading\n");
		    return(FCLOSE_FILE_CLOSED);
	      }

	      /* determine the number of ansi headers */
	      headers = read_dev.fdata.hdr34 ? 4 : 2;

	      if ( read_dev.fdata.hold ) {
		    fprintf(stderr,"dtclose: jumping back to start of file\n");
		    /* skip back to position tape just after the file's ansi headers */
		    if( tape_op( MTBSF, (read_dev.fdata.eod == TRUE) ? 2:1, read_dev.fd ) == -1 ) {
			  perror("dtclose");
			  return(TAPE_ERROR);
		    }
		    /* move to beginning of the ansi header records */
		    if ( tape_op(MTBSR, headers, read_dev.fd) < 0) {
			  perror("dtclose");
			  return(TAPE_ERROR);
		    }
	      }
	      else {
		    /* skip to next file mark */
		    fprintf(stderr,"dtclose: skipping forward to next tape file\n");
		    if (read_dev.fdata.eod == FALSE) {
			  /* need to skip passed end of data filemark */
			  if( tape_op( MTFSF, 1, read_dev.fd ) < 0 ) {
				if( verbose ) perror("dtclose");
				return(TAPE_ERROR);
			  }
		    }
		    /* read ansi trailers */
		    if( verbose) fprintf(stderr, "dtclose; reading trailers\n");
		    while ( headers-- ) {
			  if ( (i = read(read_dev.fd, read_buffer, HDR_BLOCK_LEN)) != HDR_BLOCK_LEN) {
				fprintf(stderr,"dtclose: error reading trailers\n");
				if (i < 0) perror("dtclose");
				return(FCLOSE_FAULTY_STRUCT);
			  }
		    }
		    /* skip filemark at end of trailers */
		    if( tape_op( MTFSF, 1, read_dev.fd ) < 0 ) {
			  if( verbose ) perror("dtclose");
			  return(TAPE_ERROR);
		    }
	      }
	      read_dev.fdata.eod = FALSE;
	      read_dev.file_open = FALSE;
	}
	else if (flag == WRITE_MODE) {
	      /* check to see that we actually do have a file open */
	      if ( !write_dev.file_open ) {
		    if( verbose ) fprintf(stderr, "dtclose: no file open for writing\n");
		    return(FCLOSE_FILE_CLOSED);
	      }
	      
	      /* need to flush the output buffer if necessary */
	      if ( write_buffer_pos ) {
		    memset( write_buffer + write_buffer_pos, '^', write_dev.fdata.blen - write_buffer_pos );
		    i = write( write_dev.fd, write_buffer, write_dev.fdata.blen );
		    write_dev.fdata.blocks++;
	      }

	      /* write file mark after the end of the data */
	      if( tape_op( MTWEOF, 1, write_dev.fd ) < 0 ) {	
		    if( verbose ) perror("dtclose");
		    return(TAPE_ERROR);
	      }
	      /* write the ansi trailers */
	      if (write_ansi_headers( TAIL ) == TAPE_ERROR)
		    return(TAPE_ERROR);
	      
	      /* add extra file mark for eom and backspace over it */			
	      if (tape_op( MTWEOF, 1, write_dev.fd ) < 0) {
		    perror("dtclose");
		    return(TAPE_ERROR);
	      }
	      if (tape_op( MTBSF, 1, write_dev.fd ) < 0) {
		    perror("dtclose");
		    return(TAPE_ERROR);
	      }
	      write_dev.file_open = FALSE;
	}
	else {
	      if( verbose ) fprintf(stderr, "dtclose: unknown mode %i\n", flag);
	      return(FCLOSE_UNKNOWN_MODE);
	}
	
	return(0);
}

int
raw_close(
#if NeedFunctionPrototype
	   int flag)
#else
      flag )
      int flag;	           /* 0 for READ, 1 for WRITE */
#endif
{
	if (flag ==READ_MODE) {
	      if ( !read_dev.file_open ) {
		    if( verbose ) fprintf(stderr, "dtclose: no file open for reading\n");
		    return(FCLOSE_FILE_CLOSED);
	      }
	      (void) tape_op( MTFSF, 1, read_dev.fd );
	      read_dev.file_open = FALSE;
	}
	else if (flag == WRITE_MODE) {
	      if ( !write_dev.file_open ) {
		    if( verbose ) fprintf(stderr, "dtclose: no file open for writing\n");
		    return(FCLOSE_FILE_CLOSED);
	      }
	      /* need to flush the output buffer if necessary */
	      if( write_buffer_pos ) {
		    memset( write_buffer + write_buffer_pos, 0, write_dev.fdata.blen - write_buffer_pos );
		    (void) write( write_dev.fd, write_buffer, write_dev.fdata.blen );
		    write_dev.fdata.blocks++;
	      }
	      /* add two file mark and backspace over one */			
	      if (write_dev.fdata.blocks != 0) {       
		    if (tape_op( MTWEOF, 2, write_dev.fd ) < 0) {
			  perror("dtclose");
			  return(TAPE_ERROR);
		    }
		    if (tape_op( MTBSF, 1, write_dev.fd ) < 0) {
			  perror("dtclose");
			  return(TAPE_ERROR);
		    }
	      }
	      write_dev.file_open = FALSE;
	}
	else {
	      if( verbose ) fprintf(stderr, "dtclose: unknown mode %i\n", flag);
	      return(FCLOSE_UNKNOWN_MODE);
	}
	return(0);
}

int
dtclose(
#if NeedFunctionPrototype
	   int flag)
#else
      flag )
      int flag;	           /* 0 for READ, 1 for WRITE */
#endif
{
      int nonansi = (flag == WRITE_MODE) ? write_dev.nonansi : read_dev.nonansi;

      if ( nonansi )
	    return(raw_close(flag));
      else
	    return(ansi_close(flag));
}

/*************************************************************************************/
/* returns the next record in buffer,      */
/* return value is record length, 0 on EOF */
/*  VARIABLE LENGTH RECORDS with POSSIBLE BLOCKING */
static int
dtread_vr( 
#if NeedFunctionPrototype
	  int maxbuflen, char  *buffer)
#else
      maxbuflen, buffer)
      int       maxbuflen;
      char	*buffer;
#endif
{
      int	reclen;
      
      if( !read_buffer_pos ){		/* need to read block */
	    int ibytes = read( read_dev.fd, read_buffer, read_dev.fdata.blen );
	    fix2gblimit(read_dev.fd);
	    while( ibytes != read_dev.fdata.blen ) {
		  if( ibytes ) {
			if (read_dev.fdata.eod == FALSE) {
			      fprintf(stderr, "datatape - wrong block size %d ... discarding \n",ibytes);
			      ibytes = read( read_dev.fd, read_buffer, read_dev.fdata.blen );
			      if (ibytes == -1) {
				    perror("dtread_vr");
				    return(-1);
			      }
			}
			else {
			      /* at the end of data so rewind past this last record */
			      if( tape_op( MTBSR, 1, read_dev.fd ) < 0 ) {
				    perror("dtread_vr");
				    return(-1);
			      }
			      return(0);
			}
		  }
		  else {
			/* reached end of data EOD .. just read file mark */
			read_dev.fdata.eod = TRUE;
			return(0);  
		  }
	    }
	    memcpy( rconv, read_buffer, 4 );
      }
      
      reclen = atoi(rconv) - 4;/* rconv has either stuck around since */
      /* last record, or has just been read */
      
      if( maxbuflen < reclen) {
	    fprintf(stderr,"dtread: buffer size %d smaller than record length %d\n",
		    maxbuflen, reclen);
	    return(-1);
      }
      
      memcpy( buffer, read_buffer + read_buffer_pos + 4, reclen ); 
      read_buffer_pos += (reclen + 4);
      if( read_buffer_pos + 4 > read_dev.fdata.blen )
	    read_buffer_pos = 0;
      else {
	    memcpy( rconv, read_buffer + read_buffer_pos, 4 );
	    if ( ! strncmp( rconv, "^^^^", 4 ) )
		  read_buffer_pos = 0;
      }
      return( reclen );
}

/*  FIXED LENGTH RECORDS with POSSIBLE BLOCKING */
static int
dtread_fr( 
#if NeedFunctionPrototype
	    int maxbuflen, char  *buffer)
#else
      maxbuflen, buffer)
      int       maxbuflen;
      char	*buffer;
#endif
{
    if( !read_buffer_pos ){		/* need to read block */
	int ibytes = read( read_dev.fd, read_buffer, read_dev.fdata.blen );
	fix2gblimit(read_dev.fd);
	while( ibytes != read_dev.fdata.blen ){
	    if( ibytes > 0) {
		if (read_dev.fdata.eod == FALSE) {
		    fprintf(stderr, "datatape - wrong block size "
			    "%d ... discarding \n",ibytes);
		    ibytes = read(read_dev.fd, read_buffer,
				  read_dev.fdata.blen );
		    if (ibytes == -1) {
			perror("dtread_fr");
			return(-1);
		    }
		}
		else {
		    /* at the end of data so rewind past this last record */
		    if( tape_op( MTBSR, 1, read_dev.fd ) < 0 ) {
			perror("dtread_fr");
			return(-1);
		    }
		    return(0);
		}
	    }
	    else if (ibytes == 0) {
		/* reached end of data EOD .. just read file mark */
		read_dev.fdata.eod = TRUE;
		return(0);  
	    }
	    else {
		/* Tape error */
		perror("datatape error");
		tape_op(MTNOP, 1, read_dev.fd);
		close(read_dev.fd);
		if (open_tape_device(O_RDONLY, read_dev.drive))
		{
		    perror("Unable to recover");
		    return(-1);
		}
		ibytes = read(read_dev.fd, read_buffer,
			      read_dev.fdata.blen );
		if (ibytes == -1)
		{
		    perror("datatape fatal error");
		    return(-1);
		}
	    }
	}
    }
    if( maxbuflen < read_dev.fdata.rlen) {
	fprintf(stderr,"dtread: buffer size %d smaller than record length %d\n",
		maxbuflen, read_dev.fdata.rlen);
	return(-1);
    }
    memcpy( buffer, read_buffer + read_buffer_pos, read_dev.fdata.rlen);
    read_buffer_pos += read_dev.fdata.rlen;
    if( read_buffer_pos + read_dev.fdata.rlen > read_dev.fdata.blen )
	read_buffer_pos = 0;
    return( read_dev.fdata.rlen );
}
/*************************************************************************************/
/* write reclen bytes from buffer */
/* returns reclen on success, 0 on EOT, <0 on error */
static int
dtwrite_vr( 
#if NeedFunctionPrototype
	    int	        reclen,
	    char	*buffer)
#else
      reclen, buffer)
      int	reclen;
      char	*buffer;
#endif
{
	int	ibytes;

/* check to see that we have a valid reclen */
	if( reclen > write_dev.fdata.rlen || reclen < 1 ) {
	      fprintf(stderr,"dtwrite_vr: invalid record length %d specified\n",reclen);
	      return(-1);
	}
	
/* not enough space left in block , write it */
	if( reclen + 4 + write_buffer_pos > write_dev.fdata.blen ) { /* block full, write it */
		if (write_dev.fdata.blen - write_buffer_pos)
		      memset(write_buffer + write_buffer_pos, '^',write_dev.fdata.blen - write_buffer_pos );
		if ( (ibytes = write( write_dev.fd, write_buffer, write_dev.fdata.blen) ) <= 0 ) {
		      if (ibytes < 0)
			    perror("dtwrite_vr");
		      return(ibytes);
		}
			
		write_dev.fdata.blocks++;
		fix2gblimit(write_dev.fd);
		write_buffer_pos = 0;
	}

/* update local buffer with record header and data fields */	
	sprintf( wconv, "%4.4i", reclen + 4 );
	memcpy( write_buffer + write_buffer_pos, wconv, 4 );
	memcpy( write_buffer + write_buffer_pos + 4, buffer, reclen );
	
	write_buffer_pos += reclen + 4;
	
	return( reclen );
}

/*************************************************************************************/
/* write reclen bytes from buffer to tape */
/* returns reclen on success, 0 on EOT, < 0 on error */
static int
dtwrite_fr( 
#if NeedFunctionPrototype
	    int	        reclen,
	    char	*buffer)
#else
      reclen, buffer)
      int	reclen;
      char	*buffer;
#endif
{
      int	ibytes;

      /* check to see that specified reclen is within valid range */
      if( reclen > write_dev.fdata.rlen || reclen < 1 ) {
	    fprintf(stderr,"dtwrite_fr: invalid record length %d specified\n",reclen);
	    return(-1);
      }  

      /* if reclen == blen write it straight to tape otherwise write it to our buffer */
      if (reclen == write_dev.fdata.blen) {
	    if ( (ibytes = write( write_dev.fd, buffer, write_dev.fdata.blen)) != reclen) {
		  if (ibytes < 0)
			perror("dtwrite_fr");
		  else if (ibytes > 0) {
			fprintf(stderr,"dtwrite_fr: wrote incomplete block to tape\n");
			ibytes = -ibytes;
		  }
		  return(ibytes);
	    }
	    write_dev.fdata.blocks++;
	    fix2gblimit(write_dev.fd);
      }
      else {
	    /* not enough space left in block, write it */
	    if( reclen + write_buffer_pos > write_dev.fdata.blen ) {
		  /* pad out end of block */
		  if (write_dev.fdata.blen - write_buffer_pos) {
			if (write_dev.nonansi)
			      memset(write_buffer + write_buffer_pos, 0, write_dev.fdata.blen - write_buffer_pos);
			else
			      memset(write_buffer + write_buffer_pos, '^', write_dev.fdata.blen - write_buffer_pos);
		  }
		  /* write it to tape */
		  if ( (ibytes = write( write_dev.fd, write_buffer, write_dev.fdata.blen)) <= 0 ) {
			if (ibytes < 0)
			      perror("dtwrite_fr");
			return(ibytes);
		  }     
		  write_dev.fdata.blocks++;
		  fix2gblimit(write_dev.fd);
		  write_buffer_pos = 0;
	    }
	    /* copy data to local buffer */
	    memcpy( write_buffer + write_buffer_pos, buffer, reclen );
	    write_buffer_pos += reclen;
      }
      
      return( reclen );
}

/*************************************************************************************/
#if NeedFunctionPrototype
int dtverb(int flag) {verbose = flag; return(0);}

#if 0 /* This functions is never used */
int dtvers(void) {return( DTVERSION );}
#endif
      
/* converts null-terminated string to upper case */
void toupperstring( char *s)
{
    for(; *s != '\0'; s++ )
        *s = toupper( *s );
}

#else
int dtverb(flag) int flag; {verbose = flag;}

#if 0 /* This functions is never used */
int dtvers() {return( DTVERSION );}
#endif

/* converts null-terminated string to upper case */
void toupperstring(s)
char *s
{
    for(; *s != '\0'; s++ )
        *s = toupper( *s );
}

#endif
      
/*************************************************************************************/
/*
 *   rewind a mounted tape
 */
#if 0 /* This function is never used */
void
dtrewin(
#if NeedFunctionPrototype
	     int mode)
#else
      mode)
      int mode;
#endif
{
      if (mode == WRITE_MODE) {      
	    if( tape_op(MTREW, 1, write_dev.fd ) < 0 )			
		  if( verbose ) perror( "dtrewin" );
      }
      else if (mode == READ_MODE) {
	    if( tape_op(MTREW, 1, read_dev.fd ) < 0 )			
		  if( verbose ) perror( "dtrewin" );
      }
      return;
}
#endif

/********************************************************/
/*	Newvol() - write a volume label to the tape	*/
/********************************************************/
int
newvol(
#if NeedFunctionPrototype
	     int fd, char *volume)
#else
      fd, volume)
      int  fd;
      char *volume;
#endif
{
    struct ansi_vol1 label;
    char Vol[7];
    register char *cp;

    if ( verbose)
	  fprintf(stderr,"writing ansi volume label\n");

/* clear label structure */
    (void) memset(&label, ' ', sizeof(struct ansi_vol1));
    (void) memset(Vol, ' ', 7);

/* generate upper case volume name */
    (void) memcpy(Vol, volume, strlen(volume));
    Vol[6] = '\0';
    for(cp=Vol; *cp; cp++) {		/* convert to upper case */
	  if (islower(*cp))
		*cp = toupper(*cp);
    }

/* set up label structure and write it to tape */
    (void) memcpy(label.header, "VOL1", 4);
    (void) memcpy(label.label, Vol, 6);
    (void) memcpy(label.owner, "SUNSORT",7);
    label.ansi_level = '3';
    
    if (write(fd, (char *) &label, HDR_BLOCK_LEN) != HDR_BLOCK_LEN) {
	  perror("datatape - error writing volume label");
	  return(-1);
    }
    return(0);
}


static void
fix2gblimit(
#if NeedFunctionPrototype
	 int fd )
#else
      fd)
      int fd;
#endif
{
/* hack to get round UNIX 2Gb file limit :			*/
/* just reset file pointer to zero when it goes over 1Gbyte */
	if( lseek( fd, 0L, SEEK_CUR ) > 1073741824 )
		lseek( fd, 0L, SEEK_SET );
}


