#include <stdio.h>
#include <math.h>
#include <xview/xview.h>
#include <xview/panel.h>
#include <xview/canvas.h>
#include <xview/xv_xrect.h>
#include <xview/cms.h>
#include "spec_ui.h"
#include "print_ui.h"
#include "proj_ui.h"
#include "3d_ui.h"
#include "spec_stubs.h"
#include "3d_stubs.h"
#include "print_stubs.h"
#include "contour_stubs.h"
#include "3d_redraw.h"

static double inp[3], up[3], rightp[3], scale = 288.0, persp = 5.0;
static int x_centre = 256, y_centre = 256;
static int bbllx, bblly, bburx, bbury;

struct projected {
    short llx;
    short lly;
    short ulx;
    short uly;
    short urx;
    short ury;
    short lrx;
    short lry;
};

struct polylist {
    unsigned int x:15;
    unsigned int y:15;
    unsigned int n:2;
};

static struct projected grid[MAXSPECX][MAXSPECY];
static XPoint points[(MAXSPECX > MAXSPECY) ? MAXSPECX*2 : MAXSPECY*2];

static GC gc = NULL, blackgc = NULL;
static unsigned long *index;
static Display *dip;
static Window wn;
static FILE *psFile;

static void draw_polygon(XPoint *points, int n, int c)
{
    static int lastc = -1;
    XPoint *p;
    int i;

    for(p = points, i = n; i--; p++)
	if (p->x >= 32767)
	    return;

    if (dip == NULL)
    {
	int x, y;

	if (!ferror(psFile))
	    fprintf(psFile, "%c%c", c+48-COLOUROFFSET, n+47);
	for(p = points, i = n; i--; p++)
	{
	    x = p->x + 30000;
	    y = 30000 - p->y;
	    if (x < bbllx)
		bbllx = x;
	    if (x > bburx)
		bburx = x;
	    if (y < bblly)
		bblly = y;
	    if (y > bbury)
		bbury = y;
	    if (!ferror(psFile))
		fprintf(psFile, "%c%c%c%c%c%c", (x >> 12)+48,
			((x >> 6) & 63)+48, (x & 63)+48, (y >> 12)+48,
			((y >> 6) & 63)+48, (y & 63)+48);
	}
	if (!ferror(psFile))
	    putc('\n', psFile);
    }
    else
    {
	for(p = points, i = n; i--; p++)
	    if (p->x >= bbllx || p->x < bburx || p->y >= bblly || p->y < bbury)
	    {
		if (c != lastc)
		{
		    XSetForeground(dip, gc, index[c]);
		    lastc = c;
		}
		XFillPolygon(dip, wn, gc, points, n, Convex, CoordModeOrigin);
		points[n].x = points[0].x;
		points[n].y = points[0].y;
		XDrawLines(dip, wn, blackgc, points, n+1, CoordModeOrigin);
		break;
	    }
    }
}

static void draw_top(int x, int y)
{
    int c;
    XPoint *pp;

    pp = points;
    pp->x = grid[x][y].llx;
    pp++->y = grid[x][y].lly;
    pp->x = grid[x][y].ulx;
    pp++->y = grid[x][y].uly;
    pp->x = grid[x][y].urx;
    pp++->y = grid[x][y].ury;
    pp->x = grid[x][y].lrx;
    pp->y = grid[x][y].lry;
    c = spec2dcontour[x][y];
    draw_polygon(points, 4, c);
}

static void draw_y_side(int x, int y)
{
    int c;
    XPoint *pp;

    pp = points;
    pp->x = grid[x][y].llx;
    pp++->y = grid[x][y].lly;
    pp->x = grid[x][y].lrx;
    pp++->y = grid[x][y].lry;
    pp->x = grid[x][y-1].urx;
    pp++->y = grid[x][y-1].ury;
    pp->x = grid[x][y-1].ulx;
    pp->y = grid[x][y-1].uly;
    c = (data[x][y] > data[x][y-1]) ? spec2dcontour[x][y]
	: spec2dcontour[x][y-1];
    draw_polygon(points, 4, c);
}

static void draw_x_side(int x, int y)
{
    int c;
    XPoint *pp;

    pp = points;
    pp->x = grid[x][y].llx;
    pp++->y = grid[x][y].lly;
    pp->x = grid[x][y].ulx;
    pp++->y = grid[x][y].uly;
    pp->x = grid[x-1][y].urx;
    pp++->y = grid[x-1][y].ury;
    pp->x = grid[x-1][y].lrx;
    pp->y = grid[x-1][y].lry;
    c = (data[x][y] > data[x-1][y]) ? spec2dcontour[x][y]
	: spec2dcontour[x-1][y];
    draw_polygon(points, 4, c);
}

static void draw_topline(int x, int miny, int maxy, int ysplit)
{
    int y;

    for(y=miny; y<ysplit; y++)
    {
	draw_top(x, y);
	if (y < maxy_3d)
	    draw_y_side(x, y+1);
    }
    for(y=maxy-1; y>ysplit; y--)
    {
	draw_top(x, y);
	if (y > miny_3d)
	    draw_y_side(x, y);
    }
    draw_top(x, ysplit);
}

static void draw_sideline(int x, int miny, int maxy, int ysplit)
{
    int y;
    
    for(y=miny; y<ysplit; y++)
	draw_x_side(x, y);
    for(y=maxy-1; y>ysplit; y--)
	draw_x_side(x, y);
    draw_x_side(x, ysplit);
}

/*
 * Really these find_split routines should be replaced with binary searches
 * or direct calculation for speed, but the splitting takes so little time
 * compared to the drawing that the development time would better be spent
 * elsewhere.
 */

static int find_x_split(int minx, int maxx, double cellx)
{
    int split;
    double s0x, s0y, s1x, s1y, s2x, s2y, x, z;

    for(split=minx; split<maxx-1; split++)
    {
	x = (double) (split-(minx_3d+maxx_3d)/2.0)*cellx;
	
	s0x = x*rightp[0];
	s0y = x*up[0];
	z = x*inp[0];
	z = persp/(z + persp);
	s0x *= z;
	s0y *= z;
	
	s1x = x*rightp[0] + rightp[1];
	s1y = x*up[0] + up[1];
	z = x*inp[0] + inp[1];
	z = persp/(z + persp);
	s1x *= z;
	s1y *= z;
	
	s2x = x*rightp[0] + rightp[2];
	s2y = x*up[0] + up[2];
	z = x*inp[0] + inp[2];
	z = persp/(z + persp);
	s2x *= z;
	s2y *= z;
	
	s1x -= s0x;
	s1y -= s0y;
	s2x -= s0x;
	s2y -= s0y;
	
	if (s1x*s2y < s1y*s2x)
	    break;
    }
    return split;
}

static int find_y_split(int miny, int maxy, double celly)
{
    int split;
    double s0x, s0y, s1x, s1y, s2x, s2y, y, z;

    for(split=miny; split<maxy-1; split++)
    {
	y = (double) ((miny_3d+maxy_3d)/2.0 - split)*celly;

	s0x = y*rightp[1];
	s0y = y*up[1];
	z = y*inp[1];
	z = persp/(z + persp);
	s0x *= z;
	s0y *= z;
    
	s1x = rightp[0] + y*rightp[1];
	s1y = up[0] + y*up[1];
	z = inp[0] + y*inp[1];
	z = persp/(z + persp);
	s1x *= z;
	s1y *= z;
    
	s2x = y*rightp[1] + rightp[2];
	s2y = y*up[1] + up[2];
	z = y*inp[1] + inp[2];
	z = persp/(z + persp);
	s2x *= z;
	s2y *= z;
    
	s1x -= s0x;
	s1y -= s0y;
	s2x -= s0x;
	s2y -= s0y;

	if (s1x*s2y < s1y*s2x)
	    break;
    }
    return split;
}

#define STYLE3D_MESH 0
#define STYLE3D_GRID 1
	
static void vector_cross(double out[3], double in1[3], double in2[3])
{
    out[0] = in1[1]*in2[2] - in1[2]*in2[1];
    out[1] = in1[2]*in2[0] - in1[0]*in2[2];
    out[2] = in1[0]*in2[1] - in1[1]*in2[0];
}

static int project_point(short *sx, short *sy, double x, double y, double z)
{
    double mx, my, mz;

    mx = x*rightp[0] + y*rightp[1] + z*rightp[2];
    my = x*up[0] + y*up[1] + z*up[2];
    mz = x*inp[0] + y*inp[1] + z*inp[2];

    if (mz <= -persp)
    {
/*
	fprintf(stderr, "(%6.3f, %6.3f, %6.3f) off screen\n", x, y, z);
*/
	return 0;
    }

    mz = persp/(mz + persp);

    *sx = mx*mz * scale + x_centre;
    *sy = y_centre - my*mz * scale;

/*
    fprintf(stderr, "(%6.3f, %6.3f, %6.3f) -> (%d, %d)\n", x, y, z, *sx, *sy);
*/

    return 1;
}

static void setup3d(int style, double *cx, double *cy)
{
    int midx, midy, x, y;
    double cellx, celly;
    struct projected *gp;

    if (recalculatePending)
	doRecalculate3d();

    inp[0] = -cos(el)*cos(az);
    inp[1] = -cos(el)*sin(az);
    inp[2] = -sin(el);

    rightp[0] = -sin(az);
    rightp[1] = cos(az);
    rightp[2] = 0;

    vector_cross(up, rightp, inp);
    vector_cross(rightp, inp, up);
    
    cellx = 1.0/(maxx_3d-minx_3d+1);
    celly = 1.0/(maxy_3d-miny_3d+1);
    if (cellx > celly)
	cellx = celly;
    else
	celly = cellx;

    if (cx != NULL)
	*cx = cellx;
    if (cy != NULL)
	*cy = celly;

    midx = (minx_3d + maxx_3d)/2.0;
    midy = (miny_3d + maxy_3d)/2.0;

    switch(style)
    {
    case STYLE3D_MESH:
	for(x = minx_3d; x <= maxx_3d; x++)
	    for(gp = &grid[x][miny_3d], y = miny_3d; y <= maxy_3d; y++, gp++)
	    {
		if (data[x][y] < 0 ||
		    !project_point(&gp->llx, &gp->lly, (x-midx)*cellx,
				   (midy-y)*celly, data[x][y] - 0.5))
		    gp->llx = 32767;
	    }
	break;
    default:
	for(x = minx_3d; x <= maxx_3d; x++)
	    for(gp = &grid[x][miny_3d], y = miny_3d; y <= maxy_3d; y++, gp++)
	    {
		if (data[x][y] < 0)
		    gp->llx = gp->ulx = gp->urx = gp->lrx = 32767;
		else
		{
		    if (!project_point(&gp->llx, &gp->lly,
				       (x - midx - 0.5)*cellx,
				       (midy - y + 0.5)*celly,
				       data[x][y] - 0.5))
			gp->llx = 32767;
		    if (!project_point(&gp->ulx, &gp->uly,
				       (x - midx - 0.5)*cellx,
				       (midy - y - 0.5)*celly,
				       data[x][y] - 0.5))
			gp->ulx = 32767;
		    if (!project_point(&gp->urx, &gp->ury,
				       (x - midx + 0.5)*cellx,
				       (midy - y - 0.5)*celly,
				       data[x][y] - 0.5))
			gp->urx = 32767;
		    if (!project_point(&gp->lrx, &gp->lry,
				       (x - midx + 0.5)*cellx,
				       (midy - y + 0.5)*celly,
				       data[x][y] - 0.5))
			gp->lrx = 32767;
		}
	    }
	break;
    }
}

static void drawgrid(void)
{
    int x, xsplit, ysplit;
    double cellx, celly;

    setup3d(STYLE3D_GRID, &cellx, &celly);

    xsplit = find_x_split(minx_3d, maxx_3d+1, cellx);
    ysplit = find_y_split(miny_3d, maxy_3d+1, celly);
    for(x=minx_3d; x<xsplit; x++)
    {
	draw_topline(x, miny_3d, maxy_3d+1, ysplit);
	if (x < maxx_3d)
	    draw_sideline(x+1, miny_3d, maxy_3d+1, ysplit);
    }
    for(x=maxx_3d; x>xsplit; x--)
    {
	draw_topline(x, miny_3d, maxy_3d+1, ysplit);
	if (x > minx_3d)
	    draw_sideline(x, miny_3d, maxy_3d+1, ysplit);
    }
    draw_topline(xsplit, miny_3d, maxy_3d+1, ysplit);
}

/*ARGSUSED*/
void redraw3d(Canvas canvas, Xv_window paint_window, Display *display,
	      Window xid, Xv_xrectlist *rects)
{
    int x, y;
    XPoint *pp;
    XGCValues gc_val;

    if (!xv_get(Spec_3dwindow->window, XV_SHOW))
	return;

    drawn = 1;

    if (gc == NULL)
	gc = XCreateGC(display, xid, 0, NULL);

    if (blackgc == NULL)
    {
	blackgc = XCreateGC(display, xid, 0, NULL);
	XSetForeground(display, blackgc,
		       BlackPixel(display, DefaultScreen(display)));
    }

    if (rects)
    {
	XSetClipRectangles(display, gc, 0, 0, rects->rect_array,
			   rects->count, Unsorted);
	XSetClipRectangles(display, blackgc, 0, 0, rects->rect_array,
			   rects->count, Unsorted);
	bbllx = rects->rect_array[0].x;
	bblly = rects->rect_array[0].y;
	bburx = bbllx + rects->rect_array[0].width;
	bbury = bblly + rects->rect_array[0].height;
	for(x = 1; x < rects->count; x++)
	{
	    y = rects->rect_array[x].x;
	    if (y < bbllx)
		bbllx = y;
	    y += rects->rect_array[x].width;
	    if (y > bburx)
		bburx = y;
	    y = rects->rect_array[x].y;
	    if (y < bblly)
		bblly = y;
	    y += rects->rect_array[x].height;
	    if (y > bbury)
		bbury = y;
	}
    }
    else
    {
	gc_val.clip_mask = None;
	XChangeGC(display, gc, GCClipMask, &gc_val);
	XChangeGC(display, blackgc, GCClipMask, &gc_val);
	bbllx = bblly = 0;
	bburx = bbury = 512;
    }

    index = (unsigned long *) xv_get(cms, CMS_INDEX_TABLE);
 
    if (dragging)
    {
	setup3d(STYLE3D_MESH, NULL, NULL);

	for(x = minx_3d; x <= maxx_3d; x++)
	{
	    pp = points;
	    for(y = miny_3d; y <= maxy_3d; y++)
	    {
		if (grid[x][y].llx < 32767)
		{
		    pp->x = grid[x][y].llx;
		    pp++->y = grid[x][y].lly;
		}
		else
		{
		    if (pp > points + 1)
			XDrawLines(display, xid, blackgc, points, pp - points,
				   CoordModeOrigin);
		    pp = points;
		}
	    }
	    if (pp > points + 1)
		XDrawLines(display, xid, blackgc, points, pp - points,
			   CoordModeOrigin);
	}
	
	for(y = miny_3d; y <= maxy_3d; y++)
	{
	    pp = points;
	    for(x = minx_3d; x <= maxx_3d; x++)
	    {
		if (grid[x][y].llx < 32767)
		{
		    pp->x = grid[x][y].llx;
		    pp++->y = grid[x][y].lly;
		}
		else
		{
		    if (pp > points + 1)
			XDrawLines(display, xid, blackgc, points, pp - points,
				   CoordModeOrigin);
		    pp = points;
		}
	    }
	    if (pp > points + 1)
		XDrawLines(display, xid, blackgc, points, pp - points,
			   CoordModeOrigin);
	}
	XFlush(display);
    }
    else
    {
	dip = display;
	wn = xid;

	drawgrid();
    }
}

/*ARGSUSED*/
void resize3d(Canvas canvas, int width, int height)
{
    static int o_width = 512, o_height = 512;

    x_centre = width/2;
    y_centre = height/2;
    if (width < height)
	scale = width * 0.5625;
    else
	scale = height * 0.5625;
    XClearArea(XV_DISPLAY_FROM_WINDOW(Spec_3dwindow->window),
	       (Window) xv_get(canvas_paint_window(Spec_3dwindow->canvas),
			       XV_XID), 0, 0,
	       (o_width < width) ? o_width : width,
	       (o_height < height) ? o_height : height, True);
    o_width = width;
    o_height = height;
}

/*ARGSUSED*/
void print3d(Panel_item item, Event *event)
{
    FILE *fp;
    int palette, dest, scalefac;
    double sc;
    char *msg;

    palette = xv_get(Spec_printerPopup->printerPalette, PANEL_VALUE);

    if ((fp = openPrinterFile(event_x(event), event_y(event))) == NULL)
	return;

    dest = xv_get(Spec_printerPopup->printTo, PANEL_VALUE);
    scalefac = xv_get(Spec_printerPopup->printScaleSlider, PANEL_VALUE);

    if (!ferror(fp))
	fputs("%!PS-Adobe-2.0", fp);
    if (dest == 2 && !ferror(fp))
	fputs(" EPSF-3.0", fp);
    if (!ferror(fp))
	fputs("\n%%Title: sunsort\n%%Creator: sort_spec2d\n", fp);
    if (!ferror(fp))
    {
	if (dest == 2)
	    fputs("%%BoundingBox: (atend)\n", fp);
	else
	    fputs("%%Pages: 1\n", fp);
    }
    if (!ferror(fp))
	fputs("%%EndComments\n50 dict begin\n", fp);
    fprintf(fp, "%4.2f %4.2f scale\n", scalefac*0.01, scalefac*0.01);
    writeFshowCode(fp);
    if (!ferror(fp))
	fprintf(fp, "/coord 3 string def\n"
"/colour 1 string def\n"
"0.5 setlinewidth 2 setlinejoin\n"
"/readcoord {\n"
"2 {currentfile coord readstring pop 0 get 48 sub 4096 mul coord 1 get 48\n"
"    sub 64 mul add coord 2 get 48 sub add 100 div} repeat\n"
"} bind def\n"
"/draw3d {\n"
"  /colourtable [ %d {currentfile colour readhexstring pop 0 get 255 div}\n"
"  repeat ] def currentfile colour readline pop pop {\n"
"    colourtable currentfile colour readstring pop\n"
"    dup (!) eq {pop pop exit} if\n"
"    0 get 48 sub %s\n"
"    currentfile colour readstring pop 0 get 48 sub readcoord moveto\n"
"    {readcoord lineto} repeat closepath gsave fill grestore 0 setgray\n"
"    stroke currentfile colour readline pop pop\n"
"  } loop\n"
"} bind def\n"
"draw3d\n",
		(palette == 2) ? 96 : 32,
		(palette == 2) ? "3 mul 3 getinterval aload pop setrgbcolor" :
		"get setgray");
    printPalette(fp, palette);

    sc = scale;
    scale = 30000;
    bbllx = bblly = bburx = bbury = 30000;
    dip = NULL;
    psFile = fp;
    drawgrid();
    scale = sc;

    if (!ferror(fp))
	fputs("!\n", fp);

    msg = (char *) xv_get(Spec_printerPopup->printTitle, PANEL_VALUE);
    if (*msg != '\0')
    {
	if (bbury < 63000)
	    bbury = 63000;
	if (!ferror(fp))
	    fputs("300 600 moveto ", fp);
	if (!ferror(fp))
	    psstring(fp, msg);
	if (!ferror(fp))
	    fputs(" 24 cfshow\n", fp);
    }

    msg = (char *) xv_get(Spec_printerPopup->printSubtitle, PANEL_VALUE);
    if (*msg != '\0')
    {
	if (bbury < 59500)
	    bbury = 59500;
	if (!ferror(fp))
	    fputs("300 570 moveto ", fp);
	if (!ferror(fp))
	    psstring(fp, msg);
	if (!ferror(fp))
	    fputs(" 18 cfshow\n", fp);
    }

    if (!ferror(fp))
	fputs("end\nshowpage\n%%Trailer\n", fp);
    if (dest == 2 && !ferror(fp))
	fprintf(fp, "%%%%BoundingBox: %d %d %d %d\n",
		(bbllx-199)*scalefac/10000, (bblly-199)*scalefac/10000,
		(bburx+199)*scalefac/10000, (bbury+199)*scalefac/10000);
    closePrinterFile(fp, event_x(event), event_y(event));
}
