#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <xview/xview.h>
#include <xview/panel.h>
#include <xview/xv_xrect.h>
#include <xview/cms.h>
#include <xview/notice.h>
#include "spec_ui.h"
#include "proj_ui.h"
#include "print_ui.h"
#include "3d_ui.h"
#include "spec_stubs.h"
#include "proj_stubs.h"
#include "contour_stubs.h"
#include "utilities.h"
#include "Legendre.h"
#include "sort_def.h"
#include "sort_mem.h"

void repaintProjection(Canvas, Xv_window, Display *, Window, Xv_xrectlist *);

#define POLYARRAYLEN 1024 /* must be power of 2 */
#define MAXPROJWIDTH 730 /* > sqrt(MAXSPECX^2 + MAXSPECY^2), > 256 */

static float projectAngle = 90;
static float screenAngle = M_PI/2, Xtarget = 0.0, Ytarget = 0.0;
static int polynomialOrder = 0;
static float polynomial[POLYARRAYLEN], polynomialScale = 1.0;
static unsigned int projection[MAXPROJWIDTH];
static int forceProject = 1, projectArea = 0;
int showGuidelines = 0;
static int projectAxis = 0;
float Xgain = 1.0, Ygain = 1.0, Xoffset = 0.0, Yoffset = 0.0;
extern GC gc;

/*ARGSUSED*/
Menu_item arbitraryAngle(Menu_item item, Menu_generate op)
{
    if (op != MENU_NOTIFY)
	return item;

    xv_set(Spec_projectPopup->projectPopup, FRAME_CMD_PUSHPIN_IN, TRUE, NULL);
    xv_set(Spec_projectPopup->projectPopup, XV_SHOW, TRUE, NULL);
    if (!showGuidelines)
    {
	showGuidelines = 1;
	drawGuidelines((Display *)
		       XV_DISPLAY_FROM_WINDOW(Spec_specWindow->specWindow),
		       (Window)
		       xv_get(canvas_paint_window(Spec_specWindow->spectrum),
			      XV_XID), gc);
	RRP(0);
    }
    return item;
}

static void recalculateProjection(int scale)
{
    unsigned int x, y, *up;
    int (*colp)[MAXSPECX], *sp;
    unsigned char (*wcolp)[MAXSPECX], *wp;
    int colpos, pos, dx, dy, f;

    dx = sin(screenAngle) * 65536;
    dy = cos(screenAngle) * 65536;
    colpos = (-(specy-2)*dy - specx*dx)/2 + MAXPROJWIDTH*32768;

    memset(projection, 0, sizeof(projection));

    switch(projectArea)
    {
    case 0:
	for(colp = spec2d, x = specx; x--; colp++, colpos += dx)
	    for(pos = colpos, sp = &(colp[0][0]), y = specy; y--;
		sp++, pos += dy)
	    {
		f = pos & 65535;
		projection[pos>>16] += *sp * ((65536-f)>>8);
		projection[(pos>>16)+1] += *sp * (f>>8);
	    }
	break;
    case 1:
	for(colp = spec2d, wcolp = window, x = specx; x--;
	    colp++, wcolp++, colpos += dx)
	    for(pos = colpos, sp = &(colp[0][0]), wp = &(wcolp[0][0]),
		y = specy; y--; sp++, wp++, pos += dy)
		if (*wp)
		{
		    f = pos & 65535;
		    projection[pos>>16] += *sp * ((65536-f)>>8);
		    projection[(pos>>16)+1] += *sp * (f>>8);
		}
	break;
    }

    if (scale)
    {
	for(x = 0, up = projection, y=MAXPROJWIDTH; y--; up++)
	    if (*up > x)
		x = *up;
	
	x = (x + 255) / 256;
	
	/*
	 * lint complains that comaring an usigned integer with 0 using >
	 * is suspicious. I prefer it in this case, so I'll tell lint to
	 * shut up.
	 */
	/*LINTED*/
	if (x > 0)
	    for(up = projection, y=MAXPROJWIDTH; y--; up++)
		*up /= x;
	
	forceProject = 0;
    }
}

static int calcScaleAndOffset(double *s, double *o)
{
    if ((projectAxis == 1 && (projectAngle == 0 || projectAngle == 180)) ||
	(projectAxis == 2 && projectAngle == 90))
	return -1;

    switch(projectAxis)
    {
    case 0: /* Perpendicular */
	*s = sin(screenAngle)*Xgain*sin(projectAngle*M_PI/180) +
	    cos(screenAngle)*Ygain*cos(projectAngle*M_PI/180);
	*o = (specx/2.0*Xgain+Xoffset-Xtarget)*sin(projectAngle*M_PI/180) + 
	    (-specy/2.0*Ygain-Yoffset+Ytarget)*cos(projectAngle*M_PI/180);
	break;
    case 1: /* X */
	*s = sin(screenAngle)*Xgain +
	    cos(screenAngle)*Ygain/tan(projectAngle*M_PI/180);
	*o = specx/2.0*Xgain+Xoffset +
	    (-specy/2.0*Ygain-Yoffset+Ytarget)/tan(projectAngle*M_PI/180);
	break;
    case 2: /* Y */
	*s = -cos(screenAngle)*Ygain -
	    sin(screenAngle)*Xgain*tan(projectAngle*M_PI/180);
	*o = specy/2.0*Ygain+Yoffset -
	    (specx/2.0*Xgain+Xoffset-Xtarget)*tan(projectAngle*M_PI/180);
	break;
    }
    return 0;
}

/*ARGSUSED*/
void
repaintProjection(Canvas canvas, Xv_window paint_window, Display *display,
		  Window xid, Xv_xrectlist *rects)
{
    static GC gc = NULL;
    int i, l, x, y, lx, ly;
    double s, o;
    XPoint polypoints[MAXPROJWIDTH*2], *pp;
    
    if (!showGuidelines)
	return;

    if (forceProject)
	recalculateProjection(1);

    if (gc == NULL)
	gc = XCreateGC(display, xid, 0, NULL);

    XSetFunction(display, gc, GXcopy);
    XSetForeground(display, gc, BlackPixel(display,
					   DefaultScreen(display)));
    XSetBackground(display, gc, WhitePixel(display,
					   DefaultScreen(display)));

    if (rects)
        XSetClipRectangles(display, gc, 0, 0, rects->rect_array,
                           rects->count, Unsorted);
    else
    {
        XSetClipMask(display, gc, None);
        XClearWindow(display, xid);
    }

    XSetLineAttributes(display, gc, 1, LineSolid, CapButt, JoinMiter);

    l = sqrt((double) (specx*specx + specy*specy))/2;
    lx = 0;
    for(pp = polypoints, i = -l; i < l; i++, pp += 2)
    {
	pp->x = lx;
	pp[1].x = lx =  i * 256 / l + 256;
	pp[1].y = pp->y = 255 - projection[i + MAXPROJWIDTH/2];
    }
    XDrawLines(display, xid, gc, polypoints, l*4, CoordModeOrigin);

    if (polynomialOrder > 0 && calcScaleAndOffset(&s, &o) == 0)
    {
	static char dashed[2] = { 4, 4 };

	o += s/2;
	s *= POLYARRAYLEN/180.0*l/256;
	o *= POLYARRAYLEN/180.0;
	lx = (int) (o - 256*s);
	ly = (int) (((o - 256*s) - lx) * 65536);
	lx = ((lx & (POLYARRAYLEN - 1))<<16)+ly;
	ly = (int) (s * 65536);

	for(pp = polypoints, x = 0; x < 512; x++, pp++)
	{
	    y = 255 - polynomial[lx>>16]*polynomialScale*255;
	    lx = (lx+ly) & ((POLYARRAYLEN<<16)-1);
	    pp->x = x;
	    pp->y = y;
	}
	XSetDashes(display, gc, 0, dashed, sizeof(dashed));
	XSetLineAttributes(display, gc, 0, LineOnOffDash, CapButt, JoinMiter);
	XDrawLines(display, xid, gc, polypoints, 512, CoordModeOrigin);
	XSetLineAttributes(display, gc, 0, LineSolid, CapButt, JoinMiter);
    }
}


void
drawGuidelines(Display *display, Window xid, GC gc)
{
    double dx, dy, i, l, tx, ty;
    unsigned long *index;

    if (monochrome)
	XSetForeground(display, gc,
		       BlackPixel(display, DefaultScreen(display)) ^
		       WhitePixel(display, DefaultScreen(display)));
    else
    {
	index = (unsigned long *) xv_get(cms, CMS_INDEX_TABLE);
	XSetForeground(display, gc, index[COLOUROFFSET+MAXCONTOURS-1] ^
		       index[COLOUROFFSET]);
    }
    
    XSetFunction(display, gc, GXxor);

    l = sqrt((double) specx*specx + specy*specy);

    dx = sin(screenAngle);
    dy = -cos(screenAngle);

    tx = (Xtarget - Xoffset)/Xgain;
    ty = (Ytarget - Yoffset)/Ygain;

    i = tx*dx + ty*dy;
    i -= floor(i/10)*10;

    for(i -= floor(l/10)*10; i<l; i += 10)
    {
	XDrawLine(display, xid, gc, (int) ((i*dx - l*dy + 0.5)*mag),
		  (int) ((specy - (i*dy + l*dx + 0.5))*mag),
		  (int) ((i*dx + l*dy + 0.5)*mag),
		  (int) ((specy - (i*dy - l*dx + 0.5))*mag));
    }

    switch(projectAxis)
    {
    case 0: /* Perpendicular */
	XDrawLine(display, xid, gc, (int) ((tx + l*dx + 0.5)*mag),
		  (int) ((specy - (ty + l*dy + 0.5))*mag),
		  (int) ((tx - l*dx + 0.5)*mag),
		  (int) ((specy - (ty - l*dy + 0.5))*mag));
	break;
    case 1: /* X */
	XDrawLine(display, xid, gc, 0, (int) ((specy - ty - 0.5)*mag),
		  specx*mag, (int) ((specy - ty - 0.5)*mag));
	break;
    case 2: /* Y */
	XDrawLine(display, xid, gc, (int) ((tx+ 0.5)*mag), 0,
		  (int) ((tx + 0.5)*mag), specy*mag);
	break;
    }

    XSetFunction(display, gc, GXcopy);
}

Panel_setting
valueSetProjectAngle(Panel_item item, Event *event)
{
    float num;

    if ((num = goodFloat(Spec_projectPopup->projectControls, (char *)
			 xv_get(item, PANEL_VALUE), 0.0, 180.0, event)) >= 0.0)
    {
	xv_set(Spec_projectPopup->projectAngleSlider, PANEL_VALUE,
	       (int) num*10, NULL);
	RRP(1);
	projectAngle = num;
	RRP(2);
    }
    return panel_text_notify(item, event);
}

/*ARGSUSED*/
void
sliderSetProjectAngle(Panel_item item, int value, Event *event)
{
    char buffer[6];
    double num;

    num = ((float) value) / 10;
    sprintf(buffer, "%5.1f", num);
    xv_set(Spec_projectPopup->projectAngleValue, PANEL_VALUE, buffer, NULL);
    if (projectAngle != num)
    {
	RRP(1);
	projectAngle = num;
	RRP(2);
    }
}

/*ARGSUSED*/
void
setProjectArea(Panel_item item, int value, Event *event)
{
    projectArea = value;
    RRP(0);
}

int
projectDone(Xv_opaque item)
{
    xv_set(item, XV_SHOW, FALSE, NULL);
    if (showGuidelines)
	drawGuidelines((Display *)
		       XV_DISPLAY_FROM_WINDOW(Spec_specWindow->specWindow),
		       (Window)
		       xv_get(canvas_paint_window(Spec_specWindow->spectrum),
			      XV_XID), gc);
    showGuidelines = 0;
    return XV_OK;
}

/*ARGSUSED*/
void
setProjectAxis(Panel_item item, int value, Event *event)
{
    RRP(1);
    projectAxis = value;
    RRP(2);
}

/*ARGSUSED*/
void
doProjection(Panel_item item, Event *event)
{
    int i, l;
    unsigned int y, max;
    float *p;
    double s, o;

    if (calcScaleAndOffset(&s, &o))
    {
	notice_prompt(Spec_projectPopup->projectPopup, NULL,
		      NOTICE_FOCUS_XY, event_x(event), event_y(event),
		      NOTICE_MESSAGE_STRINGS,
		      "Don't try to project parallel to the axis you want",
		      "to read the numbers off - it won't work.", NULL,
		      NOTICE_BUTTON_YES, "Ok",
		      NULL);
	return;
    }

    recalculateProjection(0);
    forceProject = 1;

    l = sqrt((double) (specx*specx + specy*specy))/2;
    strcpy(pshm->com_str, "DISPLAX 1");
    sprintf(pshm->name, "%5.1f\\So\\N", projectAngle);
    pshm->more = 0;
    pshm->size = l*4;

    max = 256;
    for(p = (float *) pshm->array, i = -l; i < l; i++, p += 2)
    {
	*p = i * s + o;
	y = projection[i + MAXPROJWIDTH/2];
	if (y > max)
	    max = y;
	p[1] = y / 256.0;
    }

    write_to_pipe(".SORT_pipe0", l*4); 

    signal_1d();

    if (polynomialOrder > 0)
    {
	double x;

	sleep(1);

	strcpy(pshm->com_str, "OVERLAX 1");
	sprintf(pshm->name, "|P%d|\\S2\\N", polynomialOrder);
	pshm->more = 0;
	pshm->size = 1024;

	s *= M_PI/180*l/256;
	o *= M_PI/180;

	for(p = (float *) pshm->array, i = 0; i < 512; i++, p += 2)
	{
	    x = ((i-256)*s + o);
	    *p = x*180/M_PI;
	    p[1] = pow(legendre(polynomialOrder, cos(x)),2.0) * polynomialScale
		* max / 256;
	}

	write_to_pipe(".SORT_pipe0", 1024);
	signal_1d();
    } 
}

Panel_setting
projectXGain(Panel_item item, Event *event)
{
    char *value = (char *) xv_get(item, PANEL_VALUE);
    double val;

    if ((val = goodFloat(Spec_projectPopup->projectControls, (char *)
			   value, 0.0, 0.0, event)) != 0.0)
    {
	xv_set(Spec_printerPopup->printXGain, PANEL_VALUE, value, NULL);
	RRP(1);
	Xgain = val;
	RRP(2);
    }
    return panel_text_notify(item, event);
}

Panel_setting
projectXOffset(Panel_item item, Event *event)
{
    char *value = (char *) xv_get(item, PANEL_VALUE);
    double val;

    if ((val = goodFloat(Spec_projectPopup->projectControls, (char *)
			   value, -1e37, 1e37, event)) >= -1e37)
    {
	xv_set(Spec_printerPopup->printXOffset, PANEL_VALUE, value, NULL);
	RRP(1);
	Xoffset = val;
	RRP(2);
    }
    return panel_text_notify(item, event);
}

Panel_setting
projectYGain(Panel_item item, Event *event)
{
    char *value = (char *) xv_get(item, PANEL_VALUE);
    double val;

    if ((val = goodFloat(Spec_projectPopup->projectControls, (char *)
			   value, 0.0, 0.0, event)) != 0.0)
    {
	xv_set(Spec_printerPopup->printYGain, PANEL_VALUE, value, NULL);
	RRP(1);
	Ygain = val;
	RRP(2);
    }
    return panel_text_notify(item, event);
}

Panel_setting
projectYOffset(Panel_item item, Event *event)
{
    char *value = (char *) xv_get(item, PANEL_VALUE);
    double val;

    if ((val = goodFloat(Spec_projectPopup->projectControls, (char *)
			   value, -1e37, 1e37, event)) >= -1e37)
    {
	xv_set(Spec_printerPopup->printYOffset, PANEL_VALUE, value, NULL);
	RRP(1);
	Yoffset = val;
	RRP(2);
    }
    return panel_text_notify(item, event);
}

void RRP(int change) /* Recalculate and Repaint Projection */
           
{
    Display *disp = (Display *)
	XV_DISPLAY_FROM_WINDOW(Spec_specWindow->specWindow);
    Window specwin = (Window)
	xv_get(canvas_paint_window(Spec_specWindow->spectrum), XV_XID);
    Window projwin = (Window)
	xv_get(canvas_paint_window(Spec_projectPopup->projection), XV_XID);

    if (change == 1)
	drawGuidelines(disp, specwin, gc);
    else if (change == 2)
    {
	if (projectAngle == 0.0 || projectAngle == 180.0)
	    screenAngle = projectAngle * M_PI / 180.0;
	else
	    screenAngle = atan(tan(projectAngle * M_PI/180) * Xgain / Ygain);
	if (screenAngle < 0)
	    screenAngle += M_PI;
	drawGuidelines(disp, specwin, gc);
    }
    forceProject = 1;
    repaintProjection(XV_NULL, XV_NULL, disp, projwin, (Xv_xrectlist *) NULL);
}

Panel_setting
projectXTarget(Panel_item item, Event *event)
{
    char *value = (char *) xv_get(item, PANEL_VALUE);
    double val;

    if ((val = goodFloat(Spec_projectPopup->projectControls, (char *)
			   value, -1e37, 1e37, event)) >= -1e37)
    {
	RRP(1);
	Xtarget = val;
	RRP(2);
    }
    return panel_text_notify(item, event);
}

Panel_setting
projectYTarget(Panel_item item, Event *event)
{
    char *value = (char *) xv_get(item, PANEL_VALUE);
    double val;

    if ((val = goodFloat(Spec_projectPopup->projectControls, (char *)
			   value, -1e37, 1e37, event)) >= -1e37)
    {
	RRP(1);
	Ytarget = val;
	RRP(2);
    }
    return panel_text_notify(item, event);
}

Panel_setting
setPolynomial(Panel_item item, Event *event)
{
    int value = (int) xv_get(item, PANEL_VALUE);
    int i;
    
    polynomialOrder = value;
    if (value > 0)
    {
	for(i = 0; i < POLYARRAYLEN; i++)
	    polynomial[i] = pow(legendre(value, cos(i*M_PI/POLYARRAYLEN)),2.0);
    }
    RRP(0);

    return panel_text_notify(item, event);
}

/*ARGSUSED*/
void
scaleUp(Panel_item item, Event *event)
{
    polynomialScale *= 1.1;
    RRP(0);
}

/*ARGSUSED*/
void
scaleDown(Panel_item item, Event *event)
{
    polynomialScale /= 1.1;
    if (polynomialScale < 1)
	polynomialScale = 1;
    RRP(0);
}

