#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <xview/xview.h>
#include <xview/panel.h>
#include <xview/xv_xrect.h>
#include <xview/cms.h>
#include <xview/wmgr.h>
#include <xview/notice.h>
#include <xview/scrollbar.h>
#include <X11/Xutil.h>
#include "spec_ui.h"
#include "3d_ui.h"
#include "print_ui.h"
#include "proj_ui.h"
#include "contour_ui.h"
#include "spec_stubs.h"
#include "proj_stubs.h"
#include "3d_stubs.h"
#include "print_stubs.h"
#include "contour_stubs.h"
#include "lookup_proc.h"
#include "utilities.h"
#include "sort_def.h"
#include "sort_mem.h"

struct windowPointList {
    int x;
    int y;
    struct windowPointList *next;
};

void repaintSpectrum(Canvas, Xv_window, Display *, Window, Xv_xrectlist *);
Menu_item abortSetWindow(Menu_item, Menu_generate);
static void newPoint(Xv_window, Event *, Notify_arg, Notify_event_type);
static void deletePoint(Xv_window, Event *, Notify_arg, Notify_event_type);
static void closeWindow(Xv_window, Event *, Notify_arg, Notify_event_type);
static void freeWindowList(struct windowPointList *);
static void store_window(void);
static int store_spectrum(char *, char *, int);
static void do_projection(Event *);
static void setMagnification(int);
static void readSpectrumFromPipe(void);
static void spectrumLoaded(void);
static void drawCrosshairs(Display *, Window, GC, int, int, int);
static int writeSpectrumToPipe(int);
static Notify_value destroy_func(Notify_client, int, Notify_signal_mode);
static Notify_value sort_sigusr2_handler(Notify_client, int,
					 Notify_signal_mode);
static Notify_value recalculateIcon(void);
static void event_proc(Xv_Window, Event *, Notify_arg);

spec_specWindow_objects	        *Spec_specWindow;
spec_contourPopup_objects	*Spec_contourPopup;
spec_levelPopup_objects	        *Spec_levelPopup;
spec_gammaPopup_objects	        *Spec_gammaPopup;
spec_rgbPopup_objects	        *Spec_rgbPopup;
spec_printerPopup_objects	*Spec_printerPopup;
spec_projectPopup_objects	*Spec_projectPopup;
spec_customPopup_objects	*Spec_customPopup;
spec_3dwindow_objects		*Spec_3dwindow;

#define WINDOW_NONE 0
#define WINDOW_OUTSIDE 1
#define WINDOW_INSIDE 2
#define WINDOW_ONLY 3
#define WINDOW_SET 4
#define MAXSPECSIZE 800

int specx = 73, specy = 73, spec2d[MAXSPECX][MAXSPECY], specMax = 0;
unsigned char spec2dcontour[MAXSPECX][MAXSPECY];
unsigned char window[MAXSPECX][MAXSPECY];
static char spectrumName[80], windowName[80];
static int windowStatus = WINDOW_NONE;
static unsigned char *iconBits;
static int iconWidth, iconHeight;
static Xv_opaque iconBitmap = XV_NULL;
static Icon iconIcon = XV_NULL;
int DBX_val = 0;
static int sort_window = 0;
static int forceRedraw;
int mag = 4;
static char *programName;
static int showCrosshairs = 0;
static int iconLine = 0;
static struct itimerval iconTimer;


/**** 16/1/95 ****/

static int Slicing;
static void do_simple_slice(Event *event, int x, int y);

#define SLICING 0x1
#define PROJECTION 0x2

/*****************/

#define SORT_TRUE   1
#define SORT_FALSE  0
#define SORT_FAIL  -1
#define SORT_NOS_ERR -2

Notify_client sort_client = (Notify_client) 104;

static void testSpectrum(void)
{
    int i,j;
    double x,y,r,t;

    for(i=0; i<specx; i++)
    {
	x = ((double) i)/(specx-1) * 2 - 1;
	for(j=0; j<specy; j++)
	{
	    y = ((double) j)/(specy-1) * 2 -1;
	    r = sqrt(x*x + y*y);
	    t = (r > 0) ? atan2(y,x) : 0 ;
	    if (r < 0.1 || (r >= 0.95 && r < 1.01)
		|| (r > 0.15 && r < 0.9 && sin(t*3) > 0))
		spec2d[i][j] = (int) (r*300) + 1;
	    else
		spec2d[i][j] = 0;
    }
  }
    specMax = 303;
    strcpy(spectrumName, "startup");
    strcpy(windowName, "slice");

}

static void getIconSize(Xv_opaque obj)
{
    Display *dpy;
    XIconSize *isp;

    dpy = (Display *) xv_get(obj, XV_DISPLAY);
    if (XGetIconSizes(dpy, DefaultRootWindow(dpy), &isp, &iconWidth) == 0)
	iconWidth = iconHeight = 64;
    else
    {
	iconWidth = isp->max_width;
	iconHeight = isp->max_height;
	while((iconWidth != iconHeight || iconWidth > 64) &&
	      iconWidth >= isp->min_width && iconHeight >= isp->min_height)
	{
	    if (iconWidth > iconHeight)
		iconWidth -= isp->width_inc;
	    else
		iconHeight -= isp->height_inc;
	}
	if (iconWidth != iconHeight || iconWidth > 64 ||
	    iconWidth < isp->min_width || iconHeight < isp->min_height)
	    iconWidth = iconHeight = 64;
    }
    if ((iconBits = malloc(iconWidth*iconHeight)) == NULL)
    {
	perror("sort_spec2d");
	exit(1);
    }
}
    
/*
 * Instance XV_KEY_DATA key.  An instance is a set of related
 * user interface objects.  A pointer to an object's instance
 * is stored under this key in every object.  This must be a
 * global variable.
 */
Attr_attribute	INSTANCE;

int main(int argc, char **argv)
{
      int i, piped = 0;

      programName = *argv;
/* check options*/
      for(i=1; i<argc; i++)
      {
	    if (!strcmp(argv[i],"-sort"))
	    {
		  sort_window = 1;
		  fprintf(stderr,"sort_spec2d started\n");
	    }
  	    else if (!strcmp(argv[i],"-dbx"))
	    {
		  DBX_val = (int) atof(argv[++i]);
		  fprintf(stderr,"sort_spec2d: diagnostics level set to %d\n",
			  DBX_val);
	    } 
	    else if (!strcmp(argv[i], "-pipe"))
		piped = 1;
      }
/*
 * Initialize XView.
 */
      xv_init(XV_INIT_ARGC_PTR_ARGV, &argc, argv, NULL);
      INSTANCE = xv_unique_key();

/* initialize the signal handler for SIGUSR2 */
      notify_set_signal_func(sort_client, sort_sigusr2_handler, SIGUSR2,
			     NOTIFY_SYNC);
      notify_set_destroy_func(sort_client, destroy_func);

      if (sort_window)
      {
#ifdef SVR4
	  sigset(SIGINT, SIG_IGN);
#else
	  signal(SIGINT, SIG_IGN);
#endif
      }
      
      Spec_specWindow = spec_specWindow_objects_initialize(NULL, NULL);
      Spec_contourPopup =
	  spec_contourPopup_objects_initialize(NULL,
					       Spec_specWindow->specWindow);
      Spec_levelPopup =
	  spec_levelPopup_objects_initialize(NULL,
					     Spec_specWindow->specWindow);
      Spec_gammaPopup =
	  spec_gammaPopup_objects_initialize(NULL,
					     Spec_specWindow->specWindow);
      Spec_rgbPopup =
	  spec_rgbPopup_objects_initialize(NULL, Spec_specWindow->specWindow); 
      Spec_printerPopup =
	  spec_printerPopup_objects_initialize(NULL,
					       Spec_specWindow->specWindow);
      Spec_projectPopup =
	  spec_projectPopup_objects_initialize(NULL,
					       Spec_specWindow->specWindow);
      Spec_customPopup =
	  spec_customPopup_objects_initialize(NULL,
					      Spec_specWindow->specWindow);
      Spec_3dwindow =
	  spec_3dwindow_objects_initialize(NULL, Spec_specWindow->specWindow);

      getIconSize(Spec_specWindow->specWindow);

      for(i=MAXSCALE*MAXSCALE+1; i--; )
      {
	  dither[i] = 0;
	  XVdither[i] = XV_NULL;
      }

      initialiseColourmap(Spec_contourPopup->contourControls);
      if (!monochrome)
      {
	  initialiseColourmap(Spec_contourPopup->contourPopup);
	  initialiseColourmap(Spec_specWindow->specWindow);
	  initialiseColourmap(Spec_specWindow->spectrumControls);
	  initialiseColourmap(Spec_specWindow->spectrum);
      }
      initialiseContourObjects();
      if (piped)
	  readSpectrumFromPipe();
      else
      {
	  testSpectrum();	
	  spectrumLoaded();
	  contourLevelDefaultSet();
	  setMagnification(4);
      }
      setContourImages();
      
      setContourArray();
      while(iconLine < iconHeight)
	  recalculateIcon();
      
      xv_set(Spec_specWindow->spectrum,
	     CANVAS_X_PAINT_WINDOW, TRUE, 
	     CANVAS_RETAINED, TRUE,
	     CANVAS_AUTO_EXPAND, FALSE,
	     CANVAS_AUTO_SHRINK, FALSE,
	     CANVAS_AUTO_CLEAR, FALSE,
	     NULL);
      xv_set(canvas_paint_window(Spec_specWindow->spectrum),
	     WIN_IGNORE_EVENTS, WIN_UP_EVENTS, NULL,
	     WIN_CONSUME_EVENTS, LOC_MOVE, LOC_DRAG, NULL,
	     NULL);
      xv_set(Spec_specWindow->specWindow, WIN_EVENT_PROC, event_proc, NULL);
      if (cmsType == XV_DYNAMIC_CMS)
      {
	  xv_set(Spec_gammaPopup->displayGammaSlider,
		 PANEL_NOTIFY_LEVEL, PANEL_ALL, NULL);
	  xv_set(Spec_gammaPopup->printoutGammaSlider,
		 PANEL_NOTIFY_LEVEL, PANEL_ALL, NULL);
	  xv_set(Spec_rgbPopup->redSlider,
		 PANEL_NOTIFY_LEVEL, PANEL_ALL, NULL);
	  xv_set(Spec_rgbPopup->greenSlider,
		 PANEL_NOTIFY_LEVEL, PANEL_ALL, NULL);
      xv_set(Spec_rgbPopup->blueSlider,
	     PANEL_NOTIFY_LEVEL, PANEL_ALL, NULL); 
      }
      xv_set(Spec_projectPopup->projectAngleSlider, PANEL_NOTIFY_LEVEL,
	     PANEL_ALL, NULL); 
      xv_set(Spec_specWindow->specWindow,
	     FRAME_LEFT_FOOTER, "Sort_spec2d version 1.07", NULL);
      if (monochrome)
	  xv_set(Spec_specWindow->specWindow,
		 FRAME_RIGHT_FOOTER, "Monochrome display - Yuk !", NULL);

      if (sort_window)
	  xv_set(Spec_specWindow->specWindow, FRAME_CLOSED, TRUE, NULL);
      
      xv_main_loop(Spec_specWindow->specWindow);
      return(0);
}

void restartIcon(void)
{
    iconLine = 0;
    iconTimer.it_value.tv_usec = iconTimer.it_interval.tv_usec = 100;
    iconTimer.it_value.tv_sec = iconTimer.it_interval.tv_sec = 0;
    notify_set_itimer_func(Spec_specWindow->specWindow,
			       recalculateIcon, ITIMER_REAL, &iconTimer, NULL);
}

static Notify_value
recalculateIcon(void)
{
    int i, k, u, t, n, x, y;
    unsigned char *p;
    double specxstep, specystep;
    int monoicon = (cms == XV_NULL) ||
	(xv_get(Spec_specWindow->specWindow, WIN_DEPTH) != 8);

    if (iconLine == iconHeight)
    {
	notify_set_itimer_func(Spec_specWindow->specWindow,
				   NOTIFY_FUNC_NULL, ITIMER_REAL, NULL, NULL);
	return NOTIFY_DONE;
    }

    specxstep = ((double) specx)/iconWidth;
    specystep = ((double) specy)/iconHeight;

    if (monoicon)
    {
	p = iconBits + iconLine*((iconWidth+7)>>3);

	u = 1;
	*p = 0;
	for(i=0; i<iconWidth; i++)
	{
	    n = t = 0;
	    for(x=i*specxstep; x<(i+1)*specxstep; x++)
		for(y=iconLine*specystep; y<(iconLine+1)*specystep; y++)
		{
		    t += spec2d[x][y];
		    n++;
		}
	    t /= n;
	    if (t > 0)
		*p |= u;
	    u <<= 1;
	    if (u > 128)
	    {
		*++p = 0;
		u = 1;
	    }
	}
    }
    else
    {
	p = iconBits + iconLine*iconWidth;
	for(i=0; i<iconWidth; i++)
	{
	    n = t = 0;
	    for(x=i*specxstep; x<(i+1)*specxstep; x++)
		for(y=iconLine*specystep; y<(iconLine+1)*specystep; y++)
		{
		    t += spec2d[x][y];
		    n++;
		}
	    t /= n;
	    for(k=usedContours-1; k;)
		if (t > contourValue[--k])
		{
		    k++;
		    break;
		}
	    *p++ = ((k == usedContours-1) ? MAXCONTOURS-1 : k) +
		COLOUROFFSET;
	}
    }

    if (++iconLine == iconHeight)
    {
	notify_set_itimer_func(Spec_specWindow->specWindow,
				   NOTIFY_FUNC_NULL, ITIMER_REAL, NULL,
				   NULL);
	if (!iconBitmap)
	    if (monoicon)
		iconBitmap = xv_create(XV_NULL, SERVER_IMAGE,
				       SERVER_IMAGE_DEPTH, 1,
				       SERVER_IMAGE_X_BITS, iconBits,
				       XV_WIDTH, iconWidth,
				       XV_HEIGHT, iconHeight,
				       NULL);
	    else
		iconBitmap = xv_create(XV_NULL, SERVER_IMAGE,
				      SERVER_IMAGE_DEPTH, 8,
				      SERVER_IMAGE_X_BITS, iconBits,
				      XV_WIDTH, iconWidth,
				      XV_HEIGHT, iconHeight,
				      SERVER_IMAGE_COLORMAP, MYCMS,
				      NULL);
	else
	    xv_set(iconBitmap, SERVER_IMAGE_X_BITS, iconBits, NULL);
	if (!iconIcon)
	    iconIcon = xv_create(XV_NULL, ICON, ICON_IMAGE, iconBitmap, NULL);
	else
	    iconIcon = xv_set(iconIcon, ICON_IMAGE, iconBitmap, NULL);
	xv_set(Spec_specWindow->specWindow, FRAME_ICON, iconIcon, NULL);
    }
    
    return NOTIFY_DONE;
}

void redrawSpectrum(void)
{
    repaintSpectrum(
	XV_NULL,
	XV_NULL,
	(Display *) XV_DISPLAY_FROM_WINDOW(Spec_specWindow->specWindow),
	(Window) xv_get(canvas_paint_window(Spec_specWindow->spectrum), 
			XV_XID),
	(Xv_xrectlist *) NULL);
/*
  XClearArea(XV_DISPLAY_FROM_WINDOW(Spec_specWindow->specWindow),
    (Window) xv_get(canvas_paint_window(Spec_specWindow->spectrum),
	       XV_XID), 0, 0, 0, 0, True);
*/
}

GC gc = NULL;
static struct windowPointList *windowPoints = NULL, *windowTail = NULL;
static int winX, winY;

static void
repaintRectangle(Display *display, Window xid, long unsigned int *index,
		 long unsigned int wincol, int minx, int miny, int maxx,
		 int maxy, int uselast)
{
    int x, y, ystart, oldcont, i;
    static char lastredraw[MAXSPECX][MAXSPECY];
    static int lastlevels = -1;
    XImage *line;
    Pixmap lbuf = None;
    XWindowAttributes wa;
    GC lgc;

    if (monochrome*usedContours != lastlevels)
    {
	forceRedraw = 1;
	lastlevels = monochrome*usedContours;
    }
    if (forceRedraw == 1 || (windowStatus == WINDOW_SET && forceRedraw != 2))
	uselast = 0;

    if (minx < 0)
	minx = 0;
    if (miny < 0)
	miny = 0;
    if (maxx > specx)
	maxx = specx;
    if (maxy > specy)
	maxy = specy;

    XGetWindowAttributes(display, xid, &wa);
/*
    fprintf(stderr, "Image at %d,%d size %dx%d\n", minx*mag, miny*mag,
	    (maxx-minx)*mag, (maxy-miny)*mag);
*/
    if (!monochrome &&
	(line = XCreateImage(display, wa.visual, wa.depth, ZPixmap, 0,
			     NULL, (maxx-minx)*mag, 1, 8, 0)) != NULL)
    {
	if (mag > 1)
	{
	    lbuf = XCreatePixmap(display, xid,  (maxx-minx)*mag, 1, wa.depth);
	    lgc =  XCreateGC(display, xid, 0, NULL);
	    XSetFunction(display, lgc, GXcopy);
	}

/*
	fprintf(stderr, "Bytes per line = %d, width = %d\n",
		line->bytes_per_line, (maxx-minx)*mag);
*/
	if ((line->data = malloc(line->bytes_per_line*wa.depth)) == NULL)
	    XDestroyImage(line);
	else
	{
	    int xl = minx*mag, xw = (maxx - minx)*mag, xp;
	    lastlevels = -1;
/*
	    fprintf(stderr, "Using new redraw.\n");
*/
	    ystart = miny*mag;
	    for(y = miny; y < maxy; y++)
	    {
		xp = 0;

		switch(windowStatus)
		{
		case WINDOW_NONE:
		case WINDOW_SET:
		{
		    register unsigned char (*sp)[MAXSPECY] =
			(unsigned char (*)[MAXSPECY]) &spec2dcontour[minx][y];
		    register long col;
		    
		    for(x = maxx - minx; x--; sp++)
		    {
			col = index[**sp];
			for(i = mag; i--; )
			    XPutPixel(line, xp++, 0, col);
		    }
		}
		break;
		case WINDOW_OUTSIDE:
		{
		    register unsigned char (*sp)[MAXSPECY] =
			(unsigned char (*)[MAXSPECY]) &spec2dcontour[minx][y];
		    register unsigned char (*wp)[MAXSPECY] =
			(unsigned char (*)[MAXSPECY]) &window[minx][y];
		    register long col;
			
		    for(x = maxx - minx; x--; sp++)
		    {
			col = (*(*wp++)) ? wincol : index[**sp];
			for(i = mag; i--; )
			    XPutPixel(line, xp++, 0, col);
		    }
		}
		break;
		case WINDOW_INSIDE:
		{
		    register unsigned char (*sp)[MAXSPECY] =
			(unsigned char (*)[MAXSPECY]) &spec2dcontour[minx][y];
		    register unsigned char (*wp)[MAXSPECY] =
			(unsigned char (*)[MAXSPECY]) &window[minx][y];
		    register long col;
			
		    for(x = maxx - minx; x--; sp++)
		    {
			col = (*(*wp++)) ? index[**sp] : wincol;
			for(i = mag; i--; )
			    XPutPixel(line, xp++, 0, col);
		    }
		}
		break;
		case WINDOW_ONLY:
		{
		    register unsigned char (*wp)[MAXSPECY] =
			(unsigned char (*)[MAXSPECY]) &window[minx][y];
		    register long col;
			
		    for(x = maxx - minx; x--; wp++)
		    {
			col = (**wp) ? index[MAXCONTOURS+COLOUROFFSET-1] :
			    index[COLOUROFFSET];
			for(i = mag; i--; )
			    XPutPixel(line, xp++, 0, col);
		    }
		}
		break;
		}
		if (mag == 1)
		    XPutImage(display, xid, gc, line, 0, 0, xl, ystart++,
			      xw, 1);
		else
		{
		    XPutImage(display, lbuf, lgc, line, 0, 0, 0, 0, xw, 1);
		    for(i=0; i<mag; i++)
			XCopyArea(display, lbuf, xid, gc, 0, 0,
				  xw, 1, xl, ystart++);
		}
	    }
	    if (mag > 1)
	    {
		XFreeGC(display, lgc);
		XFreePixmap(display, lbuf);
	    }
	    XDestroyImage(line);

	    return;
	}
    }

/*
    fprintf(stderr, "Using old redraw.\n");
*/
    for(x = minx; x < maxx; x++)
    {
	ystart = -1;
	oldcont = -1;
	for(y = miny; y < maxy; y++)
	{
	    register int newcont;

	    switch (windowStatus)
	    {
	    case WINDOW_NONE:
	    case WINDOW_SET:
		newcont = spec2dcontour[x][y];
		break;
	    case WINDOW_OUTSIDE:
		newcont = (window[x][y]) ? 127 : spec2dcontour[x][y];
		break;
	    case WINDOW_INSIDE:
		newcont = (window[x][y]) ? spec2dcontour[x][y] : 127;
		break;
	    case WINDOW_ONLY:
		newcont = (window[x][y]) ? MAXCONTOURS+COLOUROFFSET-1 :
		    COLOUROFFSET;
		break;
	    }
	    if (uselast && newcont == lastredraw[x][y])
		newcont = -1;
	    if (newcont != -1)
		lastredraw[x][y] = (char) newcont;
	    if (newcont != oldcont && oldcont != -1)
	    {
		if (monochrome)
		{
		    i = ((oldcont-COLOUROFFSET)*(mag*mag)
			 +usedContours/2)/(usedContours-1);
		    if (i > mag*mag)
			i = mag*mag;
/*
		    fflush(stderr);
 */
		    XSetStipple(display, gc, (oldcont == 127) ? windither
				: dither[i]);
		}
		else
		    XSetForeground(display, gc, (oldcont == 127) ? wincol
				   : index[oldcont]);
		XFillRectangle(display, xid, gc, x*mag, ystart*mag, mag,
			       (y-ystart)*mag);
	    }
	    if (newcont != oldcont || ystart == -1)
	    {
		oldcont = newcont;
		ystart = y;
	    }
	}
	if (oldcont != -1)
	{
	    if (monochrome)
	    {
		i = ((oldcont-COLOUROFFSET)*(mag*mag)+usedContours/2)
		    /(usedContours-1);
		if (i > mag*mag)
		    i = mag*mag;
		XSetStipple(display, gc, (oldcont == 127) ? windither
			    : dither[i]);
	    }
	    else
		XSetForeground(display, gc, (oldcont == 127) ? wincol :
			       index[oldcont]);
	    XFillRectangle(display, xid, gc, x*mag, ystart*mag, mag,
			   (y-ystart)*mag);
	}
    }
}

/*ARGSUSED*/
void
repaintSpectrum(Canvas canvas, Xv_window paint_window, Display *display,
		Window xid, Xv_xrectlist *rects)
{
    int i;
    XRectangle *r;
    XGCValues gc_val;
    unsigned long *index, wincol;
    struct windowPointList *p = windowPoints;
    char spectrum_label[80];

    switch(windowStatus)
    {
    case WINDOW_NONE:
	strcpy(spectrum_label,spectrumName);
	break;
    case WINDOW_INSIDE:
	sprintf(spectrum_label," %s inside %s",spectrumName, windowName);
	break;
    case WINDOW_OUTSIDE:
	sprintf(spectrum_label, "%s outside %s",spectrumName, windowName);
	break;
    case WINDOW_ONLY:
	sprintf(spectrum_label, "%s on %s",windowName, spectrumName);
	break;
    case WINDOW_SET:
        sprintf(spectrum_label, "Setting %s on %s", windowName, spectrumName);
        break;
    }

    xv_set(Spec_specWindow->specWindow, FRAME_LABEL, spectrum_label, NULL);

    XSetFunction(display, gc, GXcopy);

    if (rects)
    {
	XSetClipRectangles(display, gc, 0, 0, rects->rect_array,
			   rects->count, Unsorted);
    }
    else
    {
	gc_val.clip_mask = None;
	XChangeGC(display, gc, GCClipMask, &gc_val);
    }

    if (showGuidelines)
	drawGuidelines(display, xid, gc);
    if (showCrosshairs)
	drawCrosshairs(display, xid, gc, 0, 0, 0);

    if (monochrome)
    {
	XSetForeground(display, gc, BlackPixel(display,
					       DefaultScreen(display)));
	XSetBackground(display, gc, WhitePixel(display,
					       DefaultScreen(display)));
	XSetFillStyle(display, gc, FillOpaqueStippled);
	index = NULL;
    }
    else
    {
	XSetFillStyle(display, gc, FillSolid);
	index = (unsigned long *) xv_get(cms, CMS_INDEX_TABLE);
	wincol = index[COLOUROFFSET+MAXCONTOURS];
    }

    if (rects)
    {
/*
	fprintf(stderr, "repaintSpectrum rectangle list:");
*/
	i = rects->count;
	r = rects->rect_array;
	while(i--)
	{
/*
	    fprintf(stderr, " %dx%d+%d+%d", r->width, r->height, r->x, r->y);
*/
	    repaintRectangle(display, xid, index, wincol, (r->x-1)/mag,
			     (r->y-1)/mag, (r->x+r->width+mag-2)/mag,
			     (r->y+r->height+mag-2)/mag, 0);
	    r++;
	}
/*
	putc('\n', stderr);
*/
    }
    else
    {
/*
	fprintf(stderr, "repaintSpectrum full\n");
*/
	repaintRectangle(display, xid, index, wincol, 0, 0, specx, specy, 1);
    }
    
    if (monochrome)
	XSetFillStyle(display, gc, FillSolid);

    if (windowStatus == WINDOW_SET && p != NULL)
    {
	if (monochrome)
	{
	    XSetForeground(display, gc,
			   BlackPixel(display, DefaultScreen(display)) ^
			   WhitePixel(display, DefaultScreen(display)));
	    XSetFunction(display, gc, GXxor);
	}
	else
	    XSetForeground(display, gc, wincol);

        while(p->next != NULL)
        {
            XFillRectangle(display, xid, gc, p->x*mag, p->y*mag, 4, mag);
            XDrawLine(display, xid, gc, p->x*mag+mag/2, p->y*mag+mag/2,
		      p->next->x*mag+mag/2, p->next->y*mag+mag/2);
            p = p->next;
        }
        XFillRectangle(display, xid, gc, p->x*mag, p->y*mag, mag, mag);

        if (windowTail != NULL)
        {
            XSetFunction(display, gc, GXxor);
	    if (!monochrome)
		XSetForeground(display, gc, index[COLOUROFFSET+MAXCONTOURS-1]
			       ^ index[COLOUROFFSET]);
            XDrawLine(display, xid, gc, windowTail->x*mag+mag/2,
		      windowTail->y*mag+mag/2, winX*mag+mag/2, winY*mag+mag/2);
        }

    }

    if (showGuidelines)
	drawGuidelines(display, xid, gc);
    if (showCrosshairs)
	drawCrosshairs(display, xid, gc, 0, 0, 0);
	
    forceRedraw = 0;
}

Menu_item
setWindow(Menu_item item, Menu_generate op)
{
    Xv_opaque menu;

    if (op != MENU_NOTIFY)
	return item;

/* window menu */
    menu = xv_get(Spec_specWindow->windowButton, PANEL_ITEM_MENU);
    xv_set(xv_get(menu, MENU_NTH_ITEM, 1), MENU_INACTIVE, TRUE, NULL);
    xv_set(xv_get(menu, MENU_NTH_ITEM, 2), MENU_INACTIVE, TRUE, NULL);
    xv_set(xv_get(menu, MENU_NTH_ITEM, 3), MENU_INACTIVE, TRUE, NULL);
    xv_set(xv_get(menu, MENU_NTH_ITEM, 4), MENU_INACTIVE, TRUE, NULL);
    xv_set(xv_get(menu, MENU_NTH_ITEM, 5), MENU_INACTIVE, TRUE, NULL);
    xv_set(xv_get(menu, MENU_NTH_ITEM, 6), MENU_INACTIVE, TRUE, NULL);
    xv_set(xv_get(menu, MENU_NTH_ITEM, 7), MENU_INACTIVE, FALSE, NULL);

/* slice menu 16/1/95 */
    menu = xv_get(Spec_specWindow->sliceButton, PANEL_ITEM_MENU);
    xv_set(xv_get(menu, MENU_NTH_ITEM, 3), MENU_INACTIVE, TRUE, NULL);
    xv_set(xv_get(menu, MENU_NTH_ITEM, 4), MENU_INACTIVE, TRUE, NULL);
    xv_set(xv_get(menu, MENU_NTH_ITEM, 5), MENU_INACTIVE, FALSE, NULL);
    xv_set(xv_get(menu, MENU_NTH_ITEM, 7), MENU_INACTIVE, TRUE, NULL);
    xv_set(xv_get(menu, MENU_NTH_ITEM, 8), MENU_INACTIVE, TRUE, NULL);
    xv_set(xv_get(menu, MENU_NTH_ITEM, 9), MENU_INACTIVE, TRUE, NULL);

    windowStatus = WINDOW_SET;
    forceRedraw = 2;
    repaintSpectrum(XV_NULL, XV_NULL, (Display *)
		    XV_DISPLAY_FROM_WINDOW(Spec_specWindow->specWindow),
	    (Window) xv_get(canvas_paint_window(Spec_specWindow->spectrum),
			    XV_XID), (Xv_xrectlist *) NULL);

    if (windowPoints != NULL)
    {
        freeWindowList(windowPoints);
        windowPoints = NULL;
    }
    windowTail = NULL;
    xv_set(Spec_specWindow->specWindow,
	   FRAME_LEFT_FOOTER, "LEFT: add point   MIDDLE: delete last point   "
	   "RIGHT: finish", FRAME_RIGHT_FOOTER, "", NULL);
    if (showCrosshairs)
    {
	drawCrosshairs((Display *)
		       XV_DISPLAY_FROM_WINDOW(Spec_specWindow->specWindow),
		       (Window)
		       xv_get(canvas_paint_window(Spec_specWindow->spectrum),
			      XV_XID), gc, 0, 0, 0);
	showCrosshairs = 0;
    }

    return item;
}

Menu_item
showInsideWindow(Menu_item item, Menu_generate op)
{
    if (op == MENU_NOTIFY)
    {
	windowStatus = WINDOW_INSIDE;
	repaintSpectrum(XV_NULL, XV_NULL, (Display *)
			XV_DISPLAY_FROM_WINDOW(Spec_specWindow->specWindow),
			(Window) 
			xv_get(canvas_paint_window(Spec_specWindow->spectrum),
			       XV_XID), (Xv_xrectlist *) NULL);
    }

    return item;
}

Menu_item
showOutsideWindow(Menu_item item, Menu_generate op)
{
    if (op == MENU_NOTIFY)
    {
	windowStatus = WINDOW_OUTSIDE;
	repaintSpectrum(XV_NULL, XV_NULL, (Display *)
			XV_DISPLAY_FROM_WINDOW(Spec_specWindow->specWindow),
			(Window)
			xv_get(canvas_paint_window(Spec_specWindow->spectrum),
			       XV_XID), (Xv_xrectlist *) NULL);
    }

    return item;
}

Menu_item
showWindow(Menu_item item, Menu_generate op)
{
    if (op == MENU_NOTIFY)
    {
	windowStatus = WINDOW_ONLY;
	repaintSpectrum(XV_NULL, XV_NULL, (Display *)
			XV_DISPLAY_FROM_WINDOW(Spec_specWindow->specWindow),
			(Window)
			xv_get(canvas_paint_window(Spec_specWindow->spectrum),
			       XV_XID), (Xv_xrectlist *) NULL);
    }

    return item;
}

Menu_item
clearWindow(Menu_item item, Menu_generate op)
{
    int i, j;

    if (op == MENU_NOTIFY)
    {
	for(i=0; i<specx; i++)
	    for(j=0; j<specy; j++)
		window[i][j] = 0;

	repaintSpectrum(XV_NULL, XV_NULL, (Display *)
			XV_DISPLAY_FROM_WINDOW(Spec_specWindow->specWindow),
			(Window)
			xv_get(canvas_paint_window(Spec_specWindow->spectrum),
			       XV_XID), (Xv_xrectlist *) NULL);
    }

    return item;
}

Menu_item
showWithoutWindow(Menu_item item, Menu_generate op)
{
    if (op == MENU_NOTIFY)
    {
	windowStatus = WINDOW_NONE;
	repaintSpectrum(XV_NULL, XV_NULL, (Display *)
			XV_DISPLAY_FROM_WINDOW(Spec_specWindow->specWindow),
			(Window)
			xv_get(canvas_paint_window(Spec_specWindow->spectrum),
			       XV_XID), (Xv_xrectlist *) NULL);
    }

    return item;
}

Menu_item
abortSetWindow(Menu_item item, Menu_generate op)
{
    Xv_opaque menu;

    if (op != MENU_NOTIFY)
	return item;

/* window menu */    
    menu = xv_get(Spec_specWindow->windowButton, PANEL_ITEM_MENU);
    xv_set(xv_get(menu, MENU_NTH_ITEM, 1), MENU_INACTIVE, FALSE, NULL);
    xv_set(xv_get(menu, MENU_NTH_ITEM, 2), MENU_INACTIVE, FALSE, NULL);
    xv_set(xv_get(menu, MENU_NTH_ITEM, 3), MENU_INACTIVE, FALSE, NULL);
    xv_set(xv_get(menu, MENU_NTH_ITEM, 4), MENU_INACTIVE, FALSE, NULL);
    xv_set(xv_get(menu, MENU_NTH_ITEM, 5), MENU_INACTIVE, FALSE, NULL);
    xv_set(xv_get(menu, MENU_NTH_ITEM, 6), MENU_INACTIVE, FALSE, NULL);
    xv_set(xv_get(menu, MENU_NTH_ITEM, 7), MENU_INACTIVE, TRUE, NULL);

/* slice menu 16/1/95 */
    menu = xv_get(Spec_specWindow->sliceButton, PANEL_ITEM_MENU);
    xv_set(xv_get(menu, MENU_NTH_ITEM, 3), MENU_INACTIVE, FALSE, NULL);
    xv_set(xv_get(menu, MENU_NTH_ITEM, 4), MENU_INACTIVE, FALSE, NULL);
    xv_set(xv_get(menu, MENU_NTH_ITEM, 5), MENU_INACTIVE, TRUE, NULL);
    xv_set(xv_get(menu, MENU_NTH_ITEM, 7), MENU_INACTIVE, FALSE, NULL);
    xv_set(xv_get(menu, MENU_NTH_ITEM, 8), MENU_INACTIVE, FALSE, NULL);
    xv_set(xv_get(menu, MENU_NTH_ITEM, 9), MENU_INACTIVE, FALSE, NULL);

    windowStatus = WINDOW_OUTSIDE;
    forceRedraw = 1;
    repaintSpectrum(XV_NULL, XV_NULL, (Display *)
		    XV_DISPLAY_FROM_WINDOW(Spec_specWindow->specWindow),
	    (Window) xv_get(canvas_paint_window(Spec_specWindow->spectrum),
			    XV_XID), (Xv_xrectlist *) NULL);

    if (windowPoints != NULL)
    {
        freeWindowList(windowPoints);
        windowPoints = NULL;
    }
    windowTail = NULL;
    Slicing &= (~PROJECTION);

    if (Slicing & SLICING)
    {
	drawCrosshairs((Display *)
		       XV_DISPLAY_FROM_WINDOW(Spec_specWindow->specWindow),
		       (Window)
		       xv_get(canvas_paint_window(Spec_specWindow->spectrum),
			      XV_XID), gc, 0, 0, 0);
	showCrosshairs = 1;
    }
    
    xv_set(Spec_specWindow->specWindow,
	   FRAME_LEFT_FOOTER, "Window set abandoned",
	   FRAME_RIGHT_FOOTER, (Slicing & SLICING) ? "Slicing on":"", NULL);

    return item;
}

Notify_value
rubberBand(Xv_window win, Event *event, Notify_arg arg, Notify_event_type type)
{
    unsigned long *index;
    Display *display = XV_DISPLAY_FROM_WINDOW(win);
    Window xid = xv_get(win, XV_XID);
    int move = ((event_action(event) == ACTION_SELECT ||
		 event_action(event) == ACTION_ADJUST ||
		 event_action(event) == ACTION_MENU) &&
		event_is_down(event)) || event_action(event) == LOC_MOVE ||
		    event_action(event) == LOC_DRAG,
		    newX, newY;

    static char coordinates[32];

    if (move)
    {
	newX = (event_x(event)-1)/mag;
	newY = (event_y(event)-1)/mag;

	if (newX < specx && newY < specy && newX >= 0 && newY >= 0)
	{
	    sprintf(coordinates, "(%d, %d) = %d", newX, specy-newY-1,
		    spec2d[newX][newY]);
	    xv_set(Spec_specWindow->coordinates,
		   PANEL_LABEL_STRING, coordinates, NULL);
	}
    }

    if (gc == NULL)
	gc = XCreateGC(display, xid, 0, NULL);

    XSetClipMask(display, gc, None);

    if (windowStatus == WINDOW_SET && move)
    {
	if (monochrome)
	    XSetForeground(display, gc,
			   BlackPixel(display, DefaultScreen(display)) ^
			   WhitePixel(display, DefaultScreen(display)));
	else
	    index = (unsigned long *) xv_get(cms, CMS_INDEX_TABLE);
	
        if (windowTail != NULL)
        {
	    if (!monochrome)
		XSetForeground(display, gc, index[COLOUROFFSET+MAXCONTOURS-1]
			       ^ index[COLOUROFFSET]);
            XSetFunction(display, gc, GXxor);
            XDrawLine(display, xid, gc, windowTail->x*mag+mag/2,
		      windowTail->y*mag+mag/2, winX*mag+mag/2, winY*mag+mag/2);
            XSetFunction(display, gc, GXcopy);
        }
	
	if (!monochrome)
	    XSetForeground(display, gc, index[COLOUROFFSET+MAXCONTOURS]);
	
        winX = newX;
        winY = newY;
    }

    if (event_action(event) == ACTION_SELECT)
	newPoint(win, event, arg, type);
    
    if (event_action(event) == ACTION_ADJUST)
	deletePoint(win, event, arg, type);
    
    if (event_action(event) == ACTION_MENU)
	closeWindow(win, event, arg, type);
    
    if (windowStatus == WINDOW_SET && windowTail != NULL && move)
    {
	if (monochrome)
	    XSetForeground(display, gc,
			   BlackPixel(display, DefaultScreen(display)) ^
			   WhitePixel(display, DefaultScreen(display)));
	else
	{
	    index = (unsigned long *) xv_get(cms, CMS_INDEX_TABLE);
	    XSetForeground(display, gc, index[COLOUROFFSET+MAXCONTOURS-1] ^
			   index[COLOUROFFSET]);
	}
	XSetFunction(display, gc, GXxor);
	XDrawLine(display, xid, gc, windowTail->x*mag+mag/2,
		  windowTail->y*mag+mag/2, winX*mag+mag/2, winY*mag+mag/2);
	XSetFunction(display, gc, GXcopy);
    }

    if (showCrosshairs)
    {
	drawCrosshairs(display, xid, gc, 0, 0, 0);
	drawCrosshairs(display, xid, gc, 1, event_x(event), event_y(event));
    }

    /*** 16/1/95 ***/
    if ( (Slicing & SLICING) && windowStatus != WINDOW_SET) {
	if (event_is_down(event) && event_action(event) ==  ACTION_SELECT)
	    do_simple_slice(event, newX, newY);
    }

    return notify_next_event_func(win, (Notify_event) event, arg, type);
}

/*ARGSUSED*/
static void
newPoint(Xv_window win, Event *event, Notify_arg arg, Notify_event_type type)
{
    Display *display = XV_DISPLAY_FROM_WINDOW(win);
    Window xid = xv_get(win, XV_XID);

    if (windowStatus != WINDOW_SET || !event_is_down(event))
        return;

    if (monochrome)
	XSetFunction(display, gc, GXxor);
    XFillRectangle(display, xid, gc, winX*mag, winY*mag, mag, mag);
    if (windowTail != NULL)
    {
        XDrawLine(display, xid, gc, windowTail->x*mag+mag/2,
		  windowTail->y*mag+mag/2, winX*mag+mag/2, winY*mag+mag/2);
        windowTail = windowTail->next = (struct windowPointList *)
            malloc(sizeof(struct windowPointList));
    }
    else
        windowTail = windowPoints = (struct windowPointList *)
            malloc(sizeof(struct windowPointList));
    windowTail->x = winX;
    windowTail->y = winY;
    windowTail->next = NULL;
}

/*ARGSUSED*/
static void
deletePoint(Xv_window win, Event *event, Notify_arg arg,
	    Notify_event_type type)
{
    Xv_xrectlist rects;
    struct windowPointList *p = windowPoints;

    if (windowStatus != WINDOW_SET || !event_is_down(event) ||
	windowTail == NULL)
        return;


    if (p == windowTail)
    {
	rects.count = 1;
	rects.rect_array->x = windowTail->x*mag;
	rects.rect_array->y = windowTail->y*mag;
	rects.rect_array->width = (unsigned short) mag;
	rects.rect_array->height = (unsigned short) mag;

	free(windowPoints);
	windowPoints = NULL;
	windowTail = NULL;
	p = NULL;
    }
    else
    {
	rects.count = 1;
	while(p->next != windowTail)
	    p = p->next;
	rects.rect_array->x = ((windowTail->x < p->x) ? windowTail->x :
			       p->x)*mag;
	rects.rect_array->y = ((windowTail->y < p->y) ? windowTail->y :
			       p->y)*mag;
	rects.rect_array->width = ((windowTail->x > p->x) ? windowTail->x :
				   p->x)*mag - rects.rect_array->x + mag;
	rects.rect_array->height = ((windowTail->y > p->y) ? windowTail->y :
				    p->y)*mag - rects.rect_array->y+mag;

	free(windowTail);
	p->next = NULL;
	windowTail = NULL;
    }

    repaintSpectrum(XV_NULL, XV_NULL, (Display *)
		    XV_DISPLAY_FROM_WINDOW(Spec_specWindow->specWindow),
	    (Window) xv_get(canvas_paint_window(Spec_specWindow->spectrum),
			    XV_XID), &rects);

    windowTail = p;
}

/*ARGSUSED*/
static void
closeWindow(Xv_window win, Event *event, Notify_arg arg,
	    Notify_event_type type)
{
    Pixmap pix;
    Display *display = XV_DISPLAY_FROM_WINDOW(win);
    Window xid = xv_get(win, XV_XID);
    XPoint *points;
    int ps = 0, i, j;
    struct windowPointList *p = windowPoints;
    XImage *filled;
    GC gc1;
    
    if (windowStatus != WINDOW_SET || !event_is_down(event))
        return;

    if (windowPoints == NULL || windowTail == windowPoints)
    {
	abortSetWindow(XV_NULL, MENU_NOTIFY);
	return;
    }

    while(p)
    {
	ps++;
	p = p->next;
    }

    points = (XPoint *) malloc(sizeof(XPoint) * ps);

    ps = 0;

    p = windowPoints;

    while(p)
    {
	points[ps].x = p->x;
	points[ps++].y = p->y;
	p = p->next;
    }

    pix = XCreatePixmap(display, xid, specx, specy, 1);

    gc1 = XCreateGC(display, pix, 0, NULL);

    XSetForeground(display, gc1, 0);

    XSetFillStyle(display, gc1, FillSolid);
    XFillRectangle(display, pix, gc1, 0, 0, specx, specy);

    XSetForeground(display, gc1, 1);
    XSetFillRule(display, gc1, WindingRule);

    XFillPolygon(display, pix, gc1, points, ps, Complex, CoordModeOrigin);
    XDrawLines(display, pix, gc1, points, ps, CoordModeOrigin);
    XDrawLine(display, pix, gc1, windowPoints->x, windowPoints->y,
              windowTail->x, windowTail->y);

    free(points);
    XFreeGC(display, gc1);

    filled = XGetImage(display, pix, 0, 0, specx, specy, 1, XYPixmap);

    for(i=0; i<specx; i++)
	for(j=0; j<specy; j++)
	    window[i][j] = (unsigned char) XGetPixel(filled, i, j);

    XDestroyImage(filled);
    XFreePixmap(display, pix);

    if (Slicing & PROJECTION) {
	  do_projection(event);
	  Slicing &= (~PROJECTION);
    }
    else
	  store_window();

    RRP(0);
    windowChanged3d();
    abortSetWindow(XV_NULL, MENU_NOTIFY);

    xv_set(Spec_specWindow->specWindow,
	   FRAME_LEFT_FOOTER, "Window closed and stored",
	   FRAME_RIGHT_FOOTER, (Slicing & SLICING) ? "Slicing on":"", NULL);
}

static void freeWindowList(struct windowPointList *p)
{
    if (p->next != NULL)
        freeWindowList(p->next);
    free(p);
}

static int win2d;
/*ARGSUSED*/
static Notify_value
sort_sigusr2_handler(Notify_client client, int sig, Notify_signal_mode mode)
{
/* receive this signal from sort process to display 2d spectra */
    int spec_type;
    int x;

    if (DBX_val >= 3)
	fprintf(stderr,"sort_spec2d: received SIGUSR2\n");
      
/* open if in iconic state and set window on top */
    xv_set(Spec_specWindow->specWindow, FRAME_CLOSED, FALSE, NULL);
    wmgr_top(Spec_specWindow->specWindow);
      
/* read data from .SORT_pipe */
    if (read_from_pipe(".SORT_pipe") == SORT_FAIL) {
	fprintf(stderr,"sort_spec2d: Read from sort process failed");
	return NOTIFY_DONE;
    }
    if ( sscanf(pshm->com_str,"%*s %d",&spec_type) != 1) {
	fprintf(stderr,"sort_spec2d: Decoding of display spectrum "
		"command \"%s\" failed.", pshm->com_str);
	return NOTIFY_DONE;
    }
    if (spec_type != 2 && spec_type != 3) {
	fprintf(stderr,"sort_spec2d asked to display non 2d spectrum!");
	return NOTIFY_DONE;
    }

/* clear down old spectrum */
    xv_set(Spec_specWindow->windowButton,PANEL_INACTIVE,TRUE,NULL);
    xv_set(Spec_specWindow->sliceButton,PANEL_INACTIVE,FALSE,NULL);
    win2d = -1;
    memset(spec2d, 0, sizeof(spec2d));
    windowStatus = WINDOW_NONE;
    specMax = 0;

    strcpy(spectrumName,pshm->name);
    specx = (pshm->size > MAXSPECX) ? MAXSPECX : pshm->size;
    specy = specx; /* square spectra to start off with */
      
/*
 * spec2d[0][0] is the pixel on top left hand corner of the display
 * display still adheres to x=0, y=0 being in the bottom left hand corne
 * however
 */

    for(x=0; x < specx ;x++)
    {
	register int *dp = pshm->array + (x+1)*specy - 1;
	register int *sp = spec2d[x];
	register int data, y;

	for(y=0; y < specy ; y++)
	    if ((data = *sp++ = *dp--) > specMax)
		specMax = data;
    }
      
    xv_set(Spec_printerPopup->printTitle, PANEL_VALUE, pshm->name, NULL);

    if (spec_type == 3)
    {
	win2d = pshm->win2d;
	if (read_from_pipe(".SORT_pipe_W") == SORT_FAIL)
	    fprintf(stderr,"sort_spec2d: Read of window from sort process "
		    "failed");
	else if ( sscanf(pshm->com_str,"%*s %d",&spec_type) != 1)
	    fprintf(stderr,"sort_spec2d: Decoding of display window "
		    "command \"%s\" failed.", pshm->com_str);
	else if (spec_type != 3)
	    fprintf(stderr,"sort_spec2d asked to display unexpected window "
		    "type \"%s\".", pshm->com_str);
	else if (pshm->size != specx)
	    fprintf(stderr, "sort_spec2d: Window size does not match "
		    "spectrum size.\n");
	else
	{
	    memset(window, 0, sizeof(window));
	    windowStatus = WINDOW_OUTSIDE;
	    strcpy(windowName, pshm->name);
	    for(x=0; x < specx ;x++)
	    {
		register int *dp = pshm->array + (x+1)*specy - 1;
		register unsigned char *wp = window[x];
		register int y;

		for(y=0; y < specy; y++)
		    *wp++ = *dp-- != 0;
	    }
	    xv_set(Spec_specWindow->windowButton,PANEL_INACTIVE,FALSE,NULL);
	    xv_set(Spec_specWindow->sliceButton,PANEL_INACTIVE,TRUE,NULL);
	}
    }

    spectrumLoaded();
    setContoursAndRedraw();

    return NOTIFY_DONE;
}
/***  16/1/95 ***/
static void
store_window(void)
{
    if (DBX_val >= 3)
	fprintf(stderr,"sort_spec2d: storing 2d window\n");

    if (!sort_window)
	return;

    if (store_spectrum(".SORT_pipe_W", "WINDOW 4", SORT_TRUE) == -1) {
	  fprintf(stderr,"sort_spec2d: failure to write out window\n");
	  return;
    }
      
/* tell the sort process to read the window and store it */
    (void) kill(getppid(),SIGUSR2);
      
    return;
}

/*ARGSUSED*/
static Notify_value
destroy_func(Notify_client client, int sig, Notify_signal_mode mode)
{
      if (DBX_val > 3) fprintf(stderr,"sort_spec2d: received SIGTERM\n");
/*      xv_destroy_safe(Spec_specWindow->specWindow); */
      exit(0); 
#ifdef lint
      return XV_OK;
#endif
}

static float tickspacing(double range)
{
    float ts;

    ts = pow(10.0, floor(log10(range)))*2;
    if (range/ts < 3)
	ts /= 2;
    if (range/ts < 3)
	ts /= 2;
    if (range/ts < 3)
	ts /= 2.5;
    if (range/ts < 3)
	ts /= 2;
    return(ts);
}

Menu_item
autoscaleLinear(Menu_item item, Menu_generate op)
{
    int i, first, step;
    char buffer[16];

    if (op != MENU_NOTIFY)
	return item;

    step = (((float) specMax)/(usedContours-1) + 0.5);
    if (step < 1)
	step = 1;
    first = step;
    sprintf(buffer, "%d", first);
    xv_set(Spec_levelPopup->firstValue, PANEL_VALUE, buffer, NULL);
    xv_set(Spec_levelPopup->spacing, XV_SHOW, FALSE, NULL);
    sprintf(buffer, "%d", step);
    xv_set(Spec_levelPopup->step, PANEL_VALUE, buffer, XV_SHOW, TRUE, NULL);
    sprintf(buffer, "%d", usedContours);
    xv_set(Spec_levelPopup->numContours, PANEL_VALUE, buffer, NULL);
    xv_set(Spec_levelPopup->levelPopup, XV_LABEL, "Set linear contours", NULL);
    xv_set(Spec_levelPopup->levelSetApply, PANEL_NOTIFY_PROC,
	   levelSetLinear, NULL);
    for(i=1; i<usedContours-1; i++)
    {
	xv_set(contourSlider[i], PANEL_VALUE, first, NULL);
	first += step;
    }
    setContoursAndRedraw();
    
    return item;
}

Menu_item
autoscaleLogarithmic(Menu_item item, Menu_generate op)
{
    int i;
    float first=1, spacing;
    char buffer[32];

    if (op != MENU_NOTIFY)
	return item;

    spacing = pow((double) specMax, 1.0/(usedContours-2));
    xv_set(Spec_levelPopup->firstValue, PANEL_VALUE, "1", NULL);
    xv_set(Spec_levelPopup->step, XV_SHOW, FALSE, NULL);
    sprintf(buffer, "%f", spacing);
    xv_set(Spec_levelPopup->spacing, PANEL_VALUE, buffer, XV_SHOW, TRUE, NULL);
    sprintf(buffer, "%d", usedContours);
    xv_set(Spec_levelPopup->numContours, PANEL_VALUE, buffer, NULL);
    xv_set(Spec_levelPopup->levelPopup, XV_LABEL,
	   "Set pseudo-logarithmic contours", NULL);
    xv_set(Spec_levelPopup->levelSetApply, PANEL_NOTIFY_PROC,
	   levelSetPseudo, NULL);
    for(i=1; i<usedContours-1; i++)
    {
	xv_set(contourSlider[i], PANEL_VALUE, (int) first, NULL);
	first *= spacing;
    }
    setContoursAndRedraw();
    
    return item;
}

static int intcompare(const int *i, const int *j)
{
    if (*i > *j)
	return 1;
    if (*i < *j)
	return -1;
    return 0;
}

/* This is almost histogram equalisation, it ignores zeroes */
Menu_item
autoscaleHistogram(Menu_item item, Menu_generate op)
{
    int *p, *hist, x, y, (*xp)[MAXSPECX], *yp;
    
    if (op != MENU_NOTIFY)
	return item;

    if ((hist = calloc(specx*specy, sizeof(int))) == NULL)
    {
	notice_prompt(item, NULL,
/*		      NOTICE_FOCUS_XY, event_x(event), event_y(event), */
		      NOTICE_MESSAGE_STRINGS,
		      "Unable to perform histogram equalisation. Error was",
		      ERRSTRING, NULL,
		      NOTICE_BUTTON_YES, "Ok",
		      NULL);
	return item;
    }
    
    for(p = hist, x = specx, xp = spec2d; x--; xp++)
	for(y = specy, yp = *xp; y--; yp++)
	    if (*yp != 0)
		*p++ = *yp;
    
    x = p - hist;

    qsort(hist, x, sizeof(int),
	  (int (*)(const void *, const void *)) intcompare);

    for(y = 0; y < usedContours-1; y++)
	xv_set(contourSlider[y], PANEL_VALUE, hist[y*x/(usedContours-1)],
	       NULL);

    free(hist);

    setContoursAndRedraw();
    
    return item;
}

static void
setMagnification(int newmag)
{
    int wx, wy, minx, vneeded = FALSE, hneeded = FALSE;

    if (newmag > MAXSCALE)
	newmag = MAXSCALE;

    mag = newmag;

    wx = mag * specx;
    wy = mag * specy;

    xv_set(Spec_specWindow->spectrum,
	   CANVAS_WIDTH, wx, CANVAS_HEIGHT, wy,
	   NULL);

    if (newmag*specx > MAXSPECSIZE)
	hneeded = TRUE;
	
    if (newmag*specy > MAXSPECSIZE)
	vneeded = TRUE;

    if (hneeded)
    {
	wx = MAXSPECSIZE;
	if (Spec_specWindow->hscroll == XV_NULL)
	    Spec_specWindow->hscroll =
		xv_create(Spec_specWindow->spectrum, SCROLLBAR,
			  SCROLLBAR_SPLITTABLE, FALSE,
			  SCROLLBAR_DIRECTION, SCROLLBAR_HORIZONTAL,
			  NULL);
    }
    else
    {
	wx += 2;
	if (Spec_specWindow->hscroll != XV_NULL)
	{
	    xv_destroy(Spec_specWindow->hscroll);
	    Spec_specWindow->hscroll = XV_NULL;
	}
    }

    if (vneeded)
    {
	wy = MAXSPECSIZE;
	if (Spec_specWindow->vscroll == XV_NULL)
	    Spec_specWindow->vscroll =
		xv_create(Spec_specWindow->spectrum, SCROLLBAR,
			  SCROLLBAR_SPLITTABLE, FALSE,
			  SCROLLBAR_DIRECTION, SCROLLBAR_VERTICAL,
			  NULL);
    }
    else
    {
	wy += 2;
	if (Spec_specWindow->vscroll != XV_NULL)
	{
	    xv_destroy(Spec_specWindow->vscroll);
	    Spec_specWindow->vscroll = XV_NULL;
	}
    }

    if (Spec_specWindow->vscroll != XV_NULL)
	wx += xv_get(Spec_specWindow->vscroll, XV_WIDTH);
    if (Spec_specWindow->hscroll != XV_NULL)
	wy += xv_get(Spec_specWindow->hscroll, XV_HEIGHT);

    xv_set(Spec_specWindow->spectrum, XV_WIDTH, wx, XV_HEIGHT, wy, NULL);

    minx = (int) xv_get(Spec_specWindow->magnifyButton, XV_X) +
	(int) xv_get(Spec_specWindow->magnifyButton, XV_WIDTH) + 7;

    xv_set(Spec_specWindow->specWindow, XV_WIDTH, (wx < minx) ? minx : wx,
	   XV_HEIGHT,
	   wy + (int) xv_get(Spec_specWindow->spectrum, XV_Y),
	   NULL);

    if (monochrome)
	createDithers(Spec_specWindow->spectrum);
    forceRedraw = 1;

    /* You must now set reset the contour images */
}

Menu_item
menuSetMagnification(Menu_item item, Menu_generate op)
{
    if (op == MENU_NOTIFY)
    {
	xv_set(Spec_specWindow->spectrum, XV_SHOW, FALSE, NULL);
	setMagnification(atoi((char *) xv_get(item, MENU_STRING)));
	xv_set(Spec_specWindow->spectrum, XV_SHOW, TRUE, NULL);
	redrawSpectrum();
    }
    return item;
}

static void readSpectrumFromPipe(void)
{
    int x, y;
    char buffer[6];

    read(0, (char *) &specx, sizeof(int));
    read(0, (char *) &specy, sizeof(int));
    read(0, (char *) &specMax, sizeof(int));
    for(x = specx; x--; )
	read(0, (char *) spec2d[x], specy*sizeof(int));
    read(0, (char *) &usedContours, sizeof(int));
    for(x = specx; x--; )
	read(0, (char *) window[x], specy);
    read(0, (char *) &windowStatus, sizeof(int));
    read(0, (char *) &mag, sizeof(int));
    read(0, (char *) &x, sizeof(int));
    read(0, spectrumName, x);
    spectrumName[x] = '\0';
    spectrumLoaded();
    x = MAXCONTOURS-1;
    while(--x)
    {
	read(0, (char *) &y, sizeof(int));
	xv_set(contourSlider[x], PANEL_VALUE, y, NULL);
    }
    read(0, (char *) rawColour, sizeof(rawColour));
    read(0, (char *) &displayGamma, sizeof(float));

    if (strstr(spectrumName, "(copy)") == NULL)
	strcat(spectrumName, " (copy)");

    xv_set(Spec_gammaPopup->displayGammaSlider, PANEL_VALUE,
	   (int) displayGamma*100, NULL);
    sprintf(buffer, "%5.2f", displayGamma);
    xv_set(Spec_gammaPopup->displayGammaValue, PANEL_VALUE, buffer, NULL);
    brighten(rawColour, 0, MAXCONTOURS+1, displayColour, displayGamma);
    alterColours();
}

static void spectrumLoaded(void)
{
    char buf[16];
    int x;

    for(x=1; x<MAXCONTOURS-1; x++)
    {
	xv_set(contourSlider[x], PANEL_MAX_VALUE, specMax, NULL);
	if (((int) xv_get(contourSlider[x], PANEL_VALUE)) > specMax)
	    xv_set(contourSlider[x], PANEL_VALUE, specMax, NULL);
    }
    sprintf(buf,"%d",specMax);
    xv_set(contourSlider[x], PANEL_LABEL_STRING, buf, NULL);
    setMagnification(mag);
    RRP(0);
    recalculate3d();
}

/*ARGSUSED*/
void
forkProgram(Panel_item item, Event *event)
{
	int fd;
	char *buf;

	buf = malloc(strlen(programName) + 7);
	sprintf(buf, "%s -pipe", programName);
	
        if ((fd = mypopenfd(buf, "w")) == -1)
        {
	    free(buf);
	    notice_prompt(Spec_specWindow->specWindow, NULL,
			  NOTICE_FOCUS_XY, event_x(event), event_y(event),
			  NOTICE_MESSAGE_STRINGS,
			  "Unable to run copy of program. Error was",
                          ERRSTRING, NULL,
			  NOTICE_BUTTON_YES, "Ok",
			  NULL);
	    return;
        }
	free(buf);

        /*
         * Ignore SIGPIPE signals, we'll detect those by looking for errors
         * returned by writing.
         */
	trapSigpipe();

	if (writeSpectrumToPipe(fd) == -1)
	{
	    notice_prompt(Spec_specWindow->specWindow, NULL,
			  NOTICE_FOCUS_XY, event_x(event), event_y(event),
			  NOTICE_MESSAGE_STRINGS,
			  "Error whilst writing spectrum to pipe. It was",
                          ERRSTRING, NULL,
			  NOTICE_BUTTON_YES, "Ok",
			  NULL);
	    return;
        }

	(void) close(fd);
}

static int writeSpectrumToPipe(int fd)
{
    int x, y;

    if (write(fd, (char *) &specx, sizeof(int)) == -1)
	return(-1);
    if (write(fd, (char *) &specy, sizeof(int)) == -1)
	return(-1);
    if (write(fd, (char *) &specMax, sizeof(int)) == -1)
	return(-1);
    for(x = specx; x--; )
	if (write(fd, (char *) spec2d[x], specy*sizeof(int)) !=
	    specy*sizeof(int))
	{
	    if (errno == 0)
		errno = EPIPE;
	    return -1;
	}
    if (write(fd, (char *) &usedContours, sizeof(int)) == -1)
	return(-1);
    for(x = specx; x--; )
	if (write(fd, (char *) window[x], specy) != specy)
	{
	    if (errno == 0)
		errno = EPIPE;
	    return -1;
	}
    if (write(fd, (char *) &windowStatus, sizeof(int)) == -1)
	return(-1);
    if (write(fd, (char *) &mag, sizeof(int)) == -1)
	return(-1);
    x = strlen(spectrumName);
    if (write(fd, (char *) &x, sizeof(int)) == -1)
	return(-1);
    if (write(fd, spectrumName, x) == -1)
	return(-1);
    x = MAXCONTOURS-1;
    while(--x)
    {
	y = (int) xv_get(contourSlider[x], PANEL_VALUE);
	if (write(fd, (char *) &y, sizeof(int)) == -1)
	    return(-1);
    }
    if (write(fd, (char *) rawColour, sizeof(rawColour)) == -1)
	return(-1);
    if (write(fd, (char *) &displayGamma, sizeof(float)) == -1)
	return(-1);
    return(0);
}

/*ARGSUSED*/
Menu_item
toggleSlicing(Menu_item item, Menu_generate op)
{
    if (op != MENU_NOTIFY)
	return item;

    /* if Slicing is on switch it off and vice versa */
    if (Slicing & SLICING) {
	Slicing &= (~SLICING);
	xv_set(item, MENU_STRING, "Slicing On", NULL);
	xv_set(Spec_specWindow->specWindow,
	       FRAME_RIGHT_FOOTER, "", NULL);
	showCrosshairs = 0;
    }
    else {
	Slicing |= SLICING;
	xv_set(item, MENU_STRING, "Slicing Off", NULL);
	xv_set(Spec_specWindow->specWindow,
	       FRAME_RIGHT_FOOTER, "Slicing On", NULL);
	showCrosshairs = 1;
    }
    drawCrosshairs((Display *)
		       XV_DISPLAY_FROM_WINDOW(Spec_specWindow->specWindow),
		       (Window)
		       xv_get(canvas_paint_window(Spec_specWindow->spectrum),
			      XV_XID), gc, 0, 0, 0);
    return item;
}

Menu_item
setSliceWindow(Menu_item item, Menu_generate op)
{
    if (op != MENU_NOTIFY)
	return item;

    Slicing |= PROJECTION;
    setWindow(item,op);
    return item;
}

/*ARGSUSED*/
Menu_item
reslice(Menu_item item, Menu_generate op)
{
    if (op != MENU_NOTIFY)
	return item;

    do_projection(NULL);
    return item;
}

void
signal_1d(void)
{
    /* signal 1d display process to show slices */

    if (pshm->pid_1d <= 0 || kill(pshm->pid_1d, SIGUSR1) == -1) {
	pid_t pid = lookup_proc("sort_xvgr");
	
	if ((int) pid > 0) {
	    if (kill(pid, SIGUSR1) == -1) {
		perror("projecting-kill");
		fprintf(stderr,"projecting: failed to signal 1D display "
			"process pid = %d\n",
			(int) pid);
	    }
	    else
		pshm->pid_1d = pid;
	}
	else 
	    fprintf(stderr,"projecting: failed to locate 1D display proces\n");
    }
}

/*ARGSUSED*/
static void do_simple_slice(Event *event, int x, int y)
{
      int *px, i, *q, (*py)[MAXSPECX];

      strcpy(pshm->com_str, "DISPLAY 1");
      sprintf(pshm->name, "X=%d", x);
      pshm->more = 1;
      pshm->size = specy;

      for(px = spec2d[x], q = pshm->array + specy, i = specy; i--; )
	  *--q = *px++;

      write_to_pipe(".SORT_pipe0", pshm->size);

      sprintf(pshm->name, "Y=%d", specy-y-1);
      pshm->more = 0;
      pshm->size = specx;

      for(py = spec2d, q = pshm->array, i = specx; i--; )
	  *q++ = (*py++)[y];

      write_to_pipe(".SORT_pipe1", pshm->size);

      signal_1d();
}

static int
store_spectrum(char *outfile, char *command, int win)
{
    register int x, y;
      
    if (DBX_val >= 3)
	fprintf(stderr,"sort_spec2d: storing 2d spectra\n");

    /* copy spectrum window data into pshm structure*/
    strncpy(pshm->com_str,command,NAME_SIZE);
    pshm->size = (specx != specy) ? MAX(specx,specy) : specx;
    pshm->win2d = win2d;
      
    /*
     * x=0, y=0 on display is top right hand corner and 1st element of
     *  pshm->array win = TRUE store window otherwise store displayed spectrum
     */
    if (win == SORT_TRUE)
	for(x=0; x < specx ;x++)
	    for(y=0; y < specy ; y++)
		pshm->array[x*specy + (specy-1-y)] = (window[x][y]) ? 30 : 0;
    else
	for(x=0; x < specx ;x++)
	    for(y=0; y < specy ; y++)
		pshm->array[x*specy + (specy-1-y)] = spec2d[x][y];
      
/* no ipcs on system */
    if ( write_to_pipe(outfile,SHM_ARRAY_SIZE) == SORT_FAIL)
	return -1;
    return 0;
}

/*ARGSUSED*/
static void do_projection(Event *event)
{
      int *px, i, j, *q, (*py)[MAXSPECX], n;
      unsigned char *wx, (*wy)[MAXSPECX];

      strcpy(pshm->com_str, "DISPLAY 1");
      strcpy(pshm->name, "Proj onto Y");
      pshm->more = 1;
      pshm->size = specy;

      memset(pshm->array, 0, specy * sizeof(int));

      for(py = spec2d, wy = window, i = specx; i--; wy++, py++)
	  for(px = *py, wx = *wy, j = specy; j--; px++)
	      if (*wx++)
		  pshm->array[j] += *px;

      write_to_pipe(".SORT_pipe0", pshm->size);

      strcpy(pshm->name, "Proj onto X");
      pshm->more = 0;
      pshm->size = specx;

      for(py = spec2d, wy = window, q = pshm->array, i = specx; i--;
	  wy++, py++)
      {
	  for(n = 0, px = *py, wx = *wy, j = specy; j--; px++)
	      if (*wx++)
		  n += *px;
	  *q++ = n;
      }

      write_to_pipe(".SORT_pipe1", pshm->size);

      signal_1d();
}

static void drawCrosshairs(Display *display, Window xid, GC gc, int n, int x,
			   int y)
{
    static int last_x = 0, last_y = 0;
    unsigned long *index;

    if (n)
    {
	last_x = x;
	last_y = y;
    }
    else
    {
	x = last_x;
	y = last_y;
    }

    if (monochrome)
	XSetForeground(display, gc,
		       BlackPixel(display, DefaultScreen(display)) ^
		       WhitePixel(display, DefaultScreen(display)));
    else
    {
	index = (unsigned long *) xv_get(cms, CMS_INDEX_TABLE);
	XSetForeground(display, gc, index[COLOUROFFSET+MAXCONTOURS-1] ^
		       index[COLOUROFFSET]);
    }
    XSetFunction(display, gc, GXxor);
    XDrawLine(display, xid, gc, x, 0, x, specy*mag);
    XDrawLine(display, xid, gc, 0, y, specx*mag, y);
    XSetFunction(display, gc, GXcopy);

}

/*ARGSUSED*/
void
printSpectrum(Panel_item item, Event *event)
{
    FILE *fp;
    float xgain, ygain, xoffset, yoffset, xt, yt, xf, yf;
    int i, j, v, contourValue[MAXCONTOURS], palette, dest, top, scale;

    if ((xgain = goodFloat(Spec_printerPopup->printerPopup, (char *)
			   xv_get(Spec_printerPopup->printXGain, PANEL_VALUE),
			   -1e37, 1e37, event)) < -1e37)
	return;
    if ((ygain = goodFloat(Spec_printerPopup->printerPopup, (char *)
			   xv_get(Spec_printerPopup->printYGain, PANEL_VALUE),
			   -1e37, 1e37, event)) < -1e37)
	return;
    if ((xoffset = goodFloat(Spec_printerPopup->printerPopup, (char *)
			     xv_get(Spec_printerPopup->printXOffset,
				    PANEL_VALUE),
			     -1e37, 1e37, event)) < -1e37)
	return;
    if ((yoffset = goodFloat(Spec_printerPopup->printerPopup, (char *)
			   xv_get(Spec_printerPopup->printYOffset,
				  PANEL_VALUE),
			   -1e37, 1e37, event)) < -1e37)
	return;

    palette = xv_get(Spec_printerPopup->printerPalette, PANEL_VALUE);

    if ((fp = openPrinterFile(event_x(event), event_y(event))) == NULL)
	return;

    dest = xv_get(Spec_printerPopup->printTo, PANEL_VALUE);

    top = 105 + 15*usedContours;
    if (top < 495)
	top = 495;
    if (*((char *) xv_get(Spec_printerPopup->printTitle, PANEL_VALUE)) != '\0'
	&& top < 580)
	top=580;
    if (*((char *) xv_get(Spec_printerPopup->printSubtitle, PANEL_VALUE))
	!= '\0' && top < 545)
	top=545;

    scale = xv_get(Spec_printerPopup->printScaleSlider, PANEL_VALUE);

    fputs("%!PS-Adobe-2.0\n%%Title: sunsort\n%%Creator: sort_spec2d\n", fp);
    if (!ferror(fp))
    {
	if (dest == 2)
	    fprintf(fp, "%%%%BoundingBox: %d %d %d %d\n", 30*scale/100,
		    40*scale/100, 580*scale/100, top*scale/100);
	else
	    fputs("%%Pages: 1\n", fp);
    }
    if (!ferror(fp))
	fputs("%%EndComments\n/spec2ddict 64 dict def spec2ddict begin\n"
	      "/pixline 128 string def\n", fp);
    if (palette == 2)
    {
	if (!ferror(fp))
	    fprintf(fp, "/colline 384 string def\n/table %d string def\n",
		    MAXCONTOURS*3);
	if (!ferror(fp))
	    fputs(
"% Supply colorimage for printers which don't support it. This version from\n"
"% xv which got it from xwd2ps, though I've changed the red, green and blue\n"
"% multipliers to match the values in the C code\n"
"/colorimage where {pop} {/colortogray {/rgbdata exch store rgbdata length 3\n"
"idiv /npixls exch store /rgbindx 0 store /grays npixls string store 0 1\n"
"npixls 1 sub {grays exch rgbdata rgbindx get 299 mul rgbdata rgbindx 1 add\n"
"get 587 mul rgbdata rgbindx 2 add get 114 mul add add 1000 idiv put\n"
"/rgbindx rgbindx 3 add store} for grays} bind def /mergeprocs {dup length 3\n"
"-1 roll dup length dup 5 1 roll 3 -1 roll add array cvx dup 3 -1 roll 0\n"
"exch putinterval dup 4 2 roll putinterval} bind def /colorimage {pop pop\n"
"{colortogray} mergeprocs image} bind def} ifelse\n", fp);
    }
    else
	if (!ferror(fp))
	    fprintf(fp, "/table %d string def\n", MAXCONTOURS);
    writeFshowCode(fp);
    if (!ferror(fp))
	fputs(
"/format {dup dup 0 eq {pop (0.00) cvs rshow} {abs log floor cvi dup dup 6\n"
"le exch 0 ge and {pop 10 string cvs rshow} {dup 5 string cvs 3 1 roll 10\n"
"exch exp div 100 mul 0.5 add floor 100 div 0.001 add 5 string cvs 0 4\n"
"getinterval exch 0 5 rm currentfont dup 0.66666 ss exch\n"
"rshow setfont 0 -5 rm (x10) rshow rshow } ifelse} ifelse} def\n"
"/tick {/d exch def /L exch def {-10 exch h mul m 10 0 rl w 0 rm 10 0 rl\n"
"w 25 add neg 0 rm L format s /L L d add def} for /d exch def /L exch def {w\n"
"mul -10 m 0 10 rl 0 h rm 0 10 rl 0 h 30 add neg rm L pixline cvs dup\n"
"sw neg exch neg 2 div exch rm show s /L L d add def} for {-5 exch h\n"
"mul m 5 0 rl w 0 rm 5 0 rl} for {w mul -5 m 0 5 rl 0 h rm 0 5 rl} for} def\n"
"/dumptwo {gsave dup scale table currentfile exch readhexstring pop pop\n"
"currentfile pixline readline pop pop dup 1 add dup table exch 0 exch", fp);
    if (palette == 2)
	if (!ferror(fp))
	    fputs(" 3 mul", fp);
    if (!ferror(fp))
	fputs("\n"
	      "getinterval exch 1 exch 8 [0.03333 0 0 0.06667 -2 -6.6667] {}",
	      fp);
    if (!ferror(fp))
    {
	if (palette == 2)
	    fputs("false 3\ncolor", fp);
	else
	    putc('\n', fp);
    }
    if (!ferror(fp))
	fputs(
"image 0 setgray 0 setlinewidth /Helvetica findfont 10 ss -1 0 {15 mul\n"
"100 add dup 60 exch newpath m 30 0 rl 0 15 rl -30 0 rl c s 9 add 55 exch m\n"
"format} for 185 100 translate 1 setlinewidth /w 384 def /h 384 def tick w 2\n"
"div 420 m 18 cfshow w 2 div 450 m 24 cfshow 90 rotate h 2 div 65 m 24\n"
"cfshow -90 rotate w 2 div -50 m 24 cfshow\n", fp);
    if (!ferror(fp))
	fprintf(fp, "%d %d 8 [%f 0 0 %f 0 0] {currentfile pixline readline\n",
		specx, specy, specx/384.0, specy/384.0);
    if (!ferror(fp))
    {
	if (palette == 2)
	    fputs(
"pop length dup 1 sub 0 exch 1 exch {dup pixline exch get table exch 48 sub\n"
"3 mul 3 getinterval exch 3 mul exch colline 3 1 roll putinterval} for\n"
"colline exch 0 exch 3 mul getinterval} false 3 colorimage", fp);
	else
	    fputs(
"pop dup length 1 sub 0 exch 1 exch {dup pixline exch get table exch 48 sub\n"
"get pixline 3 1 roll put} for} image", fp);
    }
    if (!ferror(fp))
	fputs("\n0 0 0 h w h w 0 m l l l c s grestore} def\n"
	      "end\n%%EndProlog\n%%Page: 1\nspec2ddict begin\n", fp);
    if (!ferror(fp))
        psstring(fp, (char *) xv_get(Spec_printerPopup->printXLabel,
                                     PANEL_VALUE));
    if (!ferror(fp))
        psstring(fp, (char *) xv_get(Spec_printerPopup->printYLabel,
                                     PANEL_VALUE));
    if (!ferror(fp))
        psstring(fp, (char *) xv_get(Spec_printerPopup->printTitle,
                                     PANEL_VALUE));
    if (!ferror(fp))
        psstring(fp, (char *) xv_get(Spec_printerPopup->printSubtitle,
                                     PANEL_VALUE));

    xt=tickspacing(specx*xgain);
    if (!ferror(fp))
        fprintf(fp, "%f %f 1.0001\n", (ceil(xoffset/xt*10)*xt/10-xoffset)
                /xgain/specx, xt/specx/xgain/10);
    yt=tickspacing(specy*ygain);
    if (!ferror(fp))
        fprintf(fp, "%f %f 1.0001\n", (ceil(yoffset/yt*10)*yt/10-yoffset)
                /ygain/specy, yt/specy/ygain/10);
    xf=ceil(xoffset/xt)*xt;
    if (!ferror(fp))
        fprintf(fp, "%f %f 1.0001 %f %f\n", (xf-xoffset)/xgain/specx,
                xt/specx/xgain, xf, xt);
    yf=ceil(yoffset/yt)*yt;
    if (!ferror(fp))
        fprintf(fp, "%f %f 1.0001 %f %f\n", (yf-yoffset)/ygain/specy,
                yt/specy/ygain, yf, yt);
    loadContourValueArray(contourValue);
        for(i=0; i<usedContours-1; i++)
	    if (!ferror(fp))
		fprintf(fp, "%d\n", contourValue[i]);
    if (!ferror(fp))
	fprintf(fp, "%d\n", specMax);
    if (!ferror(fp))
        fprintf(fp, "%d %f\ndumptwo\n", usedContours-1, scale/100.0);
    printPalette(fp, palette);
    for(j=specy; j--;)
	for(i=0; i<specx; i++)
	{
	    v = spec2dcontour[i][j] - COLOUROFFSET + 48;
	    if (!ferror(fp))
		putc(v, fp);
	    if (i % 64 == 63 || i == specx-1)
		if (!ferror(fp))
                    putc('\n', fp);
	}
    if (!ferror(fp))
	fputs("end\n", fp);
    if (dest != 2 && !ferror(fp))
	fputs("showpage\n", fp);
    if (!ferror(fp))
        fputs("%%Trailer\n", fp);
    closePrinterFile(fp, event_x(event), event_y(event));
}

static void resizeSpectrum(void)
{
    static int last_w = -1, last_h = -1;
    int w, h, i, d;

    w = (int) xv_get(Spec_specWindow->specWindow, XV_WIDTH);
    h = (int) xv_get(Spec_specWindow->specWindow, XV_HEIGHT) -
	    (int) xv_get(Spec_specWindow->spectrum, XV_Y);

    /*
      This is not the prettiest way of doing this, but it works (mostly).
      The problem is that when setMagnification does a resize, this routine
      gets called twice, once with just the vertical done and once with both
      axes resized. Some of this code is to make sure we don't try and second
      guess the setMagnification resize when it's halfway through.
    */

    for(;;)
    {
	if (last_w != w)
	{
	    last_w = w;
	    
	    d = specx*mag+2;
	    if (Spec_specWindow->vscroll != XV_NULL)
		d += xv_get(Spec_specWindow->vscroll, XV_WIDTH);
	    
	    i = (int) xv_get(Spec_specWindow->spectrum, XV_WIDTH);
	    if (i == w && (w >= d || Spec_specWindow->hscroll != XV_NULL))
		continue;
	    if (i < w)
	    {
		i = d;
		if (i > w)
		    i = w;
		xv_set(Spec_specWindow->spectrum, XV_WIDTH, i, NULL);
	    }
	    else
	    {
		xv_set(Spec_specWindow->spectrum, XV_WIDTH, w, NULL);
		i = w;
	    }
	    if (i >= d)
	    {
		if (Spec_specWindow->hscroll != XV_NULL)
		{
		    xv_set(Spec_specWindow->spectrum, WIN_HORIZONTAL_SCROLLBAR,
			   FALSE, NULL);
		    Spec_specWindow->hscroll = XV_NULL;
		    last_h = -1;
		}
	    }
	    else if (Spec_specWindow->hscroll == XV_NULL)
	    {
		Spec_specWindow->hscroll = 	
		    xv_create(Spec_specWindow->spectrum, SCROLLBAR,
			      SCROLLBAR_SPLITTABLE, FALSE,
			      SCROLLBAR_DIRECTION, SCROLLBAR_HORIZONTAL,
			      NULL);
		last_h = -1;
	    }
	}

	if (last_h != h)
	{
	    last_h = h;
	    
	    d = specy*mag+2;
	    if (Spec_specWindow->hscroll != XV_NULL)
		d += xv_get(Spec_specWindow->hscroll, XV_HEIGHT);

	    i = (int) xv_get(Spec_specWindow->spectrum, XV_HEIGHT);
	    if (i == w && (h >= d || Spec_specWindow->vscroll != XV_NULL))
		continue;
	    if (i < h)
	    {
		i = d;
		if (i > h)
		    i = h;
		xv_set(Spec_specWindow->spectrum, XV_HEIGHT, i, NULL);
	    }
	    else
	    {
		xv_set(Spec_specWindow->spectrum, XV_HEIGHT, h, NULL);
		i = h;
	    }
	    if (i >= d)
	    {
		if (Spec_specWindow->vscroll != XV_NULL)
		{
		    xv_set(Spec_specWindow->spectrum, WIN_VERTICAL_SCROLLBAR,
			   FALSE, NULL);
		    Spec_specWindow->vscroll = XV_NULL;
		    last_w = -1;
		}
	    }
	    else if (Spec_specWindow->vscroll == XV_NULL)
	    {
		Spec_specWindow->vscroll = 	
		    xv_create(Spec_specWindow->spectrum, SCROLLBAR,
			      SCROLLBAR_SPLITTABLE, FALSE,
			      SCROLLBAR_DIRECTION, SCROLLBAR_VERTICAL,
			      NULL);
		last_w = -1;
	    }
	}
	
	if ((int) xv_get(Spec_specWindow->spectrum, XV_WIDTH) >= specx*mag+2 &&
	    (int) xv_get(Spec_specWindow->spectrum, XV_HEIGHT) >= specy*mag+2
	    && (Spec_specWindow->hscroll != XV_NULL ||
		Spec_specWindow->vscroll != XV_NULL))
	{
	    xv_set(Spec_specWindow->spectrum, WIN_VERTICAL_SCROLLBAR, NULL,
		   WIN_HORIZONTAL_SCROLLBAR, NULL, NULL);
	    Spec_specWindow->hscroll = Spec_specWindow->vscroll = XV_NULL;
	}

	if (last_w == w && last_h == h)
	    break;
    }

}

/*ARGSUSED*/
static void event_proc(Xv_Window window, Event *event, Notify_arg arg)
{
    if (event_action(event) == WIN_RESIZE &&
	event_xevent(event)->type == ConfigureNotify)
	resizeSpectrum();
}

