/******************************************************************************
* sort-spy.c      sort spy demon for solaris 2
*                 This task (which must run with root id)
*                 spys on the fddi or ethernet interface
*                 picks out eb-ts data packets
*                 and reconstructs data blocks in mmaped files
*/

/*****************************************************************************
version 2.3 Fri May  5 16:04:45 BST 1995, DB

now reports oversized accummulated data blocks. removed file locking. 
responds to non-zero userPid in ring buffer by sending SIGUSR.
now elevates itself to the default (minimal) priority within the RT
scheduling class. this seems largely to overcome problems with missed
network fragments, discovered by Tom Davinson (both the problem and
the solution!)

version 2.1 Thu Mar  9 11:27:09 GMT 1995, DB
changed the ring handling protocol---see the README
******************************************************************************/

#define SOLARIS2

#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <unistd.h>

#include <sys/wait.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stropts.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <sys/mman.h>
#include <sys/ethernet.h>
#include <sys/dlpi.h>
#include <sys/pfmod.h>

#include <net/if.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/udp.h>
#include <netinet/ip_icmp.h>

#include "spy.h"


#define FILTER_MODULE  "pfmod"

#define DEBUG_OUT    (debug & 1)
#define DEBUG_MISSED (debug & 2)
#define DEBUG_PUT    (debug & 4)
#define DEBUG_READ   (debug & 8)
#define DEBUG_FILTER (debug & 16)
#define DEBUG_SPY    (debug & 32)

int sortSpyfd;                /* io descriptor for sort device */
static short sourceAddress[2];   /* source IP address for filter module */
static u_long saddr;            /* for explicit filtering */

int   fd[MAX_STREAM+1];
FSTR *fstr[MAX_STREAM+1];
DATAGRAM fragments[DATA_FRAG_NUM];         /* where to put input */

char interface[128];
char host[32];
int ppa;
char fileroot[128];
int blocksize;
int ringsize;
int debug=0;

/******************************************************************************
* wakeup       signal handler to do nothing at all
*/
void wakeup (int sig) {
   return;
   }
/******************************************************************************
* fileInit     initialise shared file interface
*         open and mmap a file for each stream
*         if existing spy task then kill it
*         return OK or ERROR
*/
fileInit () {
   register int s;
   int filesize;
   int i;
   char fname[128];

   if (signal (SIGHUP, (void (*)()) wakeup) == SIG_ERR) {/* install handler */
      perror ("sort-spy - failed to install SIGHUP signal handler");
      return ERROR;
      }

   umask (0);        /* clear umask so files can have global r/w access */

   filesize=sizeof(FSTR)+blocksize*ringsize;
   for (s = 1; s <= MAX_STREAM; s++) {
      sprintf (fname, "%s-%d", fileroot, s);
      fd[s] = open (fname, O_RDWR | O_CREAT,
         S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH);
      if (fd[s] == ERROR) {
         perror ("sort-spy - failed to open shared file");
         return ERROR;
         }
      if (ftruncate (fd[s], filesize) != OK) {
         perror ("sort-spy - failed to truncate shared file");
         return ERROR;
         }
      fstr[s] = (FSTR *) mmap (NULL, filesize,
         PROT_READ | PROT_WRITE, MAP_SHARED, fd[s], 0);
      if ((int) fstr[s] == ERROR) {
         perror ("sort-spy - failed to mmap shared file");
         return ERROR;
         }

/***  if (fstr[s]->spyPid)   ***/         /* then kill existing spy task */
/***  kill (fstr[s]->spyPid, SIGKILL); ***/  /* may use SIGUSR2 to tidy up ? */

      fstr[s]->blocksize = blocksize;
      fstr[s]->ringsize = ringsize;
      fstr[s]->wrIndex = 0;
      fstr[s]->wrCount = 0;
      fstr[s]->spyPid = getpid ();
      for (i=0; i<ringsize; i++) {
         fstr[s]->buffer[i]=sizeof(FSTR)+i*blocksize;
         fstr[s]->length[i]=0;
         fstr[s]->sequence[i]=0;
         }
      }
   return OK;
   }
/******************************************************************************
* putmsgack   put message and wait for ack
*        returt OK ot ERROR
*/
putmsgack (int fd, struct strbuf *ctlptr, struct strbuf *dataptr, int putflags) {
   struct strbuf ctlbuf;
   int getflags = RS_HIPRI;
   union DL_primitives rcvbuf;

   if (putmsg (fd, ctlptr, dataptr, putflags))
      return ERROR;

   if (DEBUG_PUT)
      printf ("putmsgack phase 1 done OK\n");

   ctlbuf.maxlen = sizeof (union DL_primitives);
   ctlbuf.len    = 0;
   ctlbuf.buf    = (char *) &rcvbuf;

   if (getmsg (fd, &ctlbuf, NULL, &getflags))
      return ERROR;

   if (DEBUG_PUT)
      printf ("putmsgack phase 2 done OK\n");


   if (ctlbuf.len < sizeof (long)) {
      errno = EPROTO;
      return ERROR;
      }

   if (DEBUG_PUT)
      printf ("putmsgack phase 3 done OK\n");

   switch (rcvbuf.dl_primitive) {
      case DL_OK_ACK:

         if (DEBUG_PUT)
            printf ("putmsgack phase 4 done OK\n");

         return OK;

      case DL_ERROR_ACK:

         if (DEBUG_PUT)
            printf ("putmsgack phase 5 \n");

         if (ctlbuf.len < sizeof (dl_error_ack_t)) {    
            if (DEBUG_PUT)
               printf ("putmsgack phase 8 \n");
            errno = EPROTO;
            }
         else
            errno = rcvbuf.error_ack.dl_unix_errno;

         if (DEBUG_PUT)
            printf ("putmsgack phase 9 %x \n",rcvbuf.error_ack.dl_errno);

         return ERROR;

      case DL_BIND_ACK:

         if (DEBUG_PUT)
            printf ("putmsgack phase 6 \n");

         return OK;

      default:

         if (DEBUG_PUT)
            printf ("putmsgack phase 7 \n");

         errno = EPROTO;
         return ERROR;
      }
   }
/******************************************************************************
* readmsg      get data message
*         return message length or ERROR
*/
readmsg (int fd, char *data, int length) {
   union DL_primitives rcvbuf;
   struct strbuf ctlbuf;
   struct strbuf databuf;
   int flags = 0;
   int i;

   if (DEBUG_READ)
      printf ("readmsg phase 1 done OK\n");

   ctlbuf.maxlen = sizeof (union DL_primitives);
   ctlbuf.len    = 0;
   ctlbuf.buf    = (char *) &rcvbuf;

   databuf.maxlen = length;
   databuf.len    = 0;
   databuf.buf    = data;

   if (getmsg (fd, &ctlbuf, &databuf, &flags)<0) {
      perror ("readmsg - getmsg() error");
      return ERROR;
      }
   if (DEBUG_READ)
      printf ("readmsg phase 2 done OK\n");

   if (ctlbuf.len < sizeof (long)) {
      errno = EPROTO;
      return ERROR;
      }

   if (DEBUG_READ)
      printf ("readmsg phase 3 done OK\n");

   switch (rcvbuf.dl_primitive) {
      case DL_UNITDATA_IND:

         if (DEBUG_READ) {
            printf ("readmsg phase 4 done OK %x, \n", databuf.len);
            for (i=0; i<16; i++) printf ("%02x ", (unsigned char)data[i]);
            printf (" ...\n");
	    }
         return databuf.len;
      
      default:
         errno = EPROTO;
         return ERROR;
      }
   }
/******************************************************************************
* sortSpyInit      Initialise sorter library
*        find source internet address from host name
*        initialise hardware
*        returt OK ot ERROR
*/

#include <sys/priocntl.h>
#include <sys/rtpriocntl.h>

sortSpyInit (char *device, int ppa, char *host) {
   char message[80];
   struct hostent *hostEntry;
   struct strbuf strbuf;
   union DL_primitives ctlbuf;
   struct pcinfo pcinfobuf;
   struct pcparms pcparmsbuf;
   struct rtparms rtparmsbuf;
   long i;
   rtinfo_t rtinfobuf;

   hostEntry = gethostbyname (host);
   if (hostEntry == NULL) {
      perror ("sortSpyInit - Couldn't find host entry");
      return ERROR;
      }
   memcpy ((char *) sourceAddress, *hostEntry->h_addr_list, 4);
   saddr = *(u_long *)*hostEntry->h_addr_list;

   sortSpyfd = open (device, O_RDWR);
   if (sortSpyfd == ERROR) {
      sprintf (message, "sortSpyInit - failed to open %s", device);
      perror (message);
      return ERROR;
      }
   /* set up discard message mode */

   if (ioctl (sortSpyfd, I_SRDOPT, RMSGD) == ERROR) {
      perror ("sortSpyInit - set up to discard message mode");
      return ERROR;
      }

   /*  attach to interface    */

   strbuf.buf          = (char *) &ctlbuf;
   strbuf.len          = DL_ATTACH_REQ_SIZE;
   ctlbuf.dl_primitive      = DL_ATTACH_REQ;
   ctlbuf.attach_req.dl_ppa = ppa;

   if (putmsgack (sortSpyfd, &strbuf, NULL, RS_HIPRI)) {
      perror ("sortSpyInit - failed to attach to ppa");
      return ERROR;
      }

   /* bind */

   strbuf.len          = DL_BIND_REQ_SIZE;
   ctlbuf.dl_primitive      = DL_BIND_REQ;
   ctlbuf.bind_req.dl_sap     = ETHERTYPE_IP;
   ctlbuf.bind_req.dl_max_conind   = 0;
   ctlbuf.bind_req.dl_service_mode = DL_CLDLS;
   ctlbuf.bind_req.dl_conn_mgmt    = 0;
   ctlbuf.bind_req.dl_xidtest_flg  = 0;

   if (putmsgack (sortSpyfd, &strbuf, NULL, 0)) {
      perror ("sortSpyInit - failed to bind");
      return ERROR;
      }

   /* set to promiscuous mode */

   strbuf.len          = DL_PROMISCON_REQ_SIZE;
   ctlbuf.dl_primitive      = DL_PROMISCON_REQ;
   ctlbuf.promiscon_req.dl_level = DL_PROMISC_PHYS;

   if (putmsgack (sortSpyfd, &strbuf, NULL, 0)) {
      perror ("sortSpyInit - failed to set promiscuous mode");
      return ERROR;
      }

   /* push and configure the packet filtering module if required   */
/*
   if (ioctl (sortSpyfd, I_PUSH, FILTER_MODULE) == ERROR) {
      sprintf (message, "sortSpyInit - failed to push filter module %s",
              FILTER_MODULE);
      perror (message);
      return ERROR;
      }

   if (config_filter () == ERROR) {
      perror ("sortSpyInit - failed to configure filter");
      return ERROR;
      }
*/

#if 0
/* helpful (?) gibberish clipped from system include files */
typedef struct pcinfo {
	id_t	pc_cid;			/* class id */
	char	pc_clname[PC_CLNMSZ];	/* class name */
	long	pc_clinfo[PC_CLINFOSZ];	/* class information */
} pcinfo_t;

typedef struct pcparms {
	id_t	pc_cid;			    /* process class */
	long	pc_clparms[PC_CLPARMSZ];    /* class specific parameters */
} pcparms_t;

typedef struct rtparms {
	pri_t	rt_pri;		/* real-time priority */
	ulong	rt_tqsecs;	/* seconds in time quantum */
	long	rt_tqnsecs;	/* additional nanosecs in time quantum */
} rtparms_t;
#endif

   strncpy (pcinfobuf.pc_clname, "RT", PC_CLNMSZ);
   i=priocntl (P_PID, P_MYID, PC_GETCID, (void*)&pcinfobuf);
   if (i<0) {
      perror ("sortSpyInit - unable to get RT scheduling class id");
      return ERROR;
      }
   pcparmsbuf.pc_cid=pcinfobuf.pc_cid;

/*              added 20/6/96                               */
   (void) memcpy((void*) &rtinfobuf, (void*)pcinfobuf.pc_clinfo, sizeof(rtinfo_t));
   rtparmsbuf.rt_pri= rtinfobuf.rt_maxpri - 1;
   rtparmsbuf.rt_tqsecs=0;
   rtparmsbuf.rt_tqnsecs=RT_TQDEF;
/**            end of aadded 20/6/96                      **/

   memcpy ((void*)pcparmsbuf.pc_clparms, (void*)&rtparmsbuf, 
      sizeof(struct rtparms));
   i=priocntl (P_PID, P_MYID, PC_SETPARMS, (void*)&pcparmsbuf);
   if (i<0) {
      perror ("sortSpyInit - unable to change scheduling to RT class");
      return ERROR;
      }
   return OK;
   }
/******************************************************************************
* filter       this does the same function as the filter module
*         if only I could get that to work!
*         return OK if packet passes through otherwise ERROR
*/
filter (DATAGRAM* datagram) { 

/***
   if (DEBUG_FILTER) {
      printf ("filter phase 1 done OK %x, \n",datagram->d_hdr.i802_header1 );
      printf ("filter phase 1 done OK %x, \n",datagram->d_hdr.i802_header2 );
      printf ("filter phase 1 done OK %x, \n",datagram->d_hdr.i802_header3 );
      printf ("filter phase 1 done OK %x, \n",datagram->d_hdr.ethertype );
      }
***/
   return (datagram->ip.ip_src.s_addr != saddr    ||
      datagram->hdr.stream       == 0        ||
      datagram->rpc.program      != EGTS_RPC_PROG ||
      datagram->rpc.procedure    != EGTS_RPC_PROC ||
      datagram->udp.uh_sport     != EGEB_UDP_PORT ||
      datagram->udp.uh_dport     != EGTS_UDP_PORT ||
      datagram->ip.ip_p     != IPPROTO_UDP     ) ? ERROR : OK;
   }
/******************************************************************************
* rebuild     build up a block of data from nofrags consecutive datagrams
*             starting at fragment fragno in the fragments ring
*        check for ownership of shared file for that stream
*        copy data fragments into file
*        pass ownership back to user
*/
rebuild (int fragno, int nofrags) {
   FSTR *f;
   int wrIndex;
   DATAGRAM* d;
   int length;           /* accumulating length of data block */
   int i;
   int seq;
   unsigned short* ptr;
   d=&fragments[fragno];
   f = fstr[d->hdr.stream];     /* use shared file for relevent stream */
   wrIndex = f->wrIndex;
   length=0;
   for (i=0; i<nofrags; i++) {         /* build data block */ 
      if (d->hdr.offset != length)
         return;
      length += d->hdr.length;
      if (length > f->blocksize) {                              /* overflow */
         if (DEBUG_MISSED) printf (
"rebuild - data block discarded (length %d > ring blocksize %d)\n",
length, f->blocksize);
         return;
         }
      memcpy ((char*)f + f->buffer[f->wrIndex] + d->hdr.offset, 
         d->data, d->hdr.length);
      if (i==0) seq=d->hdr.sequence/nofrags;      
      fragno=(fragno+1) % DATA_FRAG_NUM;
      d=&fragments[fragno];
      }
   f->sequence[f->wrIndex]=seq;
   f->length[f->wrIndex] = length;
   f->wrIndex = (f->wrIndex+1) % f->ringsize;       /* update write pointer */
   f->wrCount++;
   if (f->wrCount>f->ringsize) f->wrCount=f->ringsize;

   if (DEBUG_OUT) {
      ptr=(unsigned short*)((char*)f+f->buffer[wrIndex]);
      printf (
         "rebuild - block at index %d, length %d, sequence %d, qsize %d\n", 
         wrIndex, length, seq, f->wrCount);
      for (i=0; i<8; i++) printf ("%04x ", *ptr++);
      printf (" ...\n");
      } 

   if (f->userPid) {
      i=sigsend (P_PID, f->userPid, SIGUSR1);            /* wake up task */
      if (i<0) f->userPid = 0;
      }
   }
/******************************************************************************
* sortSpy     read consecutive packets into memory
*        do explicit filtering while I cant get pfmod to work
*        rebuild data blocks when last packet is found
*        repeat until a signal occurs
*/
sortSpy () {
   int rs;
   int stream;                 /* stream number */
   int i;                    /* fragment number */
   int jj, nn;
   long* ptr;
   int seq;
   int state;
   int firstfrag;
   int nofrags;
   DATAGRAM* frag;

#define OUT_OF_PACKET 0
#define IN_PACKET 1

   i=0;
   state=OUT_OF_PACKET;
   for (;;) {
      if (i >= DATA_FRAG_NUM)      /* check for wrap around */
         i = 0;
      frag=&fragments[i];
      rs = readmsg (sortSpyfd, (char *)frag, sizeof (DATAGRAM));
      if (rs<0 && errno==EINTR) return ERROR;
      if (rs<0) continue;
      if (DEBUG_SPY) 
         for (jj = 0; jj < 1024 ;jj= jj+16) {
            ptr = (long *)frag;
            for (nn=0; nn<16; nn++)
               printf("%08x", ptr[nn+jj]);
            printf(" ...\n");
            }

/*!!!! WHEN filter module works properly remove explicit filter call !!!! */
/*!!!! needed only while I cant configure pfmod */

      if (!(filter (frag)==OK)) continue;
      if (DEBUG_SPY)
         printf ("sortSpy phase 1 done OK\n");
      if (state==IN_PACKET && frag->hdr.sequence==seq+1 && 
         nofrags<DATA_FRAG_NUM) {
         nofrags++;
         seq=frag->hdr.sequence;
         i++;
         }
      else if (frag->hdr.offset==0) {
         if (state==IN_PACKET && DEBUG_MISSED)
            printf (
"sort-Spy - resynchronising at sequence %d(%d), offset %d, discarding %d\n",
               frag->hdr.sequence, seq, frag->hdr.offset, nofrags);
         firstfrag=i;
         seq=frag->hdr.sequence;
         nofrags=1;
         state=IN_PACKET;
         i++;
         }
      else {                                      /* can't use this fragment */
         if (DEBUG_MISSED) 
            printf (
"sort-Spy - discarding fragment, sequence %d(%d), offset %d\n", 
               frag->hdr.sequence, seq, frag->hdr.offset);
         state=OUT_OF_PACKET;
         }
      if (state==IN_PACKET && (frag->hdr.options & OPT_END)) {
         rebuild (firstfrag, nofrags);
         if (DEBUG_SPY)
            printf ("sortSpy phase 2 done OK\n");
         state=OUT_OF_PACKET;
         }
      }
   }
/******************************************************************************
* config_filter       configure filtering module
*           borrowed from the manual pages and Daresbury's llc
*           return status from NIOCSETF ioctl request
*/
config_filter() {
   struct strioctl si;
   struct packetfilt pf;                /* packet filter */
   register u_short *fwp = pf.Pf_Filter;

   *fwp++ = ENF_PUSHWORD + 13;            /* internet address */
   *fwp++ = ENF_PUSHLIT | ENF_CAND;
   *fwp++ = sourceAddress[0];
   *fwp++ = ENF_PUSHWORD + 14;
   *fwp++ = ENF_PUSHLIT | ENF_CAND;
   *fwp++ = sourceAddress[1];

   *fwp++ = ENF_PUSHWORD + 42;                /* Stream Field */
   *fwp++ = ENF_PUSHZERO | ENF_CNOR;  /* should be non zero if there is data */

   *fwp++ = ENF_PUSHWORD + 27;            /* RPC Program */
   *fwp++ = ENF_PUSHLIT | ENF_CAND;
   *fwp++ = (EGTS_RPC_PROG >> 16) & 0xffff;
   *fwp++ = ENF_PUSHWORD + 28;
   *fwp++ = ENF_PUSHLIT | ENF_CAND;
   *fwp++ = EGTS_RPC_PROG & 0xffff;

   *fwp++ = ENF_PUSHWORD + 31;                /* RPC Prcedure */
   *fwp++ = ENF_PUSHLIT | ENF_CAND;
   *fwp++ = (EGTS_RPC_PROC >> 16) & 0xffff;
   *fwp++ = ENF_PUSHWORD + 32;
   *fwp++ = ENF_PUSHLIT | ENF_CAND;
   *fwp++ = EGTS_RPC_PROC & 0xffff;

   *fwp++ = ENF_PUSHWORD + 17;            /* source port */
   *fwp++ = ENF_PUSHLIT | ENF_CAND;
   *fwp++ = EGEB_UDP_PORT;
   *fwp++ = ENF_PUSHWORD + 18;            /* destination port */
   *fwp++ = ENF_PUSHLIT | ENF_CAND;
   *fwp++ = EGTS_UDP_PORT;

   *fwp++ = ENF_PUSHWORD + 6;              /* Ethertype Field */
   *fwp++ = ENF_PUSHLIT | ENF_CAND;
   *fwp++ = ETHERTYPE_IP;

   *fwp++ = ENF_PUSHWORD + 11;                /* UDP Protocol */
   *fwp++ = ENF_PUSHLIT | ENF_AND;
   *fwp++ = 0xff;
   *fwp++ = ENF_PUSHLIT | ENF_CAND;
   *fwp++ = IPPROTO_UDP;

   pf.Pf_FilterLen = fwp - pf.Pf_Filter;

   si.ic_cmd = PFIOCSETF;
   si.ic_len = (char *) fwp - (char *) &pf;
   si.ic_dp = (char *) &pf;

   return ioctl (sortSpyfd, I_STR, &si);
   }

/*****************************************************************************/

/*****************************************************************************/
void usage () {
   printf ("\
usage: sort-spy [-i interface]       (%s)\n\
                [-n devnum]          (%d)\n\
                [-h host]            (%s)\n\
                [-f fileroot]        (%s)\n\
                [-d debug]           (0)\n\
                [-b blocksize]       (%d)\n\
                [-r ringsize]        (%d)\n",
      DEFAULT_DEVICE, DEFAULT_PPA, DEFAULT_HOST, DEFAULT_FILEROOT,
      DEFAULT_BLOCKSIZE, DEFAULT_RINGSIZE);
   exit (8);
   }
/*****************************************************************************/
void processargs (int argc, char* argv[]) {
   int argno, i;
   strcpy (interface, DEFAULT_DEVICE);
   strcpy (host, DEFAULT_HOST);
   ppa=DEFAULT_PPA;
   strcpy (fileroot, DEFAULT_FILEROOT);
   debug=0;
   blocksize=DEFAULT_BLOCKSIZE;
   ringsize=DEFAULT_RINGSIZE;
   argno=1;
   while (argno<argc) {
      if (argno+1>=argc) usage ();
      if (strcmp (argv[argno], "-i")==0) {
         strcpy (interface, argv[argno+1]);    
         }
      else if (strcmp (argv[argno], "-n")==0) {
         if (sscanf (argv[argno+1], "%d", &ppa) != 1) usage ();
         }
      else if (strcmp (argv[argno], "-d")==0) {
         if (sscanf (argv[argno+1], "%d", &debug) != 1) usage ();
         }
      else if (strcmp (argv[argno], "-h")==0) {
         strcpy (host, argv[argno+1]);    
         }
      else if (strcmp (argv[argno], "-b")==0) {
         if (sscanf (argv[argno+1], "%d", &blocksize) != 1) usage ();
         if (blocksize<=0 || blocksize>MAX_BLOCKSIZE) 
            blocksize=DEFAULT_BLOCKSIZE;
         }
      else if (strcmp (argv[argno], "-r")==0) {
         if (sscanf (argv[argno+1], "%d", &ringsize) != 1) usage ();
         if (ringsize<=0 || ringsize>MAX_RINGSIZE) ringsize=DEFAULT_RINGSIZE;
         }
      else if (strcmp (argv[argno], "-f")==0) {
         strcpy (fileroot, argv[argno+1]);    
         }
      else usage ();
      argno+=2;
      }
   }

/******************************************************************************
* main       sort spy demon
*       argv[1] = network device name
*       argv[2] = name of event builder host
*       initialise the network interface
*       initialise file mapping
*       pick out eb-ts data packets
*       reconstruct data blocks in mmaped files
*/
main (int argc, char **argv) {

   printf ("sort-spy version 2.3 starting...\n");
   processargs (argc, argv);

   if (sortSpyInit (interface, ppa, host))
      return errno;

   if (fileInit ())
      return errno;

   sortSpy ();

   printf ("sort-spy version 2.3 exiting (SIGHUP received)\n");
   exit (0);
   }
/*****************************************************************************/
