/*
 * ''corr''      Angular Correlations                      Steve Chappell
 *
 *   General Subroutine to reconstruct angular correlations for
 *   decay of a resonant particle into two breakup fragments
 *   given the position of the virtual resonant particle
 *   in the event array
 *   rp* -> Heavy Ion (hi) + Light Ion (li)
 
 * Notes
 
 *   In Polar Co-ords  (hor,vert,beam)=(x,y,z)
 *   In Fixed Axial Co-ords          (xa,ya,za)
 *   In Basel Co-ords                (xb,yb,zb)
 
 * Updates
 *   18th Dec 97: Angles converted to positive range 0 -> 360
 *                Basel.phis 0->180
 *                Bug Fix in axial->Basel transform
 */

#include <stdio.h>
#include <math.h>
#include "evsubs.h"

static void corr_vrel(struct vector *vrel, struct vector *hi, double mhi,
		      struct vector *li, double mli)
{
    double rml, rmh;

    rml = 1/mli;
    rmh = 1/mhi;

    vector_set_xyz(vrel, vector_x(li)*rml - vector_x(hi)*rmh,
		   vector_y(li)*rml - vector_y(hi)*rmh,
		   vector_z(li)*rml - vector_z(hi)*rmh);
}

void corr_polar(struct correlation *tp, particle *vrp)
{
    particle *evhi, *evli;

    /*fill out phase space structure for polar co-ord frame*/    
    /*(Nb.Particles hi,li in lab, vrp in c.m. frame)*/
    evhi = ev_hi(vrp);
    evli = ev_li(vrp); 
    
    vector_copy(&tp->p_li, ev_pvec(evli));
    vector_copy(&tp->p_hi, ev_pvec(evhi));
    vector_copy(&tp->p_vrp, ev_pcmvec(vrp));
    corr_vrel(&tp->vrel, &tp->p_li, ev_m(evli), &tp->p_hi, ev_m(evhi));
}

void corr_axial(struct correlation *tp, particle *vrp)
{
    particle *evhi, *evli;
    evhi = ev_hi(vrp);
    evli = ev_li(vrp); 
    
    /*
     * FIXED ``AXIAL'' Co-ordinates (xa,ya,za)=(beam,horiz,vert)
     * Transformation from Polar Co-ord frame (x,y,z) ---> (za,xa,ya)
     */
    
    vector_set_xyz(&tp->p_li, ev_p_z(evli), ev_p_x(evli), ev_p_y(evli));
    vector_set_xyz(&tp->p_hi, ev_p_z(evhi), ev_p_x(evhi), ev_p_y(evhi));
    vector_set_xyz(&tp->p_vrp, ev_pcm_z(vrp), ev_pcm_x(vrp), ev_pcm_y(vrp));
    corr_vrel(&tp->vrel, &tp->p_li, ev_m(evli), &tp->p_hi, ev_m(evhi));
}

void corr_basel(struct correlation *tp, particle *vrp)
{
    double y, z, r;
    particle *evhi, *evli;
    double cdths, sdths;

    evhi = ev_hi(vrp);
    evli = ev_li(vrp); 
    
    /*
     * FLOATING ``BASEL'' Co-ords (xb,yb,zb) => (ths_b=90,phis_b,psi_b,chi_b)
     * ebye Rotn about xa to define trajectory in the reac plane (xb,yb)
     */
    
    y = ev_pcm_x(vrp);
    z = ev_pcm_y(vrp);
    r = sqrt(y*y + z*z);
    vector_set_xyz(&tp->p_vrp, ev_pcm_z(vrp), r, 0);

    r = 1/r;
    cdths = y*r;
    sdths = z*r;
    
    y = ev_p_x(evli);
    z = ev_p_y(evli);
    vector_set_xyz(&tp->p_li, ev_p_z(evli), y*cdths + z*sdths,
		   z*cdths - y*sdths);
    
    y = ev_p_x(evhi);
    z = ev_p_y(evhi);
    vector_set_xyz(&tp->p_hi, ev_p_z(evhi), y*cdths + z*sdths,
		   z*cdths - y*sdths);

    corr_vrel(&tp->vrel, &tp->p_li, ev_m(evli), &tp->p_hi, ev_m(evhi));
}

static struct correlation fortran_corr;

void corr_polar_(int *p)
{
    corr_polar(&fortran_corr, evhead.ev + *p);
}

void corr_axial_(int *p)
{
    corr_polar(&fortran_corr, evhead.ev + *p);
}

void corr_basel_(int *p)
{
    corr_polar(&fortran_corr, evhead.ev + *p);
}

double corr_p_li_x_() { return vector_x(&fortran_corr.p_li); }
double corr_p_li_y_() { return vector_y(&fortran_corr.p_li); }
double corr_p_li_z_() { return vector_z(&fortran_corr.p_li); }
double corr_p_li_r_() { return vector_r(&fortran_corr.p_li); }
double corr_p_li_r2_() { return vector_r2(&fortran_corr.p_li); }
double corr_p_li_t_() { return vector_t(&fortran_corr.p_li); }
double corr_p_li_p_() { return vector_p(&fortran_corr.p_li); }
double corr_p_li_td_() { return vector_td(&fortran_corr.p_li); }
double corr_p_li_pd_() { return vector_pd(&fortran_corr.p_li); }
double corr_p_li_tx_() { return vector_tx(&fortran_corr.p_li); }
double corr_p_li_ty_() { return vector_ty(&fortran_corr.p_li); }
double corr_p_li_txd_() { return vector_txd(&fortran_corr.p_li); }
double corr_p_li_tyd_() { return vector_tyd(&fortran_corr.p_li); }
double corr_p_hi_x_() { return vector_x(&fortran_corr.p_hi); }
double corr_p_hi_y_() { return vector_y(&fortran_corr.p_hi); }
double corr_p_hi_z_() { return vector_z(&fortran_corr.p_hi); }
double corr_p_hi_r_() { return vector_r(&fortran_corr.p_hi); }
double corr_p_hi_r2_() { return vector_r2(&fortran_corr.p_hi); }
double corr_p_hi_t_() { return vector_t(&fortran_corr.p_hi); }
double corr_p_hi_p_() { return vector_p(&fortran_corr.p_hi); }
double corr_p_hi_td_() { return vector_td(&fortran_corr.p_hi); }
double corr_p_hi_pd_() { return vector_pd(&fortran_corr.p_hi); }
double corr_p_hi_tx_() { return vector_tx(&fortran_corr.p_hi); }
double corr_p_hi_ty_() { return vector_ty(&fortran_corr.p_hi); }
double corr_p_hi_txd_() { return vector_txd(&fortran_corr.p_hi); }
double corr_p_hi_tyd_() { return vector_tyd(&fortran_corr.p_hi); }
double corr_p_vrp_x_() { return vector_x(&fortran_corr.p_vrp); }
double corr_p_vrp_y_() { return vector_y(&fortran_corr.p_vrp); }
double corr_p_vrp_z_() { return vector_z(&fortran_corr.p_vrp); }
double corr_p_vrp_r_() { return vector_r(&fortran_corr.p_vrp); }
double corr_p_vrp_r2_() { return vector_r2(&fortran_corr.p_vrp); }
double corr_p_vrp_t_() { return vector_t(&fortran_corr.p_vrp); }
double corr_p_vrp_p_() { return vector_p(&fortran_corr.p_vrp); }
double corr_p_vrp_td_() { return vector_td(&fortran_corr.p_vrp); }
double corr_p_vrp_pd_() { return vector_pd(&fortran_corr.p_vrp); }
double corr_p_vrp_tx_() { return vector_tx(&fortran_corr.p_vrp); }
double corr_p_vrp_ty_() { return vector_ty(&fortran_corr.p_vrp); }
double corr_p_vrp_txd_() { return vector_txd(&fortran_corr.p_vrp); }
double corr_p_vrp_tyd_() { return vector_tyd(&fortran_corr.p_vrp); }
double corr_vrel_x_() { return vector_x(&fortran_corr.vrel); }
double corr_vrel_y_() { return vector_y(&fortran_corr.vrel); }
double corr_vrel_z_() { return vector_z(&fortran_corr.vrel); }
double corr_vrel_r_() { return vector_r(&fortran_corr.vrel); }
double corr_vrel_r2_() { return vector_r2(&fortran_corr.vrel); }
double corr_vrel_t_() { return vector_t(&fortran_corr.vrel); }
double corr_vrel_p_() { return vector_p(&fortran_corr.vrel); }
double corr_vrel_td_() { return vector_td(&fortran_corr.vrel); }
double corr_vrel_pd_() { return vector_pd(&fortran_corr.vrel); }
double corr_vrel_tx_() { return vector_tx(&fortran_corr.vrel); }
double corr_vrel_ty_() { return vector_ty(&fortran_corr.vrel); }
double corr_vrel_txd_() { return vector_txd(&fortran_corr.vrel); }
double corr_vrel_tyd_() { return vector_tyd(&fortran_corr.vrel); }

double corr_ths_() { return corr_ths(fortran_corr); }
double corr_phis_() { return corr_phis(fortran_corr); }
double corr_psi_() { return corr_psi(fortran_corr); }
double corr_chi_() { return corr_chi(fortran_corr); }
