/*
 * ''detprocs''   Standard Detector procedures                Steve Chappell
 *
 * Functions called by detector_init and detector_event for each detector type
 *                                          
 * Updates
 *    15th Oct 1998
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "evsubs.h"
#include "hitsubs.h"
#include "markhits.h"

int eginit(char *);                   /*eurogam format map file initialiser*/
struct det_xstats det_xstats;         /*Extra statistics*/

static FILE *pssd_gcalfp, *pssd_tcalfp, *pssd_xlimfp, *pssd_detposfp;
static int open_pssd_files;           /*open calibration files flag*/



/*Initialisation Procedures*************************************************/


/*initialise experiment*/
/*ARGSUSED*/
int proc_expt_preinit(struct detectors *dp)
{
    if (loadgainoffset(CALIB_DIR "go.cal") < 0)
	return DET_OK;
    if (loadthresholds(CALIB_DIR "thresh.cal") < 0)
	return DET_OK;

    /* Initialise detector coordinates*/
    if (coord_init(dp->info->ndetel))
    {
	fprintf(stderr, "detprocs: Error initialising coordinates.\n");
	return DET_ABORTSORT;
    }

    open_pssd_files = 1;   /*flag subroutines to open pssd files*/

    return DET_OK;
}

/*initialise experiment*/
/*ARGSUSED*/
int proc_expt_init(struct detectors *dp)
{
    /* close any open files */
    if (pssd_gcalfp != NULL)
	fclose(pssd_gcalfp);
    if (pssd_tcalfp != NULL)
	fclose(pssd_tcalfp);
    if (pssd_xlimfp != NULL)
	fclose(pssd_xlimfp);
    if (pssd_detposfp != NULL)
	fclose(pssd_detposfp);

    memset(&det_xstats, 0, sizeof(det_xstats)); /*zero extra statistics*/
    if (ev_init(&evhead, &reac, NDET_MAX, N1_MAX, NP_MAX))
	return DET_ABORTSORT;   /*initialise event structure*/
    if (reac_init(REAC_SORT)) return DET_ABORTSORT; /*reaction definition*/

    return DET_OK;
}

/*amplifier signal*/
int proc_ampsig_init(struct detectors *dp)
{
    if (dp->threshmax > 8191)
	dp->threshmax = 8191;

    return DET_OK;
}

/*pssd detector*/
int proc_pssd_preinit(struct detectors *dp)
{
    /*open calibration files on first call to a PSSD*/
    if(open_pssd_files){
	open_pssd_files = 0;
	
	if ((pssd_gcalfp = fopen(CALIB_DIR "gsig.cal", "r")) != NULL)
	    fprintf(stderr, "detprocs: Reading file %s.\n", 
		    CALIB_DIR "gsig.cal");
	else
	    fprintf(stderr, "detprocs: No file %s, using default values.\n",
		    CALIB_DIR "gsig.cal");
	
	if ((pssd_tcalfp = fopen(CALIB_DIR "tsig.cal", "r")) != NULL)
	    fprintf(stderr, "detprocs: Reading file %s.\n", 
		    CALIB_DIR "tsig.cal");
	else
	    fprintf(stderr, "detprocs: No file %s, using default values.\n",
		    CALIB_DIR "tsig.cal");
	
	if ((pssd_xlimfp = fopen(CALIB_DIR "xlim.cal", "r")) != NULL)
	    fprintf(stderr, "detprocs: Reading file %s.\n", 
		    CALIB_DIR "xlim.cal");
	else
	    fprintf(stderr, "detprocs: No file %s, using default values.\n",
		    CALIB_DIR "xlim.cal");


	if ((pssd_detposfp = fopen(CALIB_DIR "detpos.cal", "r")) == NULL)
	{
	    fprintf(stderr, "detprocs: Can't find detector setup file \""
		    CALIB_DIR "detpos.cal\".\n");
	    return DET_ABORTSORT;
	}
	fprintf(stderr, "detprocs: Reading file " CALIB_DIR "detpos.cal.\n");  
	
    }

    if(coord_det(dp->number, pssd_detposfp)) /*generate detector coordinates*/
    {
	fprintf(stderr, "detprocs: Error generating coordinates.\n");
	return DET_ABORTSORT;
    }

    return DET_OK;
}

/*pssd strip*/
int proc_strip_init(struct detectors *dp)
{
    struct proc_strip_data *sp;
    char buffer[SUBS_BUFSIZE];

    if (dp->user == NULL)
	if ((dp->user = malloc(sizeof(struct proc_strip_data))) == NULL)
	{
	    fprintf(stderr, "detprocs: "
		    "Out of memory allocation strip data.\n");
	    return DET_ABORTSORT;
	}
    
    sp = (struct proc_strip_data *) dp->user;

    if (pssd_gcalfp == NULL)
	sp->gcal[0] = sp->gcal[1] = 1.0;
    else
    {
	if (mygetline(buffer, SUBS_BUFSIZE, pssd_gcalfp) == NULL)
	{
	    fprintf(stderr, "detprocs: Unexpected end of gcal file.\n");
	    return DET_ABORTSORT;
	}
	if (sscanf(buffer, "%lf %lf", sp->gcal, sp->gcal + 1) != 2)
	{
	    fprintf(stderr, "detprocs: Expected 2 x gcal, got \"%s\".\n",
		    buffer);
	    return DET_ABORTSORT;
	}
    }

    if (pssd_tcalfp == NULL)
	sp->toff[0] = sp->toff[1] = 0.0;
    else
    {
	if (mygetline(buffer, SUBS_BUFSIZE, pssd_tcalfp) == NULL)
	{
	    fprintf(stderr, "detprocs: Unexpected end of tcal file.\n");
	    return DET_ABORTSORT;
	}
	if (sscanf(buffer, "%lf %lf", sp->toff, sp->toff + 1) != 2)
	{
	    fprintf(stderr, "detprocs: Expected 2 x toff, got \"%s\".\n", 
		    buffer);
	    return DET_ABORTSORT;
	}
    }

    if (pssd_xlimfp == NULL)
    {
	sp->xlim[0] = 0;
	sp->xlim[1] = 256/*PSCALE*/;
    }
    else
    {
	if (mygetline(buffer, SUBS_BUFSIZE, pssd_xlimfp) == NULL)
	{
	    fprintf(stderr, "detprocs: Unexpected end of xlim file.\n");
	    return DET_ABORTSORT;
	}
	if (sscanf(buffer, "%lf %lf", sp->xlim, sp->xlim + 1) != 2)
	{
	    fprintf(stderr, "detprocs: Expected 2 x xlim, got \"%s\".\n", 
		    buffer);
	    return DET_ABORTSORT;
	}
    }


    return DET_OK;
}

/*Event Procedures***********************************************************/

/*Junk signals*/
int proc_junksig_event(int proc, struct detectors *dp)
{
    switch(proc)
    {  
    case PROC_CHECK:        /*Check hits*/
	switch(dp->hits)
	{
	case 1:		    /*partial hit*/
	    det_xstats.badjunk++;
	    return DET_OK;
	case 2:		    /*good hit*/
	{
	    det_xstats.goodjunk++;
	    return DET_OK;
	}
	default:           /*paranoia*/
	    fprintf(stderr, "detprocs: "
		    "junksig_event entered with hits = %d.\n",
		    dp->hits);
	    return DET_OK;
	}
    default:
	return DET_OK;
    }
}

/*Misc Procedures***********************************************************/


/*
 * write out the extra statistics
 */
void write_det_xstats(void)
{
    fprintf(stderr,"\n*** xtra detproc statistics ***\n\n");
    fprintf(stderr,"Channels with both energy and time : %d\n"
	    "Channels with amplitude but no time   : %d\n"
	    "Channels with time but no amplitude   : %d\n"
	    "Elements completely hit               : %d\n"
	    "Elements partially hit                : %d\n"
	    "Good junk                             : %d\n"
	    "Bad junk                              : %d\n",
	    det_xstats.chanok, det_xstats.badt, det_xstats.bada, 
	    det_xstats.detelok,det_xstats.detpartial, det_xstats.goodjunk,
	    det_xstats.badjunk);
}

int proc_expt_preinit_()
{
    return proc_expt_preinit(detector_);
}

int proc_expt_init_()
{
    return proc_expt_init(detector_);
}

int proc_ampsig_init_()
{
    return proc_ampsig_init(detector_);
}

int proc_pssd_preinit_()
{
    return proc_pssd_preinit(detector_);
}

int proc_strip_init_()
{
    return proc_strip_init(detector_);
}

int proc_junksig_event_(int *proc)
{
    return proc_junksig_event(*proc, detector_);
}


void xstats_inc_chanok_(int *val) { det_xstats.chanok += *val; }
void xstats_inc_badt_(int *val) { det_xstats.badt += *val; }
void xstats_inc_bada_(int *val) { det_xstats.bada += *val; }
void xstats_inc_detelok_(int *val) { det_xstats.detelok += *val; }
void xstats_inc_detpartial_(int *val) { det_xstats.detpartial += *val; }
void xstats_inc_goodjunk_(int *val) { det_xstats.goodjunk += *val; }
void xstats_inc_badjunk_(int *val) { det_xstats.badjunk += *val; }

void write_det_xstats_(void)
{
    write_det_xstats();
}

double proc_strip_gcal_(int *i)
{
    return ((struct proc_strip_data *) detector_->user)->gcal[*i];
}

double proc_strip_toff_(int *i)
{
    return ((struct proc_strip_data *) detector_->user)->toff[*i];
}

double proc_strip_xlim_(int *i)
{
    return ((struct proc_strip_data *) detector_->user)->xlim[*i];
}
