/* 
 * Event Structure routines
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "../evsubs.h" 

#define DEG (180/M_PI)
#define RAD (M_PI/180)

#define EV_SENSITIVE (~EV_STATE_Q2 & ~EV_STATE_PCM)

/*
#define DEBUG
*/

/*
 * create space for ev[1 to n particles] structure array
 */
int ev_init(struct evhead *evh, struct reac *reac, int numdets, int maxfold,
	    int np)
{
    int i;

    evh->reac = reac;
    if (evh->numdets != numdets)
    {
	if (evh->numdets != 0)
	{
	    free(evh->cmult);
	    free(evh->pmult);
	}
	if ((evh->cmult = (int *) calloc(numdets+1, sizeof(int))) == NULL)
	{
	    evh->numdets = 0;
	    evh->pmult = NULL;
	    return ABORT;
	}
	if ((evh->pmult = (int *) calloc(numdets+1, sizeof(int))) == NULL)
	{
	    evh->numdets = 0;
	    free(evh->cmult);
	    evh->cmult = NULL;
	    return ABORT;
	}
	evh->numdets = numdets;
    }
    if (evh->maxfold != maxfold)
    {
	if (evh->maxfold != 0)
	{
	    free(evh->n);
	    free(evh->p);
	}
	if ((evh->n = (int *) calloc(maxfold+1, sizeof(int))) == NULL)
	{
	    evh->maxfold = 0;
	    evh->p = NULL;
	    return ABORT;
	}
	if ((evh->p = (int *) calloc(maxfold+1, sizeof(int))) == NULL)
	{
	    evh->maxfold = 0;
	    free(evh->n);
	    evh->n = NULL;
	    return ABORT;
	}
	evh->maxfold = maxfold;
    }
    if (evh->numparticles != np)
    {
	if (evh->numparticles != 0)
	    free(evh->ev);
	if ((evh->ev = (particle *) calloc(np+1, sizeof(particle))) == NULL)
	{
	    evh->numparticles = 0;
	    return ABORT;
	}
	evh->numparticles = np;
	for(i=0; i<np; i++)
	    evh->ev[i].head = evh;
    }
	    
    return OK;
}

void ev_free(struct evhead *evh)
{
    if (evh->numdets != 0)
    {
	free(evh->pmult);
	free(evh->cmult);
	evh->numdets = 0;
    }
    if (evh->maxfold != 0)
    {
	free(evh->n);
	free(evh->p);
	evh->maxfold = 0;
    }
    if (evh->numparticles != 0)
    {
	free(evh->ev);
	evh->numparticles = 0;
    }
}

void ev_reset(struct evhead *evh)
{
    int i;
    particle *p;

    evh->type = 0;
    memset(evh->cmult, 0, (evh->numdets+1)*sizeof(int));
    memset(evh->pmult, 0, (evh->numdets+1)*sizeof(int));
    memset(evh->n, 0, (evh->maxfold+1)*sizeof(int));
    memset(evh->p, 0, (evh->maxfold+1)*sizeof(int));
    evh->p[1] = 1; /*singles always start at position 1*/
    for(i = evh->numparticles+1, p = evh->ev; i--; p++)
	ev_reset_particle(p);
}

void ev_reset_particle(particle *evp)
{
    evp->state = 0;
}

particle *ev_new_particle(struct evhead *evh)
{
    particle *evp;

    evh->pmult[0]++;
    evh->n[1]++;
    evh->n[0]++;
    evp = evh->ev + evh->n[1];

    ev_reset_particle(evp);
    evs_fold(evp, 1);
    evs_q(evp, 0.0);
    evs_hi(evp, NULL);
    evs_li(evp, NULL);

    return evp;
}

void evs_p_set(particle *evp)
{
#ifdef DEBUG
    fprintf(stderr, "Momentum of particle at %08x changed.\n", evp);
#endif
    evp->state = (evp->state & EV_SENSITIVE) | EV_STATE_P | EV_STATE_P_LOCKED;
    if (!(evp->state & EV_STATE_E_LOCKED))
	evp->state &= ~EV_STATE_E;
    if (!(evp->state & EV_STATE_M_LOCKED))
	evp->state &= ~EV_STATE_M;
    if (!(evp->state & EV_STATE_V_LOCKED))
	evp->state &= ~EV_STATE_V;
}

void evs_dir_set(particle *evp)
{
#ifdef DEBUG
    fprintf(stderr, "Momentum direction of particle at %08x changed.\n", evp);
#endif
    evp->state = (evp->state & EV_SENSITIVE & ~EV_STATE_P_LOCKED) | EV_STATE_P;

    if ((evp->state & EV_STATE_E_LOCKED) && (evp->state & EV_STATE_M_LOCKED))
	vector_scale_to2(ev_pvec(evp), 2.0*ev_m(evp)*ev_e(evp));
}

void evs_dir_from_coord(particle *evp, struct xy *cp)
{
#ifdef DEBUG
    fprintf(stderr, "Setting hit position of particle at %08x to %d@(%f,%f)\n",
	    evp, ev_det(evp), cp->x, cp->y);
#endif
    coord(ev_det(evp), cp, ev_pvec(evp));
    evs_dir_set(evp);
}

static void ev_debug(particle *evp)
{
    if (evp >= evhead.ev && evp <= evhead.ev + evhead.numparticles)
	fprintf(stderr, "Dump of particle %d at %08x\n", evp - evhead.ev, evp);
    else
	fprintf(stderr, "Dump of particle at %08x\n", (int) evp);
    fprintf(stderr, "    v = (%f,%f,%f)\n", evp->_private_.vx,
	    evp->_private_.vy, evp->_private_.vz);
    fprintf(stderr, "    p (raw) = (%f,%f,%f) = (%f,%f,%f)\n"
	    "        = (%f,%f,%f)\n",
	    evp->_private_.pvec.x, evp->_private_.pvec.y,
	    evp->_private_.pvec.z, evp->_private_.pvec.r,
	    evp->_private_.pvec.t, evp->_private_.pvec.p,
	    evp->_private_.pvec.r, evp->_private_.pvec.tx,
	    evp->_private_.pvec.ty);
    fprintf(stderr, "    e = %f, etot = %f, t = %f\n", evp->_private_.e,
	    evp->_private_.etot, evp->_private_.t);
    fprintf(stderr, "    pcm (raw) = (%f,%f,%f) = (%f,%f,%f)\n"
	    "        = (%f,%f,%f)\n",
	    evp->_private_.pcmvec.x, evp->_private_.pcmvec.y,
	    evp->_private_.pcmvec.z, evp->_private_.pcmvec.r,
	    evp->_private_.pcmvec.t, evp->_private_.pcmvec.p,
	    evp->_private_.pcmvec.r, evp->_private_.pcmvec.tx,
	    evp->_private_.pcmvec.ty);
    fprintf(stderr, "    q = %f, q2 = %f, m = %f, z = %d, fold = %d\n",
	    evp->_private_.q, evp->_private_.q2, evp->_private_.m,
	    evp->_private_.z, evp->_private_.fold);
    fprintf(stderr, "    hi = %08x, li = %08x, det = %d, seg = %d\n",
	    evp->_private_.hi, evp->_private_.li,
	    evp->_private_.det, evp->_private_.seg);
    fprintf(stderr, "    state:");
    if (evp->state & EV_STATE_E)
	fprintf(stderr, " E");
    if (evp->state & EV_STATE_E_LOCKED)
	fprintf(stderr, " E_LOCK");
    if (evp->state & EV_STATE_P)
	fprintf(stderr, " P");
    if (evp->state & EV_STATE_P_LOCKED)
	fprintf(stderr, " P_LOCK");
    if (evp->state & EV_STATE_M)
	fprintf(stderr, " M");
    if (evp->state & EV_STATE_M_LOCKED)
	fprintf(stderr, " M_LOCK");
    if (evp->state & EV_STATE_V)
	fprintf(stderr, " V");
    if (evp->state & EV_STATE_V_LOCKED)
	fprintf(stderr, " V_LOCK");
    if (evp->state & EV_STATE_Q2)
	fprintf(stderr, " Q2");
    if (evp->state & EV_STATE_PCM)
	fprintf(stderr, " PCM");
    putc('\n', stderr);
}

double ev_e_safe(particle *evp)
{
    if (evp->state & EV_STATE_E)
	return evp->_private_.e;

    evp->state |= EV_STATE_E;

    if ((evp->state & EV_STATE_P_LOCKED) && (evp->state & EV_STATE_M_LOCKED))
	evp->_private_.e = ev_p_r2(evp)/(2*ev_m(evp));
    else if ((evp->state & EV_STATE_V_LOCKED) &&
	     (evp->state & EV_STATE_M_LOCKED))
    {
	double x, y, z;

	x = ev_v_x(evp);
	y = ev_v_y(evp);
	z = ev_v_z(evp);
	evp->_private_.e = 0.5*ev_m(evp)*(x*x+y*y+z*z);
    }
    else
    {
	fprintf(stderr, "Unable to calculate particle energy from available "
		"information\n");
	ev_debug(evp);
	evp->_private_.e = 0.0;
    }
	
    return evp->_private_.e;
}

void evs_e(particle *evp, double val)
{
#ifdef DEBUG
    fprintf(stderr, "Setting energy of particle at %08x to %f\n", evp, val);
#endif
    evp->_private_.e = val;
    evp->state = (evp->state & EV_SENSITIVE) | EV_STATE_E | EV_STATE_E_LOCKED;
    if (!(evp->state & EV_STATE_P_LOCKED) && (evp->state & EV_STATE_P)
	&& (evp->state & EV_STATE_M_LOCKED))
	vector_scale_to2(ev_pvec(evp), 2.0*ev_m(evp)*ev_e(evp));
    if (!(evp->state & EV_STATE_M_LOCKED))
	evp->state &= ~EV_STATE_M;
    if (!(evp->state & EV_STATE_V_LOCKED))
	evp->state &= ~EV_STATE_V;
}

void evs_m(particle *evp, double val)
{
#ifdef DEBUG
    fprintf(stderr, "Setting mass of particle at %08x to %f\n", evp, val);
#endif
    evp->_private_.m = val;
    evp->state = (evp->state & EV_SENSITIVE) | EV_STATE_M | EV_STATE_M_LOCKED;
    if (!(evp->state & EV_STATE_P_LOCKED) && (evp->state & EV_STATE_P) &&
	(evp->state & EV_STATE_E_LOCKED))
	vector_scale_to2(ev_pvec(evp), 2.0*ev_m(evp)*ev_e(evp));
    if (!(evp->state & EV_STATE_E_LOCKED))
	evp->state &= ~EV_STATE_E;
    if (!(evp->state & EV_STATE_V_LOCKED))
	evp->state &= ~EV_STATE_V;
}

double ev_q2_safe(particle *evp)
{
    double mrec;
    struct reac *reac;

    if (evp->state & EV_STATE_Q2)
	return evp->_private_.q2;

    reac = evp->head->reac;

    mrec = reac->mass[1]+reac->mass[2]-ev_m(evp);
    if(mrec > 0)
        evp->_private_.q2 = ((ev_m(evp)+mrec)*ev_e(evp) -
			     (mrec-reac->mass[1])*reac->e1 -
			     reac->p1*ev_p_z(evp))/mrec;
    else
        evp->_private_.q2 = 0.0;

    return evp->_private_.q2;
}

void evs_p_xyz(particle *evp, double x, double y, double z)
{
#ifdef DEBUG
    fprintf(stderr, "Setting momentum of particle at %08x to (%f,%f,%f)\n",
	    evp, x, y, z);
#endif
    vector_set_xyz(ev_pvec(evp), x, y, z);
    evs_p_set(evp);
}

void evs_p_rtp(particle *evp, double r, double t, double p)
{
    vector_set_rtp(ev_pvec(evp), r, t, p);
    evs_p_set(evp);
}

void evs_p_rtpd(particle *evp, double r, double td, double pd)
{
    vector_set_rtpd(ev_pvec(evp), r, td, pd);
    evs_p_set(evp);
}

void evs_p_rtt(particle *evp, double r, double tx, double ty)
{
    vector_set_rtt(ev_pvec(evp), r, tx, ty);
    evs_p_set(evp);
}

void evs_p_rttd(particle *evp, double r, double txd, double tyd)
{
    vector_set_rttd(ev_pvec(evp), r, txd, tyd);
    evs_p_set(evp);
}

void evs_dir_xyz(particle *evp, double x, double y, double z)
{
#ifdef DEBUG
    fprintf(stderr, "Setting momentum direction of particle at %08x to (%f,%f,%f)\n",
	    evp, x, y, z);
#endif
    vector_set_xyz(ev_pvec(evp), x, y, z);
    evs_dir_set(evp);
}

void evs_dir_tp(particle *evp, double t, double p)
{
    vector_set_rtp(ev_pvec(evp), 1.0, t, p);
    evs_dir_set(evp);
}

void evs_dir_tpd(particle *evp, double td, double pd)
{
    vector_set_rtpd(ev_pvec(evp), 1.0, td, pd);
    evs_dir_set(evp);
}

void evs_dir_tt(particle *evp, double tx, double ty)
{
    vector_set_rtt(ev_pvec(evp), 1.0, tx, ty);
    evs_dir_set(evp);
}

void evs_dir_ttd(particle *evp, double txd, double tyd)
{
    vector_set_rttd(ev_pvec(evp), 1.0, txd, tyd);
    evs_dir_set(evp);
}

particle *ev_calc_pcm(particle *evp)
{
    if (!(evp->state & EV_STATE_PCM))
    {
	evp->state |= EV_STATE_PCM;
	
	vector_set_xyz(&(evp->_private_.pcmvec), ev_p_x(evp), ev_p_y(evp),
		       ev_p_z(evp) - ev_m(evp)*evp->head->reac->vcm);
    }

    return evp;
}

double ev_m_safe(particle *evp)
{
    if (evp->state & EV_STATE_M)
	return evp->_private_.m;

    evp->state |= EV_STATE_M;

    if ((evp->state & EV_STATE_P_LOCKED) && (evp->state & EV_STATE_E_LOCKED))
	evp->_private_.m = ev_p_r2(evp)/(2*ev_e(evp));
    else if ((evp->state & EV_STATE_P_LOCKED) &&
	     (evp->state & EV_STATE_V_LOCKED))
    {
	double x, y, z;

	x = ev_v_x(evp);
	y = ev_v_y(evp);
	z = ev_v_z(evp);
	evp->_private_.m = sqrt(ev_p_r2(evp)/(x*x+y*y+z*z));
    }
    else
    {
	fprintf(stderr, "Unable to calculate particle mass from available "
		"information\n");
	ev_debug(evp);
	evp->_private_.m = 0.0;
    }
	
    return evp->_private_.m;
}

double evhead_etot(struct evhead *evh)
{
    particle *evp;
    int i;
    double e;
    
    for(evp = evh->ev + evh->p[1], i = evh->n[1], e=0; i--; evp++)
	e += ev_e(evp);

    return e;
}

void evs_v_xyz(particle *evp, double x, double y, double z)
{
#ifdef DEBUG
    fprintf(stderr, "Setting velocity of particle at %08x to (%f,%f,%f)\n",
	    evp, x, y, z);
#endif
    evp->_private_.vx = x;
    evp->_private_.vy = y;
    evp->_private_.vz = z;
    evp->state = (evp->state & EV_SENSITIVE) | EV_STATE_V | EV_STATE_V_LOCKED;
    if (!(evp->state & EV_STATE_P_LOCKED) && (evp->state & EV_STATE_P)
	&& (evp->state & EV_STATE_M_LOCKED))
	vector_scale_to2(ev_pvec(evp), 2.0*ev_m(evp)*ev_e(evp));
    if (!(evp->state & EV_STATE_M_LOCKED))
	evp->state &= ~EV_STATE_M;
    if (!(evp->state & EV_STATE_E_LOCKED))
	evp->state &= ~EV_STATE_E;
}

