/* ev.h  event structure header file */

typedef struct ev_struct {
    struct { /* To prevent direct access to the structure */
	double vx;             /*velocity in x*/
	double vy;             /*velocity in y*/
	double vz;             /*velocity in z*/
	struct vector pvec;    /*momentum vector handled by vector package*/
	double e;              /*energy*/
	double etot;           /*running total of daughter energies*/
	double t;              /*time*/
	struct vector pcmvec;  /*Momentum vector in cm frame*/
	double q;              /*excitation q*/
	double q2;             /*two-body q*/
	double m;              /*mass*/
	int z;                 /*charge*/
	int fold;              /*particle reconstructed from fold particles*/
	struct ev_struct *hi;  /*heavy-ion daughter*/
	struct ev_struct *li;  /*light-ion daughter*/
	int det;               /*detector hit*/
	int seg;               /*detector segment hit*/
    } _private_;
    int state;
    struct evhead *head;
} particle;

struct evhead {
    int numdets;
    int maxfold;
    int numparticles;
    int *cmult; /*chan multiplicity 0=>total n=> in det n*/
    int *pmult; /*particle multiplicity 0=>total n=> in det n*/
    int *n;   /*num particles of each fold 0=>total 1=>singles..*/
    int *p;  /*particle positions in the array for each fold*/
    int type;
    particle *ev;
    struct reac *reac;
};

#define EVHEAD_STATUS_ETOT 1

#define EVHEAD_RESET { 0, 0, 0, NULL, NULL, NULL, NULL, 0, NULL, NULL}

extern struct evhead evhead;

#define EV_STATE_E 1
#define EV_STATE_E_LOCKED 2
#define EV_STATE_P 4
#define EV_STATE_P_LOCKED 8
#define EV_STATE_V 16  /* Not implemented yet */
#define EV_STATE_V_LOCKED 32 /* Not implemented yet */
#define EV_STATE_M 64
#define EV_STATE_M_LOCKED 128
#define EV_STATE_Q2 256
#define EV_STATE_PCM 512

int ev_init(struct evhead *, struct reac *, int, int, int);
void ev_reset(struct evhead *);
void wrtev(struct evhead *, int);

/*
 * All access to the ev structure must go through a wrapper. Where possible,
 * these will be implemented as macros.
 */

#define ev_v_x(ev) ((double) ((ev)->_private_.vx))
#define ev_v_y(ev) ((double) ((ev)->_private_.vy))
#define ev_v_z(ev) ((double) ((ev)->_private_.vz))
#define ev_e(ev) ((double) (((ev)->state & EV_STATE_E) ? \
			   (ev)->_private_.e : ev_e_safe(ev)))
#define ev_etot(ev) ((double) ((ev)->_private_.etot))
#define ev_t(ev) ((double) ((ev)->_private_.t))
#define ev_q(ev) ((double) ((ev)->_private_.q))
#define ev_q2(ev) ((double) (((ev)->state & EV_STATE_Q2) ? \
			    (ev)->_private_.q2 : ev_q2_safe(ev)))
#define ev_m(ev) ((double) (((ev)->state & EV_STATE_M) ? \
			   (ev)->_private_.m : ev_m_safe(ev)))
#define ev_z(ev) ((int) ((ev)->_private_.z))
#define ev_fold(ev) ((int) ((ev)->_private_.fold))
#define ev_hi(ev) ((particle *) ((ev)->_private_.hi))
#define ev_li(ev) ((particle *) ((ev)->_private_.li))
#define ev_det(ev) ((int) ((ev)->_private_.det))
#define ev_seg(ev) ((int) ((ev)->_private_.seg))

#define evs_etot(ev, val) ((ev)->_private_.etot = (val))
#define evs_t(ev, val) ((ev)->_private_.t = (val))
#define evs_q(ev, val) ((ev)->_private_.q = (val))
#define evs_z(ev, val) ((ev)->_private_.z = (val))
#define evs_fold(ev, val) ((ev)->_private_.fold = (val))
#define evs_hi(ev, val) ((ev)->_private_.hi = (val))
#define evs_li(ev, val) ((ev)->_private_.li = (val))
#define evs_det(ev, val) ((ev)->_private_.det = (val))
#define evs_seg(ev, val) ((ev)->_private_.seg = (val))

/* The momentum is handled differently via the vector package */

#define ev_pvec(ev) (&((ev)->_private_.pvec))

#define ev_p_r(ev) (vector_r(ev_pvec(ev)))
#define ev_p_x(ev) (vector_x(ev_pvec(ev)))
#define ev_p_y(ev) (vector_y(ev_pvec(ev)))
#define ev_p_z(ev) (vector_z(ev_pvec(ev)))
#define ev_p_t(ev) (vector_t(ev_pvec(ev)))
#define ev_p_p(ev) (vector_p(ev_pvec(ev)))
#define ev_p_td(ev) (vector_td(ev_pvec(ev)))
#define ev_p_pd(ev) (vector_pd(ev_pvec(ev)))
#define ev_p_tx(ev) (vector_tx(ev_pvec(ev)))
#define ev_p_ty(ev) (vector_ty(ev_pvec(ev)))
#define ev_p_txd(ev) (vector_txd(ev_pvec(ev)))
#define ev_p_tyd(ev) (vector_tyd(ev_pvec(ev)))
#define ev_p_r2(ev) (vector_r2(ev_pvec(ev)))

#define ev_pcmvec(ev) (&((((ev)->state & EV_STATE_PCM) ? (ev) : \
			  ev_calc_pcm(ev))->_private_.pcmvec))

#define ev_pcm_r(ev) (vector_r(ev_pcmvec(ev)))
#define ev_pcm_x(ev) (vector_x(ev_pcmvec(ev)))
#define ev_pcm_y(ev) (vector_y(ev_pcmvec(ev)))
#define ev_pcm_z(ev) (vector_z(ev_pcmvec(ev)))
#define ev_pcm_t(ev) (vector_t(ev_pcmvec(ev)))
#define ev_pcm_p(ev) (vector_p(ev_pcmvec(ev)))
#define ev_pcm_td(ev) (vector_td(ev_pcmvec(ev)))
#define ev_pcm_pd(ev) (vector_pd(ev_pcmvec(ev)))
#define ev_pcm_r2(ev) (vector_r2(ev_pcmvec(ev)))
#define ev_pcm_tx(ev) (vector_tx(ev_pvec(ev)))
#define ev_pcm_ty(ev) (vector_ty(ev_pvec(ev)))
#define ev_pcm_txd(ev) (vector_txd(ev_pvec(ev)))
#define ev_pcm_tyd(ev) (vector_tyd(ev_pvec(ev)))

void ev_reset_particle(particle *);
particle *ev_new_particle(struct evhead *);
void evs_p_set(particle *);
void evs_dir_set(particle *);
void evs_dir_from_coord(particle *, struct xy *);
double ev_e_safe(particle *);
void evs_e(particle *, double);
void evs_m(particle *, double);
double ev_q2_safe(particle *);
void evs_p_xyz(particle *, double, double, double);
void evs_p_rtp(particle *, double, double, double);
void evs_p_rtpd(particle *, double, double, double);
void evs_p_rtt(particle *, double, double, double);
void evs_p_rttd(particle *, double, double, double);
void evs_dir_xyz(particle *, double, double, double);
void evs_dir_tp(particle *, double, double);
void evs_dir_tpd(particle *, double, double);
void evs_dir_tt(particle *, double, double);
void evs_dir_ttd(particle *, double, double);
particle *ev_calc_pcm(particle *);
double ev_m_safe(particle *);
double evhead_etot(struct evhead *);
void evs_v_xyz(particle *, double, double, double);
