#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/mman.h>
#include "getmass.h"

#define NOMASS -1e31

#ifndef MASSFILE
#define MASSFILE "./mass.raw"
#endif

static const char *massfile = MASSFILE;

double getmass_nz(int n, int z)
{
    static void *massarea = MAP_FAILED;
    static unsigned short *index = NULL, *ip;
    static int maxz;
    int minn, maxn;
    char *p;
    double m;

    if (massarea == MAP_FAILED)
    {
	int fd, i, n;
	struct stat st;
	
	if ((fd = open(massfile, O_RDONLY, 0)) == -1)
	{
	    perror(massfile);
	    return NOMASS;
	}

	if (fstat(fd, &st) == -1)
	{
	    perror("getmass, fstat");
	    close(fd);
	    return NOMASS;
	}

	if (st.st_size > 65535)
	{
	    fprintf(stderr, "getmass: Mass file too large to be indexed.\n");
	    close(fd);
	    return NOMASS;
	}

	if ((massarea = mmap(NULL, st.st_size, PROT_READ, MAP_PRIVATE, fd,
			     0)) == MAP_FAILED)
	{
	    perror("getmass, mmap");
	    close(fd);
	    return NOMASS;
	}

	close(fd);

	for(maxz = 0, i = st.st_size, p = (char *) massarea; i > 0; maxz++)
	{
	    /*LINTED*/
	    minn = *((unsigned short *) p); 
	    /*LINTED*/
	    maxn = ((unsigned short *) p)[1];
	    n = 2*sizeof(unsigned short) + (maxn-minn+1)*sizeof(float);
	    p += n;
	    i -= n;
	}
	
	if ((index = (unsigned short *)
	     malloc((maxz+1)*sizeof(unsigned short))) == NULL)
	{
	    perror("getmass, malloc");
	    munmap(massarea, st.st_size);
	    return NOMASS;
	}

	for(ip = index, i = maxz, p = (char *) massarea; i; i--)
	{
	    /*LINTED*/
	    minn = *((unsigned short *) p);
	    /*LINTED*/
	    maxn = ((unsigned short *) p)[1];
	    *ip++ = (p - (char *) massarea);
	    p += 2*sizeof(unsigned short) + (maxn-minn+1)*sizeof(float);
	}
    }

    if (z < 0 || z > maxz)
    {
	fprintf(stderr, "getmass: Z of %d is out of range.\n", z);
	return NOMASS;
    }

    p = (char *) massarea + index[z];

    /*LINTED*/
    minn = *((unsigned short *) p);
    /*LINTED*/
    maxn = ((unsigned short *) p)[1];

    if (n < minn || n > maxn)
    {
	fprintf(stderr, "getmass: No data for N=%d, Z=%d.\n", n, z);
	return NOMASS;
    }

    /*LINTED*/
    if ((m = *(float *) (p + 2*sizeof(unsigned short) +
			 (n - minn)*sizeof(float))) < UNKNOWNMASS)
	fprintf(stderr,
		"getmass: Mass of nucleus with N=%d, Z=%d not known.\n", n, z);

    return m;
}

double getmass_az(int a, int z)
{
    return getmass_nz(a-z, z);
}

double getmass_nz_(int *n, int *z)
{
    return getmass_nz(*n, *z);
}

double getmass_az_(int *a, int *z)
{
    return getmass_az(*a, *z);
}

static const char element_names[] =
" n"
"H "                                                                       "He"
"Li""Be"                                                       "B C N O F ""Ne"
"Na""Mg"                                                       "AlSiP S Cl""Ar"
"K ""Ca"                                "ScTiV CrMnFeCoNiCuZn" "GaGeAsSeBr""Kr"
"Rb""Sr"                                "Y ZrNbMoTcRuRhPdAgCd" "InSnSbTeI ""Xe"
"Cs""Ba" "LaCePrNdPmSmEuGdTbDyHoErTmYb" "LuHfTaW ReOsIrPtAuHg" "TlPbBiPoAt""Rn"
"Fr""Ra" "AcThPaU NpPuAmCmBkCfEsFmMdNo" "LrRfDbSgBhHsMt";

static const char greek_initials[] = "nubtqphsoe";

#define NUM_ELEMENTS (sizeof(element_names)/2)

char *nucleusname_az_r(char *name, int a, int z)
{
    char *p;
    const char *q;

    if (a > 9999 || z > 999)
	return NULL;

    sprintf(name, "%d", a);
    p = name+strlen(name);
    if (z >= NUM_ELEMENTS)
    {
	*p++ = toupper(greek_initials[z/100]);
	*p++ = toupper(greek_initials[(z/10) % 10]);
	*p++ = toupper(greek_initials[z % 10]);
	*p = '\0';
    }
    else
    {
	q = element_names + z*2;
	*p++ = *q++;
	*p++ = *q;
	*p = '\0';
    }

    return name;
}

char *nucleusname_nz_r(char *name, int n, int z)
{
    return nucleusname_az_r(name, n+z, z);
}

char *nucleusname_az(int a, int z)
{
    static char name[8];

    return nucleusname_az_r(name, a, z);
}

char *nucleusname_nz(int n, int z)
{
    return nucleusname_az(n+z, z);
}

void nucleus_az(int *a, int *z, const char *name)
{
    const char *p, *q, *r;
    int c1, c2, c3;

    for(p = name; isdigit(*p); p++)
	;
    if (p == name || *p == '\0')
    {
	*a = *z = -1;
	return;
    }
    *a = atoi(name);

    if (*p == ',')
    {
	p++;
	if (!isdigit(*p))
	{
	    *a = *z = -1;
	    return;
	}
	*z = atoi(p);
	return;
    }

    if (*p == '.')
	c1 = ' ';
    else
	c1 = toupper(*p);
    if (isalpha(p[1]))
    {
	c2 = tolower(p[1]);
	c3 = p[2];
    }
    else
    {
	c2 = ' ';
	c3 = 0;
    }
    
    for(q = element_names; *q != '0'; q += 2)
    {
	if (*q == c1 && q[1] == c2)
	{
	    *z = (q - element_names)/2;
	    return;
	}
    }


    if (c3 == 0 ||
	(p = strchr(greek_initials, tolower(c1))) == NULL ||
	(q = strchr(greek_initials, c2)) == NULL ||
	(r = strchr(greek_initials, c3)) == NULL)
    {
	*a = *z = -1;
	return;
    }

    *z = (p-greek_initials)*100 + (q-greek_initials)*10 + r-greek_initials;
}

void nucleus_nz(int *n, int *z, const char *name)
{
    nucleus_az(n, z, name);
    if (*n >= 0)
	*n -= *z;
}

void nucleusname_az_(char *name, int *a, int *z)
{
    nucleusname_az_r(name, *a, *z);
}

void nucleusname_nz_(char *name, int *n, int *z)
{
    nucleusname_nz_r(name, *n, *z);
}

void nucleus_az_(int *a, int *z, const char *name)
{
    nucleus_az(a, z, name);
}

void nucleus_nz_(int *n, int *z, const char *name)
{
    nucleus_nz(n, z, name);
}

double getmass_name(char *name)
{
    int a, z;

    nucleus_az_(&a, &z, name);
    if (a < 0)
	return NOMASS;
    return getmass_az(a, z);
}

double getmass_name_(char *name)
{
    return getmass_name(name);
}

