#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#define MAXN 250
#define MAXZ 150

#define ME_UNKNOWN 0
#define ME_MEASURED 1
#define ME_CALC 2

#define BUFSIZE 256

int main()
{
    /* Files are listed in order of preference */
    static const char *file[] = { "mass_rmd.mas95", "mass93.dat"};
    static const int meposn[] = { 28, 126 };
    static const int calcposn[] = { -1, 116};
    static const int nposn[] = {5, 6};
    static const int zposn[] = {10, 1};
    static const int hash[] = { 1, 0 };
    static const int skip[] = {39, 0};
    static const double factor[] = { 0.001, 1 };

    int i, n, z, line, calc, minn, maxn, maxz;
    double me[MAXZ][MAXN], m;
    char buffer[BUFSIZE], *p, status[MAXZ][MAXN];
    FILE *fp;
    float fl;
    unsigned short sh;

    memset(status, ME_UNKNOWN, sizeof(status));

    for(i=0; i < sizeof(file)/sizeof(char *); i++)
    {
	if ((fp = fopen(file[i], "r")) == NULL)
	{
	    perror(file[i]);
	    return 1;
	}

	line = 0;

	for(n = skip[i]; n--;)
	{
	    line++;
	    if (fgets(buffer, BUFSIZE, fp) == NULL)
	    {
		fprintf(stderr, "Unexpected end of file while skipping header "
			"at line %d of file %s\n", line, file[i]);
		fclose(fp);
		return 1;
	    }
	}
	
	while(fgets(buffer, BUFSIZE, fp) != NULL)
	{
	    line++;
	    calc = 0;
	    if (hash[i])
	    {
		for(p = buffer + meposn[i], n = 10; n-- && *p != '\0'; p++)
		    if (*p == '#')
		    {
			*p = '.';
			calc = 1;
		    }
	    }
	    if (sscanf(buffer + nposn[i], "%d", &n) != 1)
	    {
		fprintf(stderr, "Unable to read N at line %d of file %s\n",
			line, file[i]);
		return 1;
	    }
	    if (n > MAXN)
	    {
		fprintf(stderr, "N out of range at line %d of file %s\n",
			line, file[i]);
		return 1;
	    }
	    if (sscanf(buffer + zposn[i], "%d", &z) != 1)
	    {
		fprintf(stderr, "Unable to read Z at line %d of file %s\n",
			line, file[i]);
		return 1;
	    }
	    if (z > MAXZ)
	    {
		fprintf(stderr, "Z out of range at line %d of file %s\n",
			line, file[i]);
		return 1;
	    }
	    buffer[meposn[i] + 10] = '\0';
	    if (sscanf(buffer + meposn[i], "%lf", &m) != 1)
	    {
		if (calcposn[i] < 0 || 
		    sscanf(buffer + calcposn[i], "%lf", &m) != 1)
		{
		    fprintf(stderr, "Unable to read mass excess at line "
			    "%d of file %s\n", line, file[i]);
		    return 1;
		}
		calc = 1;
	    }
	    m = m * factor[i];
	    switch(status[z][n])
	    {
	    case ME_UNKNOWN:
		me[z][n] = m;
		status[z][n] = calc ? ME_CALC : ME_MEASURED;
		break;
	    case ME_MEASURED:
#ifdef DEBUG
		if (!calc)
		{
		    if (fabs(m - me[z][n]) > 0.3)
			fprintf(stderr, "Mass %d,%d doubly measured, was "
				"%.2f, could be %.2f, difference %.2f\n", n, z,
				me[z][n], m, m - me[z][n]);
		}
		else
		    fprintf(stderr, "Mass %d,%d measured and calculated, "
			    "meas %.2f calc %.2f, diff %.2f\n", n, z,
			    me[z][n], m, m - me[z][n]);
#endif
		break;
	    case ME_CALC:
		if (!calc)
		{
#ifdef DEBUG
		    fprintf(stderr, "Mass %d,%d calculated and measured, "
			    "calc %.2f meas %.2f, diff %.2f\n", n, z,
			    me[z][n], m, m - me[z][n]);	
#endif
		    me[z][n] = m;
		    status[z][n] = calc ? ME_CALC : ME_MEASURED;
		}
#ifdef DEBUG
		else
		    if (fabs(m - me[z][n]) > 0.3)
			fprintf(stderr, "Mass %d,%d doubly calculated, was "
				"%.2f, could be %.2f, difference %.2f\n", n, z,
				me[z][n], m, m - me[z][n]);
#endif
		break;
	    }
	}

	fclose(fp);
    }

#ifdef DEBUG
    if ((fp = fopen("mass.dat", "w")) == NULL)
    {
	perror("mass.dat");
	return 1;
    }

    for(z = 0; z < MAXZ; z++)
	for(n = 0; n < MAXN; n++)
	    if (status[z][n] != ME_UNKNOWN)
		fprintf(fp, "%3d %3d %10.4f\n", n, z, me[z][n]);

    fclose(fp);
#endif

    for(maxz = MAXZ; maxz--; )
	for(n = 0; n < MAXN; n++)
	    if (status[maxz][n] != ME_UNKNOWN)
		goto GOT_MAXZ; /* Aargh, C doesn't have a break n command */

 GOT_MAXZ:
    ;
		
    if ((fp = fopen("mass.raw", "w")) == NULL)
    {
	perror("mass.raw");
	return 1;
    }

    for(z = 0; z <= maxz; z++)
    {
	for(minn = 0; minn < MAXN; minn++)
	    if (status[z][minn] != ME_UNKNOWN)
		break;
	for(maxn = MAXN; maxn--; )
	    if (status[z][maxn] != ME_UNKNOWN)
		break;
	sh = minn;
	fwrite(&sh, sizeof(unsigned short), 1, fp);
	sh = maxn;
	fwrite(&sh, sizeof(unsigned short), 1, fp);
	for(n = minn; n <= maxn; n++)
	{
	    if (status[z][minn] != ME_UNKNOWN)
		fl = me[z][n];
	    else
		fl = -1e31;
	    fwrite(&fl, sizeof(float), 1, fp);
	}
    }

    fclose(fp);

    return 0;
}









