/* 
 * Basel coordinate ridge projection                     SPGC
 *
 * usage "ridgeproj -s specfile [-i xll xul] [-yoff yoffset]"

 * Notes:
 *     -i xll xul to include only X in range for the projection
 *     Assumes a 2d XY spectrum with x=phis,y=chi (in degrees) as input
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#define CORR_SIZE 1000
int read_args(int, char **);     /*read command line args*/

static int xll, xul, yoff;
static char *specfile;                  /*spectrum*/

int main(int argc, char **argv)
{
    char  buf[10];
    int corr[CORR_SIZE+1],n,x,y,z,proj_bin;
    float xproj,k,l;
    char *ofile="ridgeproj.out";     /*file to write gain values into*/

    FILE *sf,*of;
    
    
    /*initialise*/
    for(n=1; n < CORR_SIZE; n++)
	corr[n] = 0;
    xll = -1;
    xul = CORR_SIZE+1;
    yoff = 0;
    
    read_args(argc,argv);

    if((sf = fopen(specfile, "r")) == NULL)
    {
	perror(specfile);
	return 1;
    }
    if((of = fopen(ofile, "w")) == NULL)
    {
	perror(ofile);
	return 1;
    }

    fprintf(stderr, "Enter k,l :");
    if(scanf("%f %f",&k, &l) != 2){
	fprintf(stderr,"Error!\n");
	return 1;
    }

    fscanf(sf,"#X-Y 2D spectrum named %s", buf);
    fprintf(stderr,"\n Analysing Spectrum %s\n", buf);
    
    while(fscanf(sf, "%d %d %d", &x, &y, &z) == 3){
	if(x >= xll && x <= xul){
	    y += yoff;
	    xproj = ((float) x) + ((float) y)*k/(l+0.5);
	    proj_bin = (int) (xproj + 0.5);
	    if(proj_bin > CORR_SIZE-1){
		fprintf(stderr,">CORR_SIZE, Error!\n");
		return 1;
	    }
	    corr[proj_bin] += z;
	}
    }
    
    for(n=1; n < CORR_SIZE; n++)
	fprintf(of, " %d %d\n",n,corr[n]); 

    fclose(sf);
    fclose(of);
    fprintf(stderr,"Output in ridgeproj.out\n");

    return 0;
}

/*
 * Read command line options
 */
int read_args(int argc, char **argv)
{
    int i; 
    
    for(i=1 ; i< argc; i++){
	if (strncmp(argv[i],"-i",2) == 0) {
	    if (argv[i+1] != NULL && argv[i+2] != NULL) {
		xll = (int) atoi(argv[i+1]);
		xul = (int) atoi(argv[i+2]);
		fprintf(stderr,"Only including range %d < X > %d \n",
			xll, xul);
		i +=2;
	    }
	    else{
		fprintf(stderr,"Error reading phi inclusion range !\n");
		return(-1);
	    }
	}
	else if (strncmp(argv[i],"-s",2) == 0) {
	    if (argv[i+1] != NULL)
		specfile = argv[i+1];
	    i++;
	}
	else if (strncmp(argv[i],"-yoff",5) == 0) {
	    if (argv[i+1] != NULL){
		yoff = (int) atoi(argv[i+1]);
		i++;
		fprintf(stderr,"Offset on Y = %d\n", yoff);
	    }
	}
	else{
	    fprintf(stderr,"ridgeproj: command line option "
		    "\"%s\" unrecognised\n",argv[i]);
	    fprintf(stderr, "\nUsage: ridgeproj -s specfile"
		    " [-i xll xul] [-yoff yoffset]\n\n");
	    return(-1);
	}
    }
    return(0);
}
