#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <malloc.h>
#include <stdlib.h>
#include "structure.h"
#include "properties.h"

struct alllist {
    struct detectors *dp;
    struct alllist *next;
};

static struct information *infohead = NULL;
static struct alllist *allhead = NULL;

static int loaddetectors(char *);
static void loaddetector(FILE *, char *);
static struct detectors *instance(char *, struct detectors *);

static void destroyall(void)
{
    struct alllist *ap = allhead, *atmp;
    struct information *ip = infohead, *itmp;
    struct namelist *np, *ntmp;

    while(ap != NULL)
    {
	if (ap->dp->user != NULL)
	    free(ap->dp->user);
	free(ap->dp);
	atmp = ap->next;
	free(ap);
	ap = atmp;
    }

    allhead = NULL;

    while (ip != NULL)
    {
	free(ip->name);
	for(np = ip->childnames; np != NULL; )
	{
	    free(np->name);
	    ntmp = np->next;
	    free(np);
	    np = ntmp;
	}
	itmp = ip->next;
	free(ip);
	ip = itmp;
    }

    infohead = NULL;
}

struct detectors *loadexperiment(char *filename)
{
    struct detectors *expt;

    if (loaddetectors(filename) < 0)
    {
	fprintf(stderr, "struct: Unable to load experiment file.\n");
	return(NULL);
    }

    expt = instance("experiment", NULL);
    expt->number = 1;

    return expt;
}

#define TOKENBUF_LEN 40
#define TOKEN_STRING 256
#define TOKEN_INTEGER 257

static int token, line;
static char tokenbuf[TOKENBUF_LEN], *currentfile;

static int gettoken(FILE *fp)
{
    int c;
    char *p = tokenbuf;

    do {
	c = getc(fp);
	if (c == '!' || c == '#')
	    while((c = getc(fp)) != '\n' && c != EOF)
		;
	if (c == '\n')
	    line++;
    } while(isspace(c));
    if (isalpha(c) || c == '_')
    {
	do {
	    if (p - tokenbuf < TOKENBUF_LEN-1)
		*p++ = tolower(c);
	    c = getc(fp);
	} while(isalnum(c) || c == '_');
	ungetc(c, fp);
	*p++ = '\0';
	return(TOKEN_STRING);
    }
    if (isdigit(c))
    {
	do {
	    if (p - tokenbuf < TOKENBUF_LEN-1)
		*p++ = c;
	    c = getc(fp);
	} while(isdigit(c));
	ungetc(c, fp);
	*p++ = '\0';
	return(TOKEN_INTEGER);
    }
    return(c);
}

static void unexpectedtoken()
{
    if (token == EOF)
	fprintf(stderr, "struct: Unexpected end of file");
    else if (token < 256)
    {
	if (!isprint(token))
	    fprintf(stderr, "struct: Unexpected character %d", token);
	else
	    fprintf(stderr, "struct: Unexpected token \"%c\"", token);
    }
    else
	fprintf(stderr, "struct: Unexpected token \"%s\"", tokenbuf);
    fprintf(stderr, " at line %d of file %s\n", line, currentfile);
}

static int loaddetectors(char *filename)
{
    FILE *fp;

    currentfile = filename;
    line = 1;

    destroyall();

    if ((fp = fopen(filename, "r")) == NULL)
    {
	perror(filename);
	return(-1);
    }

    token = gettoken(fp);
    while(token != EOF)
    {
	switch(token)
	{
	case TOKEN_STRING:
	    if (!strcmp(tokenbuf, "detector"))
		loaddetector(fp, NULL);
	    else if (!strcmp(tokenbuf, "experiment"))
		loaddetector(fp, "experiment");
	    else
	    {
		unexpectedtoken();
		token = EOF;
	    }
	    break;
	default:
	    unexpectedtoken();
	    token = EOF;
	    break;
	}
    }

    fclose(fp);
    return(0);
}

static void loaddetector(FILE *fp, char *name)
{
    struct information *ip;
    struct namelist *nametail;
    int num;

    if (name == NULL)
    {
	token = gettoken(fp);

	if (token != TOKEN_STRING)
	{
	    unexpectedtoken();
	    return;
	}
	name = tokenbuf;
    }

    if ((ip = (struct information *) malloc(sizeof(struct information)))
	== NULL)
    {
	fprintf(stderr, "struct: Out of memory at line %d of file %s.\n", line,
		currentfile);
	return;
    }

    ip->name = strdup(name);
    ip->ndetel = 0;
    ip->childnames = NULL;
    ip->preinit.c = NULL;
    ip->init.c = NULL;
    ip->preevent.c = NULL;
    ip->event.c = NULL;
    ip->next = infohead;
    infohead = ip;

    token = gettoken(fp);

    while(token == TOKEN_STRING)
    {
	if (!strcmp(tokenbuf, "comprises"))
	{
	    token = gettoken(fp);
	    if (token == TOKEN_INTEGER)
	    {
		num = atoi(tokenbuf);
		token = gettoken(fp);
		if (token != '*')
		{
		    unexpectedtoken();
		    return;
		}
		token = gettoken(fp);
	    }
	    else
		num = 1;
	    if (token != TOKEN_STRING)
	    {
		unexpectedtoken();
		return;
	    }
	    while(num--)
	    {
		if (ip->childnames == NULL)
		{
		    ip->childnames = nametail = (struct namelist *)
			malloc(sizeof(struct namelist));
		}
		else
		{
		    nametail->next = (struct namelist *)
			malloc(sizeof(struct namelist));
		    nametail = nametail->next;
		}
		nametail->name = strdup(tokenbuf);
		nametail->next = NULL;
	    }
	    token = gettoken(fp);
	}
	else if (!strcmp(tokenbuf, "detector"))
	    return;
	else if (!strcmp(tokenbuf, "experiment"))
	    return;
	else
	{
	    unexpectedtoken();
	    return;
	}
    }

    if (token != EOF)
	unexpectedtoken();
}

static struct detectors *instance(char *name, struct detectors *parent)
{
    struct detectors *dp, *child;
    struct information *ip;
    struct namelist *cname;
    struct alllist *ap;
    int num = 1;

    for(ip = infohead; ip != NULL; ip = ip->next)
	if (!strcasecmp(ip->name, name))
	    break;
    
    if (ip == NULL)
    {
	fprintf(stderr, "Detector \"%s\" is not defined.\n", name);
	destroyall();
	return(NULL);
    }

    if ((ap = (struct alllist *) malloc(sizeof(struct alllist))) == NULL)
    {
	destroyall();
	return(NULL);
    }

    if ((dp = (struct detectors *) malloc(sizeof(struct detectors))) == NULL)
    {
	fprintf(stderr, "struct: Out of memory trying to build "
		"detector structure.\n");
	free(ap);
	destroyall();
	return(NULL);
    }

    ap->dp = dp;
    ap->next = allhead;
    allhead = ap;

    dp->info = ip;
    dp->parent = parent;
    dp->children = NULL;
    initprops(dp);
    dp->next = NULL;

    for(cname = ip->childnames; cname != NULL; cname = cname->next)
    {
	if (dp->children == NULL)
	{
	    if ((dp->children = child = instance(cname->name, dp)) == NULL)
		return(NULL);
	}
	else
	{
	    if ((child->next = instance(cname->name, dp)) == NULL)
		return(NULL);
	    child = child->next;
	}
	child->number = num++;
	if (strcmp(child->info->name, "junk")) /* if name is not junk... */
	    dp->info->ndetel = child->number;
    }
    return(dp);
}

int setdetprocsmixed(char *name, void *preinit, int pit, void *init, int it,
		     void *preevent, int pet, void *event, int et)
{
    struct information *ip = infohead;

    while(ip != NULL)
    {
	if (!strcasecmp(name, ip->name))
	{
	    if (pit == DETPROC_C)
		ip->preinit.c = (int (*)(struct detectors *)) preinit;
	    else
		ip->preinit.f = (int (*)(void)) preinit;
	    if (it == DETPROC_C)
		ip->init.c = (int (*)(struct detectors *)) init;
	    else
		ip->init.f = (int (*)(void)) init;
	    if (pet == DETPROC_C)
		ip->preevent.c = (int (*)(int, struct detectors *)) preevent;
	    else
		ip->preevent.f = (int (*)(int *)) preevent;
	    if (et == DETPROC_C)
		ip->event.c = (int (*)(int, struct detectors *)) event;
	    else
		ip->event.f = (int (*)(int *)) event;
	    ip->preinit_type = pit;
	    ip->init_type = it;
	    ip->preevent_type = pet;
	    ip->event_type = et;
	    return 0;
	}
	ip = ip->next;
    }

    return 1;
}

int setdetprocs(char *name, int (*preinit)(struct detectors *),
		   int (*init)(struct detectors *),
		   int (*preevent)(int, struct detectors *),
		   int (*event)(int, struct detectors *))
{
    return setdetprocsmixed(name, (void *) preinit, DETPROC_C,
			    (void *) init, DETPROC_C,
			    (void *) preevent, DETPROC_C,
			    (void *) event, DETPROC_C);
}

int donothing_(void)
{
/* Note that this function never actually gets called, but sometimes we have
   to keep lint happy */
#ifdef lint
    return 0;
#endif
}

int setdetprocs_(char *name, int (*preinit)(void),
		   int (*init)(void),
		   int (*preevent)(int *),
		   int (*event)(int *))
{
    if (preinit == donothing_)
	preinit = NULL;
    if (init == donothing_)
	init = NULL;
    if (preevent == (int (*)(int *)) donothing_)
	preevent = NULL;
    if (event == (int (*)(int *)) donothing_)
	event = NULL;
    return setdetprocsmixed(name, (void *) preinit, DETPROC_F,
			    (void *) init, DETPROC_F,
			    (void *) preevent, DETPROC_F,
			    (void *) event, DETPROC_F);
}
