/*
 * Vector manipulation package with lazy evaluation to delay all those
 * CPU expensive trig calls until really needed and then only do them
 * once.
 */

#include <math.h>
#include <string.h>
#include "vector.h"

#define DEG (180/M_PI)
#define RAD (M_PI/180)

#define VECTOR_XYZ (VECTOR_X | VECTOR_Y | VECTOR_Z)
#define VECTOR_RTP (VECTOR_R | VECTOR_T | VECTOR_P)
#define VECTOR_RTP_DEG (VECTOR_R | VECTOR_T_DEG | VECTOR_P_DEG)
#define VECTOR_RTT (VECTOR_R | VECTOR_TX | VECTOR_TY)
#define VECTOR_RTT_DEG (VECTOR_R | VECTOR_TX_DEG | VECTOR_TY_DEG)

struct vector *vector_set_xyz(struct vector *r, double x, double y, double z)
{
    r->x = x;
    r->y = y;
    r->z = z;
    r->state = VECTOR_XYZ;

    return r;
}

struct vector *vector_set_rtp(struct vector *r, double rn, double t, double p)
{
    r->r = rn;
    r->t = t;
    r->p = p;
    r->state = VECTOR_RTP;

    return r;
}

struct vector *vector_set_rtpd(struct vector *r, double rn, double t, double p)
{
    r->r = rn;
    r->td = t;
    r->pd = p;
    r->state = VECTOR_RTP_DEG;
    
    return r;
}

struct vector *vector_set_rtt(struct vector *r, double rn, double tx,
			      double ty)
{
    r->r = rn;
    r->tx = tx;
    r->ty = ty;
    r->state = VECTOR_RTT;

    return r;
}

struct vector *vector_set_rttd(struct vector *r, double rn, double tx,
			       double ty)
{
    r->r = rn;
    r->txd = tx;
    r->tyd = ty;
    r->state = VECTOR_RTT_DEG;
    
    return r;
}

double vector_r_safe(struct vector *r)
{
    if (r->state & VECTOR_R)
	return r->r;

    r->state |= VECTOR_R;

    if (r->state & VECTOR_R2)
	return r->r = sqrt(r->r2);

    r->state |= VECTOR_R2;

    return r->r = sqrt(r->r2 = (r->x*r->x + r->y*r->y + r->z*r->z));
}

double vector_r2_safe(struct vector *r)
{
    if (r->state & VECTOR_R2)
	return r->r2;

    r->state |= VECTOR_R2;

    if (r->state & VECTOR_R)
	return r->r2 = r->r * r->r;

    return r->r2 = r->x*r->x + r->y*r->y + r->z*r->z;
}

double vector_t_safe(struct vector *r)
{
    if (r->state & VECTOR_T)
	return r->t;


    if (r->state & VECTOR_T_DEG)
    {
	r->state |= VECTOR_T;
	return r->t = r->td * RAD;
    }

    r->t = acos(vector_z(r)/vector_r(r));
    r->state |= VECTOR_T;
    return r->t;
}

double vector_p_safe(struct vector *r)
{
    double p;

    if (r->state & VECTOR_P)
	return r->p;

    r->state |= VECTOR_P;

    if (r->state & VECTOR_P_DEG)
	return r->p = r->pd * RAD;

    if ((p = atan2(r->y,r->x)) < 0)
	p += 2*M_PI;
    
    return r->p = p;
}

double vector_td_safe(struct vector *r)
{
    if (r->state & VECTOR_T_DEG)
	return r->td;

    r->td = vector_t(r) * DEG;
    r->state |= VECTOR_T_DEG;
    return r->td;
}

double vector_pd_safe(struct vector *r)
{
    if (r->state & VECTOR_P_DEG)
	return r->pd;

    r->pd = vector_p(r) * DEG;
    r->state |= VECTOR_P_DEG;
    return r->pd;
}

double vector_tx_safe(struct vector *r)
{
    double tx;

    if (r->state & VECTOR_TX)
	return r->tx;

    if (r->state & VECTOR_TX_DEG)
    {
	r->state |= VECTOR_TX;
	return r->tx = r->txd * RAD;
    }

    if ((tx = atan2(vector_x(r),vector_z(r))) < 0)
	tx += 2*M_PI;

    r->state |= VECTOR_TX;

    return r->tx = tx;
}

double vector_ty_safe(struct vector *r)
{
    if (r->state & VECTOR_TY)
	return r->ty;

    if (r->state & VECTOR_TY_DEG)
    {
	r->state |= VECTOR_TY;
	return r->ty = r->tyd * RAD;
    }

    r->ty = asin(vector_y(r)/vector_r(r));
    r->state |= VECTOR_TY;
    return r->ty;
}

double vector_txd_safe(struct vector *r)
{
    if (r->state & VECTOR_TX_DEG)
	return r->txd;

    r->txd = vector_tx(r) * DEG;
    r->state |= VECTOR_TX_DEG;
    return r->txd;
}

double vector_tyd_safe(struct vector *r)
{
    if (r->state & VECTOR_TY_DEG)
	return r->tyd;

    r->tyd = vector_ty(r) * DEG;
    r->state |= VECTOR_TY_DEG;
    return r->tyd;
}

double vector_x_safe(struct vector *r)
{
    double tmp;

    if (r->state & VECTOR_X)
	return r->x;

    if (r->state & VECTOR_XYZ_SCALE_PENDING)
    {
	r->state = (r->state &~VECTOR_XYZ_SCALE_PENDING) | VECTOR_XYZ;
	if (r->state & VECTOR_R)
	    tmp = r->r/sqrt(r->x*r->x + r->y*r->y + r->z*r->z);
	else
	    tmp = sqrt(vector_r2(r)/(r->x*r->x + r->y*r->y + r->z*r->z));
	r->y *= tmp;
	r->z *= tmp;
	return r->x *= tmp;
    }

    if ((r->state & (VECTOR_T | VECTOR_T_DEG)) && 
	(r->state & (VECTOR_P | VECTOR_P_DEG)))
    {
	double t, p;

	/*
	 * The chances are that if we want x we'll want y too, and by working 
	 * them out together we save a call to sin and a mul.
	 */
    
	r->state |= VECTOR_X | VECTOR_Y;
	
	if (r->state & VECTOR_T)
	    t = r->t;
	else
	{
	    r->state |= VECTOR_T;
	    t = r->t = (r->td * RAD);
	}
	
	if (r->state & VECTOR_P)
	    p = r->p;
	else
	{
	    r->state |= VECTOR_P;
	    p = r->p = (r->pd * RAD);
	}
	
	tmp = r->r*sin(t);
	r->y = tmp*sin(p);
	return r->x = tmp*cos(p);
    }

    {
	double tx, ty;
	
	/*
	 * The chances are that if we want x we'll want z too, and by working 
	 * them out together we save a call to sin and a mul.
	 */
	
	r->state |= VECTOR_X | VECTOR_Z;
	
	if (r->state & VECTOR_TX)
	    tx = r->tx;
	else
	{
	    r->state |= VECTOR_TX;
	    tx = r->tx = (r->txd * RAD);
	}
	
	if (r->state & VECTOR_TY)
	    ty = r->ty;
	else
	{
	    r->state |= VECTOR_TY;
	    ty = r->ty = (r->tyd * RAD);
	}
	
	tmp = r->r*cos(ty);
	r->z = tmp*cos(tx);
	return r->x = tmp*sin(tx);
    }
}

double vector_y_safe(struct vector *r)
{
    if (r->state & VECTOR_Y)
	return r->y;

    if (r->state & VECTOR_XYZ_SCALE_PENDING)
    {
	double tmp;

	r->state = (r->state &~VECTOR_XYZ_SCALE_PENDING) | VECTOR_XYZ;
	if (r->state & VECTOR_R)
	    tmp = r->r/sqrt(r->x*r->x + r->y*r->y + r->z*r->z);
	else
	    tmp = sqrt(vector_r2(r)/(r->x*r->x + r->y*r->y + r->z*r->z));
	r->x *= tmp;
	r->z *= tmp;
	return r->y *= tmp;
    }

    if ((r->state & (VECTOR_T | VECTOR_T_DEG)) && 
	(r->state & (VECTOR_P | VECTOR_P_DEG)))
    {
	double tmp, t, p;

	/*
	 * The chances are that if we want y we'll want x too, and by working 
	 * them out together we save a call to sin and a mul.
	 */
    
	r->state |= VECTOR_X | VECTOR_Y;
	
	if (r->state & VECTOR_T)
	    t = r->t;
	else
	{
	    r->state |= VECTOR_T;
	    t = r->t = (r->td * RAD);
	}
	
	if (r->state & VECTOR_P)
	    p = r->p;
	else
	{
	    r->state |= VECTOR_P;
	    p = r->p = (r->pd * RAD);
	}
	
	tmp = r->r*sin(t);
	r->x = tmp*cos(p);
	return r->y = tmp*sin(p);
    }

    {
	double ty; 

	/*
	 * Using theta-x, theta-y coordinates, y can be worked out quicker than
	 * x and z.
	 */
    
	r->state |= VECTOR_Y;

	if (r->state & VECTOR_TY)
	    ty = r->ty;
	else
	{
	    r->state |= VECTOR_TY;
	    ty = r->ty = (r->tyd * RAD);
	}

	return r->y = r->y*sin(ty);
    }
}

double vector_z_safe(struct vector *r)
{
    if (r->state & VECTOR_Z)
	return r->z;

    if (r->state & VECTOR_XYZ_SCALE_PENDING)
    {
	double tmp;

	r->state = (r->state &~VECTOR_XYZ_SCALE_PENDING) | VECTOR_XYZ;
	if (r->state & VECTOR_R)
	    tmp = r->r/sqrt(r->x*r->x + r->y*r->y + r->z*r->z);
	else
	    tmp = sqrt(vector_r2(r)/(r->x*r->x + r->y*r->y + r->z*r->z));
	r->x *= tmp;
	r->y *= tmp;
	return r->z *= tmp;
    }

    if (r->state & (VECTOR_T | VECTOR_T_DEG))
    {
	double t;

	/*
	 * In spherical polar coordinates, z can be worked out quicker than
	 * x and y and may be wanted separately.
	 */
    
	r->state |= VECTOR_Z;
	
	if (r->state & VECTOR_T)
	    t = r->t;
	else
	{
	    r->state |= VECTOR_T;
	    t = r->t = (r->td * RAD);
	}
	
	return r->z = r->r*cos(t);
    }

    {
	double tmp, tx, ty;
	
	/*
	 * The chances are that if we want z we'll want x too, and by working 
	 * them out together we save a call to sin and a mul.
	 */
	
	r->state |= VECTOR_X | VECTOR_Z;
	
	if (r->state & VECTOR_TX)
	    tx = r->tx;
	else
	{
	    r->state |= VECTOR_TX;
	    tx = r->tx = (r->txd * RAD);
	}
	
	if (r->state & VECTOR_TY)
	    ty = r->ty;
	else
	{
	    r->state |= VECTOR_TY;
	    ty = r->ty = (r->tyd * RAD);
	}
	
	tmp = r->r*cos(ty);
	r->x = tmp*sin(tx);
	return r->z = tmp*cos(tx);
    }
}

struct vector *vector_scale(struct vector *r, double fac)
{
    if ((r->state & (VECTOR_R | VECTOR_R2)) &&
	((r->state & VECTOR_XYZ) == VECTOR_XYZ ||
	 (r->state & VECTOR_XYZ_SCALE_PENDING)))
	r->state = (r->state | VECTOR_XYZ_SCALE_PENDING) & ~VECTOR_XYZ;
    else
    {
	if (r->state & VECTOR_X)
	    r->x *= fac;

	if (r->state & VECTOR_Y)
	    r->y *= fac;

	if (r->state & VECTOR_Z)
	    r->z *= fac;
    }

    if (r->state & VECTOR_R)
	r->r *= fac;

    if (r->state & VECTOR_R2)
	r->r2 *= fac*fac;

    return r;
}

struct vector *vector_normalise(struct vector *r)
{
    double fac;

    if ((r->state & VECTOR_XYZ) == VECTOR_XYZ)
	r->state = (r->state | VECTOR_XYZ_SCALE_PENDING) & ~VECTOR_XYZ;
    else if (!(r->state & VECTOR_XYZ_SCALE_PENDING) && (r->state & VECTOR_XYZ))
    {
	if (r->state & VECTOR_R)
	    fac = r->r;
	else
	{
	    if (r->state & VECTOR_R2)
		fac = sqrt(r->r2);
	    else
		fac = sqrt(r->x*r->x + r->y*r->y + r->z*r->z);
	}
	
	fac = 1.0/fac;

	if (r->state & VECTOR_X)
	    r->x *= fac;
	
	if (r->state & VECTOR_Y)
	    r->y *= fac;
	
	if (r->state & VECTOR_Z)
	    r->z *= fac;
    }

    r->state |= VECTOR_R | VECTOR_R2;
    r->r2 = r->r = 1.0;

    return r;
}

struct vector *vector_scale_to(struct vector *r, double len)
{
    double fac;

    if ((r->state & VECTOR_XYZ) == VECTOR_XYZ)
	r->state = (r->state | VECTOR_XYZ_SCALE_PENDING) & ~VECTOR_XYZ;
    else if (!(r->state & VECTOR_XYZ_SCALE_PENDING) && (r->state & VECTOR_XYZ))
    {
	if (r->state & VECTOR_R)
	    fac = r->r;
	else if (r->state & VECTOR_R2)
	    fac = sqrt(r->r2);
	else
	    fac = sqrt(r->x*r->x + r->y*r->y + r->z*r->z);
	
	fac = len/fac;

	if (r->state & VECTOR_X)
	    r->x *= fac;
	
	if (r->state & VECTOR_Y)
	    r->y *= fac;
	
	if (r->state & VECTOR_Z)
	    r->z *= fac;
    }

    r->state |= VECTOR_R | VECTOR_R2;
    r->r = len;
    r->r2 = len*len;

    return r;
}

struct vector *vector_scale_to2(struct vector *r, double len2)
{
    double fac;

    if ((r->state & VECTOR_XYZ) == VECTOR_XYZ)
	r->state = (r->state | VECTOR_XYZ_SCALE_PENDING) & ~VECTOR_XYZ;
    else if (!(r->state & VECTOR_XYZ_SCALE_PENDING) && (r->state & VECTOR_XYZ))
    {
	if (r->state & VECTOR_R2)
	    fac = r->r2;
	else if (r->state & VECTOR_R)
	    fac = r->r * r->r;
	else
	    fac = r->x*r->x + r->y*r->y + r->z*r->z;
	
	fac = sqrt(len2/fac);

	if (r->state & VECTOR_X)
	    r->x *= fac;
	
	if (r->state & VECTOR_Y)
	    r->y *= fac;
	
	if (r->state & VECTOR_Z)
	    r->z *= fac;
    }

    r->state = (r->state & ~VECTOR_R) | VECTOR_R2;
    r->r2 = len2;

    return r;
}

struct vector *vector_copy(struct vector *d, struct vector *s)
{
    return (struct vector *) memcpy(d, s, sizeof(struct vector));
}

struct vector *vector_add(struct vector *d, struct vector *a, struct vector *b)
{
    return vector_set_xyz(d, vector_x(a)+vector_x(b), vector_y(a)+vector_y(b),
		   vector_z(a)+vector_z(b));
}

struct vector *vector_sub(struct vector *d, struct vector *a, struct vector *b)
{
    return vector_set_xyz(d, vector_x(a)-vector_x(b), vector_y(a)-vector_y(b),
			  vector_z(a)-vector_z(b));
}

double vector_dot(struct vector *a, struct vector *b)
{
    return vector_x(a)*vector_x(b) + vector_y(a)*vector_y(b) +
	vector_z(a)*vector_z(b);
}

struct vector *vector_cross(struct vector *d, struct vector *a,
			    struct vector *b)
{
    double ax, ay, az, bx, by, bz;

    ax = vector_x(a);
    ay = vector_y(a);
    az = vector_z(a);
    bx = vector_x(b);
    by = vector_y(b);
    bz = vector_z(b);

    return vector_set_xyz(d, ay*bz - az*by, az*bx - ax*bz, ax*by - ay*bx);
}
