#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <fcntl.h>

#include <sys/mman.h>
#include <sys/types.h>
#include <sys/time.h>

#include <xview/xview.h>
#include <xview/panel.h>

#include "sort_def.h"
#include "sort_thread.h"
#include "version.h"
#include "gui_extern.h"
#include "gui_spectra.h"
#include "gui_filtpop.h"
#include "gui_specpop.h"
#include "gui_ui_main.h"
#include "gui_toolpop.h"
#include "gui_varpop.h"
#include "gui_winpop.h"
#include "gui_main.h"

static struct shared_data   *sdat_ptr;
static unsigned long        *in_records;
static unsigned long        *out_records;

int
connect_to_shared_data(
#if NeedFunctionPrototype
		  char *data_file, char *progname)
#else
      data_file, progname)
      char *data_file;
      char *progname;
#endif
{
      int filedes;
      int len;

/* set up shared resources */
      filedes = open(data_file, O_RDWR);
      if (filedes == -1) {
	    fprintf(stderr,"failed to open shared data file %s\n",data_file);
	    return(-1);
      }

      /*LINTED*/
      if ( (caddr_t)(sdat_ptr = (struct shared_data *)mmap(0, SHDATA_SIZE, PROT_READ,
							MAP_SHARED, filedes, 0)) == (caddr_t) -1) {
	    fprintf(stderr,"mmap error on shared data file\n");
	    perror("connect_to_shared_data");
	    (void) close(filedes);
	    return(-1);
      }
      (void) close(filedes);

/* check version compatibilty */
      if (sdat_ptr->version != SUNSORT_VERSION) {
	    fprintf(stderr,"inconsistent version numbers ... remake %s\n",
		progname);
	    return(-1);
      }
      
/* calculate size of shared data space to allow read/write access too */      
      len = (char *)sdat_ptr - (char *) &sdat_ptr->_pad;
      len = (len < 0) ? -len : len;

      if (mprotect((caddr_t) sdat_ptr, len, PROT_READ|PROT_WRITE) == -1) {
	    fprintf(stderr,"mprotect error on shared data file\n");
	    perror("connect_to_shared_data");
	    return(-1);
      }
      
/* set up pointers to shared data structures */
      sort_io_data = &sdat_ptr->sort_io_data;
      dirs_io_data = &sdat_ptr->dirs_io_data;
      in_records = &sdat_ptr->in_records;
      out_records = &sdat_ptr->out_records;
      
      return(0);
}

spec_entry_t *
get_spec_entry(
#if NeedFunctionPrototype
		       int type, int nos)	
#else
      type, nos)
      int   type;
      int   nos;	
#endif
{
      spec_entry_t  *sp;
      
      switch(type) {
	  case TYPE_var:
	    if (nos <= 0 || nos >= TAB_SIZE_VAR) {
		  fprintf(stderr,"Invalid variable number %d\n",nos);
		  return(NULL);
	    }
	    sp = &sdat_ptr->var[nos];
	    break;

	  case TYPE_1d:
	    if (nos <= 0 || nos >= TAB_SIZE_1D) {
		  fprintf(stderr,"Invalid 1D spectrum number %d\n",nos);
		  return(NULL);
	    }
	    sp = &sdat_ptr->spec1d[nos];
	    break;

	  case TYPE_2d:
	    if (nos <= 0 || nos >= TAB_SIZE_2D) {
		  fprintf(stderr,"Invalid 2D spectrum number %d\n",nos);
		  return(NULL);
	    }
	    sp = &sdat_ptr->spec2d[nos];
	    break;

	  default:
	    fprintf(stderr,"server: unknown spectrum type %d\n",type);
	    return(NULL);
      }
      return(sp);
}

static char *specnames[
(TAB_SIZE_1D > TAB_SIZE_2D)
    ? ((TAB_SIZE_1D > TAB_SIZE_VAR) ? TAB_SIZE_1D : TAB_SIZE_VAR)
    : ((TAB_SIZE_2D > TAB_SIZE_VAR) ? TAB_SIZE_2D : TAB_SIZE_VAR)];

int
read_spectra_list
#if NeedFunctionPrototype
(void)
#else
()
#endif
{
    int i;
    char buf[BUFSIZ], **sn;
    spec_entry_t *sp;

/* set up the 1d spectra */	
    sn = specnames + LIST_FIRST;
    for(i=1; i< TAB_SIZE_1D; i++) {
	if ((sp = get_spec_entry(TYPE_1d, i)) == NULL) {
	    fprintf(stderr, 
		    "sort_gui: no shared data entry for 1d spectrum %d\n", i);
	    return SORT_FAIL;
	}
	if (i != sp->nos)
	    continue;
	(void) sprintf(buf,"%4d %-15s %5d", i, sp->name, sp->size);
	if ((*sn++ = strdup(buf)) == NULL)
	{
	    for(sn = specnames + LIST_FIRST; *sn != NULL; sn++)
		free(*sn);
	    return SORT_FAIL;
	}
	if(DBX_val >= 2)
	    fprintf(stderr,
		    "sort_gui: 1d spectrum %s nos %d with size %d setup\n",
		    sp->name,i,sp->size);
    }
    *sn++ = NULL;
    spec_popup_spec1d_list_reset(specnames);

/* setup the 2d spectra */	
    sn = specnames + LIST_FIRST;
    for(i=1; i< TAB_SIZE_2D; i++) {
	if ((sp = get_spec_entry(TYPE_2d, i)) == NULL) {
	    fprintf(stderr,
		    "sort_gui: no shared data entry for 2d spectrum %d\n", i);
	    return SORT_FAIL;
	}
	if (i != sp->nos)
	    continue;
	(void) sprintf(buf,"%4d %-15s %5d", i, sp->name, sp->size);
	if ((*sn++ = strdup(buf)) == NULL)
	{
	    for(sn = specnames + LIST_FIRST; *sn != NULL; sn++)
		free(*sn);
	    return SORT_FAIL;
	}
/*
	      if (add2_2d_list(i,sp->size,sp->name) == SORT_FAIL) 
	      return(SORT_FAIL);
*/
	if(DBX_val >= 2)
	    fprintf(stderr,
		    "sort_gui: 2d spectrum %s nos %d with size %d setup\n",
		    sp->name,i,sp->size);
    }
    *sn++ = NULL;
    spec_popup_spec2d_list_reset(specnames);

/* setup the windows with a separate pass */
    sn = specnames + LIST_FIRST;
    for(i=1; i< TAB_SIZE_2D; i++) {
	if ((sp = get_spec_entry(TYPE_2d, i)) == NULL ||
	    i != sp->nos || (*sp->name != 'w' && *sp->name != 'W'))
	    continue;
	(void) sprintf(buf,"%4d %-15s %5d", i, sp->name, sp->size);
	if ((*sn++ = strdup(buf)) == NULL)
	{
	    for(sn = specnames + LIST_FIRST; *sn != NULL; sn++)
		free(*sn);
	    return SORT_FAIL;
	}
/*
	      if (add2_2d_list(i,sp->size,sp->name) == SORT_FAIL) 
	      return(SORT_FAIL);
*/
	if(DBX_val >= 6)
	    fprintf(stderr,
		    "sort_gui: window %s nos %d with size %d setup\n",
		    sp->name,i,sp->size);
    }
    *sn++ = NULL;
    window_popup_win_list_reset(specnames);

/* set up the variables */	
    sn = specnames + LIST_FIRST;
    for(i=1; i< TAB_SIZE_VAR; i++) {
	if ((sp = get_spec_entry(TYPE_var, i)) == NULL) {
	    fprintf(stderr,
		    "sort_gui: no shared data entry for variable %d\n", i);
	    return SORT_FAIL;
	}
	if (i != sp->nos)
	    continue;
	(void) sprintf(buf,"%3d %-12s %13.6g", i, sp->name, sp->size);
	if ((*sn++ = strdup(buf)) == NULL)
	{
	    for(sn = specnames + LIST_FIRST; *sn != NULL; sn++)
		free(*sn);
	    return SORT_FAIL;
	}
/*
	      if (add2_var_list(i,sp->value,sp->name) == SORT_FAIL) 
		    return(SORT_FAIL);
*/
	if(DBX_val >= 2)
	    fprintf(stderr,
		    "sort_gui: variable %s nos %d with value %g setup\n",
		    sp->name,i,sp->value);
    }
    *sn++ = NULL;
    var_popup_var_list_reset(specnames);

    win_set_name();
    
    if (DBX_val >= 2) fprintf(stderr,"sort_gui: finished reading ...\n");
    
    return 0;
}

void
refresh_gui_func
#if NeedFunctionPrototype
(int bool)
#else
(bool)
int  bool;
#endif
{
    static sort_IO_t copy_sort_data;
    sort_IO_t *ptr = &copy_sort_data;
    sort_IO_t *shared = sort_io_data;
    extern sort_set1_nh P((Panel_item, int, Event *));

    if (bool)
	memset(ptr, 0, sizeof(sort_IO_t));
		   
    if (memcmp(ptr, shared, sizeof(sort_IO_t)) != 0) {
	memcpy(ptr, shared, sizeof(sort_IO_t));
	/* refresh sort file fields */
	if (ptr->IO_state & READING_IN) {
	    char cbuffer[BUFSIZ];
	    xv_set(sort_gui_window1->sort_b1, PANEL_LABEL_STRING, "Close",
		   NULL);
	    xv_set(sort_gui_window1->sort_tf1, PANEL_VALUE, ptr->read_file,
		   NULL);
	    xv_set(sort_gui_window1->sort_setting1, PANEL_VALUE,
		   (int) ptr->sort_media, PANEL_INACTIVE, TRUE, NULL);
	    sort_set1_nh(sort_gui_window1->sort_setting1,
			 (int) ptr->sort_media, NULL);
		  
	    if (ptr->sort_media == EXABYTE_DATA) {
		strcpy(cbuffer,"Tape read file open : ");
		strncat(cbuffer,ptr->read_file,BUFSIZ - strlen(cbuffer) -1);
		xv_set(sort_gui_window1->sort_message, PANEL_LABEL_STRING,
		       cbuffer, NULL);
	    }
	    else if (ptr->sort_media == DISK_DATA) {
		strcpy(cbuffer,"Disk read file open : ");
		strncat(cbuffer,ptr->read_file,BUFSIZ - strlen(cbuffer) -1);
		xv_set(sort_gui_window1->sort_message, PANEL_LABEL_STRING,
		       cbuffer,NULL);
	    }
	    else if (ptr->sort_media == REMOTE_DATA) {
		strcpy(cbuffer,"Remote read source open : ");
		strncat(cbuffer,ptr->read_file,BUFSIZ - strlen(cbuffer) -1);
		xv_set(sort_gui_window1->sort_message, PANEL_LABEL_STRING,
		       cbuffer,NULL);
	    }
	    else if (ptr->sort_media == DEMON_DATA) {
		strcpy(cbuffer,"Demon data feed open : ");
		strncat(cbuffer,ptr->read_file,BUFSIZ - strlen(cbuffer) -1);
		xv_set(sort_gui_window1->sort_message, PANEL_LABEL_STRING,
		       cbuffer,NULL);
	    }
	    else if (ptr->sort_media == PIPE_DATA) {
		strcpy(cbuffer,"Pipe open : ");
		strncat(cbuffer,ptr->read_file,BUFSIZ - strlen(cbuffer) -1);
		xv_set(sort_gui_window1->sort_message, PANEL_LABEL_STRING,
		       cbuffer,NULL);
	    }
	}
	else {
	    xv_set(sort_gui_window1->sort_b1, PANEL_LABEL_STRING, "Open",NULL);
	    xv_set(sort_gui_window1->sort_tf1, PANEL_VALUE, NULL, NULL);
	    xv_set(sort_gui_window1->sort_message, PANEL_LABEL_STRING,
		   "State:",NULL);
	    xv_set(sort_gui_window1->sort_setting1, PANEL_INACTIVE, False,
		   NULL);
	}
	/* refresh read tape fields */      
	if (ptr->IO_state & R_TAPE_MOUNTED) {
	    xv_set(sort_gui_window1->sort_b3, PANEL_LABEL_STRING, "Unmount",
		   NULL);
	    xv_set(sort_gui_window1->sort_tf2, PANEL_VALUE, ptr->read_vol,
		   PANEL_INACTIVE, TRUE, NULL);
	    xv_set(sort_gui_window1->sort_setting3 , PANEL_VALUE,
		   (int) ptr->sort_drive, PANEL_INACTIVE, TRUE, NULL);
	}
	else {
	    xv_set(sort_gui_window1->sort_b3, PANEL_LABEL_STRING, "Mount", 
		   NULL);
 	    xv_set(sort_gui_window1->sort_tf2, PANEL_VALUE, NULL,
		   /* PANEL_INACTIVE, FALSE, */ NULL);
	    xv_set(sort_gui_window1->sort_setting3, PANEL_VALUE,
		   (int) ptr->sort_drive, /* PANEL_INACTIVE, FALSE, */ NULL);
	}
	/* refresh write tape fields */      
	if (ptr->IO_state & W_TAPE_MOUNTED) {
	    xv_set(sort_gui_filt_popup->filt_b2, PANEL_LABEL_STRING, "Unmount",
		   NULL);
	    xv_set(sort_gui_filt_popup->filt_tf2, PANEL_VALUE, ptr->write_vol,
		   PANEL_INACTIVE, TRUE, NULL);
	    xv_set(sort_gui_filt_popup->filt_setting2 , PANEL_VALUE,
		   (int) ptr->write_drive, PANEL_INACTIVE, TRUE, NULL);
	}
	else {
	    xv_set(sort_gui_filt_popup->filt_b2, PANEL_LABEL_STRING, "Mount",
		   NULL);
	    xv_set(sort_gui_filt_popup->filt_tf2, PANEL_VALUE, NULL,
		   PANEL_INACTIVE, FALSE, NULL);
	    xv_set(sort_gui_filt_popup->filt_setting2 , PANEL_VALUE,
		   (int) ptr->write_drive, PANEL_INACTIVE, FALSE, NULL);
	    }
	
	/* refresh write file fields */
	if (ptr->IO_state & WRITING_OUT) {
	    char cbuffer[BUFSIZ];
	    xv_set(sort_gui_filt_popup->filt_b1, PANEL_LABEL_STRING, "Close",
		   NULL);
	    xv_set(sort_gui_filt_popup->filt_tf1, PANEL_VALUE, ptr->write_file,
		   NULL);
	    xv_set(sort_gui_filt_popup->filt_setting1, PANEL_INACTIVE, TRUE,
		   NULL); 
	    if (ptr->write_media & EXABYTE_DATA) {
		strcpy(cbuffer,"Tape write file open : ");
		strncat(cbuffer, ptr->write_file, BUFSIZ - strlen(cbuffer) -1);
		xv_set(sort_gui_filt_popup->filt_message, PANEL_LABEL_STRING,
		       cbuffer, NULL);
	    }
	    else if (ptr->write_media & DISK_DATA) {
		strcpy(cbuffer,"Disk write file open : ");
		strncat(cbuffer, ptr->write_file, BUFSIZ - strlen(cbuffer) -1);
		xv_set(sort_gui_filt_popup->filt_message, PANEL_LABEL_STRING,
		       cbuffer, NULL);
	    }
	}
	else {
	    xv_set(sort_gui_filt_popup->filt_b1, PANEL_LABEL_STRING, "Open",
		   NULL);
	    xv_set(sort_gui_filt_popup->filt_tf1, PANEL_VALUE, NULL, NULL);
	    xv_set(sort_gui_filt_popup->filt_message, PANEL_LABEL_STRING,
		   "State:",NULL);
	    xv_set(sort_gui_filt_popup->filt_setting1, PANEL_INACTIVE, False,
		   NULL);
	}

	/* refresh format field on sorttool window */
	{
	    int value;
	    char *format;
	    for (value=0; ;value++) {
		format = (char *) xv_get(sort_gui_tool_popup->tool_setting3, 
					 PANEL_CHOICE_STRING, value);
		if (format == NULL) {
		    xv_set(sort_gui_tool_popup->tool_setting3, PANEL_VALUE,
			   -1, NULL);
		    break;
		}
		if (strncmp(ptr->decoder, format, FORMAT_NAME_LEN) == 0) {
		    xv_set(sort_gui_tool_popup->tool_setting3, PANEL_VALUE,
			   value, NULL);
		    break;
		}
	    }
	}
	
    }
}      

void
display_sort_stats(
#if NeedFunctionPrototype
		void)
#else
      )
#endif
{
      static char           dbuf[100];
      static unsigned long  last_in_records;
      static long           delta_recs;
      
      /* update sort statistics display */      
      sprintf(dbuf,"%lu",*in_records);
      xv_set(sort_gui_window1->stats_message2,PANEL_LABEL_STRING,dbuf,NULL);
      sprintf(dbuf,"%lu",*out_records);
      xv_set(sort_gui_window1->stats_message4,PANEL_LABEL_STRING,dbuf,NULL);

      delta_recs = *in_records - last_in_records;
      if (delta_recs >= 0) {
	    sprintf(dbuf,"%ld",delta_recs);
	    xv_set(sort_gui_window1->stats_message6,PANEL_LABEL_STRING,dbuf,NULL); 
      }
      last_in_records = *in_records;
}
