#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <time.h>
#include <sys/time.h>
#include <fcntl.h>
#include <thread.h>
#include <synch.h>

#include "sort_def.h"
#include "sort_thread.h"
#include "eg.h"
#include "data_io.h"
#include "main.h"
#include "version.h"
#ifdef USE_DATA_QUEUE
#include "data_queue.h"
#endif

struct shared_data    *sdat_ptr;

int
connect_to_shared_data
#if NeedFunctionPrototype
(char *data_file, char *progname)
#else
(data_file, progname)
char *data_file;
char *progname;
#endif
{
    int filedes;
    int len;
    
    /* set up shared resources */
    filedes = open(data_file, O_RDWR);
    if (filedes == -1) {
	fprintf(stderr,"failed to open shared data file %s\n",data_file);
	return -1;
    }
    
    /*LINTED*/
    if ((caddr_t) (sdat_ptr = (struct shared_data *)
		   mmap(0, SHDATA_SIZE, PROT_READ, MAP_SHARED, filedes, 0))
	== (caddr_t) -1) {
	fprintf(stderr,"mmap error on shared data file\n");
	perror("connect_to_shared_data");
	(void) close(filedes);
	return -1;
    }
    (void) close(filedes);

    /* calculate size of shared data space to allow read/write access to */
    len = (char *)sdat_ptr - (char *) &sdat_ptr->_pad;
    len = (len < 0) ? -len : len;
    
    if (mprotect((caddr_t) sdat_ptr, len, PROT_READ|PROT_WRITE) == -1) {
	fprintf(stderr,"mprotect error on shared data file\n");
	perror("connect_to_shared_data");
	return -1;
    }
    
    /* check version compatibilty */
    if (sdat_ptr->version != SUNSORT_VERSION) {
	fprintf(stderr,"inconsistent version numbers %d != %d ... remake %s\n",
		sdat_ptr->version, SUNSORT_VERSION, progname);
	return -1;
    }
    
    /* set up pointers to shared data structures */
    sort_io_data = &sdat_ptr->sort_io_data;
    dirs_io_data = &sdat_ptr->dirs_io_data;
    
    return 0;
}

spec_entry_t *
get_spec_entry(
#if NeedFunctionPrototype
		       int type, int nos)	
#else
      type, nos)
      int   type;
      int   nos;	
#endif
{
      spec_entry_t  *sp;
      
      switch(type) {
	  case TYPE_var:
	    if (nos <= 0 || nos >= TAB_SIZE_VAR) {
		  fprintf(stderr,"Invalid variable number %d\n",nos);
		  return(NULL);
	    }
	    sp = &sdat_ptr->var[nos];
	    break;

	  case TYPE_1d:
	    if (nos <= 0 || nos >= TAB_SIZE_1D) {
		  fprintf(stderr,"Invalid 1D spectrum number %d\n",nos);
		  return(NULL);
	    }
	    sp = &sdat_ptr->spec1d[nos];
	    break;

	  case TYPE_2d:
	    if (nos <= 0 || nos >= TAB_SIZE_2D) {
		  fprintf(stderr,"Invalid 2D spectrum number %d\n",nos);
		  return(NULL);
	    }
	    sp = &sdat_ptr->spec2d[nos];
	    break;

	  default:
	    fprintf(stderr,"server: unknown spectrum type %d\n",type);
	    return(NULL);
      }
      return(sp);
}

/*
 *   spectrum storage is mapped to files space (eurogam format files)
 */
/*ARGSUSED*/
int *
map_to_spectrum
#if NeedFunctionPrototype
(int spec_dim, char *name, int nos, int range)
#else
(spec_dim, name, nos, range)
int    spec_dim;    /* 1d == 1, 2d == 2 */
char   *name;
int    nos;
int    range;
#endif
{
    FILE *fp;
    long len = 0;
    char *addr = NULL;
    char file[BUFSIZ];
    SPECTRUM_HDR *sp;
      
    /* generate file name and open file */
    sprintf(file,"%s/%d.%dd",tmp_spectrum_dir,nos,spec_dim);
    if ( (fp = fopen(file,"r+")) == NULL)
	return NULL;
    
    if (fseek(fp, 0L, SEEK_END)) {
	fprintf(stderr,"sort: fseek failed in map_to_spectrum call\n");
	(void) fclose(fp);
	return(NULL);
    }
    len = ftell(fp);
    
    /* map file to memory address space of process */	  
    addr = (char *) mmap(0, (size_t)len, (PROT_READ|PROT_WRITE), MAP_SHARED,
			 fileno(fp), 0);
    if (addr == (caddr_t)-1) {
	fprintf(stderr,"sort: mmap failed in map_to_spectrum call\n");
	(void) fclose(fp);
	return(NULL);
    }
    
    /* close off file and set addr to point at beging of data */
    (void) fclose(fp);
    /*LINTED*/
    sp = (SPECTRUM_HDR *) addr;
    addr+=sp->count_base_addr;
	
    /*LINTED*/
    return((int *)addr);
}

int
unmap_spectrum(
#if NeedFunctionPrototype
		  int *data_base, int elements)
#else
      data_base, elements)
      int *data_base;
      int elements;
#endif
{
      caddr_t addr = (caddr_t) data_base - EG_MY_DATA_OFFSET;
      long    len = elements*sizeof(int) + EG_MY_DATA_OFFSET;
      
      if ( munmap(addr, len) == -1) {
	    perror("unmap_spectrum");
	    return(-1);
      }
      return(0);
}

/* pointers and buffer position counters for block I/O */
#ifndef USE_DATA_QUEUE
static int        rbuffer_pos, rdeod;
static int        wbuffer_pos;
#endif
static data_buf_t *rdptr; 
static data_buf_t *wdptr;

void
init_blockio
#if NeedFunctionPrototype
(void )
#else
()
#endif
{
    /* initialize read state I/O */
#ifndef USE_DATA_QUEUE
    rbuffer_pos = 0;
    rdeod = 0;
#endif
    rdptr = NULL;

    /* initialize write state I/O */
#ifndef USE_DATA_QUEUE
    wbuffer_pos = 0;
#endif
    wdptr = NULL;

    return;
}

short *
read_block
#if NeedFunctionPrototype
(int *bytes )
#else
(bytes)
int *bytes;
#endif
{
#ifdef USE_DATA_QUEUE
    int b;
    spec_buf_flush_safe(1);

    /* If we have a block, put it at end of empty queue */
    if (rdptr != NULL)
    {
	sdat_ptr->in_records++;
	add_to_queue(&sdat_ptr->read_empty, rdptr - sdat_ptr->read_buf);
    }

    /* Get the next block */
    while ((b = take_from_queue(&sdat_ptr->read_full, NULL)) == -1)
	;
    rdptr = sdat_ptr->read_buf + b;
    if (rdptr->data_size == 0)
    {
	add_to_queue(&sdat_ptr->read_empty, b);
	rdptr = NULL;
	return NULL;
    }

    spec_buf_flush_safe(0);
    *bytes = rdptr->data_size;
    return (short *) rdptr->data;
#else
    spec_buf_flush_safe(1);
    /* put last block on empty block queue */
    if (rdptr != NULL) {
	(void) mutex_lock(&rdptr->mp);
	if (rdptr->state == EMPTYING_BUF) {
	    rdptr->state = EMPTY_BUF;
	    (void) cond_signal(&rdptr->cv);
	    sdat_ptr->in_records++;
	}
	else {
	    (void) mutex_unlock(&rdptr->mp);
	    fprintf(stderr, "sort: read_block found shared data record in "
		    "undefined state\n");
	    exit(-1);
	}
	(void) mutex_unlock(&rdptr->mp);
	rbuffer_pos = (rbuffer_pos+1) % NOS_OF_RBUFS;
    }
    
    /* get new block of data */
    rdptr = &sdat_ptr->read_buf[rbuffer_pos];
    
    (void) mutex_lock(&rdptr->mp);
    while ( rdptr->state  != FULL_BUF) {
	    
	/* see whether we have reached the end of the data */
	if (rdptr->state == END_OF_DATA_BUF) {
	    rdeod = 1;
	    (void) mutex_unlock(&rdptr->mp);
	    if (DBX_val >= 7)
		fprintf(stderr, "sort process detected EOD for buffer %d\n",
			rbuffer_pos);
	    return(NULL);
	}
	
	/* wait for buffer to become full */
	if (cond_wait(&rdptr->cv, &rdptr->mp)) 
	    perror("sort condition variable wait");
    }
    rdptr->state = EMPTYING_BUF;
    *bytes = rdptr->data_size;
    (void) mutex_unlock(&rdptr->mp);
    
    spec_buf_flush_safe(0);
    /*LINTED*/
    return((short *) rdptr->data);	
#endif
}

/*
 *    software end of data
 *    end of data detected in sort process 
 */
int
software_eod
#if NeedFunctionPrototype
(void)
#else
()
#endif
{
    int bytes;

    /* signal to server to stop reading data if it hasn't already stopped */
    kill(getppid(),SIGUSR1); 

    /* Give server a chance to stop if it hasn't already */
#ifndef USE_DATA_QUEUE
    if (!rdeod)
#endif
	sleep(1);

    /* Flush any pending blocks */
#ifdef USE_DATA_QUEUE
    while(read_block(&bytes) != NULL)
#else
    while(rdeod == 0 && read_block(&bytes) != NULL)
#endif
	fprintf(stderr, "sort: Discarding record from input pipeline.\n");

#ifdef USE_DATA_QUEUE
    if (DBX_val >= 7)
	fprintf(stderr,"sort process signaled EOD\n");
#else
    /* ensure last block is registered as EOD block */
    if (rdptr != NULL) {
	(void) mutex_lock(&rdptr->mp);
	rdptr->state = END_OF_DATA_BUF;
	(void) cond_signal(&rdptr->cv);
	(void) mutex_unlock(&rdptr->mp);
    }
    if (DBX_val >= 7)
	fprintf(stderr,"sort process signaled EOD for buffer %d\n",
		rbuffer_pos);
#endif
      
    
    /* set sorting state to OFF */
    (void) mutex_lock(&sdat_ptr->mp_sorting);
    sdat_ptr->sorting = SORTING_OFF;
    (void) cond_signal(&sdat_ptr->cv_sorting);
    (void) mutex_unlock(&sdat_ptr->mp_sorting);
    
    return 0;
}


short *
write_block
#if NeedFunctionPrototype
(int bytes, int endflag)
#else
(bytes, endflag)
int bytes;
int endflag;
#endif
{
      static timestruc_t to;
#ifdef USE_DATA_QUEUE
      int b;
#endif

      to.tv_nsec = 0;

#ifdef USE_DATA_QUEUE
      /* put last block on full block queue */
      if (wdptr != NULL)
      {
	  wdptr->data_size = bytes;
	  add_to_queue(&sdat_ptr->write_full, wdptr - sdat_ptr->write_buf);
      }

      /* get new block of data */
      while((b = take_from_queue(&sdat_ptr->write_empty, NULL)) == -1)
	  ;
      wdptr = sdat_ptr->write_buf + b;
      /* see whether output device has had any problems */
      if (wdptr->data_size == 0)
      {
	  add_to_queue(&sdat_ptr->write_empty, b);
	  wdptr = NULL;
	  return NULL;
      }
      if (endflag)
      {
	  wdptr->data_size = 0;
	  add_to_queue(&sdat_ptr->write_full, b);
	  wdptr = NULL;
      }
      return (short *) wdptr->data;
#else      
/* put last block on full block queue */
      if (wdptr != NULL) {
	    (void) mutex_lock(&wdptr->mp);
	    if (wdptr->state == FILLING_BUF) {
		  wdptr->state = FULL_BUF;
		  wdptr->data_size = bytes;
		  (void) cond_signal(&wdptr->cv);
	    }
	    else {
		  (void) mutex_unlock(&wdptr->mp);
		  fprintf(stderr,"sort: write_block found shared data record in state\n");
		  exit(-1);
	    }
	    (void) mutex_unlock(&wdptr->mp);
	    wbuffer_pos = (wbuffer_pos+1) % NOS_OF_WBUFS;
      }

/* get new block of data */
      wdptr = &sdat_ptr->write_buf[wbuffer_pos];

      (void) mutex_lock(&wdptr->mp);
      while (wdptr->state != EMPTY_BUF) {
	    to.tv_sec = time(NULL) + WTHREAD_WAIT;
	    (void) cond_timedwait(&wdptr->cv, &wdptr->mp, &to);
	     /* see whether output device has had any problems */
	     if (wdptr->state == BAD_BUFFER) {
		   wdptr->state = EMPTY_BUF;
		   (void) mutex_unlock(&wdptr->mp);
		   return(NULL);
	     }
      }
	    
      if (endflag)
	    wdptr->state = END_OF_DATA_BUF;
      else
	    wdptr->state = FILLING_BUF;
      (void) mutex_unlock(&wdptr->mp);
      /*LINTED*/
      return((short *) wdptr->data);	
#endif
}
