/*
 *  decoding routine to unpack event by event data and 
 *  pass this on to user routines
 *  *** Data style == EUROGAM ***
 *
 *  started on new version 8/8/95
 *  modified 8/2/95
 *  Changed mask to 0x00001fff to allow starburst to write debug
 *  information in 0x0000E000 and still allow data unpacking.
 *  Much improved error logging and recovery 1/97..2/97
 */

/*
 * This file contains routines common to all Eurogam style decoders.
 * Currently this is Megha and DEMON.
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>

#include <sys/types.h>
#include <sys/time.h>

#include "sort_def.h"
#include "rdtape.h"
#include "rdtape_eg.h"

#define END_TOKEN          0xffff0000
#define MAX_EG_GLEN        0x3f

#define EG_TAPE_1          0x4d45
#define EG_TAPE_2          0x4748
#define EG_TAPE_3          0x4144
#define EG_TAPE_4          0x4154

#ifdef STYLE_MEGHA
#define MEGHA_MAX_DUMMY	   2 /* Maximum number of extra dummy cycles allowed
				in Megha format data */
#define ALLOW_EVENT_STAMPING /* Define this for debugging with Starburst
				event stamping */
#endif

#define FAST_CLEAR

/*
#define DUMP_RECORDS
#define PRINT_BLOCKS
*/
#define BAD_ONLY

#ifdef DUMP_RECORDS
#include <fcntl.h>
#define DUMP_NTH 1
static int dump_record_count = 1000;
static int dump_nth_count = DUMP_NTH;
static int dump_state = 2;
static int dump_fd;
#define DUMP_FILE "record.dump"
#endif

struct eg_word_struct {
      unsigned short type : 2;
      unsigned short addr :14;
      unsigned short data;
};

int
DECODER_NAME
#if NeedFunctionPrototype
(int filter)
#else
(filter)
int filter;
#endif
{
    register short *ebye;                     /* pointer to input buffer */
    register short event_len;
    
    register int *eptr, *header_ptr, *event_end;
    
    short *ap, *adclist = adclist_.adclist, *grouplist = adclist_.grouplist;
    time_t start_time, time_now;
    int bytes, record = 0;
    int adc_count = 0;
    long in_events = 0;
    int badfirst = 0, badlength = 0, badeob = 0, badtrunc = 0, badlong = 0;
    int recoverlong = 0, badrecords = 0, recovered = 0, badrepeat = 0;
    int repflag, badword = 0, badshort = 0, recovershort = 0, total;
    int bad, badcount = 0;
#ifdef STYLE_MEGHA
    int dummies[MEGHA_MAX_DUMMY + 2], dummy, lastdummy = 0;
    int overrun = 0, badfirstlength = 0, recoverfirstlength = 0;
    int badrepeatlength = 0, badoverrun = 0, badoverrun2 = 0;
    int recoveroverrun = 0, badstart = 0, bad16 = 0;
#endif

    /* register */ int group, hwords;
    /* register */ struct eg_word_struct *wptr;
    /* register */ int adc_bytes;
    int *end_ptr, i;
    unsigned short *ep;
#ifdef STYLE_MEGHA
    unsigned int *ip;
#endif

#ifdef DUMP_RECORDS
    if (dump_state == 2)
    {
	if ((dump_fd = open(DUMP_FILE, O_WRONLY | O_CREAT | O_EXCL, 0666)) < 0)
	{
	    perror(DUMP_FILE);
	    dump_state = 0;
	}
	else
	    dump_state = 1;
    }
#endif
    /* initialize event sorting */
    (void) memset((char *)addr2adc, 0, EG_ADDR_SIZE);
    (void) memset((char *) adcs_.bitpattern, 0, sizeof(adcs_.bitpattern));
    adc_bytes = adcnum_.nos_adcs * sizeof(short);
#ifdef FAST_CLEAR
    (void) memset((char *) adcs_.adcs, -1, adc_bytes);
#endif
#ifdef STYLE_MEGHA
    memset((char *) dummies, 0, sizeof(dummies));
#endif

    /* call user's initialization function */ 
    if (call_init())
	return filter;
      
    /* set start time */
    start_time = time(NULL);
      
    /* sort requested number of records or until end of file or error etc */
    for (record=1; ;record++) {
	if ((ebye = read_block(&bytes)) == NULL) 
	    break;
	    
	/*LINTED*/
	end_ptr = ((int *) ebye) + (bytes>>2);
	    
#ifdef PRINT_BLOCKS
	fprintf(stderr, "Block starts as follows:\n");
	{
	    for(i=0; i<256; i++)
	    {
		fprintf(stderr, "%04x", ebye[i] & 0xffff);
		if ((i & 15) == 15)
		    putc('\n', stderr);
		else
		    putc(' ', stderr);
	    }
	}
#endif

	bad = 0;

	/* check start of record header to see if it is a tape block*/
 	if (memcmp("MEGHADAT", (char *) ebye, 8) == 0 ||
 	    memcmp("EGEVENTD", (char *) ebye, 8) == 0)
	    /*LINTED*/
	    eptr = (int *) &ebye[12];
	else
	    /*LINTED*/
	    eptr = (int *) &ebye[0];

	if ((*eptr & END_TOKEN) != END_TOKEN) {
	    ep = (unsigned short *) eptr;
#ifdef STYLE_MEGHA
	    
	    if (ep[1] == 0xffff)
	    {
		fprintf(stderr, "Length missing on first event. ");
		badfirstlength++;
		bad = 1;
		ep++;
		do {
		    ep++;
		} while(ep < (unsigned short *) end_ptr && *ep != 0xffff);
		if (ep >= (unsigned short *) end_ptr)
		{
		    fprintf(stderr, "Unable to recover, skipping record.\n");
		    overrun = 0;
		    goto EVENT_LOOP_END;
		}
		if ((ep - ((unsigned short *) ebye)) & 1)
		{
		    for(ip = (unsigned int *) ebye;
			ip < (unsigned int *) end_ptr; ip++)
			*ip = (*ip>>16) | (ip[1]<<16);
		    ep -= 3;
		}
		eptr = (int *) ep;
		fprintf(stderr, "Skipping event, recovering at next event.\n");
		recoverfirstlength++;
	    }
	    else if (ep[3] == 0xffff && ep[0] != 0 && ep[1] != 0
#ifndef ALLOW_EVENT_STAMPING
		&& (ep[0] & 0xf000) == 0 && (ep[1] & 0xf000) == 0
#endif
		)
	    {
		fprintf(stderr, "Repeated event length on first event. "
			"Skipping event.\n");
		end_ptr--;
		for(ip = (unsigned int *) eptr;
		    ip < (unsigned int *) end_ptr; ip++)
		    *ip = (*ip>>16) | (ip[1]<<16);
		badrepeatlength++;
		bad = 1;
	    }
	    else
	    {
		for(; ep < (unsigned short *) end_ptr && *ep != 0xffff; ep++)
		    if ((*ep & 0xe000) != 0x2000 && (*ep & 0xe000) != 0x8000
			&& *ep != 0 &&
			(ep + 3 >= (unsigned short *) end_ptr ||
			 ep[3] != 0xffff))
			break;
		if (ep < (unsigned short *) end_ptr && *ep == 0xffff)
		{
		    if (((((char *) ep)-((char*) eptr)) & (sizeof(int)-1))
			!= 0)
		    {
			ep -= 3;
			/*LINTED*/
			for(ip = (unsigned int *) ep;
			    ip < (unsigned int *) end_ptr; ip++)
			    *ip = (*ip>>16) | (ip[1]<<16);
		    }
		    /*LINTED*/
		    eptr = (int *) ep;
		    if (*eptr == END_TOKEN)
		    {
			if (overrun)
			{
			    fprintf(stderr, "Data overrun from previous "
				    "record. Skipping record.\n");
			    badoverrun++;
			    bad = 1;
			    overrun = 0;
			    goto EVENT_LOOP_END;
			}
			fprintf(stderr, "Data overrun from unknown source."
				" Skipping record.\n");
			badoverrun2++;
			bad = 1;
			overrun = 0;
			goto EVENT_LOOP_END;
		    }
		    if (overrun)
		    {
			fprintf(stderr, "Big data overrun from previous "
				"record. Recovering some events. \n");
			badoverrun++;
			bad = 1;
			recoveroverrun++;
		    }
		    else
		    {
			fprintf(stderr, "Missing start of event. Recovering "
				"at next event.\n");
			badstart++;
			bad = 1;
		    }
		}
		else
		{
#endif
	    fprintf(stderr,"First event header is not a start token ... "
		    "trying next record.\nFirst 8 words were (HSM order): "
		    "%04x %04x %04x %04x %04x %04x %04x %04x\n", ep[1],
		    ep[0], ep[3], ep[2], ep[5], ep[4], ep[7], ep[6]);
	    badfirst++;
	    bad = 1;
	    goto EVENT_LOOP_END;
	}
#ifdef STYLE_MEGHA	
	    }
	}
	overrun = 0;
#endif
	
	/* search for end of block indication (header == END_TOKEN) */
	while( *eptr != (int) END_TOKEN) {
	    /* extract fields from event header */
	    header_ptr = eptr++;
#ifdef STYLE_MEGHA
#ifdef ALLOW_EVENT_STAMPING
		event_len = (*header_ptr & 0x00000fff) >> 1;
#else
		event_len = (*header_ptr & 0x0000ffff) >> 1;
#endif
		dummy = 0;
#else
		event_len = (*header_ptr & 0x0000ffff) >> 2;
#endif
	    
	    /* check that event length is sensible */
	    if((event_len < 0) || (event_len > 4096)) {
		fprintf(stderr,"Event length (%d  %04x) out of range ... "
			"skipping rest of record %d\n", event_len,
			event_len, record);
		badlength++;
		bad = 1;
		break;
	    }
	    event_end = header_ptr + event_len;
	    
	    if (event_end >= end_ptr)
	    {
		fprintf(stderr, "Last event in block truncated - "
			"ignoring it.\n");
		badtrunc++;
#ifdef STYLE_MEGHA
		overrun = 1;
#endif
		bad = 1;
		break;
	    }
	    
	    /* reset adcs_common block */
#ifdef FAST_CLEAR
	    for(ap = adclist_.adclist; ap < adclist; ap++)
		adcs_.adcs[*ap-1] = -1;
	    for(ap = adclist_.grouplist; ap < grouplist; ap++)
	    {
		groups[*ap].group_len = 0;
		groups[*ap].group_ptr = NULL;
	    }
#else
	    (void) memset((char *) adcs_.adcs, -1, adc_bytes);
	    (void) memset((char *) groups, 0, sizeof(groups));
#endif
	    adc_count = 0;
	    grouplist = adclist_.grouplist;	
	    adclist = adclist_.adclist;
	    repflag = 0;
 
	    /* unpack individual event ** EVENT LOOP ** */		  
	    while ( (*eptr & (int) END_TOKEN) != (int) END_TOKEN)  {
		if (eptr >= event_end) {
		    int *sptr = eptr;
		    
		    do {
			eptr++;
		    } while(eptr < end_ptr-1 &&
			    (*eptr & (int) END_TOKEN) != (int) END_TOKEN);
		    if ((*eptr & (int) END_TOKEN) != (int) END_TOKEN)
		    {
			int *ep;
			
			fprintf(stderr, "End of block at short word %d "
				"without end of block marker in record %d\n",
				((short *) sptr) - ebye, record);
			fprintf(stderr,"Words surrounding error were:\n    ");
			for(ep = eptr - 10, i = 4;
			    ep < eptr+30 && ep < end_ptr; ep++)
			{
			    if (i > 69)
			    {
				fprintf(stderr, "\n    ");
				i = 4;
			    }
			    fprintf(stderr, "%08x%c", *ep,
				    (ep == eptr-1) ? '<' : ' ');
			    i += 9;
			}
			fprintf(stderr, "\n");
			badeob++;
			bad = 1;
			goto EVENT_LOOP_END;
		    }
		    if (!bad)
		    {
			badrecords++;
			bad = 1;
		    }
		    badlong++;
		    if ((eptr-header_ptr)*4 > 1024)
		    {
			fprintf(stderr, "Expected event length %d, no start "
				"of event marker within 1024 (next at %d). "
				"Not recovering.\n",
				event_len*4, (eptr-header_ptr)*4);
			goto EVENT_LOOP_END;
		    }
		    fprintf(stderr, "Expected event length %d, actually %d, "
			    "error %d. Recovering at next event.\n",
			    event_len*4, (eptr-header_ptr)*4,
			    (eptr-header_ptr-event_len)*4);
		    recoverlong++;
		    goto EVENT_LOOP_CONTINUE;
		}
		
		/* event type defines tells us how much data is to follow*/
		wptr = (struct eg_word_struct *) eptr++;
#ifdef STYLE_MEGHA
		switch ((*(eptr-1)) >> 29) {
		case 1:           /* simple parameter word found */
		    /*LINTED*/
		    if (ap = addr2adc[wptr->addr & 0x1fff])
		    {
			if (*ap != -1)
			{
			    fprintf(stderr, "ADC %d (Eurogam channel %d) "
				    "occurs twice, was %d, now %d.\n", 
				    ap - adcs_.adcs + 1, 
				    wptr->addr & 0x1fff, *ap,
				    wptr->data & 0x1fff);
			    repflag = 1;
			    bad = 1;
			}
			*ap = wptr->data & 0x1fff;
			*adclist++ = ap - adcs_.adcs + 1;
		    }
		    adc_count++;
		    break;
		case 2:           /* group format parameter(s) found */
		    group = wptr->addr & 0x00ff;
		    hwords = (wptr->addr >> 8) & 0x003f; 
		    groups[group].group_len = hwords;
		    groups[group].group_ptr = &wptr->data;
		    eptr += (hwords / 2);
		    *grouplist++ = group;
		    break;
		case 0:	       /* possible dummy cycle */
		    if (*(eptr-1) == 0 || *(eptr-1) == 0x02000000)
		    {
			dummy++;
			break;
		    }
		    /* FALL THROUGH */
		default:           /* invalid word format found  */
		    if ((*eptr & 0xffff) == 0xffff ||
			(*eptr & 0xe000) == 0x2000)
		    {
			fprintf(stderr, "16 bit data skew in HSM "
				"(urk).\n");
			bad16++;
		    }
		    else
		    {    
			int *ep;
			
			fprintf(stderr,"Invalid word format found ... "
				"skipping rest of record %d\n",record);
			fprintf(stderr,"Words surrounding error were:"
				"\n    ");
			for(ep = eptr - 10, i = 4;
			    ep < eptr+30 && ep < end_ptr; ep++)
			{
			    if (i > 69)
			    {
				fprintf(stderr, "\n    ");
				i = 4;
			    }
			    fprintf(stderr, "%08x%c", *ep,
				    (ep == eptr-1) ? '<' : ' ');
			    i += 9;
			}
			fprintf(stderr, "\n");
			badword++;
		    }		    
		    bad = 1;
		    goto EVENT_LOOP_END;
		}
#else
		switch (wptr->type) {
		case 0:           /* simple parameter word found */
		    /*LINTED*/
		    if (ap = addr2adc[wptr->addr])
		    {
			if (*ap != -1)
			{
			    fprintf(stderr, "ADC %d (Eurogam channel %d) "
				    "occurs twice, was %d, now %d.\n", 
				    ap - adcs_.adcs + 1, 
				    wptr->addr, *ap, wptr->data);
			    repflag = 1;
			    bad = 1;
			}
			*ap = wptr->data;
			*adclist++ = ap - adcs_.adcs + 1;
		    }
		    adc_count++;
		    break;
		case 1:           /* group format parameter(s) found */
		    group = wptr->addr & 0x00ff;
		    hwords = (wptr->addr >> 8) & 0x003f; 
		    groups[group].group_len = hwords;
		    groups[group].group_ptr = &wptr->data;
		    eptr += (hwords / 2);
		    *grouplist++ = group;
		    break;
		default:           /* invalid word format found  */
		{
		    int *ep;
		    
		    fprintf(stderr,"invalid word format found ... "
			    "skipping rest of record %d\n", record);
		    fprintf(stderr,"Words surrounding error were:\n    ");
		    for(ep = eptr - 10, i = 4;
			ep < eptr+30 && ep < end_ptr; ep++)
		    {
			if (i > 69)
			{
			    fprintf(stderr, "\n    ");
			    i = 4;
			}
			fprintf(stderr, "%08x%c", *ep,
				(ep == eptr-1) ? '<' : ' ');
			i += 9;
		    }
		}
		    fprintf(stderr, "\n");
		    badword++;
		    bad = 1;
		    goto EVENT_LOOP_END;
		}
#endif
	    }

	    /* check that we are past end of event */
#ifdef STYLE_MEGHA
	    while(dummy < lastdummy && eptr != event_end)
	    {
		dummy++;
		event_end--;
		event_len--;
	    }
#endif

	    if (eptr != event_end)
	    {
		if (!bad)
		{
		    badrecords++;
		    bad = 1;
		}
		badshort++;
		recovershort++;
		fprintf(stderr, "Expected event length %d, actually %d, "
			"error %d. Recovering at next event.\n",
			event_len*2, (eptr-header_ptr)*2,
			(eptr-header_ptr-event_len)*2);
		continue;
	    }
	    
#ifdef STYLE_MEGHA
	    if (dummy > MEGHA_MAX_DUMMY)
	    {
		fprintf(stderr,
			"Too many dummy cycles, skipping event.\n");
		dummies[MEGHA_MAX_DUMMY+1]++;
		continue;
	    }
	    else
		dummies[dummy]++;
#endif

	    if (repflag)
	    {
		fprintf(stderr, "Skipping event with repeated channels.\n");
		badrepeat++;
		continue;
	    }

	    /* provide access to various information to user routine */
	    adcs_.triggernos = adc_count;
	    adcs_.event = ++in_events;
	    adcs_.record = record;
	    adcs_.wrtevt = 0;
	    
	    *adclist = -1;
	    *grouplist = -1;
	    if (bad)
		recovered++;
	    
	    /* call user provided event sorting routine */
	    if (sortin()) {
		fprintf(stderr, sortinfailmess);
		goto ABORT_SORT;
	    }	    
	    /* write out event if wrtevt set to true */
	    if (filter && adcs_.wrtevt) {
		if ( write_filt((short *)header_ptr, 2*event_len) )
		    filter = 0;
	    }

	EVENT_LOOP_CONTINUE:
	    ;

#ifdef STYLE_MEGHA	    
	    lastdummy = dummy;
#endif
	}    /* loop on ebye buffer until end token */
	
	/* come to here to jump out of faulty record */		  
    EVENT_LOOP_END:
	bytes = 0;
	
	if (bad)
	    badcount++;

#ifdef DUMP_RECORDS
	if (dump_state == 1
#ifdef BAD_ONLY
	    && bad
#endif
	    )
	{
	    if (!--dump_nth_count)
	    {
		
		write(dump_fd, ebye, 16384);
		if (!--dump_record_count)
		{
		    close(dump_fd);
		    dump_state = 0;
		}
		else
		    dump_nth_count = DUMP_NTH;
	    }
	}
#endif	    
	
    }   /* loop over requested number of records or until eof */
    
ABORT_SORT:
    
    /* lets note the time at end */
    time_now = time(NULL);
    
    /* call user provided clearup routine  */
    call_finish();
    
    /* output statistics */
    record--;
    fprintf(stderr, "\n*** sort statistics ***\n");
    fprintf(stderr, "\nsorted %ld events in %d seconds\n"
	    "Average sort rate = %g events per second\n",
	    in_events, (int) (time_now-start_time),
	    (double) in_events / (time_now-start_time));
    fprintf(stderr, "Records with missing start token       : %d (%8.4f%%)\n",
	    badfirst, 100.0*badfirst/record);
    fprintf(stderr, "Records with event length out of range : %d (%8.4f%%)\n",
	    badlength, 100.0*badlength/record);
    fprintf(stderr, "Records with invalid word format       : %d (%8.4f%%)\n",
	    badword, 100.0*badword/record);
    fprintf(stderr, "Records with incorrect event length    : %d (%8.4f%%)\n",
	    badrecords, 100.0*badrecords/record);
    fprintf(stderr, "Records without end of block marker    : %d (%8.4f%%)\n",
	    badeob, 100.0*badeob/record);
    fprintf(stderr, "Records containing truncated events    : %d (%8.4f%%)\n",
	    badtrunc, 100.0*badtrunc/record);
#ifdef STYLE_MEGHA
    fprintf(stderr, "Records containing 16 bit skew         : %d (%8.4f%%)\n",
	    bad16, 100.0*bad16/record);
    fprintf(stderr, "Records with missing first length      : %d (%8.4f%%); "
	    "recoveries: %d\n", badfirstlength, 100.0*badfirstlength/record,
	    recoverfirstlength);
    fprintf(stderr, "Records with repeated event length     : %d (%8.4f%%)\n",
	    badrepeatlength, 100.0*badrepeatlength/record);
    fprintf(stderr, "Records overrun from previous record   : %d (%8.4f%%); "
	    "recoveries: %d\n", badoverrun, 100.0*badoverrun/record,
	    recoveroverrun);
    fprintf(stderr, "Records overrun from unknown source    : %d (%8.4f%%)\n",
	    badoverrun2, 100.0*badoverrun2/record);
    fprintf(stderr, "Records with missing start             : %d (%8.4f%%)\n",
	    badstart, 100.0*badstart/record);
#endif
    bad = badfirst + badlength + badword + badeob
#ifdef STYLE_MEGHA
	+ bad16 + badfirstlength - recoverfirstlength + badoverrun
	- recoveroverrun + badoverrun2
#endif
	+ (badshort+badlong-recovershort-recoverlong);
    total = badrecords - (badshort+badlong-recovershort-recoverlong)
#ifdef STYLE_MEGHA
	+ recoverfirstlength + badrepeatlength + badstart + recoveroverrun
#endif
	;
    fprintf(stderr, "Total records in stream: %d; good: %d (%8.4f%%);\n"
	    "      recoverable: %d (%8.4f%%); bad: %d (%8.4f%%)\n", record,
	    (record-badcount), 100.0*(record-badcount)/record,
	    total, 100.0*total/record, bad, 100.0*bad/record);
    bad = badshort+badlong+badrepeat;
#ifdef STYLE_MEGHA
    bad += dummies[MEGHA_MAX_DUMMY+1];
#endif
    total = in_events+bad;
    fprintf(stderr, "Events with word count too short       : %d (%8.4f%%); "
	    "recoveries: %d\n", badshort,
	    100.0*badshort/total, recovershort);
    fprintf(stderr, "Events with word count too long        : %d (%8.4f%%); "
	    "recoveries: %d\n", badlong,
	    100.0*badlong/total, recoverlong);
    fprintf(stderr, "Events with repeated channels          : %d (%8.4f%%)\n",
	    badrepeat, 100.0*badrepeat/total);
    fprintf(stderr, "Events recovered by recovery procedures: %d (%8.4f%%)\n",
	    recovered, 100.0*recovered/total);
#ifdef STYLE_MEGHA
    for(i=0; i<=MEGHA_MAX_DUMMY; i++)
	fprintf(stderr, "Events with %d extra dummy cycle%c       : "
		"%d (%8.4f%%)\n", i, (i == 1) ? ' ' : 's', dummies[i],
		100.0*dummies[i]/total);
    fprintf(stderr, "Events with >%d extra dummy cycle%c      : "
	    "%d (%8.4f%%)\n", MEGHA_MAX_DUMMY,
	    (MEGHA_MAX_DUMMY == 1) ? ' ' : 's', dummies[MEGHA_MAX_DUMMY+1],
	    100.0*dummies[MEGHA_MAX_DUMMY+1]/total);
#endif
    fprintf(stderr, "Total events in stream: %d; good: %ld (%8.4f%%);\n"
	    "    bad: %d (%8.4f%%)\n", total, in_events, 100.0*in_events/total,
	    bad, 100.0*bad/total);
    fprintf(stderr,"\nfinished sorting data after reading %d records\n",
	    record);
    
    return filter;
}

/*
 *  user callable write event function
 *  write data to filter stream in Eurogam style format
 *
 *  first draft 15/2/95
 */

static int fbuf[NOS_ADC_WORDS+1];

int
ENCODER_NAME
#if NeedFunctionPrototype
(int size, short *array)
#else
(size, array)
int size;
short *array;
#endif
{
    register int i, j;
    register short group, glen;
    register int event_len = 1;
    register short *sptr  = (short *) &fbuf[1];

    (void) memset(fbuf, 0, sizeof(fbuf));
      
    /* ensure that we have an event length within acceptable ranges */
    if (size <= 0) {
	fprintf(stderr,"you called writefn with an invalid "
		"value of %d\n",size);
	return(-2);
    }

    /* pack event */
    for(i=0; i<size; i++) {
	if (event_len >= NOS_ADC_WORDS) {
	    fprintf(stderr,"EGAM format supports maximum event "
		    "length of %d words\n", NOS_ADC_WORDS);
	    return(-3);
	}
	/* pack simple parameter word if corresponding address exists */
	if (array[i] > 0) {
	    if (adc2addr[i] > 0) {
		*sptr++ = adc2addr[i];
		*sptr++ = array[i];
		event_len++;
	    }
	    else {
		/* check group number */
		group = array[i];
		if (group < 0 || group >= NOS_EG_GROUPS) 
		    continue;
			
		/* check group length */
		glen = -array[i+1];
		if (glen <= 0 || glen >= MAX_EG_GLEN) {
		    fprintf(stderr,"invalid group length => "
			    "%d for EGAM format\n",glen);
		    return(-4);
		}
		if (sptr + glen >= (short *) &fbuf[NOS_ADC_WORDS]) {
		    fprintf(stderr,"group length => %d too large "
			    "for event stack\n",glen);
		    return(-4);
		}
		i += 2;
			
		/* move group onto event stack */
		*sptr++ = 0x4000 | group | (glen << 8);
		for(j=0; j<glen; j++)
		    *sptr++ = array[i++];
			
		/* ensure 4 byte alignment */
		if (glen % 2 == 0) 
		    *sptr++ = 0;
			
		/* advance pointers etc appropriatley */
		event_len += (glen/2) + 1;
	    }
	}

    }

    if (event_len > 1) {
	/* setup start token */
#ifdef STYLE_MEGHA
	fbuf[0] = (event_len<<1) | END_TOKEN;
#else
	fbuf[0] = (event_len<<2) | END_TOKEN;
#endif
	    
	/* add event to output data buffer */
	if ( write_filt((short *)fbuf, 2*event_len) )
	    return(-3);
    }
    
    return(event_len);
}

