/*
 * decoding routine to unpack event by event data and 
 * pass this on to user routines
 * *** Data style == FSU PC ***
 *
 * started on 1/7/97
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/types.h>
#include <sys/time.h>

#include "sort_def.h"
#include "rdtape.h"

/*
 * Supplied documentation
 *
 * Multiparameter data is accumulated in a 2048 byte buffer by the program
 * running on the PC.  When this buffer is full, it is written out to the tape
 * drive attached to the PC, but because of a peculiarity in the exabyte tape
 * driver, the data buffer is written out in two 1024 byte buffers.
 * Consequently, when reading the tape, you must use record lengths of 1024
 * bytes.  Your data analysis program must deal with splicing the two data
 * records together, or (since an event may get split between the two records)
 * it must handle the odd event that is split as a special case.  In the data
 * analysis programs here at FSU, we have used the latter method.
 * 
 * There are two "special" buffers (2048 bytes apiece) which are also written
 * for each run, one at the beginning of each run, and one at the end.  These
 * can be distinguisehd from data buffers in that the very first word (two
 * bytes) is "FFFF" in hexadecimal.  These buffers contain in the next two
 * bytes the run number (binary), followed by 80 bytes of some trviial text
 * (ascii), and then the time and date (start time, or stop time, depending
 * the location of the buffer) in ascii.  There have been instances where the
 * first of these buffers were not written to tape, for the very first run at
 * the beginning of the tape.  Currently at FSU, we do not take advantage of
 * this information.  Our analysis programs look for the "FFFF" at the first
 * word, and then skip the rest of the buffer.
 * 
 * The event structure for your run is as follows.  EVERY event begins with a
 * 16 bit word with the high bit set.  In polarization experiements,
 * additional bits in this word are set for the polarization, but in this
 * experiment, the polarization was zero, so all the remaining bits should be
 * zero.  In our FORTRAN programs, we search for this (I*2) event header by
 * looking for a value less than zero, then the ADC values are recorded
 * sequentially following the event header.  So, for example, with 10 ADC's
 * the first two events in the buffer appear like:
 * 
 * H 1 2 3 4 5 6 7 8 9 10 H 1 2 3 4 5 6 7 8 9 10
 * 
 * where H is the header word.
 *
 * Note by SMS: Data is byte-swapped wrt Sun shorts.
 */

int
rdtape_fsupc
#if NeedFunctionPrototype
(int filter)
#else
(filter)
int filter;
#endif
{
    time_t start_time, time_now;
    int i, record, bytes, cont = 2;
    unsigned int *p;
    unsigned short *ptr, *ebye, *sp, *recordend;
    short *adclist;
    long in_events = 0;
    int adc_bytes, maxadcs = 0, adcp = 0;
    unsigned short gluebuffer[512];
    
    adc_bytes = adcnum_.nos_adcs * sizeof(short);
    
    /* call user's initialization function */      
    if (call_init())
	return filter;

    adclist = adclist_.adclist;
    *adclist_.grouplist = -1;
		
    /* set start time */
    start_time = time(NULL);
      
    /* sort requested number of records or until end of file etc */
    for (record=1; ;record++) {
	if ( (ebye = (unsigned short *) read_block(&bytes)) == NULL) 
	    break;

	if (cont == 3)
	{
	    cont = 0;
	    continue;
	}

	recordend = ebye + bytes/sizeof(short);

	if (*ebye == 0xffff && cont != 1)
	{
	    unsigned char *s = (unsigned char *) (ebye+1);
	    int n;

	    cont = 0;
	    if (s[1] < 32)
	    {
		fprintf(stderr, "Header for run %d: ",
			*s + (s[1]<<8));
		s += 2;
		n = 19;
	    }
	    else
	    {
		fprintf(stderr, "Header: ");
		n = 8;
	    }
	    recordend = ebye + 128;
	    for(; s < (unsigned char *) recordend; s++)
	    {
		if (*s == 0)
		    continue;
		if (n == 0)
		{
		    printf("    ");
		    n = 4;
		}
		switch(*s)
		{
		case 0:
		    break;
		case 9:
		    printf("\\t");
		    n += 2;
		    break;
		case 10:
		    printf("\\n");
		    n += 2;
		    break;
		case 13:
		    printf("\\r");
		    n += 2;
		    break;
		default:
		    if (*s < 32 || *s > 126)
		    {
			printf("\\%03o", *s);
			n += 4;
		    }
		    else
		    {
			putchar(*s);
			n++;
		    }
		}
		if (n > 72)
		{
		    putchar('\n');
		    n = 0;
		}
	    }
	    if (n != 0)
		putchar('\n');
	    cont = 3;
	    continue;
	}

	/*LINTED*/
	for(i = bytes>>2, p = (unsigned int *) ebye; i--; p++)	
	  *p = ((*p & 0x00ff00ff)<<8) | ((*p & 0xff00ff00)>>8);

/*
	fprintf(stderr, "Event data block starts with:\n");
	for(i = 0; i<32; i++)
	{
	  fprintf(stderr, "%04x", ebye[i]);
	  if ((i & 15) == 15)
	    putc('\n', stderr);
	  else
	    putc(' ', stderr);
	}
*/

	if (*ebye & 0x8000)
	{
	    if (cont == 1)
	    {
		fprintf(stderr, 
			"Continued event actually ended on block boundary.\n");

		adcs_.triggernos = adcp;
		adcs_.event = ++in_events;
		adcs_.record = record;
		adcs_.wrtevt = 0;
		*adclist = -1;

	        if (sortin()) 
		{
		    fprintf(stderr, sortinfailmess);
		    goto ABORT_SORT;
		}

		if (filter && adcs_.wrtevt)
		    if (write_filt((short *) gluebuffer, adcp+1))
			filter = 0;

		if (adcp > maxadcs)
		{
		    if (maxadcs != 0)
			fprintf(stderr, "Expected events with up to %d ADCs, "
				"but this one has %d.\n", maxadcs, adcp);
		    maxadcs = adcp;
		}
	    }
	    cont = 0;
	}
	else if (cont != 1)
	{
	    while(*ebye == 0 && ebye < recordend)
		ebye++;
	    if (ebye == recordend)
	    {
		fprintf(stderr, "Empty buffer ignored.\n");
		cont = 0;
		continue;
	    }
	    while((*ebye & 0x8000) == 0 && ebye < recordend)
		ebye++;
	    if (cont == 0)
	    {
		fprintf(stderr, "Unexpected continuation block, "
			"ignoring partial event at start.\n");
	    }
	    else
		cont = 0;
	    if (ebye == recordend)
	    {
		fprintf(stderr,
			"Event block did not contain any events.\n");
		continue;
	    }
	}

	for(;;)
	{
/*
	    fprintf(stderr, "Event starts with:");
	    for(i = 0; i<12; i++)
		fprintf(stderr, " %04x", ebye[i]);
	    putc('\n', stderr);
*/
	    
	    if (*ebye == 0xffff) /* Our way of signalling end of block */
		break;

	    if (cont == 0 && *ebye == 0)
		break;
	    
	    if (cont != 1)
	    {
		memset((char *) adcs_.adcs, -1, adc_bytes);
		adclist = adclist_.adclist;
		*adclist_.grouplist = -1;
		adcp = 0;
	    }
	    if (*ebye & 0x8000)
		ptr = ebye + 1;
	    else
		ptr = ebye;
	    for(; (*ptr & 0x8000) == 0 && ptr < recordend;
		ptr++)
		if (adcp < adcnum_.nos_adcs)
		{
		    adcs_.adcs[adcp++] = *ptr;
		    *adclist++ = adcp;
		}
	    
	    if (ptr == recordend)
	    {
		sp = ptr;
		while(adcp > maxadcs && *--sp == 0)
		    adcp--;
		if (adcp != maxadcs)
		{
/*
		    fprintf(stderr, "Event split over block boundary.\n");
*/
		    cont = 1;
		    for(sp = gluebuffer, ptr = ebye; ptr < recordend;)
			*sp++ = *ptr++;
		    break;
		}
	    }
	    
	    adcs_.triggernos = adcp;
	    adcs_.event = ++in_events;
	    adcs_.record = record;
            adcs_.wrtevt = 0;
	    *adclist = -1;

	    /*user supplied routine*/
	    if (sortin())
	    {
		fprintf(stderr, sortinfailmess);
		goto ABORT_SORT;
	    }
	    
	    if (filter && adcs_.wrtevt)
	    {
		if (cont == 1)
		{
		    i = ptr - ebye;
		    for(sp = gluebuffer + adcp - i, ptr = ebye; i--; )
			*sp++ = *ptr++;
		    if (write_filt((short *) gluebuffer, adcp+1))
			filter = 0;
		}
		else if (write_filt((short *) ebye, adcp+1))
		    filter = 0;
	    }

	    cont = 0;
	    
	    ebye = ptr;
	    if (adcp > maxadcs)
	    {
		if (maxadcs != 0)
		    fprintf(stderr, "Expected events with up to %d ADCs, "
			    "but this one has %d.\n", maxadcs, adcp);
		maxadcs = adcp;
	    }
	}
    }

ABORT_SORT:

    /* lets note the time at end */
    time_now = time(NULL);
    
    /* call user provided clearup routine  */
    call_finish();
  
    /* output statistics */
    record--;
    fprintf(stderr, "\n*** sort statistics ***\n");
    fprintf(stderr, "\nsorted %ld events in %d seconds\n"
	    "Average sort rate = %g events per second\n",
	    in_events, (int) (time_now-start_time),
	    (double) in_events / (time_now-start_time));

    return filter;
}

/*
 *  functions used in the filtering of data
 *  write any data to end of filtered data block
 */

/*ARGSUSED*/
static void
filt_tail
#if NeedFunctionPrototype
(short *base, short *fptr)
#else
(base, fptr)
short *base;
short *fptr;
#endif
{
    unsigned int *p;

    /*LINTED*/
    for(p = (unsigned int *) base;
	/*LINTED*/
	p < (unsigned int *) fptr; p++)
	*p = ((*p & 0x00ff00ff)<<8) | ((*p & 0xff00ff00)>>8);

    *fptr = (short) 0xffff;

    return;
}

/*
 *  write any data to the beginning of filtered data block
 */
static short *
filt_head
#if NeedFunctionPrototype
(short *base)
#else
(base)
short *base;
#endif
{
    return(base);
}

/* fsupc filter data specification */
static filtdat_t fsupc_filtdat = {
      1024,
      510,
      filt_head,
      filt_tail,
      NULL,
      "FSUPC"
};

/* routine to get pointer to fsupc_filtdat */
filtdat_t *
get_fsupc_filtdat
#if NeedFunctionPrototype
(void)
#else
()
#endif
{
      return(&fsupc_filtdat);
}
