/*
 * decoding routine to unpack event by event data and 
 * pass this on to user routines
 * *** Data style == IN2P3 ***
 *
 * started on 1/5/97
 * based on sort-shell code by Chris Pearson
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <sys/types.h>
#include <sys/time.h>

#include "sort_def.h"
#include "rdtape.h"

/*
#define DUMP
*/

/* IN2P3 format data comes from Ganil

   IN2P3 records are fixed format 16kbyte records in 16k physical blocks
   of 16 bit word orientated *little-endian* data.

   all records have a 12 byte header with a record type string
   in bytes 1-8 (byte 1 is always space leaving 7 further characters)
   followed by a 4 byte record number

   Known record type strings are ...

   " FILEH  " : file header record
   " EVENTH " : event header record
   " COMMENT" : comment record containing ascii ??
   " PARAM  " : parameter information record
   " EVENTDB" : event data record
   " SCALER " : scaler data record
   " SPECTH " : spectrum header record
   " SPECTD " : spectrum data record

   fileh record :
   -------------------------------------------------------------------
      12 byte :             " IN2P3-PN   "  : id
       4 byte :                     "   N"  : version
      16 byte :         " XXXXXXXXXXXXXXX"  : lab
      16 byte :         " XXXXXXXXXXXXXXX"  : Machine/os
      16 byte :         " XXXXXXXXXXXXXXX"  : Acquisition program name
      16 byte :                             : reserved
   -------------------------------------------------------------------
       8 byte :                 " NNNNNNN"  : Block length
       4 byte :           " MSB" or " LSB"  : Byte order
      20 byte :     " DD-MMM-YY HH/MM/SS "  : date
      48 byte :                             : reserved
   -------------------------------------------------------------------

   eventh record :
   -------------------------------------------------------------------
       8 byte :   " WORDC  " or " FFFF   "  : Event separation type
       8 byte :   " INT*2  " or " INT*4  "  : Event data item size
       8 byte :   " INT*2  " or " INT*4  "  : Word count data item size
      20 byte :     " DD-MMM-YY HH/MM/SS "  : Date
      16 byte :         " XXXXXXXXXXXXXXX"  : Run name string
       8 byte :                 " NNNNNNN"  : Run number string
       4 byte :                             : Reserved
       8 byte :                 " GANIL  "  : Structure type ?
   -------------------------------------------------------------------
      80 byte :      " blah blah blah ..."  : Message ??
   -------------------------------------------------------------------
       4 byte :                     "   N"  : Num following Comment blocks
   -------------------------------------------------------------------

   IN2P3 events are highly experiment-specific, so we take the usual cop-out
   of passing the event type word as ADC#0, etc etc

*/

#define IN2P3FILEHEAD   " FILEH  "
#define IN2P3EVENTHEAD  " EVENTH "
#define IN2P3COMMENT    " COMMENT"
#define IN2P3PARAM      " PARAM  "
#define IN2P3EVENTDATA  " EVENTDB"
#define IN2P3SCALER     " SCALER "
#define IN2P3SPECHEAD   " SPECTH "
#define IN2P3SPECDATA   " SPECTD "

/* Comparing integers is faster than comparing strings or byte arrays */

#define IN2P3FILEHEAD1  0x2046494c
#define IN2P3FILEHEAD2  0x45482020
#define IN2P3EVENTHEAD1 0x20455645
#define IN2P3EVENTHEAD2 0x4e544820
#define IN2P3COMMENT1   0x20434f4d
#define IN2P3COMMENT2   0x4d454e54
#define IN2P3PARAM1     0x20504152
#define IN2P3PARAM2     0x414d2020
#define IN2P3EVENTDATA1 0x20455645	/* Same as IN2P3EVENTHEAD1 */
#define IN2P3EVENTDATA2 0x4e544442
#define IN2P3SCALER1    0x20534341
#define IN2P3SCALER2    0x4c455220
#define IN2P3SPECHEAD1  0x20535045
#define IN2P3SPECHEAD2  0x43544820
#define IN2P3SPECDATA1  0x20535045	/* Same as IN2P3SPECHEAD1 */
#define IN2P3SPECDATA2  0x43544420

#define MAXIN2P3RECLEN  65536
#define MAXIN2P3EVENTSIZE 512

#define IN2P3ENDMARK 0

struct in2p3file_head_struct {
    char id[12];
    char version[4];
    char lab[16];
    char os[16];
    char prog[16];
    char res1[16];
    char block_len[8];
    char order[4];
    char date[20];
    char res2[48];
};

struct in2p3event_head_struct {
    char sep[8];
    char size[2][8];
    char date[20];
    char run[16];
    char number[8];
    char res[4];
    char type[8];
    char message[80];
    char comment_num[4]; 
};

struct in2p3event_struct { 
   short event_number;
   short stat1[2];
   short event_words;
   short stat2[2];
};

struct in2p3scaler_value_struct {
    unsigned int label;
    int state;
    unsigned int count;
    unsigned int freq;
    unsigned int ticks;
    unsigned int free[3];
};

struct in2p3_scaler_buffer_struct {
    unsigned int num_bytes;
    unsigned int num_scalers;
    unsigned int state;
    unsigned int free[2];
    struct in2p3scaler_value_struct *scalers;
};

union in2p3_ptrs {
    unsigned int *u;
    short *s;
    unsigned short *us;
    char *c;
    struct in2p3file_head_struct *fh;
    struct in2p3event_head_struct *eh;
    struct in2p3event_struct *e;
    struct in2p3scaler_value_struct *sv;
    struct in2p3_scaler_buffer_struct *sb;
};


int
rdtape_in2p3
#if NeedFunctionPrototype
(int filter)
#else
(filter)
int filter;
#endif
{
    register union in2p3_ptrs ebye;      /* pointer to input buffer */
    time_t start_time, time_now;
    int i, j, len, record, bytes;
    unsigned int *p;
    unsigned short *ptr, *endptr;
    short *adclist;
    char *recordend;
    long in_events = 0;
    int adc_bytes;

    adc_bytes = adcnum_.nos_adcs * sizeof(short);

    /* call user's initialization function */      
    if (call_init())
	return filter;
  
    /* set start time */
    start_time = time(NULL);
      
    /* sort requested number of records or until end of file etc */
    for (record=1; ;record++) {
	if ( (ebye.s = read_block(&bytes)) == NULL) 
	    break;

	recordend = ebye.c + bytes;

	switch(*ebye.u)
	{
	case IN2P3FILEHEAD1:
	    if (ebye.u[1] != IN2P3FILEHEAD2)
		goto BADHEADER;
	    ebye.c += 12;
	    fprintf(stderr,
		    "File header: %.12s Version %.4s %.16s %.16s\n"
		    "             %.16s %.8s %.4s %.20s\n",
		    ebye.fh->id, ebye.fh->version,
		    ebye.fh->lab, ebye.fh->os,
		    ebye.fh->prog, ebye.fh->block_len,
		    ebye.fh->order, ebye.fh->date);
	    break;
	case IN2P3EVENTHEAD1:	/* Same as IN2P3EVENTDATA1 */
	    if (ebye.u[1] == IN2P3EVENTHEAD2)
	    {
		register char *np;

		ebye.c += 12;
		fprintf(stderr,
			"Event header: %.20s %.7s %.8s (%.8s) %.8s\n"
			"              %.16s (%.8s)\n",
			ebye.eh->date, ebye.eh->type,
			ebye.eh->sep, ebye.eh->size[0],
			ebye.eh->size[1], ebye.eh->run,
			ebye.eh->number);
		for (np = ebye.eh->number, i = 8; i && !isdigit(*np);
		     i--, np++)
		    ;
		for(j = 0; i--; np++)
		    j = j*10 + *np-'0';
		filenm_.runnumber = j;
		break;
	    }
	    if (ebye.u[1] != IN2P3EVENTDATA2)
		goto BADHEADER;
	    ebye.c += 12;
	    for(i = (bytes-12)>>2, p = ebye.u; i--; p++)	
		*p = ((*p & 0x00ff00ff)<<8) | ((*p & 0xff00ff00)>>8);

#ifdef DUMP
	    fprintf(stderr, "Event data block starts with:\n");
	    for(i = 0; i<256; i++)
	    {
		fprintf(stderr, "%04x", ebye.us[i]);
		if ((i & 15) == 15)
		    putc('\n', stderr);
		else
		    putc(' ', stderr);
	    }
#endif

	    for(;;)
	    {
#ifdef DUMP
		fprintf(stderr, "Event starts with:");
		for(i = 0; i<12; i++)
		    fprintf(stderr, " %04x", ebye.us[i]);
		putc('\n', stderr);
#endif

		if ((len = *ebye.us) == 0)
		    break;

#ifdef DUMP
		fprintf(stderr, "Event ends with:");
		for(i = -12; i<0; i++)
		    fprintf(stderr, " %04x", ebye.us[len + i]);
		putc('\n', stderr);
#endif

		if (len * sizeof(short) >= MAXIN2P3EVENTSIZE)
		{
		    fprintf(stderr, "Event too long (%d > %d)\n",
			    len * sizeof(short), MAXIN2P3EVENTSIZE);
		    break;
		}
		ptr = ebye.us + 1 +
		    sizeof(struct in2p3event_struct)/sizeof(short);
		if (((char *) (ptr + len)) > recordend)
		{
		    fprintf(stderr, "Event truncated. Length = %d, bytes "
			    "left in record = %d.\n", len * sizeof(short),
			    recordend - ((char *) ptr));
		    break;
		}
		endptr = ebye.us + len;
		if (len > adcnum_.nos_adcs)
		    memcpy((char *) adcs_.adcs, ptr, adc_bytes);
		else
		{
		    memcpy((char *) adcs_.adcs, (char *) ptr,
			   len*sizeof(short));
		    memset((char *) (adcs_.adcs + len), -1,
			   adc_bytes - len*sizeof(short));
		}
		
		*adclist_.grouplist = -1;
		i = (len > adcnum_.nos_adcs) ? adcnum_.nos_adcs : len;
		adclist = adclist_.adclist + i;
		*adclist = -1;
		while(i)
		    *--adclist = i--;

		adcs_.triggernos = len;
		adcs_.event = ++in_events;
		adcs_.record = record;
                adcs_.wrtevt = 0;

		/*user supplied routine*/
        	if (sortin())
		{
		    fprintf(stderr, sortinfailmess);
		    goto ABORT_SORT;
		}

		if (filter && adcs_.wrtevt)
		    if (write_filt(ebye.s, len))
			filter = 0;

		ebye.us = endptr;
	    }
	    break;
	case IN2P3COMMENT1:
	    if (ebye.u[1] != IN2P3COMMENT2)
		goto BADHEADER;
	    fprintf(stderr, "IN2P3 Comment block: ...\n");
	    break;
	case IN2P3PARAM1:
	    if (ebye.u[1] != IN2P3PARAM2)
		goto BADHEADER;	
	    fprintf (stderr, "IN2P3 Parameter block: ...\n");
	    i = 1;
	    j = 0;
	    for (ebye.c += 12; *ebye.c != '\0' && ebye.c < recordend;
		 ebye.c++)
	    {
		if (j == 0)
		{
		    fprintf(stderr, "ADC %2d: ", i);
		    j = 1;
		}
		if (*ebye.c == '\r')
		{
		    j = 0;
		    i++;
		    putc('\n', stderr);
		}
		else
		    putc(*ebye.c, stderr);
	    }
	    if (j != 0)
		putc('\n', stderr);
	    break;
	case IN2P3SCALER1:
	    if (ebye.u[1] != IN2P3SCALER2)
		goto BADHEADER;
	    fprintf (stderr, "IN2P3 Scaler block: ...\n");
	    break;
	case IN2P3SPECHEAD1: /* same as IN2P3SPECDATA1 */
	    if (ebye.u[1] == IN2P3SPECHEAD2)
	    {
		fprintf (stderr, "IN2P3 Spectrum header block: ...\n");
		break;
	    }
	    else if (ebye.u[1] != IN2P3SPECDATA2)
		goto BADHEADER;
	    fprintf (stderr, "IN2P3 Spectrum data block: ...\n");
	    break;
	default:
	BADHEADER:
	    fprintf(stderr, "Unexpected record header:");
	    for(i=0; i<8; i++)
		fprintf(stderr, " %02x", ebye.c[i]);
	    fprintf(stderr, ".\n");
	    break;
	}
    }

ABORT_SORT:

    /* lets note the time at end */
    time_now = time(NULL);
    
    /* call user provided clearup routine  */
    call_finish();

    /* output statistics */
    record--;
    fprintf(stderr, "\n*** sort statistics ***\n");
    fprintf(stderr, "\nsorted %ld events in %d seconds\n"
	    "Average sort rate = %g events per second\n",
	    in_events, (int) (time_now-start_time),
	    (double) in_events / (time_now-start_time));

    return filter;
}

/*
 *  functions used in the filtering of data
 *  write any data to end of filtered data block
 */
/*ARGSUSED*/
static void
filt_tail
#if NeedFunctionPrototype
(short *base, short *fptr)
#else
(base, fptr)
short *base;
short *fptr;
#endif
{
    unsigned int *p;

    *fptr = 0;

    /*LINTED*/
    for(p = (unsigned int *) (base + 12/sizeof(short));
    /*LINTED*/
	p < (unsigned int *) fptr; p++)
	*p = ((*p & 0x00ff00ff)<<8) | ((*p & 0xff00ff00)>>8);

    return;
}

/*
 *  write any data to the beginning of filtered data block
 */
static short *
filt_head
#if NeedFunctionPrototype
(short *base)
#else
(base)
short *base;
#endif
{
    /*LINTED*/
    register unsigned int *ip = (unsigned int *) base;

    *ip++ =  IN2P3EVENTDATA1;
    *ip++ =  IN2P3EVENTDATA2;
    *ip++ = 0;

    return(base+12/sizeof(short));
}

/* in2p3 filter data specification */
static filtdat_t in2p3_filtdat = {
      16384,
      8190,
      filt_head,
      filt_tail,
      NULL,
      "IN2P3"
};

/* routine to get pointer to in2p3_filtdat */
filtdat_t *
get_in2p3_filtdat
#if NeedFunctionPrototype
(void)
#else
()
#endif
{
      return(&in2p3_filtdat);
}


/*************** residual sort-shell code for reference ***************/

#if 0

IN2P3Format::IN2P3Format(Stream* str)
{
    formatname = "in2p3";
    record = NULL;
    state = ERR;
    stream = str;
}

IN2P3Format::~IN2P3Format()
{
    if (record != NULL)
	delete record;
}

Result IN2P3Format::initFormat(int, char**)
{
    Result result;

    result = reset();
    return result;
}

Result IN2P3Format::reset()
{
   Buffer* buf;

   if (record != NULL)
       delete record;
   record = NULL;

   maxreclen=MAXIN2P3RECLEN;
   if (stream->medium != NULL && stream->medium->maxreclen < maxreclen)
       maxreclen = stream->medium->maxreclen;

   buf=new Buffer(maxreclen);

   if (buf->base == NULL) {
       delete buf; 
       return badResult ("unable to allocate record buffer", ERR_MEM);
   }

   record = buf;
   stream->position.zero();
//   stream->position.blockbase = stream->buffer->base;
   stream->position.recordbase = record->base;
   stream->position.eventbase = NULL;

   return Result();
}

#define IN2P3STRT  0   /* values of state variable */
#define IN2P3EVT   1
#define IN2P3END   3

Result IN2P3Format::scanRecord (Position* pos)
{
    static unsigned short *ptr; /* also block counter + other stuff ? */
    static char *rec;
    short word;
    int i;

    if (state == IN2P3STRT)
	goto labIN2P3START;
    if (state == IN2P3EVT)
	goto labIN2P3EVENT;
    if (state == IN2P3END)
	goto labIN2P3END;

  labIN2P3START:
    rec = pos->recordbase;
    ptr = (unsigned short*) pos->recordbase;
    if (memcmp(rec, IN2P3FILEHEAD, 8) == 0) {
	memcpy(&in2p3file_head, pos->recordbase + 12, sizeof(in2p3file_head));
	if (LOG_AUXREC){
	    sprintf(logbuf,
		    "%.12s Version %.4s %.16s %.16s %.16s\n%.8s %.4s %.20s\n",
		    in2p3file_head.id, in2p3file_head.version,
		    in2p3file_head.lab, in2p3file_head.os,
		    in2p3file_head.prog, in2p3file_head.block_len,
		    in2p3file_head.order, in2p3file_head.date );
	    doLog(LOG_AUXREC, logbuf);
	}
    } else if (memcmp(rec, IN2P3EVENTHEAD, 8) == 0) {
	memcpy(&in2p3event_head, pos->recordbase + 12,
	       sizeof(in2p3event_head));
	if (LOG_AUXREC) { /* type field contains non-ascii character ! */
	    sprintf (logbuf, "%.20s  %.7s %.8s (%.8s) %.8s\n%.16s (%.8s)\n",
		     in2p3event_head.date, in2p3event_head.type,
		     in2p3event_head.sep, in2p3event_head.size[0],
		     in2p3event_head.size[1], in2p3event_head.run,
		     in2p3event_head.number);
	    doLog(LOG_AUXREC, logbuf);
	}
    } else if (memcmp(rec, IN2P3COMMENT, 8) == 0) {
	sprintf(logbuf, "IN2P3 Comment block: ...\n");
	if (LOG_AUXREC)
	    doLog(LOG_AUXREC, logbuf);
    } else if (memcmp(rec, IN2P3PARAM, 8) == 0) then {
	sprintf (logbuf, "IN2P3 Parameter block: ...\n");
	if (LOG_AUXREC)
	    doLog(LOG_AUXREC, logbuf);
    } else if (memcmp(rec, IN2P3EVENTDATA, 8) == 0) then {
	swapShort(pos->recordbase, pos->recordlen);
	pos->eventbase = pos->recordbase + 12;
	pos->eventnum  = pos->eventlen = 0;
	state=IN2P3EVT;

      labIN2P3EVENT:
	eventtype=ADCEVENT;
	pos->eventbase += pos->eventlen * sizeof(short);
	pos->eventlen = *((unsigned short * ) pos->eventbase);
	memcpy(&in2p3event, pos->eventbase, sizeof(in2p3event));
	ptr = (unsigned short*) pos->eventbase +
	    sizeof(in2p3event)/sizeof(short);
	if (pos->eventlen == 0) {
	    state=IN2P3END;
	    return Result ("end of record", FEOF);   
	}
	if (pos->eventlen * sizeof(short) >= MAXIN2P3EVENTSIZE) {
	    state=IN2P3END;
	    return badResult ("event too long", ERR_REC_STRUCT);
	}
	if ((char*) ptr + pos->eventlen * sizeof(short) >
	    pos->recordbase + pos->recordlen) {
	    state=IN2P3END;
	    return badResult ("event truncated", ERR_REC_STRUCT);
	}
	noadcs=0;
	for(i=0; i<pos->eventlen - sizeof(in2p3event)/sizeof(short); i++) {
	    word = *ptr++;
	    adcs[noadcs] = i;
	    conversions[noadcs] = word;
	    noadcs++;
	}
	return Result ("adc avent", ADCEVENT);
    } else if (memcmp(rec, IN2P3SCALER, 8) == 0) {
	eventtype=SCALEREVENT;
	state=IN2P3END;
	return Result ("scaler event", SCALEREVENT);
    } else if (memcmp(rec, IN2P3SPECHEAD, 8) == 0) {
	sprintf (logbuf, "Spectrum Header\n");
	if (LOG_AUXREC)
	    doLog(LOG_AUXREC, logbuf);
    } else if ( memcmp(rec, IN2P3SPECDATA, 8) == 0) then {
	sprintf (logbuf, "Spectrum data block\n");
	if (LOG_AUXREC)
	    doLog (LOG_AUXREC, logbuf);
    } else {
	return badResult("IN2P3 block of unknown type", ERR_REC_STRUCT);
    }
  labIN2P3END:
    return Result ("end of record", FEOF);   
}

Result IN2P3Format::firstEvent (Position* pos)
{
    state = IN2P3STRT;
    return scanRecord(pos);
}

Result IN2P3Format::nextEvent(Position* pos)
{
    return scanRecord(pos);
}

Result IN2P3Format::formatEvent(char*, int)
{
    return Result();
}

Result IN2P3Format::unpackEvent(char*, int)
{
    return Result();
}

Result IN2P3Format::unpackScaler(char*, int)
{
    return Result();
}

Result IN2P3Format::packEvent(int,int nadc,int*,int *conv)
{
    return Result()
}

Result IN2P3Format::packScaler(int,int nadc,int*,int* conv)
{
    return Result()
}

Result IN2P3Format::putEvent(int type,char* evptr,int evlen)
{
    return Result()
}

Result IN2P3Format::flush()
{
    return Result();
}

Result IN2P3Format::is_destroyable()
{
    return Result();
}

#endif

