#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <thread.h>
#include <synch.h>
#include "inttypes.h" /* From here or from /usr/include */
#include "sort_def.h"
#include "sort_thread.h"
#include "spectra_buf.h"

extern spec_1d_t *tab_1d[TAB_SIZE_1D];
extern spec_2d_t *tab_2d[TAB_SIZE_2D];

/*
#define DEBUG
*/

#define MAX_SPEC_THREADS MAX_WORKERS

#ifdef SPEC_COMMAND_QUEUE
static struct data_buf_queue *spec_full[MAX_SPEC_THREADS];
static struct data_buf_queue *spec_empty[MAX_SPEC_THREADS];
static data_buf_t *spec_bufs[MAX_SPEC_THREADS];
static data_buf_t *spec_buf_p[MAX_SPEC_THREADS];
static int waiting = 1;
static thread_t thread_id[MAX_SPEC_THREADS];
static int num_threads, thread_queue[MAX_SPEC_THREADS];
static int flush_safe = 0, flush_pending = 0, alarm_set = 0;
#endif

/*
int spec_buf_incv, spec_buf_set;
*/

#ifdef SPEC_COMMAND_QUEUE
static void spec_buf_error(char *message, int *address)
{
    int i, c;

    for(i=0; i<TAB_SIZE_1D; i++)
    {
	if (address >= tab_1d[i]->data &&
	    address < tab_1d[i]->data+tab_1d[i]->sp->size)
	{
	    fprintf(stderr, "Spec buf %s at channel %d of spectrum %s (%d).\n",
		    message, address-tab_1d[i]->data, tab_1d[i]->sp->name, i);
	    return;
	}
    }
    
    for(i=0; i<TAB_SIZE_2D; i++)
    {
	if (address >= tab_2d[i]->data &&
	    address < tab_2d[i]->data+tab_2d[i]->sp->size*tab_2d[i]->sp->size)
	{
	    c = address - tab_2d[i]->data;
	    fprintf(stderr, "Spec buf %s at channel (%d,%d) of spectrum %s "
		    "(%d).\n", message, c / tab_2d[i]->sp->size,
		    c % tab_2d[i]->sp->size, tab_2d[i]->sp->name, i);
	    return;
	}
    }

    fprintf(stderr, "Spec buf %s in some random bit of memory - report "
	    "this.\n", message);
}
#endif

/*ARGSUSED*/
spec_buf_t *get_spec_com_ptr(int spec, int size)
{
#ifdef SPEC_COMMAND_QUEUE
    int b, s, q;

    if (!alarm_set)
    {
	alarm(1);
	alarm_set = 1;
    }

    q = spec % num_threads;
    size *= sizeof(spec_buf_t);
    if (spec_buf_p[q] != NULL &&
	spec_buf_p[q]->data_size + size > MAX_DATA_SIZE)
    {
#ifdef DEBUG
	fprintf(stderr, "Flushing buffer %d[%d]\n", spec_buf_p - spec_bufs,
		spec_buf_p->data_size);
#endif
	add_to_queue(spec_full[q], spec_buf_p[q] - spec_bufs[q]);
	spec_buf_p[q] = NULL;
    }

    if (spec_buf_p[q] == NULL)
    {
	while((b = take_from_queue(spec_empty[q], NULL)) == -1)
	    ;
#ifdef DEBUG
	fprintf(stderr, "Taking empty buffer %d\n", b);
#endif
	spec_buf_p[q] = spec_bufs[q] + b;
	s = 0;
    }
    else
	s = spec_buf_p[q]->data_size;

    spec_buf_p[q]->data_size = s + size;
    return (spec_buf_t *) (spec_buf_p[q]->data + s);
#else
    return NULL;
#endif
}

void spec_buf_flush(void)
{
#ifdef SPEC_COMMAND_QUEUE
    int q;

    alarm(0);
    flush_pending = 0;
    alarm_set = 0;

    for(q = 0; q < num_threads; q++)
    {
	if (spec_buf_p[q] == NULL)
	    return;
	if (spec_buf_p[q]->data_size == 0)
	    return;
#ifdef DEBUG
	fprintf(stderr, "Flushing buffer %d[%d]\n", spec_buf_p[q]-spec_bufs[q],
		spec_buf_p[q]->data_size);
#endif
	add_to_queue(spec_full[q], spec_buf_p[q] - spec_bufs[q]);
	spec_buf_p[q] = NULL;
    }
#endif
}

/*ARGSUSED*/
void spec_buf_flush_safe(int safe)
{
#ifdef SPEC_COMMAND_QUEUE
    flush_safe = safe;
    if (safe && flush_pending)
	spec_buf_flush();
#endif
}

#ifdef SPEC_COMMAND_QUEUE
static void spec_buf_flush_alarm(int sig)
{
    signal(SIGALRM, spec_buf_flush_alarm);
    if (!flush_safe)
	flush_pending = 1;
    else
	spec_buf_flush();
    alarm_set = 0;
}

static void *spec_buf_process_thread(void *data)
{
    int b, q, blocks = 0, bytes = 0, commands = 0;
    timestruc_t to;
    int *channel;
    spec_buf_t *com, *end;
    data_buf_t *bufp;

    q = *(int *) data;

    while(waiting)
    {
	to.tv_sec = time(NULL) + 1;
	if((b = take_from_queue(spec_full[q], &to)) == -1)
	    continue;
#ifdef DEBUG
	fprintf(stderr, "Taking full buffer %d\n", b);
#endif
	bufp = spec_bufs[q] + b;
	com = (spec_buf_t *) bufp->data;
	end = (spec_buf_t *) ((char *) com + bufp->data_size);
	blocks++;
	bytes += bufp->data_size;
	while(com < end)
	{
	    channel = com++->address;
	    commands++;
#ifdef SPEC_BUF_LAST_COMMAND
	    if ((uintptr_t) channel > (uintptr_t) SPEC_BUF_LAST_COMMAND)
	    {
		if (++*channel < 0)
		    spec_buf_error("inc: overflow", channel);
	    }
	    else
#endif
#ifdef SPEC_BUF_TARTAN
	    if (channel == SPEC_BUF_TARTAN)
	    {
		spec_buf_t *x, *y;
		int *data, nx, ny, n, size, sign;

		channel = com++->address;
		n = com++->value;
		size = n>>16;
		n &= 0xffff;
		sign = 0;
		for (x = com, nx = n; nx; nx--, x++)
		{
		    data = (int *) ((char *) channel + x->value * size);
		    for(y = com, ny = n; ny; ny--, y++)
			sign |= ++*(int *) ((char *) data + y->value);
		}
		com = x;
		if (sign < 0)
		    spec_buf_error("tartan: overflow", channel);
	    }
	    else
#endif
	    if (channel == SPEC_BUF_INCV)
	    {
		channel = com++->address;
		if ((*channel += com++->value) < 0)
		    spec_buf_error("incv: overflow", channel);
	    }
	    else if (channel == SPEC_BUF_SET)
	    {
		channel = com++->address;
		*channel = com++->value;
	    }
	    else
	    {
#ifdef SPEC_BUF_LAST_COMMAND
		fprintf(stderr, "spec buf: Unknown command %08x\n", channel);
#else
		if (++*channel < 0)
		    spec_buf_error("inc: overflow", channel);
#endif
	    }
	}
#ifdef DEBUG
	fprintf(stderr, "Returning empty buffer %d\n", b);
#endif
	add_to_queue(spec_empty[q], b);
    }

#ifdef ALLOW_MULTIPROCESS
    general_mutex_lock();
#endif
    fprintf(stderr, "Spectrum queue %d: %d blocks, %d bytes and "
	    "%d commands.\n", q, blocks, bytes, commands);
#ifdef ALLOW_MULTIPROCESS
    general_mutex_unlock();
#endif

    return NULL;
}
#endif

/*ARGSUSED*/
int spec_buf_init(const char *spec_buf_file)
{
#ifdef SPEC_COMMAND_QUEUE
    int fd, i;

    i = MAX_SPEC_THREADS*(2*sizeof(struct data_buf_queue) +
			  DATA_QUEUE_LEN*sizeof(data_buf_t));

    if ((fd = open(spec_buf_file, O_RDWR | O_CREAT, 0600)) == -1)
    {
	perror("spec_buf open");
	return -1;
    }

    if (ftruncate(fd, i) == -1)
    {
	perror("spec_buf truncate");
	close(fd);
	return -1;
    }

    if ((caddr_t) (spec_empty[0] = (struct data_buf_queue *)
		   mmap(NULL, i, PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0))
	== (caddr_t) -1)
    {
	perror("spec_buf mmap");
	close(fd);
	return -1;
    }
    close(fd);

    for(i=0; i<MAX_SPEC_THREADS; i++)
    {
	if (i > 0)
	    spec_empty[i] = (struct data_buf_queue *)
		(spec_bufs[i-1] + DATA_QUEUE_LEN);
	spec_full[i] = spec_empty[i] + 1;
	spec_bufs[i] = (data_buf_t *) (spec_full[i] + 1);
    }
#endif
    return 0;
}

#ifdef SPEC_COMMAND_QUEUE
static int spec_buf_reset(void)
{
    int i, q;

    for(q=0; q<MAX_SPEC_THREADS; q++)
    {
	if (queue_init(spec_full[q]))
	{
	    perror("spec full init");
	    return -1;
	}
	if (queue_init(spec_empty[q]))
	{
	    perror("spec full init");
	    return -1;
	}
	for(i=0; i<DATA_QUEUE_LEN; i++)
	    add_to_queue(spec_empty[q], i);

	spec_buf_p[q] = NULL;
    }

    return 0;
}
#endif

/*ARGSUSED*/
int spec_buf_start_threads(int t)
{
#ifdef SPEC_COMMAND_QUEUE
    int i;

    signal(SIGALRM, spec_buf_flush_alarm);

    if (t > MAX_SPEC_THREADS)
	t = MAX_SPEC_THREADS;

    num_threads = 0;
    if (spec_buf_reset() == -1)
	return -1;
    waiting = 1;
    for(i=t; i--; )
    {
	thread_queue[num_threads] = num_threads;
	if (thr_create(NULL, 0, spec_buf_process_thread,
		       thread_queue + num_threads, 0,
		       thread_id + num_threads) != 0)
	    break;
	num_threads++;
    }
    if (num_threads == 0)
	return -1;
#endif
    return 0;
}

void spec_buf_stop_threads(void)
{
#ifdef SPEC_COMMAND_QUEUE
    int i;

    waiting = 0;
    for(i=0; i<num_threads; i++)
	if (thread_id[i] != (thread_t) 0)
	    thr_join(thread_id[i], NULL, NULL);

    num_threads = 0;
#endif
}

