/*
 * This file contains the spectra routines which depend on whether we're
 * running in single ot multi processor mode.
 */

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <limits.h>

#if defined(ALLOW_MULTIPROCESS) && !defined(SPEC_COMMAND_QUEUE)
#include <thread.h>
#include <synch.h>
#define MUTEX_LOCK_SPECTRA
#endif

#ifdef ALLOW_MULTIPROCESS
#include <unistd.h>
#include <fcntl.h>
#endif

#include "sort_def.h"
#include "sort_thread.h"
#include "eg.h"
#include "data_io.h"
#include "spectra.h"
#include "spectraint.h"
#include "spectra_buf.h"

/*
 * The elements of tab_?d contain pointers to the storage locations
 * of each spectrum structure. ie nos_tab_1d[1] points to the first
 * address location of the 1d spectrum structure for spectrum number
 * 1. Element [0] will point to the last structure to be filled
 * spectrum
 */
#if defined(ALLOW_MULTIPROCESS)
static mutex_t *mutex_1d;
#ifdef MUTEX_LOCK_SPECTRA
static mutex_t *mutex_2d;
#endif
#endif

extern spec_1d_t *tab_1d[TAB_SIZE_1D];
extern spec_2d_t *tab_2d[TAB_SIZE_2D];

int allow_multiprocess(void)
{
#ifdef ALLOW_MULTIPROCESS
    return 1;
#else
    return 0;
#endif
}

/*ARGSUSED*/
int spec_mutex_init(const char *mutex_file)
{
#ifdef ALLOW_MULTIPROCESS
    int fd, i;
    mutex_t *mp;

#ifdef MUTEX_LOCK_SPECTRA
    i = (TAB_SIZE_1D+TAB_SIZE_2D)*sizeof(mutex_t);
#else
    i = sizeof(mutex_t);
#endif

    if ((fd = open(mutex_file, O_RDWR | O_CREAT, 0600)) == -1)
    {
	perror("mutex open");
	return -1;
    }

    if (ftruncate(fd, i) == -1)
    {
	perror("mutex truncate");
	close(fd);
	return -1;
    }

    if ((caddr_t) (mutex_1d = (mutex_t *)
		   mmap(NULL, i, PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0))
	== (caddr_t) -1)
    {
	perror("mutex mmap");
	close(fd);
	return -1;
    }
    close(fd);

#ifdef MUTEX_LOCK_SPECTRA
    mutex_2d = mutex_1d + TAB_SIZE_1D;

    for(mp = mutex_1d, i = TAB_SIZE_1D + TAB_SIZE_2D; i--; mp++)
	mutex_init(mp, USYNC_PROCESS, NULL);
#else
    mutex_init(mutex_1d, USYNC_PROCESS, NULL);
#endif
#endif
    return 0;
}

int general_mutex_lock(void)
{
#ifdef ALLOW_MULTIPROCESS
    return mutex_lock(mutex_1d);
#else
    return 0;
#endif
}

int general_mutex_unlock(void)
{
#ifdef ALLOW_MULTIPROCESS
    return mutex_unlock(mutex_1d);
#else
    return 0;
#endif
}

/****************************************************************************
 *           user callable spectrum incrementing subroutines etc.           *
 ****************************************************************************/

void
inc1d
#if NeedFunctionPrototype
(register int spec_nos,
register int channel)
#else
(spec_nos, channel)
register int spec_nos;
register int channel;
#endif
{
    register spec_1d_t *ptr;
    int i;
#ifdef SPEC_COMMAND_QUEUE
    spec_buf_t *p;
#endif

    if (spec_nos > 0 && spec_nos < TAB_SIZE_1D)
	/*LINTED*/
	if (ptr = tab_1d[spec_nos])
	{
	    if (channel >= 0 && channel < ptr->sp->size)
	    {
#ifdef SPEC_COMMAND_QUEUE
		p = get_spec_com_ptr(spec_nos, 1);
		p->address = ptr->data + channel;
#else
#ifdef MUTEX_LOCK_SPECTRA
		mutex_lock(&mutex_1d[spec_nos]);
#endif
		i = ++ptr->data[channel];
#ifdef MUTEX_LOCK_SPECTRA
		mutex_unlock(&mutex_1d[spec_nos]);
#endif
		if (i > 0)
		    return;
		else
		    fprintf(stderr,
			    "inc1d overflow at channel %d in spectrum %s\n",
			    channel, ptr->sp->name);
#endif
	    }
	    else if (DBX_val >= 10)
		fprintf(stderr, "inc1d to channel %d outside spectrum range\n",
			channel);
	}
	else
	    fprintf(stderr, "inc1d to undefined 1d spectrum number %d\n",
		    spec_nos);
    else
	fprintf(stderr, "inc1d to invalid 1d spectrum number %d", spec_nos);
}

void
inc2d
#if NeedFunctionPrototype
(register int spec_nos,
register int x_ch,
register int y_ch)
#else
(spec_nos, x_ch, y_ch)
register int spec_nos;
register int x_ch;
register int y_ch;
#endif
{
    register spec_2d_t *ptr;
    int i;
#ifdef SPEC_COMMAND_QUEUE
    spec_buf_t *p;
#endif

#ifdef DEBUG    
    fprintf(stderr, "inc2d(%d,%d,%d)\n", spec_nos, x_ch, y_ch);
#endif
    if (spec_nos > 0 && spec_nos < TAB_SIZE_2D)
	/*LINTED*/
	if (ptr = tab_2d[spec_nos])
	{
	    if (x_ch >= 0 && y_ch >= 0 && x_ch < ptr->sp->size &&
		y_ch < ptr->sp->size)
	    {
#ifdef SPEC_COMMAND_QUEUE
#ifdef SPEC_BUF_HASH_BY_X
		p = get_spec_com_ptr(spec_nos+x_ch, 1);
#else
		p = get_spec_com_ptr(spec_nos+1, 1);
#endif
		p->address = ptr->data + y_ch + x_ch*ptr->sp->size;
#else
#ifdef MUTEX_LOCK_SPECTRA
		mutex_lock(&mutex_2d[spec_nos]);
#endif
		i = ++ptr->data[y_ch + x_ch*ptr->sp->size];
#ifdef MUTEX_LOCK_SPECTRA
		mutex_unlock(&mutex_2d[spec_nos]);
#endif
		if (i > 0)
		    return;
		else
		    fprintf(stderr,
			    "inc2d overflow at x=%d, y=%d in spectrum %s\n",
			    x_ch, y_ch, ptr->sp->name);
#endif
	    }
	    else if (DBX_val >= 10)
		fprintf(stderr,
			"inc2d to x=%d y=%d outside range of spectrum %s\n",
			x_ch, y_ch, ptr->sp->name);
	}
	else
	    fprintf(stderr, "inc2d to undefined 2d spectrum number %d\n",
		    spec_nos);
    else
	fprintf(stderr, "inc2d to invalid 2d spectrum number %d", spec_nos);
}

void
inc2d_list
#if NeedFunctionPrototype
(register int spec_nos,
register struct spec2d_list *list,
register int n)
#else
(spec_nos, list, n)
register int spec_nos;
register struct spec2d_list *list;
register int n;
#endif
{
    register spec_2d_t *ptr;

    if (spec_nos > 0 && spec_nos < TAB_SIZE_2D)
	/*LINTED*/
	if (ptr = tab_2d[spec_nos])
	{
	    register int s = ptr->sp->size, x_ch, y_ch;
	    register int *data = ptr->data;
#ifdef SPEC_COMMAND_QUEUE
	    spec_buf_t *p;
	    int i, j;
	    register struct spec2d_list *lp;
#else
	    register int sign = 0, range = 0;
#endif

#ifdef SPEC_COMMAND_QUEUE
	    for(i=n, j=0, lp = list; i--; lp++)
	    {
		if ((x_ch = lp->x) >= 0 && x_ch < s &&
		    (y_ch = lp->y) >= 0 && y_ch < s)
		    j++;
	    }
#ifndef SPEC_BUF_HASH_BY_X
	    p = get_spec_com_ptr(spec_nos+1, j);
#endif
	    for(i=n, lp = list; i--; lp++)
	    {
		if ((x_ch = lp->x) >= 0 && x_ch < s &&
		    (y_ch = lp->y) >= 0 && y_ch < s)
		{
#ifdef SPEC_BUF_HASH_BY_X
		    p = get_spec_com_ptr(spec_nos + x_ch, 1);
#endif
		    p++->address = data + x_ch*ptr->sp->size + y_ch;
		}
	    }
#else

#ifdef MUTEX_LOCK_SPECTRA
	    mutex_lock(&mutex_2d[spec_nos]);
#endif
	    for(; n--; list++)
	    {
		if ((x_ch = list->x) >= 0 && x_ch < s &&
		    (y_ch = list->y) >= 0 && y_ch < s)
		    sign |= ++data[x_ch*s + y_ch];
		else
		    range = 1;
	    }
#ifdef MUTEX_LOCK_SPECTRA
	    mutex_unlock(&mutex_2d[spec_nos]);
#endif
#endif
	    
	    if (DBX_val >= 10 &&
#ifdef SPEC_COMMAND_QUEUE
		j != n
#else
		range
#endif
		)
		fprintf(stderr, "inc2d_list some points outside range of "
			"spectrum %s\n", ptr->sp->name);
#ifndef SPEC_COMMAND_QUEUE
	    if (sign < 0)
		fprintf(stderr, "inc2d_list some points overflowed "
			"in spectrum %s\n", ptr->sp->name);
#endif
	}
	else
	    fprintf(stderr, "inc2d_list to undefined 2d spectrum number %d\n",
		    spec_nos);
    else
	fprintf(stderr, "inc2d_list to invalid 2d spectrum number %d",
		spec_nos);
}

void
inc2d_tartan
#if NeedFunctionPrototype
(register int spec_nos,
register int *ch)
#else
(spec_nos, ch)
register int spec_nos;
register int *ch;
#endif
{
    register spec_2d_t *ptr;
    register int *x_ch, *y_ch;
    register int x;

#ifdef DEBUG    
    fprintf(stderr, "inc2d_tartan(%d", spec_nos);
    for(x_ch = ch; *x_ch != -1; x_ch++)
	fprintf(stderr, ",%d", *x_ch);
    fprintf(stderr, ")\n");
#endif

    if (spec_nos > 0 && spec_nos < TAB_SIZE_2D)
	/*LINTED*/
	if (ptr = tab_2d[spec_nos])
	{
	    register int s, n;
#ifdef SPEC_COMMAND_QUEUE
	    spec_buf_t *p;
#else
	    register int sign = 0;
#endif
	    
	    s = ptr->sp->size;

	    for(x_ch = y_ch = ch; (x = *x_ch++) > 0; )
		if (x < s)
		    *y_ch++ = x*sizeof(int);
	    n = y_ch - ch;

	    if (x_ch != y_ch && DBX_val >= 10)
		fprintf(stderr, "inc2d_tartan some points outside range of "
			"spectrum %s\n", ptr->sp->name);

	    *y_ch = -1;

#ifdef SPEC_BUF_TARTAN
	    p = get_spec_com_ptr(spec_nos+1, n+3);
	    p++->address = SPEC_BUF_TARTAN;
	    p++->address = ptr->data;
	    p++->value = n + (s<<16);
	    
	    for(x_ch = ch; (x = *x_ch++) > 0; )
		p++->value = x;
#else
#if defined(SPEC_COMMAND_QUEUE) && !defined(SPEC_BUF_HASH_BY_X)
	    p = get_spec_com_ptr(spec_nos+1, n*n);
#endif
	    
#ifdef MUTEX_LOCK_SPECTRA
	    mutex_lock(&mutex_2d[spec_nos]);
#endif
	    for(x_ch = ch; (x = *x_ch++) > 0; )
	    {
		register int yn = n;
		register int *data;
#ifdef SPEC_BUF_HASH_BY_X
		p = get_spec_com_ptr(spec_nos+(x/sizeof(int)), n);
#endif
		/*LINTED*/
		for(data = (int *) (((char *) ptr->data) + x*s),y_ch = ch;
		    yn > 0; yn--)
#ifdef SPEC_COMMAND_QUEUE
		    p++->address = (int *) (((char *) data) + *y_ch++);
#else
		    /*LINTED*/
		    sign |= ++*((int *) (((char *) data) + *y_ch++));
#endif
	    }
#ifdef MUTEX_LOCK_SPECTRA
	    mutex_unlock(&mutex_2d[spec_nos]);
#endif
#endif

#ifndef SPEC_COMMAND_QUEUE
	    if (sign < 0)
		fprintf(stderr, "inc2d_tartan some points overflowed "
			"in spectrum %s\n", ptr->sp->name);
#endif
	}
	else
	    fprintf(stderr, "inc2d_tartan to undefined 2d spectrum number "
		    "%d\n", spec_nos);
    else
	fprintf(stderr, "inc2d_tartan to invalid 2d spectrum number %d",
		spec_nos);
}

void
incv1d
#if NeedFunctionPrototype
(register int spec_nos,
register int channel,
register int value)
#else
(spec_nos, channel, value)
register int spec_nos;
register int channel;
register int value;
#endif
{
    register spec_1d_t *ptr;
    int i;
#ifdef SPEC_COMMAND_QUEUE
    spec_buf_t *p;
#endif

    if (spec_nos <= 0 || spec_nos >= TAB_SIZE_1D) {
	fprintf(stderr, "incv1d to invalid 1d spectrum number %d", spec_nos);
	return;
    }

    /*LINTED*/
    if (!(ptr = tab_1d[spec_nos])) {
	fprintf(stderr, "incv1d to undefined 1d spectrum number %d\n",
		spec_nos);
	return;
    }

    if (channel < 0 || channel >= ptr->sp->size) {
	if (DBX_val >= 10)
	    fprintf(stderr, "incv1d to channel %d outside spectrum range\n",
		    channel);
	return;
    }
#ifdef SPEC_COMMAND_QUEUE
    p = get_spec_com_ptr(spec_nos, 3);
    p++->address = SPEC_BUF_INCV;
    p++->address = ptr->data + channel;
    p->value = value;
#else
#ifdef MUTEX_LOCK_SPECTRA
    mutex_lock(&mutex_1d[spec_nos]);
#endif
    i = ptr->data[channel];
    if (i < INT_MAX - value)
    {
	ptr->data[channel] = i + value;
#ifdef MUTEX_LOCK_SPECTRA
	mutex_unlock(&mutex_1d[spec_nos]);
#endif
    }
    else
    {
#ifdef MUTEX_LOCK_SPECTRA
	mutex_unlock(&mutex_1d[spec_nos]);
#endif
	fprintf(stderr, "incv1d overflow at channel %d in spectrum %s\n",
		channel, ptr->sp->name);
    }
#endif
}

void
incv2d
#if NeedFunctionPrototype
(register int spec_nos,
register int x_ch,
register int y_ch,
register int value)
#else
(spec_nos, x_ch, y_ch, value)
register int spec_nos;
register int x_ch;
register int y_ch;
register int value;
#endif
{
    register spec_2d_t *ptr;
    int i, c;
#ifdef SPEC_COMMAND_QUEUE
    spec_buf_t *p;
#endif
    if (spec_nos <= 0 || spec_nos >= TAB_SIZE_2D) {
	fprintf(stderr, "incv2d to invalid 2d spectrum number %d", spec_nos);
	return;
    }

    /*LINTED*/
    if (!(ptr = tab_2d[spec_nos])) {
	fprintf(stderr, "incv2d to undefined 2d spectrum number %d\n",
		spec_nos);
	return;
    }

    if (x_ch < 0 || x_ch >= ptr->sp->size ||
	y_ch < 0 || y_ch >= ptr->sp->size) {
	if (DBX_val >= 10)
	    fprintf(stderr,
		    "incv2d to x=%d y=%d outside range of spectrum %s\n",
		    x_ch, y_ch, ptr->sp->name);
	return;
    }
#ifdef SPEC_COMMAND_QUEUE
#ifdef SPEC_BUF_HASH_BY_X
    p = get_spec_com_ptr(spec_nos+x_ch, 3);
#else
    p = get_spec_com_ptr(spec_nos+1, 3);
#endif
    p++->address = SPEC_BUF_INCV;
    p++->address = ptr->data + x_ch*ptr->sp->size + y_ch;
    p->value = value;
#else
#ifdef MUTEX_LOCK_SPECTRA
    mutex_lock(&mutex_2d[spec_nos]);
#endif
    c = y_ch + x_ch*ptr->sp->size;
    i = ptr->data[c];
    if (i < INT_MAX - value)
    {
	ptr->data[c] = i + value;
#ifdef MUTEX_LOCK_SPECTRA
	mutex_unlock(&mutex_2d[spec_nos]);
#endif
    }
    else
    {
#ifdef MUTEX_LOCK_SPECTRA
	mutex_unlock(&mutex_2d[spec_nos]);
#endif
	fprintf(stderr, "incv2d overflow at (%d,%d)=%d, increment=%d in "
		"spectrum %s\n", x_ch, y_ch,
		ptr->data[(y_ch)+(x_ch)*(ptr->sp->size)], value,
		ptr->sp->name);
    }
#endif
}

int
win2d
#if NeedFunctionPrototype
(register int spec_nos,
register int x_ch,
register int y_ch)
#else
(spec_nos, x_ch, y_ch)
register int spec_nos;
register int x_ch;
register int y_ch;
#endif
{
    register spec_2d_t *ptr;

    if (spec_nos <= 0 || spec_nos >= TAB_SIZE_2D) {
	fprintf(stderr, "win2d to invalid 2d spectrum number %d\n", spec_nos);
	return SORT_FALSE;
    }

    /*LINTED*/
    if (!(ptr = tab_2d[spec_nos])) {
	fprintf(stderr, "win2d to undefined 2d spectrum number %d\n",
		spec_nos);
	return SORT_FALSE;
    }

    if (!ptr->win) {
	fprintf(stderr, "win2d called with non window 2d spectrum number %d\n",
		spec_nos);
	return SORT_FALSE;
    }

    if (x_ch < 0 || x_ch >= ptr->sp->size ||
	y_ch < 0 || y_ch >= ptr->sp->size) {
	if (DBX_val >= 10)
	    fprintf(stderr,
		    "win2d to x=%d y=%d outside range of spectrum %s\n",
		    x_ch, y_ch, ptr->sp->name);
	return SORT_FALSE;
    }

    return (ptr->data[y_ch+x_ch*ptr->sp->size]) ? SORT_TRUE : SORT_FALSE;
}

void
set1d
#if NeedFunctionPrototype
(register int spec_nos,
register int channel,
register int value)
#else
(spec_nos, channel, value)
register int spec_nos;
register int channel;
register int value;
#endif
{
    register spec_1d_t *ptr;
#ifdef SPEC_COMMAND_QUEUE
    spec_buf_t *p;
#endif

    if (spec_nos <= 0 || spec_nos >= TAB_SIZE_1D) {
	fprintf(stderr, "set1d to invalid 1d spectrum number %d", spec_nos);
	return;
    }

    /*LINTED*/
    if (!(ptr = tab_1d[spec_nos])) {
	fprintf(stderr, "set1d to undefined 1d spectrum number %d\n",
		spec_nos);
	return;
    }

    if (channel < 0 || channel >= ptr->sp->size) {
	if (DBX_val >= 10)
	    fprintf(stderr, "set1d to channel %d outside spectrum range\n",
		    channel);
	return;
    }
#ifdef SPEC_COMMAND_QUEUE
    p = get_spec_com_ptr(spec_nos, 3);
    p++->address = SPEC_BUF_SET;
    p++->address = ptr->data + channel;
    p->value = value;
#else
#ifdef MUTEX_LOCK_SPECTRA
    mutex_lock(&mutex_1d[spec_nos]);
#endif
    ptr->data[channel] = value;
#ifdef MUTEX_LOCK_SPECTRA
    mutex_unlock(&mutex_1d[spec_nos]);
#endif
#endif
}

void
set2d
#if NeedFunctionPrototype
(register int spec_nos,
register int x_ch,
register int y_ch,
register int value)
#else
(spec_nos, x_ch, y_ch, value)
register int spec_nos;
register int x_ch;
register int y_ch;
register int value;
#endif
{
    register spec_2d_t *ptr;
#ifdef SPEC_COMMAND_QUEUE
    spec_buf_t *p;
#endif

    if (spec_nos <= 0 || spec_nos >= TAB_SIZE_2D) {
	fprintf(stderr, "set2d to invalid 2d spectrum number %d", spec_nos);
	return;
    }

    /*LINTED*/
    if (!(ptr = tab_2d[spec_nos])) {
	fprintf(stderr, "set2d to undefined 2d spectrum number %d\n",
		spec_nos);
	return;
    }

    if (x_ch < 0 || x_ch >= ptr->sp->size ||
	y_ch < 0 || y_ch >= ptr->sp->size) {
	if (DBX_val >= 10)
	    fprintf(stderr,
		    "set2d to x=%d y=%d outside range of spectrum %s\n",
		    x_ch, y_ch, ptr->sp->name);
	return;
    }
#ifdef SPEC_COMMAND_QUEUE
#ifdef SPEC_BUF_HASH_BY_X
    p = get_spec_com_ptr(spec_nos+x_ch, 3);
#else
    p = get_spec_com_ptr(spec_nos+1, 3);
#endif
    p++->address = SPEC_BUF_SET;
    p++->address = ptr->data + x_ch*ptr->sp->size + y_ch;
    p->value = value;
#else
#ifdef MUTEX_LOCK_SPECTRA
    mutex_lock(&mutex_2d[spec_nos]);
#endif
    ptr->data[y_ch + x_ch*ptr->sp->size] = value;
#ifdef MUTEX_LOCK_SPECTRA
    mutex_unlock(&mutex_2d[spec_nos]);
#endif
#endif
}

int
val1d
#if NeedFunctionPrototype
(register int spec_nos,
register int channel)
#else
(spec_nos, channel)
register int spec_nos;
register int channel;
#endif
{
    register spec_1d_t *ptr;

    if (spec_nos <= 0 || spec_nos >= TAB_SIZE_1D) {
	fprintf(stderr, "val1d to invalid 1d spectrum number %d", spec_nos);
	return -1;
    }

    /*LINTED*/
    if (!(ptr = tab_1d[spec_nos])) {
	fprintf(stderr, "val1d to undefined 1d spectrum number %d\n",
		spec_nos);
	return -1;
    }

    if (channel < 0 || channel >= ptr->sp->size) {
	if (DBX_val >= 10)
	    fprintf(stderr, "val1d to channel %d outside spectrum range\n",
		    channel);
	return -1;
    }
    return ptr->data[channel];
}

int
val2d
#if NeedFunctionPrototype
(register int spec_nos,
register int x_ch,
register int y_ch)
#else
(spec_nos, x_ch, y_ch)
register int spec_nos;
register int x_ch;
register int y_ch;
#endif
{
    register spec_2d_t *ptr;

    if (spec_nos <= 0 || spec_nos >= TAB_SIZE_2D) {
	fprintf(stderr, "val2d to invalid 2d spectrum number %d", spec_nos);
	return -1;
    }

    /*LINTED*/
    if (!(ptr = tab_2d[spec_nos])) {
	fprintf(stderr, "val2d to undefined 2d spectrum number %d\n",
		spec_nos);
	return -1;
    }

    if (x_ch < 0 || x_ch >= ptr->sp->size ||
	y_ch < 0 || y_ch >= ptr->sp->size) {
	if (DBX_val >= 10)
	    fprintf(stderr,
		    "val2d to x=%d y=%d outside range of spectrum %s\n",
		    x_ch, y_ch, ptr->sp->name);
	return -1;
    }
    return ptr->data[y_ch + x_ch*ptr->sp->size];
}

