#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include "data_pipe.h"
#include "sort_def.h"
#include "sort_threadint.h"

static int pfd = -1;
static pid_t pid = 0;

int pipeopen(char *command, int reclen)
{
    int fd[2], i;

    if (reclen == 0)
    {
	fprintf(stderr, "Variable length records not supported on pipes.\n");
	return -1;
    }

    if (socketpair(AF_UNIX, SOCK_STREAM, 0, fd) == -1)
	return(-1);
    switch(pid = fork1())
    {
    case -1:
	close(fd[0]);
	close(fd[1]);
	return(-1);
    case 0:
	close(fd[0]);
	dup2(fd[1], 1);
	for(i=getdtablesize(); i>2; i--)
	    close(i);
	for(i=NSIG+1; i--; )
	    signal(i, SIG_DFL);
	/* Let sh worry about parsing the command line */
	_exit(system(command));
	break;
    default:
	close(fd[1]);
	break;
    }

    pfd = fd[0];

    init_read_from_pipe(reclen);

    return 0;
}

static int processdead(pid_t pid)
{
    int i;

    do {
	i = waitpid(pid, NULL, WNOHANG);
    } while (i == -1 && errno == EINTR);
    switch(i)
    {
    case -1:
	if (errno != ECHILD)
	    perror("pipeclose");
	return 1;
    case 0:
	return 0;
    }

    return 1;
}

int pipeclose(void)
{
    if (pfd != -1)
	if (close(pfd) == -1)
	    perror("pipeclose");
    pfd = -1;

    if (pid != 0)
    {
	if (!processdead(pid))
	{
	    /* Give it a second to die of natural causes */
	    sleep(1);
	    if (!processdead(pid))
	    {
		/* Hurry it along */
		kill(pid, SIGINT);
		sleep(1);
		if (!processdead(pid))
		{ 
		    /* Hurry it along again */
		    kill(pid, SIGHUP);
		    sleep(1);
		    if (!processdead(pid))
		    {
			/* Terminate with extreme predjudice */
			kill(pid, SIGKILL);
			while(waitpid(pid, NULL, 0) == -1 && errno == EINTR)
			    ;
		    }
		}
	    }
	}
	pid = 0;
    }

   return 0;
}

int piperead(int buflen, char *buffer)
{
    int n;

    if ((n = read_block_from_pipe(pfd, buffer, buflen)) == -1)
    {
	if (errno != EINTR && errno != ENOSPC)
	    perror("piperead");
    }

    return n;
}

static int bs, skip;

void pipestatus(void)
{
    fprintf(stderr, "Read pipe fd = %d, PID = %d, block size = %d\n", pfd,
	    pid, bs);
}

void init_read_from_pipe(int reclen)
{
    bs = reclen;
    skip = 0;
}

/*
 * Attempt to read len bytes from a pipe. Can read less than len. May
 * read 0 bytes, in which case it will return 0 which does not mean EOF.
 * EOF is signalled with a return value of -1 and errno set to ENOSPC
 * (not a perfect choice of error number, but it will do).
 */

static int read_partial_from_pipe(int fd, char *buffer, int len)
{
    int n;
    fd_set fds;
    struct timeval timeout;

    if (Stop_interupt(0))
    {
	errno = EINTR;
	return -1;
    }

    FD_ZERO(&fds);
    FD_SET(fd, &fds);
    timeout.tv_sec = 0;
    timeout.tv_usec = 100000;
    if (select(fd+1, &fds, NULL, NULL, &timeout) == -1)
    {
	if (errno == EINTR)
	    return 0;
	return -1;
    }
    if (FD_ISSET(fd, &fds))
    {
	switch(n = read(fd, buffer, len))
	{
	case -1:
	    if (errno == EINTR)
		return 0;
	    return -1;
	case 0:
	    errno = ENOSPC;
	    return -1;
	default:
	    return n;
	}
    }
    
    return 0;
}

/*
 * The call reads the minimum of buflen and bs (the block size given to the
 * open call) from the given file descriptor into the given buffer. If bs is
 * greater than buflen, then bs - buflen bytes are thrown away before the next
 * read. If EOF or an error occurs patway through a block then the partial
 * block is returned. If EOF or an error occurs before any bytes are read then
 * -1 is returned. Errno is set to the error type EINTR means that stop
 * interrupt was set and ENOSPC means that EOF was reached.
 */

int read_block_from_pipe(int fd, char *buffer, int buflen)
{
    int n, len, slen;

    while(skip)
    {
	len = (skip < buflen) ? skip : buflen;

	if ((n = read_partial_from_pipe(fd, buffer, len)) == -1)
	    return -1;

	skip -= n;
    }
    
    skip = bs;
    slen = len = (bs < buflen) ? bs : buflen;
	
    while(len > 0)
    {
	if ((n = read_partial_from_pipe(fd, buffer, len)) == -1)
	    break;

	buffer += n;
	len -= n;
	skip -= n;
    }
    
    if (len == slen)
	return -1;

    return slen - len;
}
