#include <stdio.h>
#include <thread.h>
#include <synch.h>

#include "sort_def.h"
#include "sort_thread.h"

/* #define DEBUG */

/*
 * Buffer states:
 *
 * head == -1:   Queue is full. Read next block from tail (and reset head)
 * head == tail: Queue is empty (put next block at head)
 * head != tail: Queue is partially empty (put next block at head, read
 *		 next block from tail).
 */

void queue_diagnostic(struct data_buf_queue *queue)
{
    int i;

    fprintf(stderr, "Diagnostic for queue at %08x:", (unsigned int) queue);
    for(i = 0; i < DATA_QUEUE_LEN; i++)
	fprintf(stderr, " %d", queue->queue[i]);
    fprintf(stderr, " head: %d, tail: %d.\n", queue->head, queue->tail);
}

/* Add a buffer to a queue */

void add_to_queue(struct data_buf_queue *queue, int buffer)
{
    int head;

    /*
     * We guarantee that there's always enough room to add a buffer to
     * a queue. The only way this won't happen is if something's gone
     * horribly wrong. 
     */

    mutex_lock(&queue->mutex);
#ifdef DEBUG
    fprintf(stderr, "Process %d adding buffer %d to queue at %08x.\n",
	    getpid(), buffer, queue);
    sleep(1);
#endif
    if ((head = queue->head) == -1)
    {
	fprintf(stderr, "add_to_queue: Attempt to add buffer to full queue. "
		"This shouldn't happen!\n");
	queue_diagnostic(queue);
	return;
    }
    
    queue->queue[head] = buffer;
    head = (head + 1) % DATA_QUEUE_LEN;
    if (head == queue->tail)
	head = -1;
    queue->head = head;
    cond_signal(&queue->wait_for_data);
#ifdef DEBUG
    queue_diagnostic(queue);
#endif
    mutex_unlock(&queue->mutex);
}

/*
 * Take the lead buffer from a queue. Return -1 in event of error (for
 * example, signal or timeout).
 */

int take_from_queue(struct data_buf_queue *queue, timestruc_t *abstime)
{
    int head, tail;
    int buf;

    mutex_lock(&queue->mutex);
#ifdef DEBUG
    sleep(1);
    fprintf(stderr, "Process %d attempting to take from queue at %08x.\n",
	    getpid(), queue);
#endif
    if ((tail = queue->tail) == (head = queue->head))
    {
	if (abstime == NULL)
		cond_wait(&queue->wait_for_data, &queue->mutex);
	else
	    cond_timedwait(&queue->wait_for_data, &queue->mutex, abstime);
	if ((tail = queue->tail) == (head = queue->head))
	{
	    mutex_unlock(&queue->mutex);
	    return -1;
	}
    }
    buf = queue->queue[tail];
    if (head == -1)
	queue->head = tail;
    queue->tail = (tail+1) % DATA_QUEUE_LEN;
#ifdef DEBUG
    sleep(1);
#ifdef DEBUG
    queue_diagnostic(queue);
#endif
    fprintf(stderr, "Process %d got buffer %d from queue at %08x.\n",
	    getpid(), buf, queue);
#endif
    mutex_unlock(&queue->mutex);
    return buf;
}

int queue_init(struct data_buf_queue *queue)
{
    if (mutex_init(&queue->mutex, USYNC_PROCESS, NULL))
	return -1;
    
    if (cond_init(&queue->wait_for_data, USYNC_PROCESS, NULL))
	return -1;

    queue->head = queue->tail = 0;

    return 0;
}

int queue_flush(struct data_buf_queue *queue)
{
    if (mutex_trylock(&queue->mutex))
	return -1;

    queue->head = queue->tail = 0;

    return mutex_unlock(&queue->mutex);
}

int queue_destroy(struct data_buf_queue *queue)
{
    if (mutex_destroy(&queue->mutex))
	return -1;
    
    if (cond_destroy(&queue->wait_for_data))
	return -1;

    return 0;
}

