/*
 *  sunsort commands functions
 *  prev modified 14/2/95
 *  prev modified 27/3/95
 *  last modified 17/7/95 
 *
 */

#include <stdio.h>
#include <string.h>
#include <dirent.h>
#include <signal.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>

#include <sys/param.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/time.h>

#include <thread.h>
#include <synch.h>
#include <fcntl.h>

#include "sort_def.h"
#include "sort_threadint.h"
#include "sort_mem.h"
#include "datatape.h"
#include "../sunsort_lib/formats.h"
#include "data_pipe.h"
#include "demon.h"
#include "read_spec.h"
#include "discsubs.h"
#include "sort_main.h"
#include "sort_notify.h"
#include "sort_comms.h"
#include "data_tape.h"
#include "data_disk.h"
#include "data_remote.h"

/**************************************************************************************************/

typedef struct IO_format {
      char     format_name[FORMAT_NAME_LEN];    /* name of predefined format */
      int      wtape_format;      /* write format adopted with data tape */

      int      rtape_blen;        /* block length for tape reading (not needed for ansi tapes) */
      int      rtape_rlen;        /* record length for tape reading (not needed for ansi tapes) */
      int      wtape_blen;        /* block length for tape writing */
      int      wtape_rlen;        /* record length for tape writing */

      int      rdisk_blen;        /* block length for disk reading */
      int      rdisk_rlen;        /* record length for disk reading */
      int      wdisk_blen;        /* block length for tape writing */
      int      wdisk_rlen;        /* record length for tape writing */
} IO_format_t;

/* known format types */
static IO_format_t   formats[] = {FMT_PARAM_LIST,
				  "", 0, 0, 0, 0, 0, 0, 0, 0, 0};

/* current format state */
static IO_format_t   IOfmt;

static char actionchar = 'Y';

/**************************************************************************************************/
/*
 * open read sort file
 * NB read tape doesn't require a tape format specifying
 */
int
open_sort_file
#if NeedFunctionPrototype
(char *opt_str, char *reply)
#else
(opt_str, reply)
char *opt_str;
char *reply;
#endif
{
    char source[BUFSIZ], read_file[BUFSIZ], *rf;
    int comm;
    sort_IO_t   *ptr = sort_io_data;
      
    if (DBX_val >= 5)
	fprintf(stderr, "server: open_sort_file called with arg %s\n",
		opt_str);
    
    /* check that not trying to sort */      
    if (Sorting_data(0) != SORT_FALSE) {
	sprintf(reply, "invalid command while sorting\n");
	return SORT_FALSE;
    }
    
    /* make sure we haven't already got a read file open */		  
    if ( ptr->IO_state & READING_IN) {
	sprintf(reply, "OPEN command issued with %s already open!\n",
		ptr->read_file);
	return SORT_FALSE;
    }
    if ( sscanf(opt_str, "%s %n%s", source, &comm, read_file) < 2) {
	sprintf(reply, "invalid command line for OPEN command => %s\n",
		opt_str);
	return SORT_FALSE;
    }
    rf = read_file;
    if (strncmp(source, "disk", 4) == 0) {
	/* open disk run file */			
	if (ddopen(read_file, READ_MODE, IOfmt.rdisk_blen, IOfmt.rdisk_rlen)
	    == -1) {
	    sprintf(reply, "error whilst opening disk file %s\n", read_file);
	    return SORT_FALSE;
	} 
	ptr->IO_state |= READING_DISK;
	ptr->sort_media = (short) DISK_DATA;
	sprintf(reply, "disk file %s open\n", read_file);
    }		  
    else if (strncmp(source, "tape", 4) == 0) {
	int tape_opt = 0;
	if (!(ptr->IO_state & R_TAPE_MOUNTED)) {
	    sprintf(reply, "tape not mounted for OPEN\n");
	    return SORT_FALSE;
	}
	/* check to see if rewind or hold options set */
	if (source[4] == '/') {
	    if (isdigit((int) source[5]))
		sscanf(source, "tape/%d", &tape_opt);
	    if (DBX_val >= 5)
		fprintf(stderr, "server: tape option = %d\n", tape_opt);
	}
	/* open tape run file */
	(void) Stop_interupt(SORT_SET, SORT_FALSE);
	if (dtopen(read_file, READ_MODE, IOfmt.rtape_blen, IOfmt.rtape_rlen, 
		   tape_opt)) {
	    sprintf(reply, "error whilst opening tape file %s\n", read_file);
	    return SORT_FALSE;
	} 
	ptr->IO_state |= READING_TAPE;
	ptr->sort_media = (short) EXABYTE_DATA;
	sprintf(reply, "tape file %s open\n", read_file);
    }
    else if (strncmp(source, "remote", 6) == 0) {
	/* open tape run file */			
	if ( remote_op(OPEN_REMOTE) == -1) {
	    sprintf(reply, "error whilst opening remote stream\n");
	    return SORT_FALSE;
	} 
	ptr->IO_state |= READING_REMOTE;
	ptr->sort_media = (short) REMOTE_DATA;
	sprintf(reply, "Remote read source open\n");
	strcpy(read_file, "REMOTE");
    }
    else if (strncmp(source, "demon", 5) == 0) {
	if (demonopen(read_file, IOfmt.rdisk_rlen))
	{
	    sprintf(reply, "Error whilst opening Demon data feed from %s.\n", 
		    read_file);
	    return SORT_FALSE;
	}
	sprintf(reply, "Demon data feed from %s open.\n", read_file);
	ptr->IO_state |= READING_DEMON;
	ptr->sort_media = (short) DEMON_DATA;
    }
    else if (strncmp(source, "pipe", 4) == 0) {
	rf = opt_str + comm;
	if (pipeopen(rf, IOfmt.rdisk_rlen))
	{
	    sprintf(reply, 
		    "Error whilst opening pipe from \"%s\".\n", rf);
	    return SORT_FALSE;
	}
	sprintf(reply, "Pipe from \"%s\" open.\n", rf);
	ptr->IO_state |= READING_PIPE;
	ptr->sort_media = (short) PIPE_DATA;
    }
    else {
	sprintf(reply, "error during OPEN command => %s\n", opt_str);
	return SORT_FALSE;
    }
    
    strncpy(ptr->read_file, rf, MED_SUNSORT_BUFFER);
    ptr->read_file[MED_SUNSORT_BUFFER] = '\0';
    *sorted_records = 0;
      
    fprintf(stderr, "*** %s", reply);
    return SORT_TRUE;
}

/*
 *  close read sort file
 */
int
close_sort_file(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      sort_IO_t   *ptr = sort_io_data;
      
      if (DBX_val >= 5)
	    fprintf(stderr,"server: close_sort_file called with arg %s\n",opt_str);
      
/* check that not trying to sort */      
      if (Sorting_data(0) != SORT_FALSE) {
	    sprintf(reply,"invalid command while sorting\n");
	    return(SORT_FALSE);
      }

      if (ptr->IO_state & READING_TAPE) {
	    if ( dtclose(READ_MODE) ) {
		  /* panic unmount of data tape */
		  ptr->IO_state &= (~READING_TAPE);
		  sprintf(opt_str,"r");
		  (void) eject_func(opt_str,reply);
		  sprintf(reply,"Error closing tape file %s ... EJECTING TAPE\n",ptr->read_file);
		  return(SORT_FALSE);
	    }
	    ptr->IO_state &= (~READING_TAPE);
      }
      else if (ptr->IO_state & READING_DISK) {
	    if ( ddclose(READ_MODE) == -1) {
		  sprintf(reply,"Error closing disk file %s\n",ptr->read_file);
		  return(SORT_FALSE);
	    }
	    ptr->IO_state &= (~READING_DISK);
      }
      else if (ptr->IO_state & READING_REMOTE) {
	    if ( remote_op(CLOSE_REMOTE) != 0 ) {
		  sprintf(reply,"Error closing remote data source\n");
		  return(SORT_FALSE); 
	    }
	    ptr->IO_state &= (~READING_REMOTE);
      }
      else if (ptr->IO_state & READING_DEMON) {
	  demonclose();
	  ptr->IO_state &= (~READING_DEMON);
      }
      else if (ptr->IO_state & READING_PIPE) {
	  pipeclose();
	  ptr->IO_state &= (~READING_PIPE);
      }
      else {
	    sprintf(reply,"CLOSE command issued with no read file open\n");
	    return(SORT_FALSE);
      }

      if ( empty_R_buffers() == -1) 
	    fprintf(stderr,"server: error flushing read stream buffers\n");

      sprintf(reply,"read file CLOSED\n");
      return(SORT_TRUE);
}

/*
 *  open file to wite filter events to
 */
int
open_filt_file(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      char   source[BUFSIZ], write_file[BUFSIZ];
      sort_IO_t   *ptr = sort_io_data;

      if (DBX_val >= 5)
	    fprintf(stderr,"server: open_filt_file called with arg %s\n",opt_str);

      if (Sorting_data(0) != SORT_FALSE) {
	    sprintf(reply,"invalid command while sorting\n");
	    return(SORT_FALSE);
      }
      
      /* make sure we haven't already got a read file open */		  
      if ( ptr->IO_state & WRITING_OUT) {
	    sprintf(reply,"WOPEN command issued with %s already open!\n",ptr->write_file);
	    return(SORT_FALSE);
      }
      if ( sscanf(opt_str,"%s %s",source,write_file) < 2) {
	    sprintf(reply,"invalid command line for WOPEN command => %s\n",opt_str);
	    return(SORT_FALSE);
      }
      
      if (strncmp(source,"disk",4) == 0) {
	    if ( ddopen(write_file,WRITE_MODE,IOfmt.wdisk_blen,IOfmt.wdisk_rlen) == -1) {
		  sprintf(reply,"failed to WOPEN disk file %s\n",write_file);
		  return(SORT_FALSE);
	    }
	    ptr->IO_state |= WRITING_DISK;
	    ptr->write_media = (short) DISK_DATA;
	    sprintf(reply,"write disk file %s open\n",write_file);
      }		  
      else if (strncmp(source,"tape",4) == 0) {
	    if ( ! (ptr->IO_state & W_TAPE_MOUNTED)) {
		  sprintf(reply,"tape not mounted for WOPEN\n");
		  return(SORT_FALSE);
	    }
	    /* open tape run file */			
	    if ( dtopen(write_file,WRITE_MODE,IOfmt.wtape_blen,IOfmt.wtape_rlen,IOfmt.wtape_format)) {
		  sprintf(reply,"error whilst opening tape %s\n",write_file);
		  return(SORT_FALSE);
	    } 
	    ptr->IO_state |= WRITING_TAPE;
	    ptr->write_media = (short) EXABYTE_DATA;
	    sprintf(reply,"write tape file %s open\n",write_file);
      }
      else {
	    sprintf(reply,"error during WOPEN command => %s\n",opt_str);
	    return(SORT_FALSE);
      }
      strncpy(ptr->write_file,write_file,MED_SUNSORT_BUFFER);
      ptr->write_file[MED_SUNSORT_BUFFER] = '\0';
      *filtered_records = 0;

      return(SORT_TRUE);
}

/*
 *  close file for ouputing of filtered events
 */
int
close_filt_file(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      sort_IO_t   *ptr = sort_io_data;

      if (DBX_val >= 5)
	    fprintf(stderr,"server: close_filt_file called with arg %s\n",opt_str);

      if (Sorting_data(0) != SORT_FALSE) {
	    sprintf(reply,"invalid command while sorting\n");
	    return(SORT_FALSE);
      }

      if (ptr->IO_state & WRITING_TAPE) {
	    if ( dtclose(WRITE_MODE) ) {
		  /* panic unmount of data tape */
		  ptr->IO_state &= (~WRITING_TAPE);
		  sprintf(opt_str,"w");
		  (void) eject_func(opt_str,reply);
		  sprintf(reply,"Error closing tape file %s ... EJECTING TAPE\n",ptr->write_file);
		  return(SORT_FALSE);  
	    }
	    ptr->IO_state &= (~WRITING_TAPE);
      }
      else if (ptr->IO_state & WRITING_DISK) {
	    if ( ddclose(WRITE_MODE) == -1) {
		  sprintf(reply,"Error closing disk file %s\n",ptr->write_file);
		  return(SORT_FALSE);
	    }
	    ptr->IO_state &= (~WRITING_DISK);
      }
      else {
	    sprintf(reply,"WCLOSE command issued with no write file open\n");
	    return(SORT_FALSE);
      }
      
      if ( empty_W_buffers() == -1) 
	    fprintf(stderr,"server: error flushing read stream buffers\n");

      *filtered_records = 0;
      sprintf(reply,"write file CLOSED\n");
      
      return(SORT_TRUE);
}

/*
 *  sort the data from the preconnected sort file
 */
int
sort_data(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      int skip,stop;

      if (DBX_val >= 5)
	    fprintf(stderr,"server: sort_data called with arg %s\n",opt_str);

/* check that not already trying to sort */      
      if (Sorting_data(0) != SORT_FALSE) {
	    sprintf(reply,"invalid command while sorting\n");
	    return(SORT_FALSE);
      }

/* check that sort process is alive */
      if (sort_proc_OK != SORT_TRUE) {
	    sprintf(reply,"no sort process running ... perform loadsort\n");
	    return(SORT_FALSE);
      }

/* check that a read file has been opened */
      if ( !(sort_io_data->IO_state & READING_IN)) {
	    sprintf(reply,"SORT command issued without open read file\n");
	    return(SORT_FALSE);
      }
      
/* get arguments to rdtape  */
      if (sscanf(opt_str,"%d %d",&skip,&stop) != 2) {
	    sprintf(reply,"invalid command options %s\n",opt_str);
	    return(SORT_FALSE);
      }

/* set sort state parameters */
      if (Stop_interupt(SORT_SET, SORT_FALSE) != SORT_FALSE) {
	    sprintf(reply,"error while trying to set state variable Stop_interupt!\n");
	    return(SORT_FALSE);
      }  
      skip_records = skip;
      sort_records = (stop < 0) ? 9999999 : stop;
      
/* and start the main sort thread */
      if ( thr_setconcurrency(3)) {
	    sprintf(reply,"error while trying to start sorting\n");
	    return(SORT_FALSE);
      }
      
      if (Sorting_data(SORT_SET, SORT_TRUE) != SORT_TRUE) {
	    sprintf(reply,"error while trying to set state variable Sorting_data\n");
	    return(SORT_FALSE);
      }

      if (thr_create(NULL, 0, sort_thread, (void *) sort_io_data, THR_DETACHED, NULL )) {
	    (void) Sorting_data(SORT_SET, SORT_FALSE);
	    sprintf(reply,"error while trying to start main sort thread\n");
	    return(SORT_FALSE);
      }

      sprintf(reply,"starting sort functions ... ");
      return(SORT_TRUE);   
}

/*
 *  mount an exabyte
 */
int
tapemount(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      char vol[BUFSIZ];
      char access_mode[BUFSIZ];
      int drive;
      sort_IO_t   *ptr = sort_io_data;

      if (DBX_val >= 5)
	    fprintf(stderr,"server: tapemount called with arg %s\n",opt_str);

      if (DBX_val >= 1)
	    dtverb(1);
      
      if (Sorting_data(0) != SORT_FALSE) {
	    sprintf(reply,"invalid command while sorting\n");
	    return(SORT_FALSE);
      }

      if (sscanf(opt_str,"%s %d %s",vol, &drive, access_mode) != 3) {
	    sprintf(reply,"invalid options for tapemount %s\n",opt_str);
	    return(SORT_FALSE);
      }
      
      if (access_mode[0] == 'r') {
	    if ( ptr->IO_state & R_TAPE_MOUNTED) {
		  sprintf(reply,"tapemount command issued with tape already mounted\n");
		  return(SORT_FALSE);
	    }
	    /* mount tape */
	    if (dtmount(vol,drive,READ_MODE) != 0) {
		  sprintf(reply,"failed mount for tape %s on drive %d\n",vol,drive);
		  return(SORT_FALSE);
	    }
	    ptr->IO_state |= R_TAPE_MOUNTED;
	    strncpy(ptr->read_vol,vol,MED_SUNSORT_BUFFER);
	    ptr->read_vol[MED_SUNSORT_BUFFER] = '\0';
	    ptr->sort_drive = (short) drive;
      }
      else if (access_mode[0] == 'w') {
	    if (ptr->IO_state & W_TAPE_MOUNTED) {
		  sprintf(reply,"tapemount command issued with tape already mounted\n");
		  return(SORT_FALSE);
	    }
	    /* mount tape */
	    if (dtmount(vol,drive,WRITE_MODE) != 0) {
		  sprintf(reply,"failed mount for tape %s on drive %d\n",vol,drive);
		  return(SORT_FALSE);
	    }
	    ptr->IO_state |= W_TAPE_MOUNTED;
	    strncpy(ptr->write_vol,vol,MED_SUNSORT_BUFFER);
	    ptr->write_vol[MED_SUNSORT_BUFFER] = '\0';
	    ptr->write_drive = (short) drive;
      }
      else {
	    sprintf(reply,"tapemount command issued with unknown access mode %s\n",access_mode);
	    return(SORT_FALSE);
      }
      sprintf(reply,"tape %s mounted\n",vol);      

      return(SORT_TRUE);
}

/*
 *  label and mount an exabyte for writing
 */
int
newWtape(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      char vol[BUFSIZ];
      int drive, ierr;
      sort_IO_t   *ptr = sort_io_data;

      if (DBX_val >= 5)
	    fprintf(stderr,"server: newWtape called with arg %s\n",opt_str);

      if (DBX_val >= 1)
	    dtverb(1);
      
      if (Sorting_data(0) != SORT_FALSE) {
	    sprintf(reply,"invalid command while sorting\n");
	    return(SORT_FALSE);
      }

      if (sscanf(opt_str,"%s %d",vol, &drive) != 2) {
	    sprintf(reply,"invalid options for newWtapel %s\n",opt_str);
	    return(SORT_FALSE);
      }
      
      if (ptr->IO_state & W_TAPE_MOUNTED) {
	    sprintf(reply,"newWtapel command issued with tape already mounted\n");
	    return(SORT_FALSE);
      }
      
      /* label tape but first check if already labelled */
      fprintf(stderr,"server: checking that tape is blank ...\n");
      if ((ierr = dtmount(vol,drive,WRITE_MODE)) != 0) {
	    if ( ierr == MOUNT_NOT_ANSI_FORMAT) {
		  if (dtmount(vol,drive,NEWVOL_MODE) != 0) {
			sprintf(reply,"failed to label tape as %s on drive st%d\n",vol,drive);
			return(SORT_FALSE);
		  }
	    }
	    else if (ierr == MOUNT_WRONG_TAPE) {
		  sprintf(reply,"tape on drive st%d already has a label!\n",drive);
		  return(SORT_FALSE);
	    }
	    else {
		  sprintf(reply,"error during newtape command on drive st%d\n",drive);
		  return(SORT_FALSE);
	    }
      }

      ptr->IO_state |= W_TAPE_MOUNTED;
      strncpy(ptr->write_vol,vol,MED_SUNSORT_BUFFER);
      ptr->write_vol[MED_SUNSORT_BUFFER] = '\0';
      ptr->write_drive = (short) drive;

      sprintf(reply,"tape with label %s has been mounted on drive %d\n",vol,drive);      
      return(SORT_TRUE);
}

/*
 * unmount an exabyte
 */
int
tapeumount(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      char access_mode[BUFSIZ];
      sort_IO_t   *ptr = sort_io_data;

      if (DBX_val >= 5)
	    fprintf(stderr,"server: tapeumount called with arg %s\n",opt_str);
      
      if (Sorting_data(0) != SORT_FALSE) {
	    sprintf(reply,"invalid command while sorting\n");
	    return(SORT_FALSE);
      }

      if (sscanf(opt_str,"%s",access_mode) != 1) {
	    sprintf(reply,"invalid options for tapemount %s\n",opt_str);
	    return(SORT_FALSE);
      }

      if (access_mode[0] == 'r') { /* READ TAPE */
	    if ( ! (ptr->IO_state & R_TAPE_MOUNTED)) {
		  sprintf(reply,"tapeumount command issued with tape not mounted\n");
		  return(SORT_FALSE);
	    }
	    if ( ptr->IO_state & READING_TAPE) {
		  sprintf(reply,"tapeumount command issued with read tape file open\n");
		  return(SORT_FALSE);
	    }
	    if ( dtumount(READ_MODE, 1)) {
		  sprintf(reply,"tapeumount command failed\n");
		  return(SORT_FALSE);	
	    }
	    (void) memset(ptr->read_vol,'\0',MED_SUNSORT_BUFFER);
	    ptr->IO_state &= (~R_TAPE_MOUNTED);
      }
      else if (access_mode[0] == 'w') { /* WRITE TAPE */
	    if ( ! (ptr->IO_state & W_TAPE_MOUNTED)) {
		  sprintf(reply,"tapeumount command issued with tape not mounted\n");
		  return(SORT_FALSE);
	    }
	    if ( ptr->IO_state & WRITING_TAPE) {
		  sprintf(reply,"tapeumount command issued with write tape file open\n");
		  return(SORT_FALSE);
	    }	    
	    if ( dtumount(WRITE_MODE, (access_mode[1] == '-') ? 1:0) ) {
		  sprintf(reply,"tapeumount command failed\n");
		  return(SORT_FALSE);		  
	    }
	    (void) memset(ptr->write_vol,'\0',MED_SUNSORT_BUFFER);
	    ptr->IO_state &= (~W_TAPE_MOUNTED);
      }
      else {
	    sprintf(reply,"tapeumount command issued with unknown access mode %s\n",access_mode);
	    return(SORT_FALSE);
      }
      sprintf(reply,"tape unmounted\n");

      return(SORT_TRUE);
}

/*
 * unmount an exabyte
 */
int
eject_func(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      char access_mode[BUFSIZ];

      if (DBX_val >= 5)
	    fprintf(stderr,"server: eject called with arg %s\n",opt_str);
      
      if (Sorting_data(0) != SORT_FALSE) {
	    sprintf(reply,"invalid command while sorting\n");
	    return(SORT_FALSE);
      }

      if (sscanf(opt_str,"%s",access_mode) != 1) {
	    strcpy(access_mode, "r");
      }
      
      return( tapeumount(access_mode, reply));
}

/*
 * move to new position on mounted tape
 */
int
tapemove(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      int nosfiles;
      sort_IO_t   *ptr = sort_io_data;
      
      if (DBX_val >= 5)
	    fprintf(stderr,"server: tapemove called with arg %s\n",opt_str);
      
      if (Sorting_data(0) != SORT_FALSE) {
	    sprintf(reply,"invalid command while sorting\n");
	    return(SORT_FALSE);
      }

      if (sscanf(opt_str,"%d",&nosfiles) != 1) {
	    sprintf(reply,"invalid options for tapemove <<%s>>\n",opt_str);
	    return(SORT_FALSE);
      }
      
      if ( ! (ptr->IO_state & R_TAPE_MOUNTED)) {
	    sprintf(reply,"tapemove command issued with tape not mounted\n");
	    return(SORT_FALSE);
      }
      
      if ( ptr->IO_state & READING_TAPE) {
	    sprintf(reply,"tapemove command issued with read tape file open\n");
	    return(SORT_FALSE);
      }

      if (skip_files(nosfiles)) {
	    sprintf(reply,"tapemove command failed\n");
	    return(SORT_FALSE);
      }

      sprintf(reply,"tapemove command finished OK\n");
      return(SORT_TRUE);
}

/*
 *   save spectrum data to disk
 */
int
save_spectra(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      dirs_IO_t       *ptr = dirs_io_data;
      char            save_dir[BUFSIZ];
      char            option[BUFSIZ];
      char            file_fmt[BUFSIZ];
      int             mode;
      
      if (DBX_val >= 5)
	    fprintf(stderr,"server: save_spectra called with arg %s\n",opt_str);

      if (sscanf(opt_str,"%s %s %s",option, save_dir, file_fmt) != 3) {
	    sprintf(reply,"invalid save options %s\n",opt_str);
	    return(SORT_FALSE);
      }
      ptr->IO_state = SAVING_DATA;
      strncpy(ptr->save_dir,save_dir,MED_SUNSORT_BUFFER);
      ptr->save_dir[MED_SUNSORT_BUFFER] = '\0';
/* determine save format */
      if (strncmp(file_fmt,"sdb",3) == 0)
	    ptr->IO_state |= SDB_TYPE;
      else if (strncmp(file_fmt,"xy",2) == 0)
	    ptr->IO_state |= XY_TYPE;
      else if (strncmp(file_fmt,"eg",2) == 0)
	    ptr->IO_state |= EG_TYPE;
      else {
	    sprintf(reply,"invalid save format %s in save command\n",file_fmt);
	    return(SORT_FALSE);
      }

/* check whether overwriting is permitted */
      if (sscanf(opt_str,"%s %s %s %d",option, save_dir, file_fmt, &mode) != 4) 
	    mode = 0;
      if (mode)
	    ptr->IO_state |= OVER_WRITE;
      
/* use option to specify which save routine to call */
      if (strncmp(option,"all",3) == 0)
	    mode = Save_all(ptr);
      else if (strncmp(option,"windows",7) == 0)
	    mode = Save_windows(ptr);
      else if (strncmp(option,"variables",9) == 0)
	    mode = Save_variables(ptr);
      else if (strncmp(option,"selected",8) == 0)
	    mode = Save_selected(ptr);
      else {
	    sprintf(reply,"invalid save option => %s\n",option);
	    return(SORT_FALSE);
      }
      
      if (mode == SORT_TRUE) {
	    sprintf(reply,"save operation complete\n");
	    return(SORT_TRUE);
      }
      else
	    sprintf(reply,"save operation failed\n");
      return(SORT_FALSE);
}

/*
 * load data from disk into spectra
 */
int
load_spectra(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      dirs_IO_t       *ptr = dirs_io_data;
      char            load_dir[BUFSIZ];
      char            option[BUFSIZ];
      char            file_fmt[BUFSIZ];
      int             mode;
      
      if (DBX_val >= 5)
	    fprintf(stderr,"server: load_spectra called with arg %s\n",opt_str);

      if (sscanf(opt_str,"%s %s %s",option, load_dir, file_fmt) != 3) {
	    sprintf(reply,"invalid load options %s\n",opt_str);
	    return(SORT_FALSE);
      }
      ptr->IO_state = LOADING_DATA;
      strncpy(ptr->load_dir,load_dir,MED_SUNSORT_BUFFER);
      ptr->load_dir[MED_SUNSORT_BUFFER] = '\0';
/* determine save format */
      if (strncmp(file_fmt,"sdb",3) == 0)
	    ptr->IO_state |= SDB_TYPE;
      else if (strncmp(file_fmt,"xy",2) == 0)
	    ptr->IO_state |= XY_TYPE;
      else if (strncmp(file_fmt,"eg",2) == 0)
	    ptr->IO_state |= EG_TYPE;
      else {
	    sprintf(reply,"invalid load format %s in save command\n",file_fmt);
	    return(SORT_FALSE);
      }

/* check whether overwriting is permitted */
      if (sscanf(opt_str,"%s %s %s %d",option, load_dir, file_fmt, &mode) != 4) 
	    mode = 0;
      if (mode)
	    ptr->IO_state |= LOAD_N_ADD;
      
/* use option to specify which save routine to call */
      if (strncmp(option,"all",3) == 0)
	    mode = Load_all(ptr);
      else if (strncmp(option,"windows",7) == 0)
	    mode = Load_windows(ptr);
      else if (strncmp(option,"variables",9) == 0)
	    mode = Load_variables(ptr);
      else if (strncmp(option,"selected",8) == 0)
	    mode = Load_selected(ptr);
      else {
	    sprintf(reply,"invalid load option => %s\n",option);
	    return(SORT_FALSE);
      }
      
      if (mode == SORT_TRUE) {
	    sprintf(reply,"load operation complete\n");
	    /* refresh gui display list */
	    if (nogui != SORT_TRUE && gui_pid > 0) {
		  kill(gui_pid, SIGUSR2);
	    }
	    /* refresh info in sort process */ 
	    if (sort_proc_OK == SORT_TRUE) {
		  write_to_sortproc("check spectra");
	    }
	    return(SORT_TRUE);
      }
      else
	    sprintf(reply,"load operation failed\n");
      return(SORT_FALSE);
}

/*
 *  send data to 1d display process
 */
int displayed_spectrum[MAX_SPEC_DISPLAY+2];

int
Display_1d(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      spec_1d_t	      **pp1d;
      register int    more;
      int             size, nos;
      int             no[MAX_SPEC_DISPLAY];
      char            file[SMALL_SUNSORT_BUFFER];
      
      if (DBX_val >= 5)
	    fprintf(stderr,"server: Display_1d called with arg %s\n",opt_str);

/* check that graphics process is alive */	
      if (! graf_proc1_OK) {
	    sprintf(reply,"1d graphics process not running\n"); 
	    return(SORT_FALSE);
      }
      
/* pass information on display processes ids */      
      pshm->pid_1d = graf_1d_pid;
      pshm->pid_2d = graf_2d_pid;

/* decode option string */
      if ( (more = sscanf(opt_str,"%d %d %d %d",&no[0],&no[1],&no[2],&no[3])) <= 0) {
	    sprintf(reply,"no valid spectrum number provided to Display1d\n");
	    return(SORT_FALSE);
      }
      displayed_spectrum[MAX_SPEC_DISPLAY] = more;
      
/* setup file name to communicate data in 1st spectrum */	
      strcpy(file,".SORT_pipe0");
      for(more-- ; more >= 0; more--) {
	    nos = no[more];
	    /* get pointer to spectrum storage table */
	    pp1d = get_nametab_1d();
	    if ( nos >= TAB_SIZE_1D || nos <= 0 || (! pp1d[nos])) {
		  sprintf(reply,"no data associated with 1d spectrum number %d\n",nos);
		  return(SORT_FALSE);
	    }
	    
	    /* copy spectrum data into memory segment */
	    strcpy(pshm->name,pp1d[nos]->sp->name);
	    pshm->size = size = pp1d[nos]->sp->size;
	    pshm->more = more;
	    if ( DBX_val >= 5) fprintf(stderr,"server: %s, name %d\n",pshm->name,pshm->size);
	    
	    size = (size > SHM_ARRAY_SIZE) ? SHM_ARRAY_SIZE : size;
	    (void) memcpy((char *)pshm->array, (char *)pp1d[nos]->data, sizeof(int)*size);
	    
	    /* non ipc system ... make sure more is never too large*/
	    if ( write_to_pipe(file,size) == SORT_FAIL) 
		  sprintf(reply,"error during writing of data to graf process\n");
	    sprintf(file,".SORT_pipe%d",more);
	    displayed_spectrum[more] = nos;
      } 
      
/* signal graphics process to read shared memory segment and display */
      if (kill(graf_1d_pid,SIGUSR1) == -1)
	    perror("Display_1d SIGUSR1");

      sprintf(reply,"\n");
      return(SORT_TRUE);
}

/*
 *  request the display of a 1d spectra
 */
int
display_1d(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      if (DBX_val >= 5)
	    fprintf(stderr,"server: display_1d called with arg %s\n",opt_str);
      (void) strcpy(pshm->com_str,"DISPLAY 1"); /*set xvgr exec process from menu*/
      pshm->com_str[6] = actionchar;
      return(Display_1d(opt_str, reply));
}

/*
 *  request the overlay of a 1d spectra
 */
int
overlay_1d(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      if (DBX_val >= 5)
	    fprintf(stderr,"server: overlay_1d called with arg %s\n",opt_str);
      (void) strcpy(pshm->com_str,"OVERLAY 1");
      return(Display_1d(opt_str, reply));
}

int Display_2d
#if NeedFunctionPrototype
(char *opt_str, char *reply, int window)
#else
(opt_str, reply, window)
char *opt_str;
char *reply;
int window;
#endif
{
    spec_2d_t **pp2d;
    int nos;
    int size;
    
    if (DBX_val >= 5)
	fprintf(stderr,"server: Display_2d called with arg %s\n",opt_str);

    /* check that graphics process is alive */
    if (! graf_proc2_OK) {
	sprintf(reply,"2d graphics process not running\n"); 
	return SORT_FALSE;
    }
    
    /* pass information on display processes ids */      
    pshm->pid_1d = graf_1d_pid;
    pshm->pid_2d = graf_2d_pid;
      
    /* decode option string */
    if ( sscanf(opt_str,"%d",&nos) != 1) {
	sprintf(reply,"no valid spectrum number provided to Display2d\n");
	return SORT_FALSE;
    }
	  
    /* get pointer to spectrum storage table */
    pp2d = get_nametab_2d();
    if ( nos >= TAB_SIZE_2D || nos <= 0 || (! pp2d[nos])) {
	sprintf(reply,"no data associated with 2d spectrum number %d\n",nos);
	return SORT_FALSE;
    }
      
    /* copy spectrum data into memory segment */
    (void) strcpy(pshm->name,pp2d[nos]->sp->name);
    pshm->size = size = pp2d[nos]->sp->size;
    
    size = size*size;
    size = (size > SHM_ARRAY_SIZE) ? SHM_ARRAY_SIZE : size;
    memcpy((char *)pshm->array, (char *)pp2d[nos]->data, sizeof(int)*size);

    /* non ipc system */
    if ( write_to_pipe(window ? ".SORT_pipe_W" : ".SORT_pipe", size)
	 == SORT_FAIL) 
	sprintf(reply,"error during writing of data to graf process\n");

    /* signal graphics process to read shared memory segment and display */
    if (!window)
    {
	if (kill(graf_2d_pid,SIGUSR2) == -1)
	    perror("Display_2d SIGUSR2");
	
	displayed_spectrum[MAX_SPEC_DISPLAY+1] = nos;
    }
    sprintf(reply,"\n");
    return SORT_TRUE;
}

int
display_2d(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      if (DBX_val >= 5)
	    fprintf(stderr,"server: Display_2d called with arg %s\n",opt_str);
      (void) strcpy(pshm->com_str,"DISPLAY 2");
      return(Display_2d(opt_str,reply,0));
}

/*
 *  clear the spectra of any counts
 */
int
clear_spectra(
#if NeedFunctionPrototype
		   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      char    sbuf[BUFSIZ];
      int     ret_code;
      int     number;
      char    *cptr = NULL;
      
      if (DBX_val >= 5)
	    fprintf(stderr,"server: clear_spectra called with arg %s\n",opt_str);
      
/* decode option string */
      if ((ret_code = sscanf(opt_str,"%s %d", sbuf, &number)) !=2) {
	    if (ret_code !=1) {
		  sprintf(reply,"invalid options string in clear spectra call\n");
		  return(SORT_FALSE);
	    }
      }
      else {
	    cptr = strstr(opt_str,sbuf);
	    cptr += strlen(sbuf);
      }

/* see which spectra type we want clearing */       
      if (strncmp(sbuf,"1d",2) == 0) {
	    /* if we are using terminal for input query this request */
	    if (nogui == SORT_TRUE) {
		  if (query_request("OK to clear") == SORT_FALSE) {
			sprintf(reply,"abandoned\n");
			return(SORT_FALSE);
		  }
	    }
	    ret_code = Clear_1d_spectra(cptr);
      }
      else if (strncmp(sbuf,"2d",2) == 0) {
	    /* if we are using terminal for input query this request */
	    if (nogui == SORT_TRUE) {
		  if (query_request("OK to clear") == SORT_FALSE) {
			sprintf(reply,"abandoned\n");
			return(SORT_FALSE);
		  }
	    }
	    ret_code = Clear_2d_spectra(cptr);
      }
      else if (strncmp(sbuf,"all",3) == 0) {
	    /* if we are using terminal for input query this request */
	    if (nogui == SORT_TRUE) {
		  if (query_request("OK to clear") == SORT_FALSE) {
			sprintf(reply,"abandoned\n");
			return(SORT_FALSE);
		  }
	    }
	    ret_code = Clear_1d_spectra(cptr);
	    if ( (ret_code = SORT_TRUE) != 0)
		  ret_code = Clear_2d_spectra(cptr);
      }
      else {
	    sprintf(reply,"invalid clear option => %s\n",sbuf);
	    return(SORT_FALSE);
      }
      sprintf(reply,"\n");
      return(ret_code);
}

/*
 *   request the display of a 2d spectrum on which to create a 2d window
 */
int
setwindow
#if NeedFunctionPrototype
(char *opt_str, char *reply)
#else
(opt_str, reply)
char *opt_str;
char *reply;
#endif
{
    int  win_nos;
    int  spec_nos;
    spec_2d_t	**pp2d;
      
    if (DBX_val >= 5)
	fprintf(stderr,"server: setwindow called with arg %s\n",opt_str);
    
    if (Sorting_data(0) != SORT_FALSE) {
	sprintf(reply,"invalid command while sorting\n");
	return SORT_FALSE;
    }
    
    /* decode option string */
    if (sscanf(opt_str,"%d on %d", &win_nos, &spec_nos) !=2) {
	sprintf(reply,"invalid options string in set window call\n");
	return SORT_FALSE;
    }
    
/* check that spectrum number refers to 2d window */
    pp2d = get_nametab_2d();
    if (win_nos > 0 && win_nos < TAB_SIZE_2D) {
	if (!pp2d[win_nos]) {
	    sprintf(reply,"no data associated with 2d spectrum number %d\n",
		    win_nos);
	    return SORT_FALSE;
	}
	if (pp2d[win_nos]->win == 0) {
	    sprintf(reply,"2d spectrum %d is not a window!\n",win_nos);
	    return SORT_FALSE;
	}
	strcpy(pshm->com_str,"WINDOW 3");
	sprintf(opt_str, "%d", win_nos);
	if (Display_2d(opt_str, reply, 1) != SORT_TRUE)
	    return SORT_FALSE;
	strcpy(pshm->com_str,"DISPLAY 3");
	pshm->win2d = win_nos;
	sprintf(opt_str,"%d",spec_nos);
	return(Display_2d(opt_str,reply,0));
    }
    sprintf(reply,"window number %d outside possible range\n",win_nos);
    return(SORT_FAIL);
}

/*
 * modify the current value of one of the variables var(n) used in sorting
 */
int
change_var(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      int             var_nos;
      float           var_value;
      var_str_t	      **ptr;
      
      if (DBX_val >= 5)
	    fprintf(stderr,"server: change_var called with arg %s\n",opt_str);
      
      if (Sorting_data(0) != SORT_FALSE) {
	    sprintf(reply,"invalid command while sorting\n");
	    return(SORT_FALSE);
      }

      if (sscanf(opt_str,"%d %g", &var_nos, &var_value) !=2) {
	    sprintf(reply,"invalid options string in change variable call\n");
	    return(SORT_FALSE);
      }

/* get pointer to last structure in variable list */
      if ( !(ptr = get_nametab_var()) ) {
	    sprintf(reply,"variable data table not found...seek help\n");
	    return(SORT_FALSE);
      }

/* update variable value in list */      
      if (var_nos < TAB_SIZE_VAR && ptr[var_nos] != NULL) {
	    ptr[var_nos]->sp->value = var_value;
	    ptr[var_nos]->sp->version++;
	    sprintf(reply,"variable %d (%s) changed to %g\n", var_nos,
	        ptr[var_nos]->sp->name, var_value);
	    return(SORT_TRUE);
      }
      else 
	    sprintf(reply,"failed to modify variable %d\n",var_nos);
      return(SORT_FALSE);
}

/*ARGSUSED*/
int
update_gui(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
    if (nogui != SORT_TRUE && gui_pid > 0)
        kill(gui_pid, SIGUSR2);
    sprintf(reply, "GUI updated.\n");
    return(SORT_TRUE);
}

/*
 * display on stderr the current value of the variables var(n) used in sorting
 */
/*ARGSUSED*/
int
print_vars(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      var_str_t	      **pptr, *pvar;
      
/* get pointer to last structure in variable list */
      if ( !(pptr = get_nametab_var()) ) {
	    sprintf(reply,"variable data table not found...seek help\n");
	    return(SORT_FALSE);
      }
      
      for(pvar = pptr[0]; pvar != NULL; pvar = pvar->next) {
	    fprintf(stderr,"%d   %-16s   %g\n",pvar->nos,pvar->sp->name,pvar->sp->value);
      }
      sprintf(reply,"\n");
      return(SORT_TRUE);
}

/*
 *  reset graphics or graphical user interface
 */
int
reset_command(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      char *option;
      char *argv[2];
      int  argc = 1;
      argv[0] = "sunsort";
      argv[1] = (char *)0;
      
      if (DBX_val >= 5)
	    fprintf(stderr,"server: reset_command called with arg %s\n",opt_str);

      option = rm_leading_blanks(opt_str,BUFSIZ);
      if (strncasecmp(option,"graphics",8) == 0) {
	    if (nograf == SORT_FALSE) {
		  /* kill the graphics process */
		  if (graf_proc1_OK && graf_1d_pid > 0) {
			kill(graf_1d_pid,SIGKILL);
			graf_proc1_OK = SORT_FALSE;
			(void) waitpid(graf_1d_pid, NULL, 0);
		  }
		  if (graf_proc2_OK && graf_2d_pid > 0) {
			kill(graf_2d_pid,SIGKILL);
			graf_proc2_OK = SORT_FALSE;
			(void) waitpid(graf_2d_pid, NULL, 0);
		  }
	    }
	    (void) signal(SIGUSR2, sigwin2d_handler);
	    start_graphics(argc, argv);
	    nograf = SORT_FALSE;
      }
      else if (strncasecmp(option,"gui",3) == 0) {
	    /* if existing gui alive kill it and reset streams */
	    if (nogui == SORT_FALSE && gui_pid > 0 ) {
		  kill(gui_pid,SIGKILL);
		  (void) waitpid(gui_pid, NULL, 0);
		  if (FDIN != STDIN_FILENO)
			(void) close(FDIN);
		  if (FDOUT != STDOUT_FILENO)
			(void) close(FDOUT);
		  FDIN = STDIN_FILENO;
		  FDDAT = FDOUT = STDOUT_FILENO;
	    }
	    if ( (gui_pid = start_gui("sort_gui", argc, argv)) == -1) {
		  nogui = SORT_TRUE;
		  FDIN = STDIN_FILENO;
		  FDDAT = FDOUT = STDOUT_FILENO;
	    }
	    nogui = SORT_FALSE;
      }
      else {
	    sprintf(reply,"unrecognised option for reset command\n");
	    return(SORT_FALSE);
      }
      sprintf(reply,"reset ok\n");
      return(SORT_TRUE);
}

/*
 *  set input from file specified in opt_str
 *  EOF should reset input to orginal input source
 */

int
batch_read(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      int bfp;
      char *file;
      
      if (DBX_val >= 5)
	    fprintf(stderr,"server: batch_read called with arg %s\n",opt_str);

      if (Sorting_data(0) != SORT_FALSE) {
	    sprintf(reply,"invalid command while sorting\n");
	    return(SORT_FALSE);
      }
      if (batch_mode == SORT_TRUE) {
	    sprintf(reply,"invalid command while using batch\n");
	    return(SORT_FALSE);
      }
      
      file = rm_leading_blanks(opt_str,BUFSIZ);

      if ( (bfp = open(file,O_RDONLY)) == -1) {
	    perror("server");
	    sprintf(reply,"failed to open batch file %s\n",file);
	    return(SORT_FALSE);
      }
      write_reply(SORT_TRUE,"batch file opened\n",-1);
      
/* set streams for batch processing */
      FDBAT = bfp;
      FDDAT = STDOUT_FILENO;
      
      batch_mode = SORT_TRUE;
      sprintf(reply,"batch file %s opened\n",opt_str);
      return(SORT_TRUE);
}

int
end_batch(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      if (DBX_val >= 5)
	    fprintf(stderr,"server: end_batch called with arg %s\n",opt_str);

/* tidy up after batch processing */
      if (batch_mode == SORT_TRUE) {
	    (void) close(FDBAT);
	    batch_mode = SORT_FALSE;
	    FDDAT = FDOUT;
	    FDBAT = STDIN_FILENO;
	    /* wait for sort thread to exit or timeout if sorting */
	    while (Sorting_data(0) == SORT_TRUE) {
		  sleep(1);
	    }
	    /* if gui is alive signal that batch mode has stopped */
	    if (nogui != SORT_TRUE && gui_pid > 0)
		  kill(gui_pid,SIGUSR1);
      }
      sprintf(reply, "batch processing ended\n");
      return(SORT_FAIL);    /* return sort_fail to suppress further output */
}

/*
 *  exit sorting program and terminate any associated processes
 */
/*ARGSUSED*/
int
exit_sunsort(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      sort_IO_t   *ptr = sort_io_data;
      
/* signal all child processes of this process group to terminate */      
/*      (void) signal(SIGTERM, SIG_IGN);
      kill(0,SIGTERM); */

/* set timeout limit */
      (void) signal(SIGALRM, SIG_DFL);
      alarm(5);
/* wait for the graphics to die */      
      if (graf_proc1_OK) {
	    if (graf_1d_pid > 0) kill(graf_1d_pid,SIGTERM);
	    waitpid(graf_1d_pid, NULL, 0);
      }
      if (graf_proc2_OK){
	    if (graf_2d_pid > 0) kill(graf_2d_pid,SIGTERM);
	    waitpid(graf_2d_pid, NULL, 0);
      }
/* wait for gui to die */
      if (nogui != SORT_TRUE) {
	    if (gui_pid > 0) kill(gui_pid,SIGKILL);
	    waitpid(gui_pid, NULL, 0);
      }
/* wait for sort process to die */      
      if (sort_proc_OK) {
	    if (sort_pid > 0) kill(sort_pid,SIGTERM);
	    waitpid(sort_pid, NULL, 0);
      }
      
/* if filtering data ensure data buffer flushed and closed */
      if (ptr->IO_state & WRITING_OUT) {
	    char opt_str2[BUFSIZ], reply2[BUFSIZ];
	    fprintf(stderr,"closing write file %s\n",ptr->write_file);
	    sprintf(opt_str2,"close on exit");
	    (void) close_filt_file(opt_str2,reply2);
      }

/* remove shared data directory */
      sprintf(reply,"rm -r %s",tmp_spectrum_dir);
      (void) system(reply);
      
/* remove communication files to display processes */
      (void) system("rm .SORT_pipe* 2> /dev/null");

      (void) signal(SIGTERM, SIG_DFL);
      exit(0);
#ifdef lint
      return SORT_TRUE;
#endif
}

/*
 *  regard line beginning with # as a comment line
 *  and just print it to stderr
 */
int
comment_line(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      fprintf(stderr,"server: comment line => %s\n",opt_str);
      sprintf(reply,"\n");
      return(SORT_TRUE);
}

/*
 *  regard line beginning with # as a comment line
 *  and just print it to stderr
 */
int
sleep_func(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      int  time;
      if (DBX_val >= 5)
	    fprintf(stderr,"server: sleep function called with args => %s\n",opt_str);
/* decode option string */
      if ( sscanf(opt_str,"%d",&time) != 1 || time <= 0) {
	    sprintf(reply,"invalid sleep argument %d\n",time);
	    return(SORT_FALSE);
      }
/* pause process for specified period */
      fprintf(stderr,"sleeping for %d seconds\n",time);
      sleep(time);
      sprintf(reply,"sleep finished\n");
      return(SORT_TRUE);
}

/*
 *  update displayed spectra every refresh_time while sorting
 */
int
refresh_func(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      int  time;
      if (DBX_val >= 5)
	    fprintf(stderr,"server: refresh function called with args => %s\n",opt_str);
/* decode option string */
      if ( sscanf(opt_str,"%d",&time) != 1 ) {
	    sprintf(reply,"invalid refresh argument %d\n",time);
	    return(SORT_FALSE);
      }
/* set global time variable to an appropriate value */      
      if (time <= 0) {
	    refresh_time = REFRESH_OFF;
	    setup_intervaltimer(SORT_FALSE);
	    sprintf(reply,"refresh off\n");
	    return(SORT_TRUE);
      }
      else if (time < 4) {
	    refresh_time = 4;
      }
      else {
	    refresh_time = time;
      }
      setup_intervaltimer(SORT_TRUE);
      
      sprintf(reply,"refresh period = %d seconds\n",refresh_time);
      return(SORT_TRUE);
}

/*
 *  set up IOfmt structure with info corresponding to predefined EbyE data format
 */
int
set_format
#if NeedFunctionPrototype
(const char *opt_str)
#else
(opt_str)
char     *opt_str;
#endif
{
    char format_name[BUFSIZ];
    int i;
    
    if (DBX_val >= 5)
	fprintf(stderr,"server: set_format function called with args => %s\n",
		opt_str);
    
/* check that not trying to sort */      
    if (Sorting_data(0) != SORT_FALSE) {
	fprintf(stderr,"invalid command while sorting\n");
	return(SORT_FALSE);
    }
    
/* decode option string */
    if ( sscanf(opt_str,"%s",format_name) != 1) {
	fprintf(stderr,"invalid set_format argument %s\n",opt_str);
	return(SORT_FALSE);
    }
    
/* choose one of the formats */
    for (i=0; strlen(formats[i].format_name) != 0; i++) {
	if (!strncmp(format_name, formats[i].format_name, FORMAT_NAME_LEN)) {
	    (void) memcpy(&IOfmt, &formats[i], sizeof(IO_format_t));
	    strcpy(sort_io_data->decoder, formats[i].format_name);
	    return(SORT_TRUE);
	}
    }
    
    fprintf(stderr,"Unknown format type %s\n",format_name);
    return(SORT_FALSE);
}

/*
 *  set one of the paramters in the IOfmt structure
 *  these value will be passed as parameters to the IO open funtions
 *  but may be ignored by these functions
 */
void
set_IOfmt_value(
#if NeedFunctionPrototype
	   char *reply, char *name, int value)
#else
      reply, name, value)
      char     *reply;
      char     *name;
      int      value;
#endif
{
      if (strncmp(name,"blen_R_tape",11) == 0) {
	    IOfmt.rtape_blen = value;
	    sprintf(reply,"%s has been set to %d\n",name,value);
      }
      else if (strncmp(name,"rlen_R_tape",11) == 0) {
	    IOfmt.rtape_rlen = value;
	    sprintf(reply,"%s has been set to %d\n",name,value);
      }
      else if (strncmp(name,"blen_W_tape",11) == 0) {
	    IOfmt.wtape_blen = value;
	    sprintf(reply,"%s has been set to %d\n",name,value);
      }
      else if (strncmp(name,"rlen_W_tape",11) == 0) {
	    IOfmt.wtape_rlen = value;
	    sprintf(reply,"%s has been set to %d\n",name,value);
      }
      else if (strncmp(name,"blen_R_disk",11) == 0) {
	    IOfmt.rdisk_blen = value;
	    sprintf(reply,"%s has been set to %d\n",name,value);
      }
      else if (strncmp(name,"rlen_R_disk",11) == 0) {
	    IOfmt.rdisk_rlen = value;
	    sprintf(reply,"%s has been set to %d\n",name,value);
      }
      else if (strncmp(name,"blen_W_disk",11) == 0) {
	    IOfmt.wdisk_blen = value;
	    sprintf(reply,"%s has been set to %d\n",name,value);
      }
      else if (strncmp(name,"rlen_W_disk",11) == 0) {
	    IOfmt.wdisk_rlen = value;
	    sprintf(reply,"%s has been set to %d\n",name,value);
      }
      else
	    sprintf(reply,"%s not recognised as a IO format parameter\n",name);
      return;
}


/*
 *  set sunsort environment variables 
 */
int
set_sortenv(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      char variable[BUFSIZ];
      char value[BUFSIZ];
      
      if (DBX_val >= 5)
	    fprintf(stderr,"server: set_sortenv function called with args => %s\n",opt_str);

/* check that not trying to sort */      
      if (Sorting_data(0) != SORT_FALSE) {
	    sprintf(reply,"invalid command while sorting\n");
	    return(SORT_FALSE);
      }
      
/* decode option string */
      if ( sscanf(opt_str,"%s %s",variable,value) != 2) {
	    sprintf(reply,"invalid set_sortenv argument %s\n",opt_str);
	    return(SORT_FALSE);
      }

/* check against list of known variables */
      if (strncmp(variable,"dbxlevel",8) == 0) {
	    /* change detail of debug information printed to screen */
	    DBX_val = atoi(value);
	    sprintf(reply,"dbxlevel set to %d\n",DBX_val);
      }
      else if (strncmp(variable,"format",6) == 0) {
	    /* reject request if read/write data file already open */
	    if (sort_io_data->IO_state & (READING_IN | WRITING_OUT)) {
		  sprintf(reply,"can not change format with i/o data stream open!\n");
		  return(SORT_FALSE);
	    }
	    /* change default format specifications and decoder for io streams */
	    if (set_format(value) == SORT_TRUE)
		  sprintf(reply,"format %s selected\n",value);
	    else {
		  sprintf(reply,"error selecting format %s\n",value);
		  return(SORT_FALSE);
	    }
      }
      else if (strncmp(variable,"decoder",7) == 0) {
	    /* change decoder used in sort process */
	    strncpy(sort_io_data->decoder, value, FORMAT_NAME_LEN);
	    sprintf(reply,"will request sort process to use decoder %s\n",value);
      }
      else if (strncmp(variable,"blen_",5) == 0 || strncmp(variable,"rlen_",5) == 0) {
	    /* change io format length */
	    set_IOfmt_value(reply,variable,atoi(value));
      }
      else {
	    sprintf(reply,"unknown variable %s in setsenv call\n",variable);
	    return(SORT_FALSE);
      }

      return(SORT_TRUE);
}

/*
 *  print sunsort environment variables
 */
int
print_sortenv(
#if NeedFunctionPrototype
	   char *opt_str, char *reply)
#else
      opt_str, reply)
      char     *opt_str;
      char     *reply;
#endif
{
      
      if (DBX_val >= 5)
	    fprintf(stderr,"server: print_sortenv function called with args => %s\n",opt_str);

      fprintf(stderr,"\ndbxlevel = %d\n",DBX_val);
      fprintf(stderr,"format = %s\n\n",IOfmt.format_name);
      fprintf(stderr,"decoder = %s\n",sort_io_data->decoder);

      fprintf(stderr,"*** Parameters passed to IO function when OPENING data streams ***\n");
      fprintf(stderr,"For reading ...  blen_R_tape = %d, \t rlen_R_tape = %d\n", IOfmt.rtape_blen, IOfmt.rtape_rlen);
      fprintf(stderr,"                 blen_R_disk = %d, \t rlen_R_disk = %d\n", IOfmt.rdisk_blen, IOfmt.rdisk_rlen);
      fprintf(stderr,"For writing ...  blen_W_tape = %d, \t rlen_W_tape = %d\n", IOfmt.wtape_blen, IOfmt.wtape_rlen);
      fprintf(stderr,"                 blen_W_disk = %d, \t rlen_W_disk = %d\n", IOfmt.wdisk_blen, IOfmt.wdisk_rlen);
      
      sprintf(reply,"\n");
      return(SORT_TRUE);
}

int
display_action
#if NeedFunctionPrototype
(char *opt_str, char *reply)
#else
(opt_str, reply)
char     *opt_str;
char     *reply;
#endif
{
    char *p = opt_str;

    while(*p == ' ' || *p == '\t')
	p++;

    if (DBX_val >= 5)
	fprintf(stderr,"server: Display_action called with arg %s\n",opt_str);

    if (!strcasecmp(p, "display"))
    {
	actionchar = 'Y';
	sprintf(reply, "Now just displaying spectra.\n");
	return SORT_TRUE;
    }

    if (!strcasecmp(p, "mstick") || !strcasecmp(p, "matchstick"))
    {
	actionchar = 'M';
	sprintf(reply, "Now auto-matchsticking displayed spectra.\n");
	return SORT_TRUE;
    }

    if (!strcasecmp(p, "pkfind") || !strcasecmp(p, "peakfind"))
    {
	actionchar = 'P';
	sprintf(reply, "Now auto-peakfinding displayed spectra.\n");
	return SORT_TRUE;
    }

    if (!strcasecmp(p, "Paper"))
    {
	actionchar = 'E';
	sprintf(reply, "Now auto-writing Phys Rev Lett! Hit display!\n");
	return SORT_TRUE;
    }

    sprintf(reply, "Unknown display action \"%s\".\n", p);
    return SORT_FALSE;
}

/*
 *   write down pipe connected to sort process
 */
int
write_sort
#if NeedFunctionPrototype
(char *opt_str, char *reply)
#else
(opt_str, reply)
char     *opt_str;
char     *reply;
#endif
{
    int i;

    if (sort_proc_OK != SORT_TRUE) {
	sprintf(reply, "server: attempt to communicate with nonexistent "
		"sort process\n");
	return SORT_FALSE;
    }
    i = strlen(opt_str);
    opt_str[i] = '\n';
    if (write(FDSUSEROUT, opt_str+1, i) != i)
    {
	sprintf(reply, "server: attempt to communicate with sort process "
		"failed\n");
	return SORT_FALSE;
    }

    *reply = '\0';
    return SORT_TRUE;
}

int
processes
#if NeedFunctionPrototype
(char *opt_str, char *reply)
#else
(opt_str, reply)
char     *opt_str;
char     *reply;
#endif
{
    int w;

    if (DBX_val >= 5)
	fprintf(stderr,"server: processes function called with args => %s\n",
		opt_str);
    
    if (sscanf(opt_str, " %d", &w) < 1)
    {
	w = sysconf(_SC_NPROCESSORS_ONLN);
	fprintf(stderr, "Running on a machine with %d processor%s.\n",
		w, (w == 1) ? "" : "s");
    }
    else
    {
	workers = w;
    }
    
    if (workers == 0)
	sprintf(reply, "Next loadsort will match number of worker "
		"processes to number of processors.\n");
    else
	sprintf(reply, "Next loadsort will use %d worker process%s.\n",
		workers, (workers == 1) ? "" : "es");

    return SORT_TRUE;
}
