/*
 *   routines dealing with memory control shared between sort
 *   process and this control process (sunsort)
 
 *   prev modified 15/6/95
 *   last modified 26/7/95
 */
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <malloc.h>
#include <signal.h>
#include <stdarg.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <sys/wait.h>

#include <time.h>
#include <fcntl.h>

#include "sort_def.h"
#include "sort_threadint.h"
#include "sort_main.h"
#include "data_disk.h"
#include "data_tape.h"
#include "data_remote.h"
#include "data_pipe.h"
#include "demon.h"
#include "version.h"

/* local variables */
static struct shared_data *sdat_ptr;
static struct worker_data *wdat;

static IOfunc       get_data_func = NULL;
static IOfunc       put_data_func = NULL;

/* variables shared with other source modules */
int                sort_records = 0;
int                skip_records = 0;
unsigned long      *sorted_records;
unsigned long      *filtered_records;
sort_IO_t          *sort_io_data;
dirs_IO_t          *dirs_io_data;

static int worker_init(const char *data_file_dir)
{
    const char name[] = "/.worker_data";
    char *worker_file;
    int fd, i;

    worker_file = (char *) malloc(strlen(data_file_dir)+strlen(name)+1);
    if (worker_file == NULL) {
	perror("worker_init");
	fprintf(stderr, "server: failed to allocate memory for worker data "
		"file name\n");
	return -1;
    }
    sprintf(worker_file,"%s%s",data_file_dir,name);
    
    if ((fd = open(worker_file, O_RDWR | O_CREAT, 0600)) == -1)
    {
	perror("worker_init open");
	free(worker_file);
	return -1;
    }

    free(worker_file);

    if (ftruncate(fd, sizeof(struct worker_data)) == -1)
    {
	perror("worker_init truncate");
	close(fd);
	return -1;
    }

    if ((caddr_t) (wdat = (struct worker_data *)
		   mmap(NULL, sizeof(struct worker_data),
			PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0))
	== (caddr_t) -1)
    {
	perror("worker_init mmap");
	close(fd);
	return -1;
    }
    close(fd);

    for(i = 0; i < MAX_WORKERS; i++)
	wdat->worker_pid[i] = (pid_t) 0;

    if (mutex_init(&wdat->worker_mutex, USYNC_PROCESS, NULL))
    {
	perror("worker_init mutex_init");
	return -1;
    }

    return 0;
}

void kill_all_workers P((void))
{
    int i;

    mutex_lock(&wdat->worker_mutex);
    for(i = 0; i < MAX_WORKERS; i++)
	if (wdat->worker_pid[i] != (pid_t) 0)
	    kill(wdat->worker_pid[i], SIGKILL);
    mutex_unlock(&wdat->worker_mutex);
}

/*
 *  create directory in which shared data file and spectra files shall reside
 */
int
create_shared_data
#if NeedFunctionPrototype
(const char *data_file_dir)
#else
(data_file_dir)
char *data_file_dir
#endif
{
    const char name[] = "/.shared.dat";
    char *data_file;
    int filedes;
    mode_t pmask;
#ifndef USE_DATA_QUEUE
    mutex_t *mp;
    cond_t  *cv;
    int i;
#endif
      
/* create name of shared data file */
    data_file = (char *) malloc(strlen(data_file_dir)+strlen(name)+1);
    if (data_file == NULL) {
	perror("create_shared_data");
	fprintf(stderr, "server: failed to allocate memory for shared data "
		"file name\n");
	return -1;
    }
    sprintf(data_file,"%s%s",data_file_dir,name);
      
/* set up shared resources */
    pmask = umask(0077);
    filedes = open(data_file, O_RDWR|O_CREAT,0600);
    if (filedes == -1) {
	fprintf(stderr, "server: failed to create shared data file %s\n",
		data_file);
	free(data_file);
	return -1;
    }
    (void) umask(pmask);

    free(data_file);

    if (ftruncate(filedes,SHDATA_SIZE) == -1) {
	perror("ftruncate");
	return -1;
    }
      
    /*LINTED*/
    if ((caddr_t) (sdat_ptr = (struct shared_data *)
		   mmap(0, SHDATA_SIZE, PROT_READ | PROT_WRITE,
			MAP_SHARED, filedes, 0)) == (caddr_t) -1) {
	fprintf(stderr, "server: mmap error on shared data file\n");
	return -1;
    }

    (void) close(filedes);

/* set version information */
    sdat_ptr->version = SUNSORT_VERSION;

/* set up other pointers to data structure */      
    sort_io_data = &sdat_ptr->sort_io_data;
    dirs_io_data = &sdat_ptr->dirs_io_data;
    sorted_records = &sdat_ptr->in_records;
    filtered_records = &sdat_ptr->out_records;

/* initialize sorting state mutexes and conditional variables */
    if (mutex_init(&sdat_ptr->mp_sorting, USYNC_PROCESS, NULL)) {
	perror("mutex sorting state initialization");
	return -1;
    }
    if (cond_init(&sdat_ptr->cv_sorting, USYNC_PROCESS, NULL)) {
	perror("conditional variable sorting state initialization");
	return -1;
    }
    sdat_ptr->sorting = SORTING_OFF;

#ifdef USE_DATA_QUEUE
    if (queue_init(&sdat_ptr->read_full))
    {
	perror("read full queue");
	return -1;
    }
    if (queue_init(&sdat_ptr->read_empty))
    {
	perror("read full queue");
	return -1;
    }
    if (queue_init(&sdat_ptr->write_full))
    {
	perror("write full queue");
	return -1;
    }
    if (queue_init(&sdat_ptr->write_empty))
    {
	perror("write full queue");
	return -1;
    }
#else
/* initialize read buffer mutexes and conditional variables */
      for(i=0; i< NOS_OF_RBUFS; i++) {
	    mp = &sdat_ptr->read_buf[i].mp;
	    if (mutex_init(mp, USYNC_PROCESS, NULL)) {
		  perror("mutex read initialization");
		  return(-1);
	    }
	    cv = &sdat_ptr->read_buf[i].cv;
	    if (cond_init(cv, USYNC_PROCESS, NULL)) {
		  perror("conditional variable read initialization");
		  return(-1);
	    }
	    sdat_ptr->read_buf[i].state = EMPTY_BUF;
      }

/* initialize write buffer mutexes and conditional variables */
      for(i=0; i< NOS_OF_WBUFS; i++) {
	    mp = &sdat_ptr->write_buf[i].mp;
	    if (mutex_init(mp, USYNC_PROCESS, NULL)) {
		  perror("mutex  write initialization");
		  return(-1);
	    }
	    cv = &sdat_ptr->write_buf[i].cv;
	    if (cond_init(cv, USYNC_PROCESS, NULL)) {
		  perror("conditional variable read initialization");
		  return(-1);
	    }
	    sdat_ptr->write_buf[i].state = EMPTY_BUF;
      }
#endif

      return worker_init(data_file_dir);
}

/*
 *  set the state of each read buffer to empty
 */
int
empty_R_buffers
#if NeedFunctionPrototype
(void )
#else
()
#endif
{
#ifdef USE_DATA_QUEUE      
    int i;

    if (queue_flush(&sdat_ptr->read_full))
    {
	perror("read full queue flush");
	return -1;
    }
    if (queue_flush(&sdat_ptr->read_empty))
    {
	perror("read full queue flush");
	return -1;
    }

    for(i = 0; i < NOS_OF_RBUFS; i++)
	add_to_queue(&sdat_ptr->read_empty, i);
    
#else
    data_buf_t *dptr;
    int i;
    
    for(i=0; i< NOS_OF_RBUFS; i++) {
	dptr = &sdat_ptr->read_buf[i];
	if (mutex_trylock(&dptr->mp)) {
	    perror("mutex read lock -- reset_buffers");
		  return -1;
	}
	dptr->state = EMPTY_BUF;
	if (mutex_unlock(&dptr->mp)) {
	    perror("mutex read unlock-- reset_buffers");
	    return -1;
	    }
    }
#endif
    return 0;
}
/*
 *  set the state of each write buffer to empty
 */
int
empty_W_buffers
#if NeedFunctionPrototype
(void )
#else
()
#endif
{
#ifdef USE_DATA_QUEUE      
    int i;

    if (queue_flush(&sdat_ptr->write_full))
    {
	perror("write full queue flush");
	return -1;
    }
    if (queue_flush(&sdat_ptr->write_empty))
    {
	perror("write full queue flush");
	return -1;
    }

    for(i = 0; i < NOS_OF_RBUFS; i++)
    {
	sdat_ptr->write_buf[i].data_size = MAX_DATA_SIZE;
	add_to_queue(&sdat_ptr->write_empty, i);
    }
#else
      data_buf_t *dptr;
      int i;

      for(i=0; i< NOS_OF_WBUFS; i++) {
	    dptr = &sdat_ptr->write_buf[i];
	    if (mutex_trylock(&dptr->mp)) {
		  perror("mutex write lock -- reset_buffers");
		  return(-1);
	    }
	    dptr->state = EMPTY_BUF;
	    if (mutex_unlock(&dptr->mp)) {
		  perror("mutex write  unlock-- reset_buffers");
		  return(-1);
	    }
      }
#endif
      return 0;
}

/*
 *  set the state of each read buffer to empty
 */
int
diagnostic_R_buffers
#if NeedFunctionPrototype
(void)
#else
()
#endif
{
    data_buf_t *dptr;
    int i;
      
#ifdef USE_DATA_QUEUE
    fprintf(stderr, "Read empty queue: ");
    queue_diagnostic(&sdat_ptr->read_empty);
    fprintf(stderr, "Read full queue: ");
    queue_diagnostic(&sdat_ptr->read_full);
    fprintf(stderr, "Buffer status:");
    for(i=0, dptr = sdat_ptr->read_buf; i < NOS_OF_RBUFS; dptr++, i++)
	fprintf(stderr, " %d@%08x=%d", i, (unsigned long) dptr,
		dptr->data_size);
    putc('\n', stderr);
#else
    for(i=0; i< NOS_OF_RBUFS; i++) {
	dptr = &sdat_ptr->read_buf[i];
	if (mutex_trylock(&dptr->mp)) {
	    perror("mutex read lock-- diagnostic_buffers");
	    continue;
	}
	fprintf(stderr,"server: Read buffer %d in state %d\n", i, dptr->state);
	if (mutex_unlock(&dptr->mp)) {
	    perror("mutex read unlock-- diagnostic_buffers");
	    continue;
	}
    }
#endif
    return 0;
}

/*
 * reset state of each shared buffer to EMPTY, re-inititialize locking variables
 */
int
reset_all_buffers
#if NeedFunctionPrototype
(void)
#else
()
#endif
{
/* reset read multexes and CV's */      
#ifdef USE_DATA_QUEUE
    if (queue_destroy(&sdat_ptr->read_full))
    {
	perror("read full queue destroy");
	return -1;
    }
    if (queue_destroy(&sdat_ptr->read_empty))
    {
	perror("read empty queue destroy");
	return -1;
    }
    if (queue_init(&sdat_ptr->read_full))
    {
	perror("read full queue init");
	return -1;
    }
    if (queue_init(&sdat_ptr->read_empty))
    {
	perror("read empty queue init");
	return -1;
    }
#else
    data_buf_t *dptr;
    int i;

    for(i=0; i< NOS_OF_RBUFS; i++) {
	dptr = &sdat_ptr->read_buf[i];
	if (mutex_destroy(&dptr->mp)) {
	    perror("read mutex destroy");
	    return(-1);
	    }
	if (mutex_init(&dptr->mp, USYNC_PROCESS, NULL)) {
	    perror("read mutex re-initialization");
	    return(-1);
	}
	if (cond_destroy(&dptr->cv)) {
	    perror("read conditional variable destroy");
	    return(-1);
	    }	
	if (cond_init(&dptr->cv, USYNC_PROCESS, NULL)) {
	    perror("read conditional variable re-initialization");
	    return(-1);
	}	    
    }
#endif
/* set state of all read buffers to empty */      
    if (empty_R_buffers() == -1)
	return(-1);

#ifdef USE_DATA_QUEUE
    if (queue_destroy(&sdat_ptr->write_full))
    {
	perror("write full queue destroy");
	return -1;
    }
    if (queue_destroy(&sdat_ptr->write_empty))
    {
	perror("write empty queue destroy");
	return -1;
    }
    if (queue_init(&sdat_ptr->write_full))
    {
	perror("write full queue init");
	return -1;
    }
    if (queue_init(&sdat_ptr->write_empty))
    {
	perror("write empty queue init");
	return -1;
    }
#else
/* reset write multexes and CV's */        
    for(i=0; i< NOS_OF_WBUFS; i++) {
	dptr = &sdat_ptr->write_buf[i];
	if (mutex_destroy(&dptr->mp)) {
	    perror("write mutex destroy");
	    return(-1);
	}
	if (mutex_init(&dptr->mp, USYNC_PROCESS, NULL)) {
		  perror("write mutex re-initialization");
		  return(-1);
	}
	if (cond_destroy(&dptr->cv)) {
	    perror("write conditional variable destroy");
	    return(-1);
	}	
	if (cond_init(&dptr->cv, USYNC_PROCESS, NULL)) {
	    perror("write conditional variable re-initialization");
		  return(-1);
	}	    
    }
#endif
/* set state of all write buffers to empty */       
    if (empty_W_buffers() == -1)
	return(-1);
    
    return(0);
}


/*
 * update the entry in the shared data file for a spectrum database
 * with type = var, 1d, 2d
 */
spec_entry_t *
update_shared_list(
#if NeedFunctionPrototype
		       int type, int nos, float size, char *name, int opt)	
#else
      type, nos, size, name, opt)
      int   type;
      int   nos;	
      float size;	
      char *name;
      int   opt;
#endif
{
      spec_entry_t  *sp;
      
      switch(type) {
	  case TYPE_var:
	    if (nos <= 0 || nos >= TAB_SIZE_VAR) {
		  fprintf(stderr,"server: Invalid variable number %d ... '%s'\n",nos,name);
		  return(NULL);
	    }
	    sp = &sdat_ptr->var[nos];
	    sp->value = size;
	    break;

	  case TYPE_1d:
	    if (nos <= 0 || nos >= TAB_SIZE_1D) {
		  fprintf(stderr,"server: Invalid 1D spectrum number %d ... '%s'\n",nos,name);
		  return(NULL);
	    }
	    sp = &sdat_ptr->spec1d[nos];
	    sp->size = (int) size;
	    break;

	  case TYPE_2d:
	    if (nos <= 0 || nos >= TAB_SIZE_2D) {
		  fprintf(stderr,"server: Invalid 2D spectrum number %d ... '%s'\n",nos,name);
		  return(NULL);
	    }
	    sp = &sdat_ptr->spec2d[nos];
	    sp->size = (int) size;
	    sp->win = opt;
	    break;

	  default:
	    fprintf(stderr,"server: unknown spectrum type %d\n",type);
	    return(NULL);
      }
      
      sp->nos = nos;
      sp->version++;
      (void) strncpy(sp->name, name, NAME_SIZE);
      return(sp);
}

/*
 *  set and/or return current sorting status
 *  incoporates lock to prevent more than one thread accessing at once
 */
int
Sorting_data(
#if NeedFunctionPrototype
		  int set_flag, ...)
#else
      set_flag, ...)
      int set_flag;
#endif
{
      static mutex_t sorting_data_mp;
      static int     sorting_data = SORT_FALSE;
      int            retcode;
      
      if (mutex_lock(&sorting_data_mp)) {
	    perror("mutex sorting_data lock");
	    return(SORT_FAIL);
      }
      if (set_flag == 0) {
	    retcode = sorting_data;
      }
      else {
	    va_list ap;
	    va_start(ap, set_flag);
	    retcode = sorting_data = va_arg(ap, int);
	    va_end(ap);
      }
      if (mutex_unlock(&sorting_data_mp)) {
	    perror("mutex sorting_data lock");
	    return(SORT_FAIL);
      }
      return(retcode);
}
/*
 *  set and/or return current ^C interupt status
 *  incoporates lock to prevent more than one thread accessing at once
 */
int
Stop_interupt(
#if NeedFunctionPrototype
		  int set_flag, ...)
#else
      set_flag, ...)
      int set_flag;
#endif
{
      static mutex_t stop_interupt_mp;
      static int     stop_interupt = SORT_FALSE;
      int            retcode;
      
      (void) mutex_lock(&stop_interupt_mp);
      if (set_flag == 0) {
	    retcode = stop_interupt;
      }
      else {
	    va_list ap;
	    va_start(ap, set_flag);
	    retcode = stop_interupt = va_arg(ap, int);
	    va_end(ap);
      }
      (void) mutex_unlock(&stop_interupt_mp);
      return(retcode);
}


/*
 * read data records from source medium and put in shared data area
 */
/*ARGSUSED*/
static void *
read_thread
#if NeedFunctionPrototype
(void *data)
#else
(data)
void *data;
#endif
{
    data_buf_t *dptr = &sdat_ptr->read_buf[0];
    char command[2*FORMAT_NAME_LEN+12];
    int buffer_pos = 0;
    int records;
    int bytes;
    timestruc_t to;
    to.tv_nsec = 0;
    
    if (DBX_val >= 5)
	fprintf(stderr,"server: Starting read thread\n");
    
    /* skip requested number of records */
    for (records = 1 ;records <= skip_records; records++) {
	bytes = (*get_data_func)(MAX_DATA_SIZE, (char *) dptr->data);
	if (bytes <= 0) {
	    /* exit read thread */
	    if (bytes == 0 && DBX_val < 1)
		fprintf(stderr, "read on data source returned %d (eof = 0) "
			"at record %d\n", bytes, records);
	    thr_exit(NULL);
	}
    }

#ifdef USE_DATA_QUEUE
    empty_R_buffers();
#endif
      
    /* set sorting state to ON */
    (void) mutex_lock(&sdat_ptr->mp_sorting);
    sdat_ptr->sorting = SORTING_ON;
    (void) mutex_unlock(&sdat_ptr->mp_sorting);
      
    /* tell sort process to start sorting */
    if (sort_io_data->IO_state & WRITING_OUT) 
	(void) sprintf(command,"sortf %s %s",
		       sort_io_data->decoder, sort_io_data->decoder);
    else 
	(void) sprintf(command,"sort %s %s",
		       sort_io_data->decoder, sort_io_data->decoder);
    write_to_sortproc(command);
    
    /* 
     * sort requested number of records
     * note: a continue call before the very bottom of the loop does not
     * advance the buffer position.
     */
#ifdef USE_DATA_QUEUE
    for(records = 1; records <= sort_records && !Stop_interupt(0); )
    {
	to.tv_sec = time(NULL) + RTHREAD_WAIT;
	if ((buffer_pos = take_from_queue(&sdat_ptr->read_empty, &to)) == -1)
	    continue;

	dptr = sdat_ptr->read_buf + buffer_pos;
	bytes = (*get_data_func)(MAX_DATA_SIZE, (char *) dptr->data);
	if (bytes <= 0) {
	    fprintf(stderr, "read on data source returned %d (eof = 0) at "
		    "record %d\n", bytes, records+skip_records);
	    dptr->data_size = 0;
	    add_to_queue(&sdat_ptr->read_full, buffer_pos);
	    break;
	}
	dptr->data_size = bytes;
	records++;
	add_to_queue(&sdat_ptr->read_full, buffer_pos);
    }

#else
    for (records = 1 ; records <= sort_records && !Stop_interupt(0);
	 dptr = &sdat_ptr->read_buf[buffer_pos])
    {

	/* check state of next buffer in shared data storage */	    
	(void) mutex_lock(&dptr->mp);
	if (dptr->state == EMPTY_BUF) {
	    dptr->state = FILLING_BUF;
	}
	else {
	    to.tv_sec = time(NULL) + RTHREAD_WAIT;
	    (void) cond_timedwait(&dptr->cv, &dptr->mp, &to);
	    (void) mutex_unlock(&dptr->mp);
	    continue;  
	}
	(void) mutex_unlock(&dptr->mp);

	/* copy block into shared memory buffer */
	bytes = (*get_data_func)(MAX_DATA_SIZE, (char *) dptr->data);
	if (bytes <= 0) {
	    fprintf(stderr, "read on data source returned %d (eof = 0) at "
		    "record %d\n", bytes, records+skip_records);
	    break;
	}
	dptr->data_size = bytes;
	records++;
	    
	/* update state of next buffer in shared data storage */	  
	(void) mutex_lock(&dptr->mp);
	if (dptr->state == FILLING_BUF) {
	    dptr->state = FULL_BUF;
	    (void) cond_signal(&dptr->cv);
	}
	else {
	    dptr->state = END_OF_DATA_BUF;
	    (void) mutex_unlock(&dptr->mp);
	    (void) Stop_interupt(SORT_SET,SORT_EXIT);
	    fprintf(stderr, "server: data buffer in wrong state, junking "
		    "record %d\n", records+skip_records);
	    break;
	}
	(void) mutex_unlock(&dptr->mp);
	
	buffer_pos = (buffer_pos+1) % NOS_OF_RBUFS;
    }
#endif

    /* exit prematurely from thread if requested to do so */     
    if (Stop_interupt(0) == SORT_EXIT)
    {
	fprintf(stderr,
		"server: Emergency stop of read thread at after %d records.\n",
		records-1);
	thr_exit(NULL);
    }

#ifdef USE_DATA_QUEUE
    if (DBX_val >= 5)
	fprintf(stderr,"server: read thread EOD buffer %d after %d "
		"records\n",buffer_pos,records-1);
      
    for(;;)
    {
	mutex_lock(&sdat_ptr->mp_sorting);
	if (sdat_ptr->sorting  == SORTING_ON && Stop_interupt(0) != SORT_EXIT)
	{
	    mutex_unlock(&sdat_ptr->mp_sorting);
	    to.tv_sec = time(NULL) + RTHREAD_WAIT;
	    if ((buffer_pos = take_from_queue(&sdat_ptr->read_empty, &to))
		!= -1)
	    {
		sdat_ptr->read_buf[buffer_pos].data_size = 0;
		add_to_queue(&sdat_ptr->read_full, buffer_pos);
	    }
	}
	else
	{
	    mutex_unlock(&sdat_ptr->mp_sorting);
	    break;
	}
    }
#else
    /* set current buffer to end of data state */
    (void) mutex_lock(&dptr->mp);
    while (dptr->state & (FULL_BUF|EMPTYING_BUF)) {
	to.tv_sec = time(NULL) + RTHREAD_WAIT;
	(void) cond_timedwait(&dptr->cv, &dptr->mp, &to);
	if (Stop_interupt(0) == SORT_EXIT) 
	    break;
    }
    dptr->state = END_OF_DATA_BUF;
    (void) cond_signal(&dptr->cv);
    (void) mutex_unlock(&dptr->mp);
    
    if (DBX_val >= 5)
	fprintf(stderr,"server: read thread EOD buffer %d after %d "
		"records\n",buffer_pos,records-1);
      
    /* wait until sorting process has finished */
    (void) mutex_lock(&sdat_ptr->mp_sorting);
    while (sdat_ptr->sorting  == SORTING_ON) {
	to.tv_sec = time(NULL) + RTHREAD_WAIT;
	(void) cond_timedwait(&sdat_ptr->cv_sorting, &sdat_ptr->mp_sorting,
			      &to);
	if (Stop_interupt(0) == SORT_EXIT) 
	    break;
    }
#endif
    sdat_ptr->sorting = SORTING_OFF;
    (void) mutex_unlock(&sdat_ptr->mp_sorting);
    
    if (DBX_val >= 10)
	(void) diagnostic_R_buffers();
    
    /* exit read thread now */
    (void) empty_R_buffers();
    if (DBX_val >= 5) 
	fprintf(stderr,"server: Ending read thread\n");
    
    thr_exit(NULL);
    return(NULL);
}

/*
 * read filtered data records from shared data source and write them to
 * storage medium
 */
/*ARGSUSED*/
static void *
write_thread
#if NeedFunctionPrototype
(void *data)
#else
(data)
void *data;
#endif
{
    data_buf_t *dptr;
    int records, b;
    int bytes;
    timestruc_t to;
    to.tv_nsec = 0;
#ifndef USE_DATA_QUEUE    
    int buffer_pos = 0;
#endif

    if (DBX_val >= 5)
	fprintf(stderr,"server: Starting write thread\n");

#ifdef USE_DATA_QUEUE
    empty_W_buffers();
#endif
      
    for(records=0; Stop_interupt(0) != SORT_EXIT; )
    {
#ifdef USE_DATA_QUEUE
	to.tv_sec = time(NULL) + WTHREAD_WAIT;
	if ((b = take_from_queue(&sdat_ptr->write_full, &to)) == -1)
	    continue;
	dptr = sdat_ptr->write_buf + b;

	if (dptr->data_size == 0)
	    break;

	bytes = (*put_data_func)(dptr->data_size, (char *) dptr->data);
	if (bytes <= 0) {
	    fprintf(stderr, "server: ERROR write returned %d at record %d\n",
		    bytes, records);
	    dptr->data_size = 0;
	    add_to_queue(&sdat_ptr->write_empty, b);
	    break;
	}
	sdat_ptr->out_records++;
	records++;
	dptr->data_size = MAX_DATA_SIZE;
	add_to_queue(&sdat_ptr->write_empty, b);
#else    
	dptr = &sdat_ptr->write_buf[buffer_pos];
	
/* check state of next buffer in shared data storage */	    
	(void) mutex_lock(&dptr->mp);
	if (dptr->state == FULL_BUF) {
	    dptr->state = EMPTYING_BUF;
	}
	else if (dptr->state == END_OF_DATA_BUF) {
	    dptr->state = EMPTY_BUF;
	    (void) mutex_unlock(&dptr->mp);
	    break;
	}
	else {
	    to.tv_sec = time(NULL) + WTHREAD_WAIT;
	    (void) cond_timedwait(&dptr->cv, &dptr->mp, &to);
	    (void) mutex_unlock(&dptr->mp);
	    continue;
	} 
	(void) mutex_unlock(&dptr->mp);
	
/* copy block from shared memory buffer to output device */
	bytes = (*put_data_func)(dptr->data_size, (char *) dptr->data);
	if (bytes <= 0) {
	    fprintf(stderr,"server: ERROR write returned %d at record %d\n",bytes,records);
	    (void) mutex_lock(&dptr->mp);
	    dptr->state = BAD_BUFFER;
	    (void) mutex_unlock(&dptr->mp);
	    break;
	}
	sdat_ptr->out_records++;
	records++;
	
/* update state of next buffer in shared data storage */	  
	(void) mutex_lock(&dptr->mp);
	if (dptr->state == EMPTYING_BUF) {
	    dptr->state = EMPTY_BUF;
	    if (cond_signal(&dptr->cv)) {
		perror("conditional variable signal");
	    }
	}
	else {
	    fprintf(stderr,"server: data buffer in wrong state, record %d\n",records);
	    (void) mutex_unlock(&dptr->mp);
	    continue;
	}
	(void) mutex_unlock(&dptr->mp);
	
	buffer_pos = (buffer_pos+1) % NOS_OF_WBUFS;
#endif
    }
    
/* exit write thread now */      
    if (DBX_val >= 5)
	fprintf(stderr,"server: Ending write thread after %d records written out\n",records);
    thr_exit(NULL);
    return(NULL);
}

/*
 *  tidy up after sort thread has decided to terminate
 */
static void
sort_thr_exit(
#if NeedFunctionPrototype
		  int retcode)
#else
      retcode)
      int retcode;
#endif
{
      if (DBX_val >= 1)
	    fprintf(stderr,"server: sort thread exited with return code %d\n",retcode);

/* if gui is alive signal that sorting has stopped */
      if (nogui != SORT_TRUE && gui_pid > 0)
	    kill(gui_pid,SIGINT); 

/* signal end of sorting */      
      if (Sorting_data(SORT_SET,SORT_FALSE) != SORT_FALSE) {
	    fprintf(stderr,"server: error while ending sorting\n");
      }

      thr_exit(NULL);
}

/*
 *   thread controlling read and write threads
 */
void *
sort_thread(
#if NeedFunctionPrototype
		  void *data)
#else
      data)
      void *data;
#endif
{
      sort_IO_t   *io_ptr = (sort_IO_t *) data;
      thread_t    rthread, wthread; 
      sigset_t maskset;
      int filter; 

      if (DBX_val >= 5)
	    fprintf(stderr,"server: Starting sort thread\n");

/* set up signal mask to ignore most signals */
      if (sigfillset(&maskset) == -1) {
	    perror("sort thread sigfillset");
	    sort_thr_exit(1);
      }
      if (sigdelset(&maskset, SIGSEGV) == -1 || sigdelset(&maskset, SIGBUS) == -1) {
	    perror("sort thread sigdelset");
	    sort_thr_exit(1);
      }   
      if (thr_sigsetmask(SIG_BLOCK, &maskset, NULL)) {
	    perror("sort thread sigsetmask");
	    sort_thr_exit(2);
      }
      
/* initialize read state IO data */   
      if (io_ptr->IO_state & READING_DISK) {
	    get_data_func = ddiofunc(0);
	    fprintf(stderr,"server: reading from disk ...\n");
      }
      else if (io_ptr->IO_state & READING_TAPE) {
	    get_data_func =  dtiofunc(0);
	    fprintf(stderr,"server: reading from tape ...\n");
      }
      else if (io_ptr->IO_state & READING_REMOTE) {
	    get_data_func = remoteget;
	    fprintf(stderr,"server: reading from remote ethernet source ...\n");
      }
      else if (io_ptr->IO_state & READING_DEMON) {
	    get_data_func = demonread;
	    fprintf(stderr,"server: reading from demon data feed ...\n");
      }
      else if (io_ptr->IO_state & READING_PIPE) {
	    get_data_func = piperead;
	    fprintf(stderr,"server: reading from pipe ...\n");
      }
      else {
	    fprintf(stderr,"server: The function for reading data has not been internally specified !\n");
	    fprintf(stderr,"server: This shouldn't happen ... help\n");
	    sort_thr_exit(3);
      }
      
/* initialize write state IO data */
      if ((filter = io_ptr->IO_state & WRITING_OUT) != 0) {
	    if (filter & WRITING_DISK) {
		  put_data_func = ddiofunc(1);
	    }
	    else if (filter & WRITING_TAPE) {
		  put_data_func = dtiofunc(1);
	    }
	    else {
		  fprintf(stderr,"server: The function for writing data has not been specified !\n");
		  fprintf(stderr,"server: This shouldn't happen ... help\n");
		  sort_thr_exit(4);
	    }
      }

/* and sort */
      if (filter) {
	    /* start write thread if filtering data */
	    if (thr_create(NULL, 0, write_thread, data, 0, &wthread)) {
		  (void) Sorting_data(SORT_SET, SORT_FALSE);
		  fprintf(stderr,"server: error while trying to start write thread\n");
		  sort_thr_exit(-5);
	    }
      }
      
      if (thr_create(NULL, 0, read_thread, data, 0, &rthread)) {
	    (void) Sorting_data(SORT_SET, SORT_FALSE);
	    fprintf(stderr,"server: error while trying to start read thread\n");
	    if (filter)
		  (void) thr_kill(wthread, SIGKILL);
	    sort_thr_exit(-6);
      }

/* wait for read (and write) thread(s) to exit */
      if (thr_join(rthread, NULL, NULL)) {
	    perror("sort rthread thr_join");
      }
      if (filter) {
	    if (thr_join(wthread, NULL, NULL)) {
		  perror("sort wthread thr_join");
	    }
      }

      sort_thr_exit(0);
      return(NULL);
}

/*
 *  try and stop read and write threads thereby causing sort thread to exit
 *  reset shared data buffers to reduce the chances of foul ups
 */
void 
stop_sort_threads(
#if NeedFunctionPrototype
		  void )
#else
      )
#endif
{
      (void) Stop_interupt(SORT_SET,SORT_EXIT);
/* wait for sort thread to signal exit */      
      while(Sorting_data(0) == SORT_FALSE)
	    thr_yield();
      if (reset_all_buffers() == -1)
	    fprintf(stderr,"server:  reset buffers in kill threads failed\n");
      return;
}

/*
 *  start a background process and wait for it to end
 *  void *data = "argv[0] argv[1] ...."
 */
void *
background_thread(
#if NeedFunctionPrototype
		  void *data)
#else
      data)
      void *data;
#endif
{
      sigset_t maskset;
      char     *my_argv[100];
      char     *token, *lasts;
      int      statusp;
      int      i;

      if (DBX_val >= 5)
	    fprintf(stderr,"server: Starting background thread\n=>%s\n",(char *)data);

/* set up signal mask to ignore most signals */
      if (sigfillset(&maskset) == -1) {
	    perror("background thread sigfillset");
	    free(data);
	    thr_exit(NULL);
      }
      if (sigdelset(&maskset, SIGSEGV) == -1 || sigdelset(&maskset, SIGBUS) == -1) {
	    perror("background thread sigdelset");
	    free(data);
	    thr_exit(NULL);
      }   
      if (thr_sigsetmask(SIG_BLOCK, &maskset, NULL)) {
	    perror("background thread sigsetmask");
	    free(data);
	    thr_exit(NULL);
      }

/* setup argument vector */
      token = my_argv[0] = strtok_r(data, " ", &lasts);
      for(i=1; token != NULL || i<99; i++) {
	    token = my_argv[i] = strtok_r(NULL, " ", &lasts);
      }
      my_argv[i] = (char *)0;
      
/* start background job  */      
      if ( (i = do_process(my_argv[0], my_argv)) == -1) {
	    fprintf(stderr,"server: failed to start process within background thread\n");
	    free(data);
	    thr_exit(NULL);	    
      }
      (void) waitpid(i, &statusp, 0);
      free(data);

      if (DBX_val >= 5)
	    fprintf(stderr,"server: background thread exiting\n");
      
      thr_exit(NULL);
      return(NULL); /* make lint happier */
}


